structure Execute :> EXECUTE =
struct
  structure A = Absyn
  structure R = Response

  structure M = CML

  exception CloseConnection
  exception doAuth of Sasl.conn * string * string * string option 
                                * R.response CML.chan

  structure ContextMap = 
    ListMapFn(struct type ord_key = string
		     val compare = String.compare
	      end)

  (* internal error *)
  type conn = {user : Auth.ident option, 
	       comm : R.response CML.chan,
	       contexts : (bool * bool * Context.context) ContextMap.map ref,
	       saslconn : Sasl.conn}

  fun start comm sasl = ({user=NONE, comm=comm, contexts=ref ContextMap.empty,
			  saslconn=sasl} : conn)

  fun authAs ({user, comm, contexts, saslconn} : conn) u =
    (TextIO.print ("authenticating as " ^ (Auth.toString u) ^ "...\n");
     {user=SOME u, comm=comm, 
      contexts=contexts, saslconn=saslconn} : conn)

  exception BadSearchCriteria
  exception BadSearchModifier
  exception BadAttributePattern
  exception BadAttributeMetadata
  exception NoSuchContext
  exception BadEntryPath
  exception BadStoreModifier
  exception BadStoreMetadata

  exception AbortSearch

  exception InternalError of string

  fun expandPath user d =
    let
      val path = 
	case String.fields (fn c => c = #"/") d of
	  nil => raise BadEntryPath
	| (""::path) => ((if (List.last path) <> "" then raise BadEntryPath
			  else path) handle _ => raise BadEntryPath)
	| _ => raise BadEntryPath

      fun doSub s = if s = "~" then "/" ^ (Auth.toString user) else ("/" ^ s)
    in
      (String.concat (List.map doSub path))
    end

  fun deleteContext contexts c =
    (let
      val (others, (_, _, ctxt)) = ContextMap.remove (!contexts, c)
     in
       Context.destroy ctxt;
       contexts := others
     end handle LibBase.NotFound => raise NoSuchContext)

  fun compileDSearch ident spec =
    let
      val myfetch = Entry.fetch ident

      fun compileDSearch' A.All = (fn e => true)
	| compileDSearch' (A.Equal (attr, (A.Comparator comp), v)) =
	let
	  val comp = Comparator.lookup Comparator.Equal comp
	  fun f e = (case myfetch e attr of
		       {value,...} => (comp(v, value) = EQUAL))
	in f end
	| compileDSearch' (A.Compare (attr, (A.Comparator comp), v)) =
	let
	  val comp = Comparator.lookup Comparator.Order comp
	  fun f e = (case myfetch e attr of
		       {value,...} => (comp(v, value) <> GREATER))
	in f end
	| compileDSearch' (A.Prefix (attr, (A.Comparator comp), v)) =
	let
	  val comp = Comparator.lookup Comparator.Prefix comp
	  fun f e = (case myfetch e attr of
		       {value,...} => (comp(v, value) = EQUAL))
	in f end
	| compileDSearch' (A.Range _) = raise BadSearchCriteria
	| compileDSearch' (A.Strict (attr, (A.Comparator comp), v)) =
	let
	  val comp = Comparator.lookup Comparator.Order comp
	  fun f e = (case myfetch e attr of
		       {value,...} => (comp(v, value) = LESS))
	in f end
	| compileDSearch' (A.Substr (attr, (A.Comparator comp), v)) =
	let
	  val comp = Comparator.lookup Comparator.Substring comp
	  fun f e = (case myfetch e attr of
		       {value,...} => (comp(v, value) = EQUAL))
	in f end
	| compileDSearch' (A.Not crit) =
	let
	  val f = compileDSearch' crit
	in not o f end
	| compileDSearch' (A.Or (crit1, crit2)) =
	let
	  val f = compileDSearch' crit1
	  val g = compileDSearch' crit2
	in
	  (fn e => f e orelse g e)
	end
	| compileDSearch' (A.And (crit1, crit2)) =
	let
	  val f = compileDSearch' crit1
	  val g = compileDSearch' crit2
	in
	  (fn e => f e andalso g e)
	end
    in
      compileDSearch' spec
    end

  (* returns a search fun and a time that the search is good up to *)
  fun compileCSearch ident enum spec =
    let
      val myfetch = Entry.fetch ident

      fun compileCSearch' A.All = (fn e => true, AcapTime.future)
	| compileCSearch' (A.Equal (attr, (A.Comparator comp), v)) =
	let
	  val comp = Comparator.lookup Comparator.Equal comp
	  fun f (_,e) = (case myfetch e attr of
			   {value,...} => (comp(v, value) = EQUAL))
	in (f, AcapTime.future) end
	| compileCSearch' (A.Compare (attr, (A.Comparator comp), v)) =
	let
	  val comp = Comparator.lookup Comparator.Order comp
	  fun f (_,e) = (case myfetch e attr of
			   {value,...} => (comp(v, value) <> GREATER))
	in (f, AcapTime.future) end
	| compileCSearch' (A.Prefix (attr, (A.Comparator comp), v)) =
	let
	  val comp = Comparator.lookup Comparator.Prefix comp
	  fun f (_,e) = (case myfetch e attr of
			   {value,...} => (comp(v, value) = EQUAL))
	in (f, AcapTime.future) end
	| compileCSearch' (A.Range (s, e, t)) = 
	  if not enum then raise BadSearchCriteria
	  else let
		 fun f (i, _) = s <= i andalso i <= e
	       in
		 (f, AcapTime.fromString t)
	       end
	| compileCSearch' (A.Strict (attr, (A.Comparator comp), v)) =
	let
	  val comp = Comparator.lookup Comparator.Order comp
	  fun f (_,e) = (case myfetch e attr of
			   {value,...} => (comp(v, value) = LESS))
	in (f, AcapTime.future) end
	| compileCSearch' (A.Substr (attr, (A.Comparator comp), v)) =
	let
	  val comp = Comparator.lookup Comparator.Substring comp
	  fun f (_,e) = (case myfetch e attr of
			   {value,...} => (comp(v, value) = EQUAL))
	in (f, AcapTime.future) end
	| compileCSearch' (A.Not crit) =
	let
	  val (f, t) = compileCSearch' crit
	in (not o f, t) end
	| compileCSearch' (A.Or (crit1, crit2)) =
        let
	  val (f, t1) = compileCSearch' crit1
	  val (g, t2) = compileCSearch' crit2
	  val t = 
	    (case AcapTime.compare (t1, t2) of
	       GREATER => t2
	     | LESS => t1
	     | EQUAL => t1)
	in
	  ((fn e => f e orelse g e), t)
	end
	| compileCSearch' (A.And (crit1, crit2)) =
	let
	  val (f, t1) = compileCSearch' crit1
	  val (g, t2) = compileCSearch' crit2
	  val t = 
	    (case AcapTime.compare (t1, t2) of
	       GREATER => t2
	     | LESS => t1
	     | EQUAL => t1)
	in
	  ((fn e => f e andalso g e), t)
	end
    in
      compileCSearch' spec
    end

  fun tripletOrder ((_, a, _), (_, b, _)) = String.compare (a, b)
  fun doubleOrder ((a, _), (b, _)) = String.compare (a, b)

  fun compileSort ident (getEntry, baseComp) sortlist =
    let
      val myfetch = (Entry.fetch ident) o getEntry

      fun compileSort' nil = baseComp
	| compileSort' (A.SortItem(attr, A.Comparator comp)::l) =
	let
	  val comp = Comparator.lookup Comparator.Order comp
	  val f = compileSort' l
	  fun g (a, b) = 
	    case comp (#value (myfetch a attr), 
		       #value (myfetch b attr)) of
	      LESS => LESS
	    | EQUAL => f (a, b)
	    | GREATER => GREATER
	in
	  g
	end
    in
      compileSort' sortlist
    end
      
  (* returns a function to apply to an entry;
   that function returns a string suitable for appending to an ENTRY
   list *)
  fun compileMetaData ident nil = (fn e => "")
    | compileMetaData ident metadata_list = 
      let
	val myfetch = Entry.fetch ident
	val mysearch = Entry.search ident
	
	(* we have to figure out what attributes to pull; we use
	 search for patterns, fetch otherwise *)
	datatype tomatch = pat of string | lit of string

	(* should probably verify this is an ok attribute o/w *)
	fun isPat nil s = lit (String.implode (List.rev s))
	  | isPat [#"*"] s = pat (String.implode (List.rev s))
	  | isPat [c] s = lit (String.implode (List.rev (c::s)))
	  | isPat (#"*"::s) s2 = raise BadAttributePattern
	  | isPat (c::s) s2 = isPat s (c::s2)
	  
        (* construct a function to extract the appropriate metadata out *)
	fun getmeta nil = (fn _ => nil)
	  | getmeta ("attribute"::l) =
	  let val rest = getmeta l in
	    (fn (md as {attribute,...}:Entry.attribute) =>
	     ("\"" ^ attribute ^ "\"")::(rest md))
	  end
	  | getmeta ("value"::l) =
	  let val rest = getmeta l in
	    (fn (md as {value,...}:Entry.attribute) =>
	     (Value.toString value)::(rest md))
	  end
	  | getmeta ("acl"::l) =
	  let val rest = getmeta l in
	    (fn (md as {acl,...}:Entry.attribute) =>
	     (Acl.toString acl)::(rest md))
	  end
	  | getmeta ("myrights"::l) =
	  let val rest = getmeta l in
	    (fn (md as {acl,...}:Entry.attribute) => 
	     (AclRights.toString (Acl.myrights ident acl))::(rest md))
	  end
	  | getmeta ("size"::l) = 
	  let val rest = getmeta l in
	    (fn (md as {size,...}:Entry.attribute) =>
	     (map Int.toString size) @ (rest md))
	  end
	  | getmeta (_::l) = raise BadAttributeMetadata
	  
	(* we must have at least one metadata item ! *)
	(* formatMeta : string list -> string *)
	fun formatMeta nil = raise (InternalError "need one metadata item")
	  | formatMeta [s] = s
	  | formatMeta (s::l) = "(" ^ s ^ 
	        (List.foldr (fn (s, acc) => (" " ^ s ^ acc)) ")" l)
	  
        (* this function returns a list of functions we apply to the entry.
	 we concat them all together to get the return-data-list.

	 this would be more efficient if we could do one fetch for all
	 the static data items; right now that's too hard and i punt *)
	fun fetchit nil fl = List.rev fl
	  | fetchit (A.MetaData(a, md)::l) fl = 
	  (case isPat (String.explode a) nil of
	     (pat s) => let 
	                  (* what attributes to print out *)
			  val match = String.isPrefix s 
			    
			  (* what metadata do we want from these attributes? *)
			  val print = if List.null md then 
			                ["attribute", "value"]
				      else md
					
			  (* val sl : attribute -> string *)
			  val sl = formatMeta o (getmeta print)

                          (* Entry.search e match will give us a list of
			   attributes. *)

		    (* we map sl over it to get a list of return-metalist,
		     which we then put together to form a return-attr-list,
		     which is part of the return-data-list *)
			  val thisFn = 
			  (fn e => case List.map sl (mysearch e match) of
			             nil => "()"
				   | [a] => "(" ^ a ^ ")"
				   | (a::l) => "(" ^ a ^ 
		       (List.foldr (fn (s,acc) => (" " ^ s ^ acc)) "" l) ^ ")")
			in fetchit l (thisFn::fl)
			end
	   | (lit s) => let
			(* hopefully easier than above; requested a single
			 attribute *)

			  (* default metadata *)
			  val print = if List.null md then ["value"]
				      else md

			  (* val sl : attribute -> string *)
			  val sl = formatMeta o (getmeta print) 

                          val thisFn = 
			    (fn e => sl (myfetch e s))
			in fetchit l (thisFn::fl)
			end)

	(* ok, for those of you not paying attention, we now apply
	 fetchit to our metadata list to give us a list of functions
	 we must apply to the entry; when an entry is returned, we apply
	 those functions and smoosh it all together to form a
	 return-data-list, the last parameter to R.ENTRY *)
	val fl = fetchit metadata_list nil

	fun makeString e =
	  let
	    val retdatalist = map (fn f => f e) fl
	    val retdatalist = 
	      case retdatalist of
		nil => ""
	      | [s] => s
	      | (s::l) => s ^ (List.foldr (fn (s,acc) => (" " ^ s ^ acc)) "" l)
	  in
	    retdatalist
	  end
      in
	makeString
      end

  (* this is how we return stuff back *)
  fun donePass (tag, comm) =
    let
      val retmodtime = R.TAGMODTIME(tag,
				    (AcapTime.toString o AcapTime.now) ())
	
      fun donePass (Throttle.OK) = 
	(M.send (comm, retmodtime);
	 M.send (comm, R.OK(SOME tag, nil, "search completed")))
    in
      donePass
    end

  fun doneSoft (tag, comm) =
    let
      val retmodtime = R.TAGMODTIME(tag,
				    (AcapTime.toString o AcapTime.now) ())
	
      fun doneSoft (Throttle.S_OK) = 
	(M.send (comm, retmodtime);
	 M.send (comm, R.OK(SOME tag, nil, "search completed")))
	| doneSoft (Throttle.S_TOOMANY i) =
	(M.send (comm, retmodtime);
	 M.send (comm, R.OK(SOME tag, [R.TOOMANY i], "search completed")))
    in
      doneSoft
    end


  fun searchdset (conn as {comm,contexts,user=(SOME user),saslconn=saslconn},
		  tag, dsetpath, sm, sc) =
    (let
       val dsetpath = expandPath user dsetpath

       (* extract options *)
       fun get nil (opts as (depth, context, hard, soft, ret, sort, doinh)) = 
	                   opts
	 | get ((A.Depth x)::l) (NONE,c,h,s,r,t,i) = 
			   get l (SOME x, c,h,s,r,t,i)
	 | get ((A.HardLimit x)::l) (d,c,NONE,s,r,t,i) = 
			   get l (d,c,SOME x,s,r,t,i)
	 | get ((A.Limit x)::l) (d,c,h,NONE,r,t,i) = 
			   get l (d,c,h,SOME x,r,t,i)
	 | get ((A.MakeContext x)::l) (d,NONE,h,s,r,t,i) = 
			   get l (d,SOME x,h,s,r,t,i)
	 | get ((A.Return x)::l) (d,c,h,s,NONE,t,i) = 
			   get l (d,c,h,s,SOME x,t,i)
	 | get ((A.Sort x)::l) (d,c,h,s,r,NONE,i) = 
			   get l (d,c,h,s,r,SOME x,i)
	 | get ((A.Noinherit)::l) (d,c,h,s,r,t,NONE) =
			   get l (d,c,h,s,r,t, SOME false)
						     
	 | get _ _ = raise BadSearchModifier

       val (depth, context, hard, soft, ret, sortCrit, doinh) = 
	     get sm (NONE, NONE, NONE, NONE, NONE, NONE, NONE)

       val doinh = getOpt (doinh, true)

       (* depth defaults to 1 *)
       val showWholePath = isSome depth
       val depth = getOpt (depth, 1)

       (* compile the search criteria *)
       val search = compileDSearch user sc

       (* compile sort criteria *)
       val sort = Option.map (compileSort user (fn (_,_,x)=>x,
						tripletOrder)) sortCrit

       (* compile the return attributes; yields a function that 
	sets up the string to send out *)
       val metaret = Option.map (compileMetaData user) ret

       val ret =
	 let
	   val metaret = case metaret of
	     NONE => (fn _ => "")
	   | (SOME metaret) => metaret
	 in
	   if showWholePath then (* report the whole path *)
	     (fn (dsetname, n, e) =>
	      M.send (comm, R.ENTRY(tag, dsetname ^ n, metaret e)))
	   else (* report just the shortname *)
	     (fn (_, n, e) => M.send (comm, R.ENTRY(tag, n, metaret e)))
	 end

       (* compile the soft limit *)
       val softThrottle = 
	 case soft of
	   NONE => Throttle.createpass ret (donePass (tag, comm))
	 | (SOME (a,b)) => Throttle.createsoft (a,b) ret 
	                                      (doneSoft (tag, comm))

       (* compile the sort throttle, if needed *)
       val sortThrottle =
	 case sort of
	   NONE => softThrottle (* skip it *)
	 | (SOME f) => 
	     Throttle.createsort f (Throttle.call softThrottle) 
	                           (fn OK => Throttle.done softThrottle)

       (* compile the context *)
       val context = 
	 case context of
	   NONE => NONE (* just pass it through *)
	 | SOME(enum, false, name) => (* no notifications *)
	     let
	       val _ = (deleteContext contexts name)
		               handle NoSuchContext => ()

	       val sort = Option.map (compileSort user (fn (_,x)=>x,
							doubleOrder)) sortCrit
	       val c = Context.create search (sort, NONE, enum)
	     in
	       SOME (name, enum, false, c)
	     end
	 | SOME(enum, true, name) => (* notifications *)
	     let
	       val _ = (deleteContext contexts name)
			       handle NoSuchContext => ()

	       (* create output channel *)
	       val outchan = CML.channel ()

	       val metaret = 
		 case metaret of
		   NONE => (fn e => "")
		 | (SOME m) => m

	       (* function to convert notifications into output *)
	       fun conv (Context.CNEW (n, e, i)) =
		 M.send (comm, R.ADDTO(name, n, i, metaret e))
		 | conv (Context.CCHANGE (n, e, (i, j))) =
		 M.send (comm, R.CHANGE(name, n, i, j, metaret e))
		 | conv (Context.CDELETE (n, e, i)) =
		 M.send (comm, R.REMOVEFROM(name, n, i))
		 | conv (Context.CTIME time) =
		 M.send (comm, R.MODTIME(name, AcapTime.toString time))
		 | conv (Context.CDIE) = CML.exit ()
		 
	       (* and a process to convert between the two *)
	       fun lconv ch = (conv (CML.recv ch); lconv ch)
		 
	       val _ = CML.spawnc lconv outchan
		 
	       val sort = Option.map (compileSort user (fn (_,x)=>x,
							doubleOrder)) sortCrit
	       val c = Context.create search (sort, SOME outchan, enum)
	     in
	       SOME (name, enum, true, c)
	     end

       val contextThrottle =
	 case context of
	   NONE => sortThrottle (* skip it *)
	 | (SOME (_, _, _, cxt)) => Throttle.createpass 
	     (fn (E as (d,n,e)) => (Context.fastInsert cxt (d^n,e);
	               Throttle.call sortThrottle E)) 
	     (fn Throttle.OK => (Context.doneInitial cxt;
				 Throttle.done sortThrottle))

       fun doneHard f (Throttle.H_OK) = f () (* call the done on the next fn *)
	 | doneHard f (Throttle.H_WAYTOOMANY) =
	 let
	   (* ok, this should kill the search in progress, report failure,
	    and kill the context we're building, if any *)

	   (* send NO *)
	   val _ = M.send (comm, R.NO(SOME tag, [R.WAYTOOMANY],
			              "hard limit exceeded"))

	   (* remove context *)
	   val _ = (case context of SOME (_,_,_,c) => (Context.destroy c)
                           	  | NONE => ())
	 in
	   (* finally, tell the search to stop *)
	   raise AbortSearch
	 end

       (* compile the hard throttle *)
       val hardThrottle = 
	 (case hard of
	    NONE => contextThrottle (* skip it *)
	  | (SOME x) => Throttle.createhard x (Throttle.call contextThrottle) 
	                  (doneHard (fn () => Throttle.done contextThrottle)))

       val contextHook = 
	 case context of
	   (SOME (_, _, true, c)) => (* notification context *)
	     SOME (fn (path, port) => Context.addDSet c port)
	 | _ => NONE

       fun sendrefers (dset, urls) =
	 M.send (comm, R.REFER (tag, dset, urls))

       val return = Throttle.call hardThrottle
       fun doneOut () = Throttle.done hardThrottle
       (* ok, we now can do the search ! *)
     in
       TextIO.print "starting search...\n";

       (* would be nice to spawn a thread here, but then we can't catch
	AbortSearch below; we should think about this *)
       Dataset.search user (dsetpath, depth, doinh) 
                           search 
                           contextHook 
                           (return, sendrefers)
			   doneOut;
       case context of
         NONE => ()
       | (SOME (n, enum, notify, ctxt)) => 
	   contexts := ContextMap.insert(!contexts, n, (enum, notify, ctxt));
       {comm=comm, contexts=contexts, user=SOME user,
	saslconn=saslconn} before
       TextIO.print "done search\n"
     end
       handle (Dataset.NoSuchDataset |
	       BadEntryPath) => (M.send (comm, R.NO(SOME tag, 
					[R.NOEXIST ("\"" ^ dsetpath ^ "\"")],
					"no such dataset"));
					conn)
	    | Dataset.DatasetRefer l => (M.send (comm,
					       R.NO(SOME tag,
			      [R.REFERCODE l], "see other server"));
				       conn)
	    | BadSearchCriteria => (M.send (comm, R.BAD(SOME tag,
					       	"bad search criteria")); conn)
	    | BadSearchModifier => (M.send (comm, R.BAD(SOME tag,
						"bad search modifier")); conn)
	    | Comparator.NoSuchComparator s => 
	                           (M.send (comm, R.BAD(SOME tag,
				       "no such comparator " ^ s)); conn)
	    | BadAttributePattern => (M.send (comm, R.BAD(SOME tag,
			               "bad attribute pattern")); conn) 
	    | BadAttributeMetadata => (M.send (comm, R.BAD(SOME tag,
				       "no such metadata")); conn)
	    | AbortSearch => conn
	    | LSyncVar.DeadLock => (M.send (comm, R.NO(SOME tag,
						       [R.TRYLATER],
						       "deadlock detected"));
				    conn))
    | searchdset _ = raise (InternalError "unauthed user calling searchdset")

  (* there's a large amount of duplication between here and above *)
  fun searchcontext(conn as {comm, contexts, user=(SOME user),...}:conn,
		    tag, cname, sm, sc) =
    (let
       (* find the context *)
       val (enum, notify, text) = 
	 (case ContextMap.find (!contexts, cname) of
	    NONE => raise NoSuchContext
	  | (SOME x) => x)

       (* extract options *)
       fun get nil (opts as (depth, context, hard, soft, ret, sort, doinh)) = 
	             opts
	 | get ((A.Depth x)::l) (NONE,c,h,s,r,t,i) = 
	             raise BadSearchModifier
	 | get ((A.HardLimit x)::l) (d,c,NONE,s,r,t,i) = 
		     get l (d,c,SOME x,s,r,t,i)
	 | get ((A.Limit x)::l) (d,c,h,NONE,r,t,i) = 
		     get l (d,c,h,SOME x,r,t,i)
	 | get ((A.MakeContext x)::l) (d,NONE,h,s,r,t,i) = 
	             raise BadSearchModifier (* XXX? *)
	 | get ((A.Return x)::l) (d,c,h,s,NONE,t,i) = 
		     get l (d,c,h,s,SOME x,t,i)
	 | get ((A.Sort x)::l) (d,c,h,s,r,NONE,i) = 
		     get l (d,c,h,s,r,SOME x,i)
	 | get ((A.Noinherit)::l) (d,c,h,s,r,t,NONE) =
		     get l (d,c,h,s,r,t,SOME false)
	 | get _ _ = raise BadSearchModifier

       val (_, _, hard, soft, ret, sort, doinh) = 
	 get sm (NONE, NONE, NONE, NONE, NONE, NONE, NONE)

       val doinh = getOpt (doinh, true)

       (* compile search criteria *)
       val (search, time) = compileCSearch user enum sc

       (* compile sort criteria *)
       val sort = Option.map (compileSort user (fn (_,x)=>x, doubleOrder)) sort

       (* compile the return attributes; yields a function that sends to
	the client *)
       val metaret = Option.map (compileMetaData user) ret

       val ret = 
	 let
	   val metaret = case metaret of
	     NONE => (fn _ => "")
	   | (SOME m) => m
	 in
	   (fn (n, e) => M.send (comm, R.ENTRY(tag, n, metaret e)))
	 end

       val retmodtime = R.TAGMODTIME(tag,
				     (AcapTime.toString o AcapTime.now) ())

       (* compile the soft limit *)
       val softThrottle = 
	 case soft of
	   NONE => Throttle.createpass ret (donePass (tag, comm))
	 | (SOME(a,b)) => Throttle.createsoft (a,b) ret (doneSoft (tag, comm))

       val sortThrottle =
	 (case sort of
	    NONE => softThrottle
	  | (SOME f)  => 
	      Throttle.createsort f (Throttle.call softThrottle)
	                          (fn OK => Throttle.done softThrottle))

       fun doneHard f (Throttle.H_OK) = f () (* call the done on the next fn *)
	 | doneHard f (Throttle.H_WAYTOOMANY) =
	 let
	   (* ok, this should kill the search in progress & report failure *)

	   (* send NO *)
	   val _ = M.send (comm, R.NO(SOME tag, [R.WAYTOOMANY] ,
				      "hard limit exceeded"))
	 in
	   (* finally, tell the search to stop *)
	   raise AbortSearch
	 end

       val hardThrottle =
	 (case hard of
	    NONE => sortThrottle (* skip it *)
	  | (SOME x) => Throttle.createhard x (Throttle.call sortThrottle)
	                  (doneHard (fn () => Throttle.done sortThrottle)))
	    
       fun cancelTime t =
	 M.send (comm, R.NO(SOME tag, [R.MODIFIED (AcapTime.toString t)],
			    "context has been modified"))
     in
       Context.search text (search, hardThrottle, time);
       conn
     end
       handle NoSuchContext => (M.send (comm, R.NO(SOME tag, nil,
						   "no such context"));
	                        conn)
	    | BadSearchCriteria => (M.send (comm, R.BAD(SOME tag,
					       	"bad search criteria")); conn)
	    | BadSearchModifier => (M.send (comm, R.BAD(SOME tag,
						"bad search modifier")); conn)
	    | Comparator.NoSuchComparator s =>
	                           (M.send (comm, R.BAD(SOME tag,
				       "no such comparator " ^ s)); conn) 
	    | AcapTime.BadTimeString => (M.send (comm, R.BAD(SOME tag,
				       "bad time string")); conn)
	    | BadAttributePattern => (M.send (comm, R.BAD(SOME tag,
			               "bad attribute pattern")); conn) 
	    | BadAttributeMetadata => (M.send (comm, R.BAD(SOME tag,
				       "no such metadata")); conn)
	    | (Context.Modified t) => 
              (M.send (comm, R.NO(SOME tag,[R.MODIFIED (AcapTime.toString t)],
				  "context has been modified")); conn)
	    | AbortSearch => conn)
    | searchcontext _ = 
       raise (InternalError "unauthed user calling searchcontext")

  (* stores are suppose to be atomic across the whole command.
   that sucks, so we're atomic across entries only *)
  fun storedata (conn as {comm,...} : conn) tag nil = 
      (M.send (comm, R.OK(SOME tag, nil, "store completed"));
       conn)
    | storedata (conn as {comm,user=SOME user,...}) tag 
      ((A.StoreEntry(path, sm, attr))::rest) =
    (let
       val (dsetpath, entry) = Substring.splitr (fn n => n <> #"/")
	                                        (Substring.all path)

       (* val dsetpath = Substring.string (Substring.trimr 1 dsetpath) *)
       val dsetpath = expandPath user (Substring.string dsetpath)

       val entry = Substring.string entry

       (* extract options *)
       fun get nil (opts as (nocreate, usince)) = opts
	 | get ((A.NoCreate)::l) (NONE, usince) = get l (SOME true, usince)
	 | get ((A.UnchangedSince t)::l) (n,NONE) = get l (n, SOME t)
	 | get _ _ = raise BadStoreModifier

       val (nocreate, unchangedsince) = get sm (NONE, NONE)

       val nocreate = getOpt (nocreate, false)
       val unchangedsince = Option.map AcapTime.fromString unchangedsince

       fun convattr nil (acl, value) = (acl, value)
	 | convattr (("acl",av)::l) (NONE,v) = 
	           convattr l (SOME (Acl.fromValue av), v)
	 | convattr (("value",av)::l) (a,NONE) = 
		   convattr l (a, SOME av)
	 | convattr _ _ = raise BadStoreMetadata

       fun makestore (A.AttrStoreValue(a, v)) = {acl=NONE, attribute=a,
					       value=(SOME v)}
	 | makestore (A.AttrStoreMeta(a, l)) =
	 let
	   val (acl, value) = convattr l (NONE, NONE)
	 in
	   {acl=acl, attribute=a, value=value} : Entry.storedata
	 end

       (* we pass the full name ! *)
       val _ = Dataset.store user (dsetpath, (not nocreate), unchangedsince)
                             (entry, (List.map makestore attr))
    in
      storedata conn tag rest;
      conn
    end
      handle (Entry.Modified t) => 
                (M.send (comm, R.NO(SOME tag,
				    [R.MODIFIED (AcapTime.toString t)], 
				    "entry modified"));
		 conn)
           | (Entry.EnforcedValidation (e,a)) => 
                (M.send (comm, R.NO(SOME tag,
				    [R.INVALID (e,a)], "enforced validation"));
		 conn)
	   | (Entry.RenameNotImpemented) => 
		(M.send (comm, R.NO(SOME tag, nil, "rename not implemented"));
		 conn)
           | (Dataset.SyntaxError msg) => (M.send (comm, R.BAD(SOME tag, msg));
					 conn)
	   | (Dataset.DatasetRefer l) => (M.send (comm, R.NO(SOME tag,
				  [R.REFERCODE l], "see other dataset"));
					  conn)
	   | (Entry.NotPermitted) => (M.send (comm, R.NO(SOME tag, 
				     [R.PERMISSION "\"\""], "no permission"));
					conn)
	   | Dataset.NoSuchDataset => (M.send (comm, R.NO(SOME tag, nil,
				     ("no such dataset")));
				       conn)
	   | BadEntryPath => (M.send (comm, R.BAD(SOME tag, "bad entry path"));
			      conn)
	   | BadStoreModifier => (M.send (comm, R.BAD(SOME tag,
						      "bad store modifier"));
				  conn)
	   | BadStoreMetadata => (M.send (comm, R.BAD(SOME tag,
						      "bad store metadata"));
				  conn)
	   | (Acl.SyntaxError s) => (M.send (comm, R.BAD(SOME tag, 
							"syntax error: " ^ s));
				     conn)
	   | (AcapTime.BadTimeString) => (M.send (comm, R.BAD(SOME tag,
							  "bad time string"));
					  conn)
	   | LSyncVar.DeadLock => (M.send (comm, R.NO(SOME tag,
						       [R.TRYLATER],
						       "deadlock detected"));
				    conn))
    | storedata _ _ _ = raise (InternalError "unauthed user calling storedata")

  fun run (conn as ({comm,...} : conn)) (A.Error (t, s)) =
    (M.send (comm, R.BAD(t, s));
     conn)
    | run (conn as ({comm,...})) (A.BareString _) =
    (M.send (comm, R.BAD(NONE, "bare string not expected"));
     conn)
    | run (conn as ({comm,...} : conn)) (A.Line(tag, A.Noop)) =
    (M.send (comm, R.OK(SOME tag, nil, "noop noop noop"));
     conn)
    | run (conn as {comm,...}) (A.Line(tag, A.Logout)) =
    (M.send (comm, R.BYE("have a nice day"));
     M.send (comm, R.OK(SOME tag, nil, "LOGOUT completed"));
     raise CloseConnection)
    | run (conn as {comm,...}) (A.Line(tag, A.Lang(_))) =
    (M.send (comm, R.BAD(SOME tag, "you're stuck with what you have"));
     conn)
    | run (conn as {comm,contexts,saslconn,user=NONE}) 
          (A.Line(tag, A.Authenticate (s,s2))) =
    (M.send (comm, R.OK(SOME tag, nil, "pre-authentication accepted"));
     authAs conn (Auth.fromString (getOpt(s2, s))))
    | run (conn as {comm,user=NONE,...}) (A.Line(tag, _)) =
    (M.send (comm, R.BAD(SOME tag, "authenticate first"));
     conn)
    | run (conn as {comm,...}) (A.Line(tag, A.Authenticate(_,_))) =
    (M.send (comm, R.BAD(SOME tag, "already authenticated"));
     conn)
    | run (conn as {comm,contexts,user,saslconn}) 
          (A.Line(tag, A.FreeContext(c))) =
    (let
       val _ = deleteContext contexts c
     in
       M.send (comm, R.OK(SOME tag, nil, "context nuked"));
       {comm=comm, contexts=contexts, user=user, saslconn=saslconn}
     end handle
       NoSuchContext => (M.send (comm, R.BAD(SOME tag, "no such context"));
			 conn))
    | run (conn as {comm,contexts,...}:conn) 
          (A.Line(tag, A.UpdateContext(cl))) =
    (let
       fun doupdate c = case ContextMap.find (!contexts, c) of 
	 (SOME (_, _, ctxt)) => Context.updatecontext ctxt
       | NONE => raise NoSuchContext
     in
       List.app doupdate cl;
       M.send (comm, R.OK(SOME tag, nil, "updating contexts"));
       conn
     end handle
       NoSuchContext => (M.send (comm, R.BAD(SOME tag, "no such context"));
			 conn))
    | run (conn as {comm,...} : conn) (A.Line(tag, A.Search(n, sm, sc))) =
      (* figure out if it's a context search or a dataset search *)
    if (String.size n <> 0) then 
	if (String.sub(n, 0) = #"/") then searchdset(conn, tag, n, sm, sc)
	else searchcontext(conn, tag, n, sm, sc)
    else (M.send (comm, R.BAD(SOME tag, "dataset or context required"));
	  conn)
    | run (conn) (A.Line(tag, A.Store(sel))) =
      storedata conn tag sel
    | run (conn as {user=SOME user,comm,...}) 
          (A.Line(tag, A.DeleteAcl(delobj, ident))) =
      (let
	 val (A.AclObj(dset, attr, ent)) = delobj
	 val aclobj = {dataset=expandPath user dset,
		       attribute=attr,
		       entry=ent}

	 val _ = Dataset.modifyacl user 
	   (aclobj, Dataset.DELACL(ident))
       in
	M.send (comm, R.OK(SOME tag, nil, "acl deleted"))
       end
	 handle Entry.NotPermitted =>
	   M.send (comm, R.NO(SOME tag, nil, "not authorized"))
	      | Dataset.NoSuchDataset => (* R.NOEXIST! *)
	   M.send (comm, R.BAD(SOME tag, "no such dataset"));
       conn)
    | run (conn as {user=SOME user,comm,...}) 
          (A.Line(tag, A.ListRights(aobj, auser))) =
      (let 
	 val (A.AclObj(dset, attr, ent)) = aobj
	 val aclobj = {dataset=expandPath user dset,
		       attribute=attr,
		       entry=ent}

	 val acl = Dataset.modifyacl user (aclobj, Dataset.GETACL)
	 val auser = Auth.fromString auser

	 val rights = if AclRights.has (Acl.myrights auser acl, 
					AclRights.ACLa) 
			then ["r", "w", "x", "a", "i"]
		      else nil
       in
	 M.send (comm, R.LISTRIGHTS(tag, "", rights));
	 M.send (comm, R.OK(SOME tag, nil, "Listrights completed"))
       end 
	 handle Entry.NotPermitted => 
	   M.send (comm, R.NO(SOME tag, nil, "not authorized"))
	      | Dataset.NoSuchDataset => (* R.NOEXIST! *)
	   M.send (comm, R.BAD(SOME tag, "no such dataset"));
	conn)
    | run (conn as {comm,user=SOME user,...}) 
          (A.Line(tag, A.MyRights(aobj))) =
      (let 
	 val (A.AclObj(dset, attr, ent)) = aobj
	 val aclobj = {dataset=expandPath user dset,
		       attribute=attr,
		       entry=ent}

	 val acl = Dataset.modifyacl user (aclobj, Dataset.GETACL)
	 val rights = Acl.myrights user acl
       in
	 M.send (comm, R.MYRIGHTS(tag, AclRights.toString rights));
	 M.send (comm, R.OK(SOME tag, nil, "Myrights completed"))
       end
	 handle Entry.NotPermitted =>
	   M.send (comm, R.NO(SOME tag, nil, "not authorized"))
	      | Dataset.NoSuchDataset => 
	   M.send (comm, R.BAD(SOME tag, "no such dataset"))
	      | (Dataset.SyntaxError s) => 
	   M.send (comm, R.BAD(SOME tag, "syntax error: " ^ s));
       conn)
    | run (conn as {comm,user=SOME user,...}) 
          (A.Line(tag, A.SetAcl(aobj, aid, arght))) =
      (let
	 val (A.AclObj(dset, attr, ent)) = aobj
	 val aclobj = {dataset=expandPath user dset,
		       attribute=attr,
		       entry=ent}

	 val rights = AclRights.fromString arght

	 val acl = Dataset.modifyacl user (aclobj, Dataset.SETACL(aid, rights))
       in
	 M.send (comm, R.OK(SOME tag, nil, "Setacl completed"))
       end
	 handle AclRights.InvalidRights =>
	   M.send (comm, R.BAD(SOME tag, "bad rights"))
	      | (Acl.SyntaxError s) => 
	   M.send (comm, R.BAD(SOME tag, "syntax error: " ^ s))
	      | Entry.NotPermitted => 
	   M.send (comm, R.NO(SOME tag, nil, "not authorized"))
	      | Dataset.NoSuchDataset => 
	   M.send (comm, R.BAD(SOME tag, "no such dataset"));
       conn)
    | run (conn as {comm,...}:conn) (A.Line(tag, _)) =
      (M.send (comm, R.NO(SOME tag, nil, "not implemented"));
       conn)

  (* should i attempt to kill all the contexts i'm holding? *)
  fun error (conn as ({comm,...} : conn)) msg =
    (M.send (comm, R.ALERT(msg));
     M.send (comm, R.BYE("serious internal error"));
     conn)

  fun stop (conn as ({comm,contexts,...} : conn)) =
    let
      fun destroy (_, _, ctxt) = Context.destroy ctxt
    in
      ContextMap.app destroy (!contexts)
    end

end
