structure AcapTime :> ACAPTIME =
struct
  (* think about using type acaptime = string ? *)
  type acaptime = Time.time * int

  exception BadTimeString

  open Date

  val old = (Time.zeroTime, 0 : int)

  val future = (Time.fromSeconds (valOf Int32.maxInt), 0 : int)

  val tick = Time.fromSeconds 15

  val reqCh = (CML.channel ()) : acaptime SyncVar.ivar CML.chan

  val round = Time.fromSeconds o Time.toSeconds
  val now = round o Time.now

  fun server (time as (cur, t), timeout) = 
    let
      val respCh = CML.channel ()

      val tevt = CML.wrap(CML.atTimeEvt timeout, 
			  fn () => let val new = now () 
				   in server ((new, 0), Time.+(new, tick)) end)

      val revt = CML.wrap(CML.recvEvt reqCh, 
			  fn (replyV) => (SyncVar.iPut (replyV, time);
					  server ((cur, t+1), timeout)))
    in
      CML.select [tevt, revt]
    end

  fun init () = 
    let
      val new = now ()
    in ignore (CML.spawnc server ((new, 0), Time.+(new, tick))) end

  fun now () = 
    let
      val replyV = SyncVar.iVar ()
    in
      CML.send (reqCh, replyV);
      SyncVar.iGet replyV
    end

  fun compare ((t1, c1), (t2, c2)) = 
    case Time.compare (t1, t2) of
      EQUAL => Int.compare (c1, c2)
    | other => other

  fun intToString p = (StringCvt.padLeft #"0" p) o Int.toString
  fun int32ToString p = (StringCvt.padLeft #"0" p) o Int32.toString

  fun monthToString Jan = "01"
    | monthToString Feb = "02"
    | monthToString Mar = "03"
    | monthToString Apr = "04"
    | monthToString May = "05"
    | monthToString Jun = "06"
    | monthToString Jul = "07"
    | monthToString Aug = "08"
    | monthToString Sep = "09"
    | monthToString Oct = "10"
    | monthToString Nov = "11"
    | monthToString Dec = "12"

  val iTS4 = intToString 4
  val iTS2 = intToString 2
  val i32TS6 = intToString 6

  fun toString (t, count) = (* sigh *)
    let
      val d = Date.fromTimeUniv t
    in
      (iTS4 (year d)) ^ 
      (monthToString (month d)) ^ 
      (iTS2 (day d)) ^
      (iTS2 (hour d)) ^
      (iTS2 (minute d)) ^
      (iTS2 (second d)) ^
      (i32TS6 count)
    end

  exception BadMonth

  fun fromString s =
    let
      fun splitString n s = (String.extract (s, 0, SOME n),
			     String.extract (s, n, NONE))
      
      fun convMonth "01" = Jan
	| convMonth "02" = Feb
	| convMonth "03" = Mar
	| convMonth "04" = Apr
	| convMonth "05" = May
	| convMonth "06" = Jun
	| convMonth "07" = Jul
	| convMonth "08" = Aug
	| convMonth "09" = Sep
	| convMonth "10" = Oct
	| convMonth "11" = Nov
	| convMonth "12" = Dec
	| convMonth _ = raise BadMonth

      val (year, s) = splitString 4 s
      val (month, s) = splitString 2 s
      val (day, s) = splitString 2 s
      val (hour, s) = splitString 2 s
      val (minute, s) = splitString 2 s
      val (second, smallt) = splitString 2 s

      val conv = (valOf o Int.fromString) 
      val year = conv year
      val month = convMonth month
      val day = conv day
      val hour = conv hour
      val minute = conv minute
      val second = conv second
      val smallt = getOpt(Int.fromString smallt, 0)

      val year = if year < 1970 then 1970 else year
      val date = Date.date {year=year, month=month, day=day, hour=hour,
			    minute=minute, second=second, 
			    offset=(SOME Time.zeroTime)}

      val time = (Date.toTime date, smallt)
    in
      time
    end handle _ => raise BadTimeString
    
  fun toValue t = Value.Single (toString t)
end
