structure Value :> VALUE = 
struct
  datatype value =
    Single of string
  | List of string list
  | Nil
  | Default

  exception DefaultValue

  val debugLiteral = ref false

  (* this would also be legal to just look for " \ CR LF *)
  fun safeChar c = (Char.isPrint c) andalso c <> #"\"" andalso c <> #"\\"

  fun stringAll f = 
    let
      fun g NONE = true
        | g (SOME (c, ss)) = f c andalso g (Substring.getc ss)
    in
      g o Substring.getc o Substring.all
    end

  val safe = stringAll safeChar

  fun wrap s = 
    let
      val l = String.size s

      val isquote = l < 256 andalso safe s 

      val s = if isquote then concat ["\"", s, "\""]
	      else concat ["{", Int.toString l, "}\r\n", s]

      val _ = if !debugLiteral andalso not isquote then
	        TextIO.print (concat ["[", s, "]\n"])
	      else ()
    in s end

  fun toString (Single s) = wrap s
    | toString (List nil) = toString (Nil)
    | toString (List (a::l)) = "(" ^ (wrap a) ^
             (List.foldl (fn (s, acc) => (" " ^ (wrap s) ^ acc)) ")" l)
    | toString (Nil) = "NIL"
    | toString (Default) = raise DefaultValue

  fun size (Single s) = [String.size s]
    | size (List l) = List.map String.size l
    | size Nil = nil
    | size Default = nil

  fun eat "" getc = (fn strm => (true, strm))
    | eat s getc = 
    let
      val n = String.size s
      fun isPrefix (i, strm) =
	if (i = n) then SOME strm
	else (case getc strm
		of (SOME(c, strm)) => if (String.sub(s, i) = c)
					then isPrefix(i+1, strm)
				      else NONE
	      | NONE => NONE)
      fun eat' strm = (case isPrefix (0, strm)
			 of (SOME strm) => (true, strm)
		       | NONE => (false, strm))
    in
      eat'
    end

  (* this is the "safe" implementation *)
  fun grab len getc strm =
    let
      val arr = CharArray.array (len, #"\000")
      fun grab' (n, strm) =
	if n = len then SOME (CharArray.extract (arr, 0, NONE), strm)
	else 
	  (case getc strm of
	   (SOME (c, strm)) => (CharArray.update (arr, n, c);
				grab' (n+1, strm))
	 | NONE => NONE)
    in
      grab' (0, strm)
    end
  
  (* but this one is faster *)
  fun grab len getc strm =
    let
      val arr = Unsafe.CharVector.create len
      fun grab' (n, strm) =
	if n = len then SOME (arr, strm)
	else 
	  (case getc strm of
	     (SOME (c, strm)) => (Unsafe.CharVector.update (arr, n, c);
				  grab' (n+1, strm))
	   | NONE => NONE)
    in
      grab' (0, strm)
    end

  fun scanOne getc strm =
    case getc strm of
      (SOME (#"{", strm)) =>
	let (* literal *)
	  val scanInt = Int.scan StringCvt.DEC getc
	in
	  case scanInt strm of
	    NONE => NONE
	  | (SOME (len, strm)) => 
	      (case eat "}\r\n" getc strm of
		 (true, strm) => (* grab len bytes *)
	           (case grab len getc strm of
		      NONE => NONE
		    | s => s)
     	       | (false, strm) => NONE)
	end
    | (SOME (#"\"", strm)) => 
	     let
	       val (s, strm) = StringCvt.splitl (fn c => c <> #"\"") getc strm
	     in
	       case (getc strm) of
                 NONE => NONE
	       | (SOME (#"\"", strm)) => SOME (s, strm)
	       | _ => NONE
	     end
    | _ => NONE

  fun scanList getc = ListFormat.scan {init="(", sep=" ", final=")", scan=scanOne} getc

  (* val scan : (char, 'a) StringCvt.reader -> (value, 'a) StringCvt.reader *)
  fun scan getc strm =
    case getc strm of
      (SOME (#"(", _)) => 
	(case scanList getc strm of
	   NONE => NONE
	 | (SOME (l, strm)) => SOME (List l, strm))
    | (SOME (#"N", strm')) => 
        (case eat "NIL" getc strm of
           (true, strm) => SOME (Nil, strm)
	 | (false, _) => NONE)
    | (SOME (_, strm')) => 
	(case scanOne getc strm of
	   NONE => NONE
	 | (SOME (s, strm)) => SOME (Single s, strm))
    | NONE => NONE

  val fmt = toString
    
end
