structure Acl :> ACL =
struct
  structure Rights = AclRights

  structure V = Value

  exception SyntaxError of string

  datatype entry = POS of (string * Rights.rights)
                 | NEG of (string * Rights.rights)

  type acl = entry list

  val empty = (nil : entry list)

  fun formatPair (POS(ident, rights)) = 
    ident ^ "\t" ^ (Rights.toString rights)
    | formatPair (NEG(ident, rights)) =
    "-" ^ ident ^ "\t" ^ (Rights.toString rights)

  fun toString nil = "NIL"
    | toString (a::l) =
    let
      fun f (pair, acc) = ("\"" ^ (formatPair pair) ^ "\" ") ^ acc
    in
      "(" ^ (foldl f (formatPair a) l) ^ ")"
    end

  fun toValue nil = V.Nil
    | toValue l = V.List (List.map formatPair l)

  fun optToString (SOME acl) = toString acl
    | optToString NONE = "NIL"

  fun stringToIdent s =
    case Substring.getc (Substring.all s) of
      NONE => raise (SyntaxError "zero length identifier")
    | (SOME (#"-", substr)) => (NEG, Substring.string substr)
    | (SOME _) => (POS, s)
    
  fun fromString' s = 
    case String.fields (fn c => c = #"\t") s of
      [ident, rights] => 
	let
	  val (cons, ident) = stringToIdent ident

	  val rights = (Rights.fromString rights 
			handle Rights.InvalidRights => Rights.empty)
	in
	  cons(ident, rights)
	end
    | _ => raise SyntaxError "not tab divided"

  fun fromString s = [fromString' s]

  fun fromValue (V.Single _) = raise (SyntaxError "acls are multivalued")
    | fromValue (V.List l) = map fromString' l
    | fromValue _ = nil

  val union = op @

  val all = List.foldl Rights.union Rights.empty 
    [Rights.ACLx, Rights.ACLr, Rights.ACLw, Rights.ACLi, Rights.ACLa]

  fun myrights ident = 
    if Auth.equal (ident, Auth.master) then
      fn l => all
    else let 
      val must = if Auth.memberof (ident, Auth.admins) then Rights.ACLa
		 else Rights.empty

      fun myrights' (nil, pos, neg) = 
	Rights.union (must, Rights.diff (pos, neg))
	| myrights' ((POS(i, r))::l, pos, neg) =
	if Auth.memberof (ident, i) then 
	  myrights' (l, Rights.union (pos, r), neg)
	else myrights' (l, pos, neg)
	| myrights' ((NEG(i, r))::l, pos, neg) =
	if Auth.memberof (ident, i) then 
	  myrights' (l, pos, Rights.union (neg, r))
	else myrights' (l, pos, neg)
    in
      fn l => myrights' (l, Rights.empty, Rights.empty)
    end

  fun update nil (ident, rights) =
    let
      val (cons, ident) = stringToIdent ident
    in
      [cons(ident, rights)]
    end
    | update l (ident, rights) =
    let
      fun matches (POS(i, _) | NEG(i, _)) = (i = ident)

      val (this, others) = List.partition matches l

      val (cons, ident) = stringToIdent ident
    in
      (cons(ident, rights)::others)
    end

  fun scanEntry getc =
    let
      fun grabRights strm =
	let
	  fun gr (v as (acc, strm)) =
	    case getc strm of
	      (SOME (#"x", strm)) => gr (Rights.union (acc, Rights.ACLx), strm)
	    | (SOME (#"r", strm)) => gr (Rights.union (acc, Rights.ACLr), strm)
	    | (SOME (#"w", strm)) => gr (Rights.union (acc, Rights.ACLw), strm)
	    | (SOME (#"i", strm)) => gr (Rights.union (acc, Rights.ACLi), strm)
	    | (SOME (#"a", strm)) => gr (Rights.union (acc, Rights.ACLa), strm)
	    | _ => v
	in
	  gr (Rights.empty, strm)
	end

      fun scanEntry' strm =
	case (getc strm) of
	  NONE => NONE
	| (SOME(c, strm')) => 
	    let
	      val strm = if c = #"-" then strm' else strm

	      val (ident, strm) = (StringCvt.splitl (fn c => c <> #"\t") 
				         getc strm)

	      val (rights, strm) = grabRights strm 
	    in
	      if c = #"-" then 
		SOME(NEG(ident, rights), strm)
	      else SOME(POS(ident, rights), strm)
	    end
    in
      scanEntry'
    end
  
  (* val scan : (char, 'a) StringCvt.reader -> (acl, 'a) StringCvt.reader *)
  fun scan getc = 
    ListFormat.scan {init="{", sep=" ", final="}", scan=scanEntry} getc

  (* val fmt : acl -> string *)
  val fmt = ListFormat.fmt {init="{", sep=" ", final="}", fmt=formatPair}
  
end
