(* ACAP entries 

   Generally, entries are the thing that are passed around by the
   various routines.
*)

signature ENTRY =
sig
  type entry
  type entry' (* entry without filled-in acls *)
  type attribute = {acl : Acl.acl, attribute : string,
		    size : int list, value : Value.value}
  type attribute' = {acl : Acl.acl option, attribute : string,
		     size : int list, value : Value.value}
  type storedata = {acl : Acl.acl option, attribute : string, 
		    value : Value.value option}

  (* used for transferring information (dataset->dataset, dataset->context) *)
  datatype updatemsg =
    NEW of string * entry
  | CHANGE of (string * entry) * (string * entry)
  | DELETE of string * entry

  (* raised when an attempt to store an illegal value is made to this
   dataset---returns attribute raised on *)
  exception EnforcedValidation of (string * string)

  (* raised when an attempt is made to store with a UNCHANGEDSINCE;
     returns the time it was last modified *)
  exception Modified of AcapTime.acaptime

  (* renames aren't implemented *)
  exception RenameNotImpemented

  (* ACL problem *)
  exception NotPermitted
   
  (* read an entry *)
  val scan : (char, 'strm) StringCvt.reader 
    -> (string * entry', 'strm) StringCvt.reader

  (* saves an entry, given it's name *)
  val save : TextIO.outstream -> (string * entry') -> unit

  (* creates an entry, given it's complete name *)
  val create : Auth.ident -> (string -> Acl.acl) -> 
    string -> storedata list -> entry'
  
  (* stores a list of attributes in this entry; this is where
     enforced validation takes place. returns NONE if this entry
     was deleted *)
  (* requires current user and an default attrname -> acl function *)
  val store : Auth.ident -> (string -> Acl.acl) -> entry' -> 
    (storedata list * AcapTime.acaptime option) -> entry' option

  (* gets the attributes *)
  val fetch : Auth.ident -> entry -> string -> attribute

  (* gets the matching attributes *)
  val search : Auth.ident -> entry -> (string -> bool) -> attribute list

  (* return the time of last modification *)
  val gettime : entry -> AcapTime.acaptime

  (* --- needed for inheritance --- *)

  (* given an (ancestor, entry) pair, create an inherited entry *)
  val union : (entry * entry) -> entry
    
  (* given an entry and an acl function, produce a fully qualified entry *)
  val fillAcls : (string -> Acl.acl) -> entry' -> entry

  val getattr' : entry' -> string -> attribute' option
end
