(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

(* notify contexts should probably only queue events that are actually
 going to effect the context.

 also, it would be really nice if we don't send notifications that don't convey any information.  this is hard. *)
structure Context :> CONTEXT =
struct
  structure SV = LSyncVar
  structure MC = RMulticast

  structure Map = RedBlackMapFn(type ord_key = string * string
				fun compare ((s1, s2), (t1, t2)) = 
				  case String.compare (s2, t2) of
				    LESS => LESS
				  | EQUAL => String.compare (s1, t1)
				  | GREATER => GREATER)

  (* outgoing messages are batched up and sent no later than X seconds
   after generated *)
  val batchDelay = Time.fromSeconds 30

  datatype outMsg = 
    CNEW of {dset : string, name : string, entry : Entry.entry} * int
  | CCHANGE of {dset : string, name : string, entry : Entry.entry} * 
    (int * int)
  | CDELETE of {dset : string, name : string, entry : Entry.entry} * int
  | CTIME of AcapTime.acaptime
  | CDIE

  exception ContextInconsistency
  exception Modified of AcapTime.acaptime
  exception NotNotifyContext
  exception NotifyContextBadState

  datatype ctlMsg =
    E of Entry.updatemsg		(* entry to consider adding *)
  | U of unit SV.ivar			(* flush buffered updates,
					 issue modtime, and return *)
  | F					(* flush, noop if empty *)
  | AE of Entry.updatemsg MC.port	(* add dataset to listen to *)
  | Q					(* let the context die *)

  (* invariant: the enumeration (if it exists) holds whatever is currently 
                in the data map. *)
  type staticcontext = {data : Entry.entry Map.map,
			enumeration : {dset : string, 
				       name : string, 
				       entry : Entry.entry} Enum.enum option,
			die : unit SV.ivar}
                              SV.mvar

  type notifycontext = {guard : Entry.entry -> bool,
			notify : outMsg CML.chan,
			listen : Entry.updatemsg MC.port list,
			lastupdate : AcapTime.acaptime,
			ctl : ctlMsg CML.chan option,
			data : Entry.entry Map.map,
			enumeration : {dset : string, 
				       name : string, 
				       entry : Entry.entry} Enum.enum option,
			die : unit SV.ivar}
                              SV.mvar

  datatype context = S of staticcontext | N of notifycontext

  fun defaultSort ({dset=s1, name=s2, entry=_}, 
		   {dset=t1, name=t2, entry=_}) = 
    case String.compare (s2, t2) of
      LESS => LESS
    | EQUAL => String.compare (s1, t1)
    | GREATER => GREATER

  fun create _ (_, NONE, false) =
    S (SV.mVarInit {data=Map.empty, enumeration=NONE, die=SV.iVar ()})
    | create _ (sort, NONE, true) =
    S (SV.mVarInit {data=Map.empty, 
		    enumeration=SOME (Enum.empty (getOpt(sort, defaultSort))),
		    die=SV.iVar ()})
    | create guard (_, SOME notify, false) =
    N (SV.mVarInit {guard=guard, notify=notify, 
		    listen=nil, lastupdate=AcapTime.old,
		    ctl=NONE,
		    data=Map.empty, enumeration=NONE,
		    die=SV.iVar ()})
    | create guard (sort, SOME notify, true) =
    N (SV.mVarInit {guard=guard, notify=notify, 
		    listen=nil, lastupdate=AcapTime.old,
		    ctl=NONE,
		    data=Map.empty, 
		    enumeration=SOME (Enum.empty (getOpt(sort, defaultSort))),
		    die=SV.iVar ()})

  (* if there's an enumeration, returns the new one and a position;
     otherwise NONE and zero *)
  fun oEnumIns (_, NONE) = (NONE, 0)
    | oEnumIns (a, SOME enum) = 
    let
      val (enum, pos) = Enum.insert (enum, a)
    in
      ((SOME enum), pos)
    end

  fun oEnumRem (_, NONE) = (NONE, 0)
    | oEnumRem (a, SOME enum) = 
    let
      val (enum, pos) = Enum.remove (enum, a)
    in
      ((SOME enum), pos)
    end

  fun fastInsert (S ctxt) (new as {dset, name, entry}) =
    let
      val {data, enumeration, die} = SV.mTake ctxt
    in
      SV.mPut (ctxt, {data=Map.insert (data, (dset, name), entry),
		      enumeration=(#1 (oEnumIns (new, enumeration))),
		      die=die})
    end
    | fastInsert (N ctxt) (new as {dset, name, entry}) =
    let
      val {guard, notify, listen, lastupdate, ctl, data, enumeration, die} = 
	SV.mTake ctxt
    in
      SV.mPut (ctxt, {guard=guard, notify=notify,
		      listen=listen, lastupdate=lastupdate,
		      ctl=ctl,
		      data=Map.insert (data, (dset, name), entry),
		      enumeration=(#1 (oEnumIns (new, enumeration))),
		      die=die})
    end

  (* attempt to insert new entry into context; give notification *)
  (* val insert : context -> string * Entry.entry -> unit *)
  fun insert (ctxt : notifycontext) (new as {dset, name, entry}) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, data, enumeration,
		 die}) = 
	SV.mTake ctxt
    in
      if guard entry then (* add it *)
	let
	  val data = Map.insert (data, (dset, name), entry)
	  val (enumeration, pos) = oEnumIns (new, enumeration)
	in
	  CML.send (notify, (CNEW(new, pos)));
	  SV.mPut (ctxt, {guard=guard, notify=notify,
			  listen=listen, lastupdate=lastupdate,
			  ctl=ctl,
			  data=data, enumeration=enumeration,
			  die=die})
	end
      else SV.mPut (ctxt, c)
    end

  (* change an entry that may or may not be in the context *)
  (* val change : context -> (string * Entry.entry) * (string * Entry.entry) 
    -> unit *)
  fun change (ctxt : notifycontext) (new as {dset, name, entry}) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, 
		 data, enumeration, die}) = SV.mTake ctxt

      fun rem (data, key) =
	let val (data, ent) = Map.remove (data, key)
	in (data, SOME ent) end handle LibBase.NotFound => (data, NONE)
      
      val (data, oldent) = rem (data, (dset, name))
    in
      case oldent of
	(SOME oldent) =>
	if guard entry then (* CHANGE *)
	  let
	    val (enumeration, oldpos) = oEnumRem ({dset=dset, name=name,
						   entry=oldent}, 
						  enumeration)
	    val data = Map.insert (data, (dset, name), entry)
	    val (enumeration, newpos) = oEnumIns (new, enumeration)
	  in
	    CML.send (notify, (CCHANGE(new, (oldpos, newpos))));
	    SV.mPut (ctxt, {guard=guard, notify=notify,
			    listen=listen, lastupdate=lastupdate,
			    ctl=ctl,
			    data=data, enumeration=enumeration,
			    die=die})
	  end
	else (* DELETE *)
	  let
	    val old = {dset=dset, name=name, entry=oldent}
	    val (enumeration, pos) = oEnumRem (old, enumeration)
	  in
	    CML.send (notify, (CDELETE(old, pos)));
	    SV.mPut (ctxt, {guard=guard, notify=notify,
			    listen=listen, lastupdate=lastupdate,
			    ctl=ctl,
			    data=data, enumeration=enumeration,
			    die=die})
	  end
      | NONE => 
	if guard entry then (* ADD *)
	  let
	    val data = Map.insert (data, (dset, name), entry)
	    val (enumeration, pos) = oEnumIns (new, enumeration)
	  in
	    CML.send (notify, (CNEW(new, pos)));
	    SV.mPut (ctxt, {guard=guard, notify=notify,
			    listen=listen, lastupdate=lastupdate,
			    ctl=ctl,
			    data=data, enumeration=enumeration,
			    die=die})
	  end
	else (* NOOP *)
	  SV.mPut (ctxt, c)
    end

  (* delete an entry that may or may not be in the context *)
  (* val delete : context -> (string * Entry.entry) -> unit *)
  fun delete (ctxt : notifycontext) (old as {dset, name}) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, 
		 data, enumeration, die}) = 
	SV.mTake ctxt

      fun rem (data, key) =
	let val (data, ent) = Map.remove (data, key)
	in (data, SOME ent) end handle LibBase.NotFound => (data, NONE)
      
      val (data, oldent) = rem (data, (dset, name))
    in
      case oldent of
	(SOME oldent) =>
	let
	  val old = {dset=dset, name=name, entry=oldent}
	  val (enumeration, pos) = oEnumRem (old, enumeration)
	in
	  CML.send (notify, (CDELETE(old, pos)));
	  SV.mPut (ctxt, {guard=guard, notify=notify,
			  listen=listen, lastupdate=lastupdate,
			  ctl=ctl,
			  data=data, enumeration=enumeration,
			  die=die})
	end
      | NONE => SV.mPut (ctxt, c)
    end

  fun processE ctx (Entry.NEW new) = insert ctx new
    | processE ctx (Entry.CHANGE chg) = change ctx chg
    | processE ctx (Entry.DELETE old) = delete ctx old

  fun sendTime (ctxt : notifycontext) =
    let
      val ({notify,lastupdate,...}) = SV.mGet ctxt
    in
      CML.send (notify, CTIME(lastupdate))
    end

  fun updateTime (ctxt : notifycontext) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, 
		 data, enumeration, die}) = 
	SV.mTake ctxt

      val lastupdate = AcapTime.now ()
    in
      CML.send (notify, CTIME(lastupdate));
      SV.mPut (ctxt, {guard=guard, notify=notify,
		      listen=listen, lastupdate=lastupdate,
		      ctl=ctl,
		      data=data, enumeration=enumeration,
		      die=die})
    end

  fun listener (ctx, evts, ctl) = 
    let
      (* process all the entry events waiting *)
      val process = Fifo.app (processE ctx)

      fun timerThread () = (CML.sync(CML.timeOutEvt batchDelay);
			    CML.send (ctl, F))
	
      (* two-state listener; we queue up events until we get a flush
       or we time out *)
      fun empty evts =
	case CML.select evts of
	  (E x) => (CML.spawn timerThread; 
		    nonempty (evts, Fifo.enqueue (Fifo.empty, x)))
	| (U sv) => (sendTime ctx; 
		     SV.iPut (sv, ()); 
		     empty evts)
	| F => (* already flushed! *) (empty evts)
	| (AE p) => empty (CML.wrap (MC.recvEvt p, E)::evts)
	| Q => ()
      and nonempty (evts, fifo) =
	case CML.select evts of
	  (E x) => nonempty (evts, Fifo.enqueue (fifo, x))
	| (U sv) => (process fifo; 
		     updateTime ctx; 
		     SV.iPut (sv, ()); 
		     empty evts)
	| F => (process fifo; 
		updateTime ctx; 
		empty evts)
	| (AE p) => nonempty (CML.wrap (MC.recvEvt p, E)::evts, fifo)
	| Q => ()
    in
      empty evts
    end

  fun startListener (ctxt, ctl, listen) =
    let
      val ctlEvt = CML.recvEvt ctl
      val evts = List.map (fn p => CML.wrap (MC.recvEvt p, E)) listen
    in
      ignore (CML.spawnc listener (ctxt, ctlEvt::evts, ctl))
    end

  fun doneInitial (S ctxt) = ()
    | doneInitial (N ctxt) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, 
		 data, enumeration, die}) = 
	SV.mTake ctxt

      val ctl = CML.channel ()
      val _ = startListener (ctxt, ctl, listen)
    in
      SV.mPut (ctxt, {guard=guard, notify=notify,
		      listen=listen, lastupdate=lastupdate,
		      ctl=SOME ctl,
		      data=data, enumeration=enumeration,
		      die=die})
    end

  fun updatecontext (N ctxt) = 
    let
      val ctl = 
	case SV.mGet ctxt of
	  ({ctl=SOME ctl,...}) => ctl
	| _ => raise NotifyContextBadState

      val sv = SV.iVar ()
    in 
      CML.send (ctl, U sv);
      SV.iGet sv
    end
    | updatecontext (S _) = raise NotNotifyContext

  (* add a new dataset to listen to *)
  (* val addDSet : context -> Entry.updatemsg RMulticast.port -> unit *)
  fun addDSet (N ctxt) port =
    let
      val {guard, notify, listen, lastupdate, ctl, 
	   data, enumeration, die} = SV.mTake ctxt
    in
      (case ctl of
	 NONE => ()
       | (SOME ctl) => CML.send (ctl, AE port));
	 SV.mPut (ctxt, {guard=guard, notify=notify,
			 listen=port::listen, lastupdate=lastupdate,
			 ctl=ctl,
			 data=data, enumeration=enumeration,
			 die=die})
    end
    | addDSet (S _) _ = raise NotNotifyContext

  fun destroyEvent (N ctxt) = (SV.iGetEvt (#die (SV.mGet ctxt)))
    | destroyEvent (S ctxt) = (SV.iGetEvt (#die (SV.mGet ctxt)))

  fun destroy (N ctxt) =
    let
      val (c as {guard, notify, listen, lastupdate, ctl, 
		 data, enumeration, die}) = 
	SV.mTake ctxt

      val ctl = case ctl of (SOME c) => c | NONE => raise NotifyContextBadState
    in
      CML.send (ctl, Q);
      CML.send (notify, CDIE);
      SV.iPut (die, ());
      List.app MC.release listen
    end
    | destroy (S ctxt) = (SV.iPut (#die (SV.mGet ctxt), ()))

  (* val search : context -> ((int * Entry.entry) -> bool) * 
      (string * Entry.entry) Throttle.throttle * AcapTime.acaptime -> unit *)
  local 
    fun searchData (data, lastupdate) (sfun, out, time) =
      let
	val out' = Throttle.call out
	fun win (thing as ((dset, name), entry)) = 
	  if sfun (0, entry) then out' {dset=dset, name=name, entry=entry}
	  else ()
      in
      if AcapTime.compare (time, lastupdate) = LESS then
	raise (Modified lastupdate)
      else (Map.appi win data;
	    Throttle.done out)
      end
    
    fun searchEnum (enum, lastupdate) (sfun, out, time) =
      let
	val out' = Throttle.call out
	fun win (thing as {dset, name, entry}, pos) =
	  if sfun (pos, entry) then out' thing else ()
      in
	if AcapTime.compare (time, lastupdate) = LESS then
	  raise (Modified lastupdate)
	else (Enum.app win enum;
	      Throttle.done out)
      end
  in
    fun search (N ctxt) sparams =
      let
	val ({lastupdate, data, enumeration, ...}) = SV.mGet ctxt
      in
	case enumeration of
	  NONE => searchData (data, lastupdate) sparams
	| (SOME enum) => searchEnum (enum, lastupdate) sparams
      end
      | search (S ctxt) sparams = 
      let
	val ({data, enumeration, ...}) = SV.mGet ctxt
      in
	case enumeration of
	  NONE => searchData (data, AcapTime.old) sparams
	| (SOME enum) => searchEnum (enum, AcapTime.old) sparams
      end
  end

end
