(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

structure Dataset :> DATASET =
struct
  structure T = Throttle

  (* shortcuts! *)
  structure SV = LSyncVar
  structure MC = RMulticast
  structure V = Value

  structure Dict = RedBlackMapFn(type ord_key = String.string
				 val compare = String.compare)
  structure EL = Dict

  datatype aclmod = 
    SETACL of string * AclRights.rights
  | DELACL of string option
  | GETACL

  datatype 'a response =
    Res of 'a
  | Exn of exn

  fun sync (ivar) =
    case SV.iGet ivar of
      (Res x) => x
    | (Exn exn) => raise exn

  datatype request = 
      (* when the search of this dataset concludes, the ivar will be put *)
      SEARCH of ((Entry.entry -> bool) 
	      * ({dset : string, name : string, entry : Entry.entry} -> unit)
	      * ((string * Entry.updatemsg RMulticast.port -> unit) option)
	      * (unit response SV.ivar))
    | SEARCHnoinherit of ((Entry.entry -> bool)
              * ({dset : string, name : string, entry : Entry.entry} -> unit)
	      * ((string * Entry.updatemsg RMulticast.port -> unit) option)
	      * (unit response SV.ivar))
    | STORE of (Auth.ident * 
		AcapTime.acaptime option *
		string * 
		Entry.storedata list *
		unit response SV.ivar)
    | ACLentry of (Auth.ident * 
		   string *
		   string *
		   aclmod *
		   Acl.acl response SV.ivar)
    | ACLset of (Auth.ident *
		 string option *
		 aclmod *
		 Acl.acl response SV.ivar)
    | INFO of Entry.updatemsg
    | GETDATA of (Entry.entry EL.map * Entry.updatemsg MC.port) SV.ivar
    | FLUSH of unit
    | DELETE

  and dataset =
    Local of request CML.chan
  | Remote of string list
	
  structure DSetList =
    struct
      exception Impossible = Config.Impossible
      
      structure SV = LSyncVar
      structure MC = RMulticast
	
      datatype DatasetDict = EMPTY
      | DD of ((dataset Dict.map) SV.mvar *
	       (string * Entry.updatemsg MC.port) MC.mchan)
	
      val dict = ref EMPTY
	
      (* functions to manipulate the dataset file *)
      fun writeDSets mydict = 
	let
	  val fmtRem = List.foldl (fn (url, acc) => url ^ "\t" ^ acc) ""
	  fun fmt (Local _) = ("\tlocal\n")
	    | fmt (Remote urls) = ("\tremote\t" ^ (fmtRem urls) ^ "\n")
	    
	  val fname = !Config.FNAME_dsets
	  val fnamenew = fname ^ ".NEW"
	  val f = TextIO.openOut fnamenew
	    
	  fun out (key, data) = (TextIO.output (f, key);
				 TextIO.output (f, fmt data))
	in
	  Dict.appi out mydict;
	  TextIO.closeOut f;
	  Posix.FileSys.rename {old=fnamenew, new=fname}
	end

      fun readDSets () = 
	(let
	   (* blank out dict *)
	   val _ = dict := EMPTY
	     
	   val scanName = ParserComb.seqWith (#1) 
	     (ParserComb.token (fn c => c <> #"\t"), ParserComb.char #"\t")
	   fun newlocal (dsetname, _) = 
	     (dsetname, Local (CML.channel ()))
	   val scanLocal = ParserComb.seqWith newlocal 
	     (scanName,	ParserComb.string "local")
	   fun newremote (_, remstr) = (Remote remstr)
	   val url = ParserComb.seqWith (#1) 
	     (ParserComb.token (fn c => c <> #"\t"),
	      ParserComb.char #"\t")
	   val scanRemote = 
	     ParserComb.seqWith newremote (ParserComb.string "remote\t",
					   ParserComb.zeroOrMore (url))
	   val scanRemote = ParserComb.seq (scanName, scanRemote)
	   val scanDSet = ParserComb.or (scanLocal, scanRemote)
	   val scanEntry = ParserComb.seqWith (#1) 
	     (scanDSet, ParserComb.char #"\n")
	     
	   fun makedict getc =
	     let 
	       val scanEntry = scanEntry getc
	       fun makedict' (sin, dict) =
		 case scanEntry sin of
		   NONE => dict
		 | (SOME (pair, sin)) => 
		     makedict' (sin, Dict.insert' (pair, dict))
	     in
	       makedict'
	     end
	   
	   val f = TextIO.openIn (!Config.FNAME_dsets)
	   val sin = TextIO.getInstream f
	   val d = makedict TextIO.StreamIO.input1 (sin, Dict.empty)
	 in
	   dict := (DD(SV.mVarInit d, MC.mChannel ()));
	   TextIO.closeIn f;
	   true
	 end
       handle _ => false)

      fun dict' () =
	case !dict of
	  EMPTY => raise (Impossible "no dset dictionary")
	| (DD (dict, _)) => dict
	    
      fun getdict () = SV.mGet (dict' ())
      fun takedict () = SV.mTake (dict' ())
      fun putdict (mydict) = (writeDSets mydict;
			      SV.mPut (dict' (), mydict))
			      
      fun initdict mydict =
	dict := DD(SV.mVarInit mydict, MC.mChannel ())
    
      fun broadcast (p as (dsetname, port)) =
	case !dict of
	  EMPTY => raise (Impossible "no dset dictionary")
	| (DD (_, MC)) => MC.multicast (MC, p)
	    
      fun port () =
	case !dict of
	  EMPTY => raise (Impossible "no dset dictionary")
	| (DD (_, MC)) => MC.port MC
	    
    end

  exception NoSuchDataset
  exception DatasetRefer of string list
  exception SyntaxError of string
  exception DiskError of string

  (* random number server *)
  val randCh = (CML.channel ()) : int CML.chan

  datatype scanning =
      gotE of (string * Entry.entry') 
    | delE of (string * AcapTime.acaptime)
    | DONE

  fun readSet (dsetname) =
    let
      val scanOpen = ParserComb.string "<delete name=\""
      val scanName = ParserComb.token (fn c => c <> #"\"")
      val scanTime = ParserComb.seqWith (AcapTime.fromString o #2)
		      (ParserComb.string "\" time=\"",
		       ParserComb.token (fn c => c <> #"\""))
      val scanClose = ParserComb.string "\"></delete>"

      val scanDelete = ParserComb.seqWith #2 (scanOpen, scanName)
      val scanDelete = ParserComb.seq (scanDelete,
				       ParserComb.seqWith #1 (scanTime,
							      scanClose))
      val scanDelete = ParserComb.skipBefore Char.isSpace scanDelete
      val scanDelete = ParserComb.wrap (scanDelete, delE)
      val scanEntry = ParserComb.skipBefore Char.isSpace Entry.scan
      val scanEntry = ParserComb.wrap (scanEntry, gotE)
      val scanNone = ParserComb.result (DONE)

      val scanOne = ParserComb.or' [scanEntry, scanDelete, scanNone]

      fun doRemove (data, n) = 
        (let 
	   val (data, _) = EL.remove (data, n)
	 in
	   data
	 end handle LibBase.NotFound => data)

      fun scanData data DONE = ParserComb.result data
	| scanData data (gotE (e as (n, _))) =
	(ParserComb.bind (scanOne, scanData (EL.insert' (e, data))))
	| scanData data (delE (n, time)) =
	(ParserComb.bind (scanOne, scanData (doRemove (data, n))))

      val scan = ParserComb.bind (scanOne,
				  scanData EL.empty)

      val _ = TextIO.print (concat ["loading ", dsetname, "\n"])
      val file = Config.makeDataFile dsetname
      val f = TextIO.openIn file

      val data = 
	case TextIO.scanStream scan f of
	  NONE => raise (DiskError "badly formed dataset")
	| (SOME d) => d

      val _ = TextIO.closeIn f
    in
      data
    end

  datatype change = 
    addE of (string * Entry.entry')
  | delE of (string * AcapTime.acaptime)
  (* val writeSet : (string * entry' EL.map * change) *)
  fun writeSet (dsetname, data, change) =
    let
      val file = Config.makeDataFile dsetname

      fun rewriteFile () =
	let
	  val _ = TextIO.print ("compacting " ^ dsetname ^ "...\n")
	  val newfile = file ^ ".NEW"
	  val strm = TextIO.openOut newfile
	  val _ = EL.appi (Entry.save strm) data
	in
	  TextIO.closeOut strm;
	  Posix.FileSys.rename {old=newfile, new=file}
	end

      fun saveDelete sout (name, time) =
	TextIO.output (sout, 
          "<delete name=\"" ^ name ^ "\" time=\"" ^ (AcapTime.toString time)
		            ^ "\"></delete>\n")

      fun appendEntry () =
	let
	  val strm = TextIO.openAppend file
	  val _ = 
	    case change of 
	      (addE pair) => Entry.save strm pair
	    | (delE pair) => saveDelete strm pair
	in
	  TextIO.closeOut strm
	end

      val setsize = EL.numItems data

      val rand = (CML.recv randCh) mod setsize
    in
      if rand = 0 then (* write the whole thing *)
	rewriteFile ()
      else (* just append this entry *)
	appendEntry ()
    end
		       
  val defaultAclf = (fn s => Acl.empty)

  (* create a new root dataset *)
  fun createRoot () =
    let
      val acl = Value.List ["$anyone\txrwia"]

      val null = Entry.create Auth.master defaultAclf
	"" [{acl=NONE, attribute="dataset.owner", 
	      value=SOME (Value.Single (Auth.toString Auth.master))},
	     {acl=NONE, attribute="dataset.acl",
	      value=SOME acl}]

      val acl = Acl.fromValue acl
      fun aclf s = acl

      val data = EL.singleton ("", null)
	
      (* commit it to disk *)
      val _ = writeSet ("/", data, addE("", null))
      val ctl = CML.channel ()

      val root = Local ctl
      val mydict = Dict.insert (Dict.empty, "/", root)
    in
      DSetList.initdict mydict;
      DSetList.writeDSets mydict
    end
  
  (* states of a dataset: )
   - on disk
   - in memory
   - in memory w/ listeners

   operations to support:
   - changes in inherited dataset
   - searching
   - storing
   - acl modifications
   *)
  fun fetchdata dsetname =
    let
      val dict = DSetList.getdict ()
      val sv = SV.iVar ()
    in
      case Dict.find (dict, dsetname) of
	(SOME (Local d)) => (CML.send (d, GETDATA sv);
			     (SOME dsetname, SV.iGet sv))
      | _ => (NONE, (EL.empty, MC.nullport))
    end

  fun doInheritanceEntry (name, e, inherits) =
    case fetchdata name of
      (NONE, _) => e
    | (SOME _, (d, p)) => 
    let
      val _ = MC.release p
    in
      case EL.find (d, name) of
	NONE => e
      | (SOME ancestor) => Entry.union (ancestor, e)
    end

  type dsetdata = {data : Entry.entry' EL.map, 
		   idata : Entry.entry EL.map,
		   inherits : string option, 
		   acl : string -> Acl.acl,
		   port : Entry.updatemsg MC.port}

  (* should give implicit "a" rights to dataset.owner *)
  fun genAclFun (null) =
    fn attrname =>
    case Entry.getattr' null ("dataset.acl." ^ attrname) of
      NONE => (case Entry.getattr' null "dataset.acl" of
		 NONE => Acl.empty
	       | (SOME {value=aclval,...}) => Acl.fromValue aclval)
    | (SOME {value=aclval,...}) => Acl.fromValue aclval
	
  fun dataset (dsetname, ctl, channel) =
    let
      (* print out an evil message and spawn a new thread to take
       care of this dataset *)
      val ctlevt = CML.recvEvt ctl
      val flushevt = CML.wrap (CML.timeOutEvt (!Config.SweepTime), FLUSH)

      fun barf s = (TextIO.print (String.concat ["dataset ", dsetname,
						 " barfing: ", s, "\n"]);
		    dataset (dsetname, ctl, channel);
		    CML.exit ())

      fun execute (evts, ds as {idata, ...}, SEARCH (sfun, out, mess, ivar)) =
	let
	  fun win (n, e) = if sfun e then 
	                     out {dset=dsetname, name=n, entry=e}
			   else ()

	  val _ = case mess of
	            (SOME mess) => mess (dsetname, MC.port channel)
		  | NONE => ()
	in
	  (EL.appi win idata;
	   SV.iPut (ivar, Res ()))
	  handle exn => SV.iPut (ivar, Exn exn);
	  inmem (evts, ds)
	end

	| execute (evts, ds as {data, idata, inherits, acl, ...}, 
		   SEARCHnoinherit (sfun, out, mess, ivar)) = 
	let
	  fun win' (n, e') = 
	    let val e = Entry.fillAcls acl e'
	    in if sfun e then out {dset=dsetname, name=n, entry=e} else ()
	    end 

	  fun win (n, e) = if sfun e then 
	                     out {dset=dsetname, name=n, entry=e}
			   else () 
	in
	  ((if isSome inherits then EL.appi win' data
	    else EL.appi win idata);
	   SV.iPut (ivar, Res ()))
	  handle exn => SV.iPut (ivar, Exn exn);
	  inmem (evts, ds)
	end

	(* store an entry; watch the special cases for the null entry *)
	| execute (evts, ds as {data, idata, 
				inherits, acl, port}, 
		   STORE (ident, time, name, storedata, ivar)) =
	inmem (* following expression will return new state *)
	(let
	  fun hassub NONE = NONE
	    | hassub (SOME e') = 
	    case (Entry.getattr' e' "subdataset") of
	      NONE => NONE
	    | (SOME {value=(Value.List l), ...}) => SOME l
	    | _ => NONE

	  val oldentry' = EL.find (data, name)
	  val oldentry = EL.find (idata, name)

	  val newentry' =
	    case oldentry' of
	      NONE => SOME (Entry.create ident acl name storedata)
	    | (SOME e) => Entry.store ident acl e (storedata, time)

	  val _ = if not (isSome newentry') andalso name = "" then
	            raise (SyntaxError "can't delete null entry")
		  else ()

	  val data = (* insert the new entry into the dataset *)
	    case newentry' of
	      NONE => #1 (EL.remove (data, name))
	    | (SOME e) => EL.insert (data, name, e)

	  val _ = (* write it to disk *)
	    case newentry' of
	      NONE => writeSet (dsetname, data, delE(name, AcapTime.now ()))
	    | (SOME e) => writeSet (dsetname, data, addE(name, e))

	  (* do inheritance *)
	  fun doinh e' = doInheritanceEntry (name, Entry.fillAcls acl e', 
					     inherits)
	  val newentry = 
	    case inherits of
	      NONE => Option.map (Entry.fillAcls acl) newentry'
	    | (SOME inherits) => Option.map doinh newentry'

	  val idata = (* add the new entry into the inherited data *)
	    case newentry of
	      NONE => #1 (EL.remove (idata, name))
	    | (SOME e) => EL.insert (idata, name, e)

	  (* send change notification *)
	  val _ = 
	    case (oldentry, newentry) of
	      (NONE, NONE) => ()
	    | (NONE, SOME new) => 
		MC.multicast (channel, Entry.NEW{dset=dsetname, 
						 name=name, entry=new})
	    | (SOME _, SOME new) => 
		MC.multicast (channel, Entry.CHANGE{dset=dsetname, 
						    name=name, entry=new})
	    | (SOME _, NONE) => 
		MC.multicast (channel, Entry.DELETE{dset=name, name=name})

	  (* create/delete subdataset
	     null entry can't have subdatasets *)
	  val _ = if name <> "" then
	    case (hassub oldentry', hassub newentry') of
	      (NONE, NONE) => ()
	    | (SOME ["."], SOME ["."]) => ()
	    | (_, SOME ["."]) => 
		(case EL.find (data, "") of
		   NONE => barf "no null entry for dataset"
		 | (SOME null) => createDataset (dsetname, name, 
						 ident, null, acl))
	    | (_, SOME l) => createRemote (dsetname, name, l)
	    | (SOME _, NONE) => deleteDataset (dsetname, name)
		  else ()

	  val acl = if name = "" then
	              genAclFun (valOf newentry')
		    else acl

	  fun redoinh (NONE, data, acl) =
	    let
	      val _ = MC.release port
	    in
	      (data, EL.map (Entry.fillAcls acl) data, NONE, MC.nullport)
	    end
	    | redoinh (SOME inh, data, acl) =
	    let
	      val (inherits, (idata, port)) = fetchdata inh
	    in
	      (data, EL.unionWith Entry.union (idata,
			 EL.map (Entry.fillAcls acl) data), inherits, port)
	    end

	  val (data, idata, inherits, port) =
	    if name <> "" then 
	      (data, idata, inherits, port)
	    else (* we're storing to the null entry *)
	      case (inherits, 
		    Entry.getattr' (valOf newentry') "dataset.inherit") of
  		    (NONE, NONE) => (data, idata, inherits, port)
		  | (SOME old, SOME {value=V.Single new, ...}) => 
		      if old = new then (data, idata, inherits, port)
		      else redoinh (SOME new, data, acl)
		  | (NONE, SOME {value=V.Single new, ...}) => 
			redoinh (SOME new, data, acl)
		  (* dataset.inherit is malformed *)
		  | _ => redoinh (NONE, data, acl)

	  val updateevt = CML.wrap (MC.recvEvt port, INFO)
	in
	  SV.iPut (ivar, Res ());
	  ([updateevt, flushevt, ctlevt], {data=data, idata=idata,
					   inherits=inherits,
					   acl=acl, port=port})
	end handle exn => (SV.iPut (ivar, Exn exn);
			   (evts, ds)))

	| execute (evts, ds as {data, acl, ...}, 
		   ACLentry (ident, entry, attribute, GETACL, ivar)) =
	   (case EL.find (data, entry) of
	      NONE => SV.iPut (ivar, Exn (SyntaxError "no such entry"))
	    | (SOME e) => 
		case Option.mapPartial #acl 
		                   (Entry.getattr' e attribute) of
		  NONE => SV.iPut (ivar, Res (acl attribute))
		| (SOME acl) => SV.iPut (ivar, Res (acl));
	    inmem (evts, ds))

	| execute (evts, ds as {acl, ...},
		   ACLset (ident, attribute, GETACL, ivar)) =
          (case attribute of
	     NONE => SV.iPut (ivar, Res (acl "\000"))
	   | (SOME a) => SV.iPut (ivar, Res (acl a));
	   inmem (evts, ds))
	       
	| execute (evts, ds as {data, idata, inherits, acl, port},
		   ACLentry (ident, entry, attribute, aclmod, ivar)) =
	 (case EL.find (data, entry) of 
	    NONE => (SV.iPut (ivar, Exn (SyntaxError "no such entry"));
		     inmem (evts, ds))
	  | (SOME e') => 
	  let
	    val attr = Entry.getattr' e' attribute
	    val oacl = case Option.mapPartial #acl attr of
	                NONE => acl attribute
		      | (SOME a) => a
	    val nacl = case aclmod of
	      (SETACL update) => Acl.update oacl update
	    | (DELACL (SOME id)) => Acl.update oacl (id, AclRights.empty)
	    | (DELACL NONE) => Acl.empty
	    | (GETACL) => raise (Config.Impossible "shouldn't happen")

	    (* translate vars *)
	    val ivar' = SV.iVar ()
	    fun t () = (case SV.iGet ivar' of
			  (Res _) => SV.iPut (ivar, Res nacl)
			| (Exn exn) => SV.iPut (ivar, Exn exn))
	    val _ = CML.spawn t
	   in
	     execute (evts, ds, STORE (ident, NONE, entry,
				       [{acl=SOME nacl, attribute=attribute,
					 value=NONE}], ivar'))
	   end)

	| execute (evts, ds as {acl, ...}, 
		   ACLset (ident, attribute, aclmod, ivar)) =
	let
	  val aclattr = case attribute of 
	    NONE => "dataset.acl"
	  | (SOME a) => "dataset.acl." ^ a

	  val oacl = acl (getOpt (attribute, "\000"))
	  val nacl = case aclmod of
	    (SETACL update) => Acl.update oacl update
	  | (DELACL (SOME id)) => Acl.update oacl (id, AclRights.empty)
	  | (DELACL NONE) => Acl.empty
	  | (GETACL) => raise (Config.Impossible "shouldn't happen")
	  val ivar' = SV.iVar ()
	  fun t () = (case SV.iGet ivar' of
			(Res _) => SV.iPut (ivar, Res nacl)
		      | (Exn exn) => SV.iPut (ivar, Exn exn))
	  val _ = CML.spawn t
	in
	  execute (evts, ds, STORE (ident, NONE, "",
			    [{attribute=aclattr, acl=NONE,
			      value=SOME (V.Single (Acl.toString nacl))}],
				    ivar'))
	end


	(* the dataset we're listening to just did something, so let's
	 reflect it in our dataset *)
	| execute (evts, ds, INFO x) = update (evts, ds, x)

	(* return our data and a port *)
	| execute (evts, ds as {idata, ...}, GETDATA iv) = 
	(SV.iPut (iv, (idata, MC.port channel)); 
	 inmem (evts, ds))

	(* we can flush if we don't have any listeners or if we aren't
	 inheriting from someone *)
	| execute (evts, ds as {inherits, port, ...}, FLUSH ()) = 
	if MC.listeners channel = 0 orelse not (isSome inherits)
	  then (TextIO.print (concat ["flushing ", dsetname, "\n"]);
		MC.release port;
		ondisk ([ctlevt]))
	else inmem (evts, ds)

	(* if we're being deleted, delete all subdatasets and stop processing
	 on this thread. *)
	| execute (_, _, DELETE) = ()

      (* adding an entry and changing one are fundamentally the same
       thing *)
      and update (evts, 
		  ds as {data, idata, 
			 inherits,
			 acl, port}, 
		  (Entry.NEW{dset, name, entry} | 
		   Entry.CHANGE{dset, name, entry})) =
	let
	  val base = EL.find (data, name)
	  val old = EL.find (idata, name)
	  val new = 
	    case base of
	      NONE => entry
	    | (SOME base) => Entry.union (entry, Entry.fillAcls acl base)
	  val idata = EL.insert (idata, name, new)
	in
	  (case old of
	    NONE => MC.multicast (channel, Entry.NEW{dset=dsetname,
						     name=name,
						     entry=new})
	  | (SOME _) => (MC.multicast (channel,
				       Entry.CHANGE{dset=dsetname,
						    name=name,
						    entry=new})));
	  inmem (evts, {data=data,
			idata=idata,
			inherits=inherits,
			acl=acl,
			port=port})
	end
	| update (evts, ds as {data, idata, 
			       inherits,
			       acl, port}, Entry.DELETE{dset, name}) =
	let
	  val base = EL.find (data, name)
	  val (idata, _) = EL.remove (idata, name)
	  val new = case base of 
	    NONE => NONE
	  | (SOME base) => SOME (Entry.fillAcls acl base)
	  val idata = case new of
	    NONE => idata
	  | (SOME new) => EL.insert (idata, name, new)
	in
	  (case new of
	     NONE => MC.multicast (channel, Entry.DELETE({dset=dsetname,
							  name=name}))
	   | (SOME new) => MC.multicast (channel, 
					 Entry.CHANGE ({dset=dsetname,
							name=name,
							entry=new})));
	  inmem (evts, {data=data,
			idata=idata,
			inherits=inherits,
			acl=acl,
			port=port})
	end

      and ondisk (evts) =
	let
	  (* block waiting on a request *)
	  val msg = CML.select evts

	  (* ok, we just got a request; let's read the dataset into memory *)
	  val data = readSet (dsetname)
	  val null = case EL.find (data, "") of
	    NONE => barf "no null entry for dataset"
	  | (SOME null) => null
	      
	  val (inherits, (idata, port)) = 
	    case Entry.getattr' null "dataset.inherit" of
	      NONE => (NONE, (EL.empty, MC.nullport))
	    | (SOME {value=V.Single inherit,...}) => fetchdata inherit
	    | (SOME {value=(V.Nil | V.Default),...}) => 
		(NONE, (EL.empty, MC.nullport))
	    (* dataset.inherit is malformed *)
	    | _ => (NONE, (EL.empty, MC.nullport))

	  val updateevt = CML.wrap (MC.recvEvt port, INFO)

	  (* xxx should give implicit "a" rights to dataset.owner *)
	  val acl = genAclFun null

	  (* calculate inheritance---first, fill in the acls on 
	   the existing data, and then do the entry-by-entry union *)
	  val idata = EL.unionWith Entry.union (idata, 
			 EL.map (Entry.fillAcls acl) data)
	in
	  execute ([updateevt, ctlevt, flushevt], {data=data,
						   idata=idata,
						   inherits=inherits,
						   port=port,
						   acl=acl}, msg)
	end

      and inmem (evts, ds) = execute (evts, ds, CML.select evts)

    in
      ignore (CML.spawnc ondisk ([CML.recvEvt ctl]))
    end

  (* create the dataset name *)
  and createDataset (path, name, auth, null, aclf) = 
    (let
       (* create the directory *)
       val dir = String.concat[!Config.FDIR_spool, path, name]
       val flags = Posix.FileSys.S.flags [Posix.FileSys.S.irwxu,
					  Posix.FileSys.S.irgrp,
					  Posix.FileSys.S.ixgrp,
					  Posix.FileSys.S.iroth,
					  Posix.FileSys.S.ixoth]
       val _ = Posix.FileSys.mkdir (dir, flags)

       (* create the new null entry *)

       (* adjust inheritance *)
       val null = 
	 case Entry.getattr' null "dataset.inherit" of
	   (SOME {value=V.Single inherit, ...}) => 
	     valOf (Entry.store Auth.master defaultAclf null
		    ([{acl=NONE, attribute="dataset.inherit",
		    value=SOME(V.Single(String.concat[inherit, name, "/"]))}],
		     NONE))
	 | _ => null
	 
       (* adjust owner *)
       val null = valOf (Entry.store Auth.master defaultAclf null
			([{acl=NONE,attribute="dataset.owner",
			  value=SOME (V.Single (Auth.toString auth))}],
			 NONE))

       (* create data, save it to disk *)
       val data = EL.singleton ("", null)
       val dsetname = String.concat [path, name, "/"]
       val _ = writeSet (dsetname, data, addE("", null))
       val channel = MC.mChannel ()
       val ctl = CML.channel ()

       (* lock dataset dict *)
       val mydict = DSetList.takedict ()

       (* announce creation of dataset *)
       val _ = DSetList.broadcast (dsetname, MC.port' channel)

       (* announce insertion of null entry *)
       val _ = MC.multicast (channel, 
			     Entry.NEW{dset=dsetname, name="",
				       entry=Entry.fillAcls aclf null})

       val mydict = Dict.insert (mydict, dsetname, Local ctl)

       val _ = DSetList.putdict mydict
     in
       (* start the dataset thread and return *)
       dataset (dsetname, ctl, channel)
     end
       handle (OS.SysErr (s, _)) => TextIO.print (String.concat 
			["error creating ", path, name, ": ", s]))

  (* create a new remote dataset accessible via urls *)
  and createRemote (path, name, urls) =
    let
      val dsetname = String.concat [path, name, "/"]
	 
      (* lock dset *)
      val mydict = DSetList.takedict ()
	
      (* find old dataset, if any *)
      val ctl = 
	case Dict.find (mydict, dsetname) of
	  (SOME (Local ctl)) => SOME ctl
	| _ => NONE
	    
      val mydict = Dict.insert (mydict, dsetname, Remote urls)
      val _ = DSetList.putdict mydict
      val _ = (* delete old dataset, if any *)
	case ctl of NONE => () | (SOME ctl) => CML.send (ctl, DELETE)
    in
      ()
    end

  (* delete the dataset name *)
  and deleteDataset (path, name) =
    (let
       val dsetname = String.concat [path, name, "/"]
	 
       (* lock dset *)
       val mydict = DSetList.takedict ()
	 
       (* find old dataset, if any *)
       val (mydict, old) = Dict.remove (mydict, dsetname)
	     
       val _ = DSetList.putdict (mydict)
       val _ = case old of (Local ctl) => CML.send (ctl, DELETE)
                         | _ => ()
     in
       ()
     end
       handle LibBase.NotFound => 
	 TextIO.print (String.concat ["erk! expected to find ", path, name,
				      " but didn't!\n"]))

  fun randServer () =
    let
      val rand = 
	Random.rand (134, Int32.toInt (Time.toSeconds (Time.now ()) div 2))

      fun loop () =
	(CML.send (randCh, Random.randInt rand);
	 loop ())
    in
      ignore (CML.spawn loop)
    end

  (* initialize the dictionary; read off of disk *)
  fun init () = 
    let
      val _ = randServer ()

      val _ = if DSetList.readDSets () then () else createRoot ()
      val mydict = DSetList.getdict ()
      fun start (name, Local ctl) = dataset (name, ctl, MC.mChannel ())
	| start (name, Remote _) = ()
      val _ = Dict.appi start mydict
    in
      ()
    end

  fun search ident (dset, depth, doinherit)
             sfun messenger (out, referral) finish =
    let
      (* figure out what datasets should be searched *)
      val countSlash = 
	(Substring.foldl (fn (c, a) => if c = #"/" then a+1 else a) 0)
      o (Substring.all)
      val allowed = depth + (countSlash dset)
      val sub = String.isPrefix dset
      val good = if depth = 0 then sub
		 else fn name => sub name andalso countSlash name < allowed

      fun check (SOME (messenger, pleaseDie)) =
	let
	  val port = DSetList.port ()
	  val dieEvt = CML.wrap (pleaseDie, CML.exit)
	  val msg = MC.recvEvt port

	  fun f (n, p) =
	    if good n then
	      messenger (n, MC.copy p)
	    else ()

	  fun loop () =
	    (f (CML.select [dieEvt, msg]);
	     loop ())
	in
	  ignore (CML.spawn loop)
	end
	| check (NONE) = ()

      (* grab a list of current datasets *)
      val mydict = DSetList.getdict ()

      (* make sure dset exists! *)
      val _ = if Dict.inDomain (mydict, dset) then ()
	      else raise NoSuchDataset

      (* start a thread to sort out new datasets *)
      val _ = if depth <> 1 then check messenger
	      else ()
		
      val stype = if doinherit then SEARCH else SEARCHnoinherit
      fun f (name, Local ctl) =
	if good name then 
	  let
	    val iv = SV.iVar ()
	    val _ = CML.send (ctl, stype (sfun, out, 
					  Option.map #1 messenger, iv))
	  in
	    sync iv
	  end 
	else ()
	| f (name, Remote urls) =
	if good name then referral (name, urls)
	else ()
    in
      Dict.appi f mydict;
      finish ()
    end

  (*********
   * STORE *
   *********
   * val store : Entry.Acl.Auth.ident 
   *                 -> string * bool * AcapTime.acaptime option
   *                 -> (string * Entry.storedata list)
   *                 -> unit
   *
   * find the dataset:
   * - if docreate is set, create if necessary, otherwise abort
   *
   * attempt to store to the relevant entry
   *)
  fun store ident (dset, docreate, time) (name, sd) = 
    let
      val _ = print ("starting store...\n")
      val mydict = DSetList.getdict ()

      fun createit' (mydict, dset) =
	let
	  val _ = TextIO.print (concat ["trying to create ",
					Substring.string dset, "\n"])

	  val _ = if Substring.size dset < 2 then 
	            raise (Config.Impossible "trying to create root")
		  else ()
	  val (dsetpath, entry) = Substring.splitr (fn n => n <> #"/") dset
				  
	  val d = Dict.find (mydict, Substring.string dsetpath)

	  val d = case d of
	                 NONE => Dict.find 
			   (createit' (mydict, Substring.trimr 1 dsetpath),
			    Substring.string dsetpath)
		       | x => x
	in
	  case d of
	    NONE => raise NoSuchDataset
	  | (SOME (Local ctl)) => 
	      let
		val sv = SV.iVar ()
	      in
		CML.send (ctl, STORE (ident, time, Substring.string entry, 
				      [{acl=NONE, attribute="subdataset",
					value=SOME (V.List ["."])}], sv));
		sync sv;
		DSetList.getdict ()
	      end
	  | (SOME (Remote urls)) => raise (DatasetRefer urls)
	end

      fun createit dset =
	let
	  val mydict =
	    createit' (mydict, 
		       Substring.substring (dset, 0, String.size dset - 1))
	in
	  case Dict.find (mydict, dset) of
	    NONE => raise NoSuchDataset
	  | (SOME (Local ctl)) => ctl
	  | (SOME (Remote urls)) => raise (DatasetRefer urls)
	end

      val ctl = 
	case Dict.find (mydict, dset) of
	  NONE => if docreate then createit dset
		  else raise NoSuchDataset
	| (SOME (Local ctl)) => ctl
	| (SOME (Remote urls)) => raise (DatasetRefer urls)
		    
      val iv = SV.iVar ()
    in
      CML.send (ctl, STORE (ident, time, name, sd, iv));
      sync iv
    end

  (* acl stuff *)
  type aclobj = {dataset : string,
		 attribute : string option,
		 entry : string option}

  (* val modifyacl : Entry.Acl.Auth.ident -> (aclobj * aclmod) -> 
                     Entry.Acl.acl *)
  fun modifyacl ident ({dataset, attribute, entry}, aclmod) = 
    let
      val mydict = DSetList.getdict ()

      val ctl = case Dict.find (mydict, dataset) of
	NONE => raise NoSuchDataset
      | (SOME (Remote urls)) => raise (DatasetRefer urls)
      | (SOME (Local ctl))  => ctl
	  
      val sv = SV.iVar ()
    in
      (case entry of
	NONE => CML.send (ctl, ACLset (ident, attribute, aclmod, sv))
      | (SOME e) => 
	  case attribute of
	    NONE => raise (SyntaxError "need attribute and entry")
	  | (SOME a) => CML.send (ctl, ACLentry (ident, e, a, aclmod, sv)));
      sync sv
    end
end
