(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

signature PARSE =
sig
  val create : (string CML.chan * Execute.conn) -> CML.thread_id
end

structure Parse :> PARSE =
struct
  structure AcapLrVals = AcapLrValsFun(structure Token = LrParser.Token)
  structure Lex = AcapLexFun(structure Tokens = AcapLrVals.Tokens)
  structure AcapP = Join(structure ParserData = AcapLrVals.ParserData
			 structure Lex = Lex
			 structure LrParser = LrParser)

  fun exnerror exn =
    String.concat ["Serious internal error: ",
		   (exnMessage exn),
		   " history: ",
		   (foldr (fn (s, acc) => (acc ^ s ^ ";")) "" 
		           (SMLofNJ.exnHistory exn))]

  fun create (inch, conn) =
    let 
      fun get _ = CML.recv inch
      val myerror = ref ""
      fun parseerror (s, p1, p2) = (myerror := s)
      val lexer = AcapP.makeLexer get
      val dummyEOF = AcapLrVals.Tokens.EOF(0,0)
      val dummyCRLF = AcapLrVals.Tokens.CRLF(0,0)

      fun errorAdvance (nextToken, lexer) =
	if AcapP.sameToken(nextToken, dummyCRLF) orelse
	  AcapP.sameToken(nextToken, dummyEOF) then (nextToken, lexer)
	else (* we want to advance the lexer until we get there *)
	   errorAdvance (AcapP.Stream.get lexer)

      fun doauth (conn, lexer, (saslconn, tag, mech, first, response)) =
	let
	  val lexr = ref lexer

	  fun getstring (Absyn.BareString s) = SOME s
	    | getstring _ = NONE

	  fun get () =
	    let val (absyn, lexer) = (AcapP.parse(0, !lexr, parseerror, ()))
	        val (nextToken, lexer) = AcapP.Stream.get lexer
		val (nextToken, lexer) = errorAdvance (nextToken, lexer)
	    in lexr := lexer;
	      getstring absyn
	    end
	in case 
	  Sasl.doauth saslconn (tag, mech, first) (get, response)
	  of (SOME user) => (Execute.authAs conn 
			          (Auth.fromString user), !lexr)
           | NONE => (conn, !lexr)
	end

      fun (*getTag (AcapLrVals.ParserData.Token.TOKEN 
		   (_, AcapLrVals.Token.TAG s)) = SOME s
	| *)getTag _ = NONE

      fun loop lexer conn = 
	(let
	   val _ = print "ready to parse line...\n"

	   val (absyn, lexer) = (AcapP.parse(0, lexer, parseerror, ())
				 handle AcapP.ParseError =>
				   let
				     val (tok, _) = AcapP.Stream.get lexer
				   in
				    (Absyn.Error (getTag tok, !myerror), lexer)
				   end
				      | Lex.LexError =>
				   (Absyn.Error (NONE, "unrecognized token"),
				    lexer))

	   (* get the character that ended our parse *)
	   val (nextToken, lexer) = AcapP.Stream.get lexer

	   val (nextToken, lexer) = errorAdvance(nextToken, lexer)

	   val (conn, lexer) = 
	     if AcapP.sameToken(nextToken, dummyEOF) then
	       (Execute.stop conn; CML.exit ())
	     else (Execute.run conn absyn, lexer)
	       handle Execute.CloseConnection =>
		 (print "user logged out\n";
		  Execute.stop conn;
		  CML.exit ())
		    | Execute.doAuth stuff => 
		 doauth (conn, lexer, stuff)
		    | exn => 
		 (print "caught exception...\n";
		  print ((exnMessage exn) ^ "\n");
		  Execute.error conn (exnerror exn);
		  CML.exit ())
	 in
	   loop lexer conn
	 end
	   handle e => print ((exnMessage e) ^ "\n"))
    in
      print "spawning parser...\n";
      CML.spawn (fn () => loop lexer conn)
    end
end
