(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

structure Response :> RESPONSE =
struct
  datatype okcode =
    SASL of string
  | TOOMANY of int

  datatype nocode =
    AUTHTOOWEAK
  | ENCRYPTNEEDED
  | INVALID of (string * string) (* invalid entry/attr *)
  | MODIFIED of string (* entry modified *)
  | NOEXIST of string (* dataset *)
  | PERMISSION of string
  | QUOTACODE
  | REFERCODE of string list
  | TOOOLD
  | TRANSITIONNEEDED
  | TRYLATER
  | TRYFREECONTEXT
  | WAYTOOMANY

  datatype response =
    OK of string option * okcode list * string
  | NO of string option * nocode list * string
  | BAD of string option * string
  | CONTINUE of string
  | ALERT of string
  | BYE of string
  | ADDTO of string * string * int * string
  | CHANGE of string * string * int * int * string
  | DELETED of string * string (* tagged *)
  | ENTRY of string * string * string (* tagged *)
  | LANG of string
  | LISTRIGHTS of string * string * string list (* tagged *)
  | TAGMODTIME of string * string (* tagged *)
  | MODTIME of string * string
  | MYRIGHTS of string * string
  | QUOTA of string * string * string
  | REFER of string * string * string list
  | REMOVEFROM of string * string * int

  val formatSL = List.foldl (fn (s, acc) => " \"" ^ s ^ "\"" ^ acc) ""

  fun formatOK nil out = out
    | formatOK ((SASL s)::l) out = formatOK l (("SASL " ^ s)::out)
    | formatOK ((TOOMANY i)::l) out = formatOK l (("TOOMANY " ^ 
						   (Int.toString i))::out)

  fun formatNO nil out = out
    | formatNO (AUTHTOOWEAK::l) out = formatNO l ("AUTH-TOO-WEAK"::out)
    | formatNO (ENCRYPTNEEDED::l) out = formatNO l ("ENCRYPT-NEEDED"::out)
    | formatNO ((INVALID (e, a))::l) out = formatNO l 
                             (("INVALID \"" ^ e ^ "\" \"" ^ a ^ "\"")::out)
    | formatNO ((MODIFIED e)::l) out = formatNO l (("MODIFIED " ^ e)::out)
    | formatNO ((NOEXIST d)::l) out = formatNO l (("NOEXIST " ^ d)::out)
    | formatNO ((PERMISSION s)::l) out = formatNO l (("PERMISSION " ^s)::out)
    | formatNO ((QUOTACODE)::l) out = formatNO l ("QUOTA"::out)
    | formatNO ((REFERCODE sl)::l) out = formatNO l
        (("REFER" ^ (formatSL sl)):: out)
    | formatNO (TOOOLD::l) out = formatNO l ("TOOOLD"::out)
    | formatNO (TRANSITIONNEEDED::l) out = formatNO l
	                         ("TRANSITION-NEEDED"::out)
    | formatNO (TRYLATER::l) out = formatNO l ("TRYLATER"::out)
    | formatNO (TRYFREECONTEXT::l) out = formatNO l ("TRYFREECONTEXT"::out)
    | formatNO (WAYTOOMANY::l) out = formatNO l ("WAYTOOMANY"::out)

  fun code f nil = ""
    | code f l = 
    let
      val l = f l nil
    in
      case l of
	nil => ""
      | [a] => "(" ^ a ^ ") "
      | a::l => "(" ^ a ^ (List.foldl (fn (s, acc) => " " ^ s ^ acc) ") " l)
    end

  fun format (OK(NONE, l, s)) = 
    ["* OK ", code formatOK l, "\"", s, "\"\r\n"]

    | format (OK(SOME t, l, s)) = 
    [t, " OK ", code formatOK l, "\"", s, "\"\r\n"]

    | format (NO(NONE, l, s)) = 
    ["* NO ", code formatNO l, "\"", s, "\"\r\n"]

    | format (NO(SOME t, l, s)) = 
    [t, " NO ", code formatNO l, "\"", s, "\"\r\n"]

    | format (BAD(NONE, s)) = 
    ["* BAD \"", s, "\"\r\n"]

    | format (BAD(SOME t, s)) =
    [t, " BAD \"", s, "\"\r\n"]

    | format (CONTINUE "") =
    ["+ \"\"\r\n"]

    | format (CONTINUE s) = 
    ["+ ", s, "\r\n"]

    | format (ALERT s) = 
    ["* ALERT \"", s, "\"\r\n"]
    
    | format (BYE s) = 
    ["* BYE \"", s, "\"\r\n"]
    
    | format (ADDTO (s1, s2, i, s3)) = 
    ["* ADDTO \"", s1, "\" \"", s2, "\" ", Int.toString i, " ", s3, "\r\n"]

    | format (CHANGE (s1, s2, i3, i4, s5)) = 
    ["* CHANGE \"", s1, "\" \"", s2, "\" ", Int.toString i3, " ",
     Int.toString i4, " ", s5, "\r\n"]

    | format (DELETED (s1, s2)) = 
    [s1, " DELETED ", s2, "\r\n"]

    | format (ENTRY (s1, s2, "")) = 
    [s1, " ENTRY \"", s2, "\"\r\n"]

    | format (ENTRY (s1, s2, s3)) = 
    [s1, " ENTRY \"", s2, "\" ", s3, "\r\n"]
    
    | format (LANG s) = 
    ["* LANG ", s, "\r\n"]

    | format (LISTRIGHTS (s1, s2, sl3)) = 
    [s1, " LISTRIGHTS \"", s2, "\"", formatSL sl3, "\r\n"]

    | format (TAGMODTIME (s1, s2)) = 
    [s1, " MODTIME \"", s2, "\"\r\n"]

    | format (MODTIME (s1, s2)) = 
    ["* MODTIME \"", s1, "\" \"", s2, "\"\r\n"]

    | format (MYRIGHTS (s1, s2)) = 
    [s1, " MYRIGHTS \"", s2, "\"\r\n"]
    
    | format (QUOTA (s1, s2, s3)) = 
    ["* QUOTA ", s1, " ", s2, " ", s3, "\r\n"]

    | format (REFER (s1, s2, sl3)) = 
    [s1, " REFER ", s2, formatSL sl3, "\r\n"]

    | format (REMOVEFROM (s1, s2, i)) = 
    ["* REMOVEFROM \"", s1, "\" \"", s2, "\" ", Int.toString i, "\r\n"]

  fun create (outchan, pleaseDie, banner) =
    let
      val inchan = Mailbox.mailbox ()

      val evts = [CML.wrap ((Mailbox.recvEvt inchan), SOME),
		  CML.wrap ((SyncVar.iGetEvt pleaseDie, fn () => NONE))]
      fun printLoop () = 
	case CML.select evts
	  of (SOME response) => (CML.send (outchan, 
					   String.concat (format response));
				 printLoop ())
	   | NONE => (TextIO.print "response thread dying\n")
	    
      fun print () = (CML.send (outchan, banner);
		      printLoop ())

    in
      CML.spawn(print);
      inchan
    end

  fun create (outchan, pleaseDie, banner) =
    let
      val inchan = CML.channel ()

      val evts = [CML.wrap ((CML.recvEvt inchan), SOME),
		  CML.wrap ((SyncVar.iGetEvt pleaseDie, fn () => NONE))]
      fun printLoop () = 
	case CML.select evts
	  of (SOME response) => (CML.send (outchan,
					   String.concat (format response));
				 printLoop ())
	   | NONE => (TextIO.print "response thread dying\n")
	    
      fun print () = (CML.send (outchan, banner);
		      printLoop ())

    in
      CML.spawn(print);
      inchan
    end

end
