/* authize.c -- authorization module for IMSP
 *
 *	(C) Copyright 1993-1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman <chrisn+@cmu.edu>
 * Start Date: 6/28/93
 */

#include <stdio.h>
#include <string.h>
#include "util.h"
#include "acl.h"
#include "syncdb.h"
#include "option.h"
#include "auth.h"
#include "authize.h"

/* from OS: */
extern char *malloc(), *realloc();
extern char *crypt();

/* name of anonymous user */
static char anonymous[] = "anonymous";

/* standard options for setting authentication levels */
static char opt_subsadmin[] = "imsp.admin.subs";
static char opt_bbadmin[]   = "imsp.admin.bboards";
static char opt_alladmin[]  = "imsp.admin.all";

/* standard error messages */
static char err_nomem[] = "Not enough memory to log in";
static char err_nologin[] = "Login failed";

/* generate an auth_id from a login-name and password.
 *  "idptr" points to the current authentication for the user (which may be
 *               NULL, and updates it as appropriate.  No change is made on
 *               a failed login.
 *  "user" is the IMSP login name, and points to a dynamically allocated
 *               string which may be freed by the caller.
 *  "pass" is what the user entered for the password field.  It will be
 *               zeroed and freed by the caller upon return.  If NULL,
 *               authentication is already completed.
 *  "reply" is set to a user readable string which may not contain CR or LF
 *               characters.
 *  Returns -1 for login failed, 0 for success
 */
int auth_login(idptr, user, pass, reply)
    auth_id **idptr;
    char *user, *pass, **reply;
{
    int loginok = 0;
    char *rptr;
    auth_id *id = *idptr;
    static char replybuf[256];

    /* check password */
    if (!pass) {
	loginok = 2;
    } else if (!*pass && id
	       && (id->level == AUTH_ADMIN || id->level == AUTH_SUBS)) {
	loginok = 3;
#ifdef ANONYMOUS_LOGIN
    } else if (!strcmp(anonymous, user)) {
	loginok = 2;
#endif
    } else if (login_plaintext(user, pass, reply) == 0) {
	loginok = 2;
    }

    /* make space for auth_id and initialize it */
    if (loginok > 0) {
	if (!id) {
	    *idptr = id = (auth_id *) malloc(sizeof (auth_id));
	    id->level = AUTH_NONE;
	}
	if (!id) {
	    loginok = -1;
	} else {
	    strcpy(id->user, user);
	    strcpy(id->pass, pass ? pass : "");
	    auth_setid(id->user);
	}
    }

    /* set the reply */
    switch (loginok) {
	case -1:		/* memory error */
	    *reply = err_nomem;
	    break;
	case 0:			/* auth failed */
	    *reply = err_nologin;
	    break;
	case 1:			/* auth succeeded, reply already set */
	    break;
	case 2:			/* standard login */
	    strcpy(replybuf, "User `");
	    rptr = beautify_copy(replybuf + strlen(replybuf), user);
	    strcpy(rptr, "' Logged in");
	    *reply = replybuf;
	    break;
	case 3:			/* administrator switch-user */
	    strcpy(replybuf, "Administrator switch to user `");
	    rptr = beautify_copy(replybuf + strlen(replybuf), user);
	    strcpy(rptr, "' successful.");
	    *reply = replybuf;
	    break;
    }

    /* determine the authorization level */
    if (loginok > 0 && id->level != AUTH_ADMIN && id->level != AUTH_SUBS) {
	id->level = AUTH_USER;
	if (option_lookup("", opt_subsadmin, 1, id->user)) {
	    id->level = AUTH_SUBS;
	}
	if (option_lookup("", opt_bbadmin, 1, id->user)) {
	    id->level = AUTH_BB;
	}
	if (option_lookup("", opt_alladmin, 1, id->user)) {
	    id->level = AUTH_ADMIN;
	}
    }

    return (loginok > 0 ? 0 : -1);
}

/* this frees any resources used by an auth_id, it must zero out passwords
 *  and will be called before the program exits.  If the argument is NULL,
 *  no action should be taken.
 */
void auth_free(id)
    auth_id *id;
{
    if (id) {
	bzero(id->pass, sizeof (id->pass));
	free((char *)id);
    }
}

/* Return the user name of an identity.  The user name is used to identify
 * the appropriate mailboxes/bb-subscriptions/options/address-book/etc to
 * use.  It is also used for error messages.  The string may not be longer
 * than AUTH_USERMAX.
 * If "id" is NULL, this should return a string representing an anonymous user.
 */
char *auth_username(id)
    auth_id *id;
{
    return (id ? id->user : anonymous);
}

/* return the access level for a given id
 */
int auth_level(id)
    auth_id *id;
{
    return (id ? id->level : AUTH_NONE);
}

/* get login name & password for use with a proxy IMAP command
 *  returns -1 on failure, 0 on success
 */
int auth_proxy(id, host, user, pass)
    auth_id *id;
    char *host, **user, **pass;
{
    /* if no auth, try anonymous proxy */
    if (!id || id->level == AUTH_NONE) {
	*user = "anonymous";
	*pass = "imap";
	return (0);
    }
    *user = id->user;
    *pass = id->pass;

    return (0);
}
