/* main.c -- main entry point for imsp server
 *
 *	(C) Copyright 1993-1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman <chrisn+@cmu.edu>
 * Start Date: 2/16/93
 */

#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "version.h"
#include "dispatch.h"
#include "imsp.h"
#include "imsp_routines.h"

struct sockaddr_in imspd_localaddr, imspd_remoteaddr;

static char forkfailed[] = "* BYE IMSP server is currently overloaded\015\012";

/* cleanup a child
 */
static void cleanup_child()
{
    while (waitpid(-1, NULL, WNOHANG) > 0);
}

/* start server socket
 */
static void start_server()
{
    int sock, len, pid, newfd, socksz, tries = 0;
    struct sockaddr_in server;
    struct servent *svent;
    struct hostent *hent;
    char *iaddr;
    char host[MAXHOSTNAMELEN];

    /* fork a new process and for the server and establish a new session id */
#ifndef DEBUG
    pid = fork();
    if (pid == 0) {                   /* we are the child */
        setsid((pid_t)0);	/* disassociate session */
    } else if (pid < 0) {
	(void) fprintf(stderr, "imspd: failed to fork server process\n");
        perror("fork");
	exit(1);
    } else {			/* parent process - quit */
	printf("IMSP server started\n");
	exit(0);
    }
#endif

    /* set up signal handler for child processes */
    signal(SIGCHLD, cleanup_child);

    /* open IMSP service port */
    sock = socket(AF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
	perror("socket");
	exit(1);
    }
    server.sin_family = AF_INET;
    server.sin_addr.s_addr = htonl(INADDR_ANY);
    svent = getservbyname(IMSP_PORTNAME, IMSP_PROTOCOL);
    server.sin_port = svent ? svent->s_port : htons(IMSP_PORT);
    while (bind(sock, (struct sockaddr *) &server, sizeof (server)) < 0) {
	if (errno != EADDRINUSE || ++tries > 30) {
	    perror("bind");
	    exit(1);
	}
	sleep(1);
    }
    (void) listen(sock, 5);
#ifdef DEBUG
    printf("server ready\n");
#endif
    for (;;) {
	/* wait for connection */
	if (dispatch_loop(sock, 0) < 0) {
	    perror("select");
	    exit(1);
	}

	/* accept connection */
	len = sizeof (struct sockaddr_in);
	newfd = accept(sock, (struct sockaddr *) &server, &len);
	if (newfd < 0) {
	    perror("accept");
	    return;
	}

	/* find out hostname of client */
	strcpy(host, "unknown-host");
	socksz = sizeof (struct sockaddr_in);
	getsockname(newfd, (struct sockaddr *) &imspd_localaddr, &socksz);
	socksz = sizeof (struct sockaddr_in);
	getpeername(newfd, (struct sockaddr *) &imspd_remoteaddr, &socksz);
	if ((iaddr = inet_ntoa(imspd_remoteaddr.sin_addr))) {
	    strcpy(host, iaddr);
	}
	if ((hent = gethostbyaddr((char *) &imspd_remoteaddr.sin_addr,
				  sizeof (struct in_addr), AF_INET)) != NULL) {
	    strcpy(host, hent->h_name);
	}

	/* fork server process */
#ifndef DEBUG
	pid = fork();
	if (pid == 0) {
	    (void) close(sock);
	    im_start(newfd, host);
	    exit(0);
	} else if (pid < 0) {
	    (void) write(newfd, forkfailed, sizeof (forkfailed));
	    perror("fork");
	}
	(void) close(newfd);
#else
	(void) close(sock);
	im_start(newfd, host);
#endif
    }
}

main()
{
    printf("Cyrus IMSP server version %s\n", VERSION);
    if (sdb_init() < 0) {
	fprintf(stderr, "imspd: Failed to initialize db module.\nYou may need to create the PREFIX directory as defined in syncdb.c\n");
	exit(1);
    }
    sdb_create("abooks");
    dispatch_init();
    if (sdb_check("mailboxes") < 0) bb_init();
    start_server();
}
