/* syncdb.c -- synchronized data base access for IMSP via disk files & locking
 *
 *	(C) Copyright 1993-1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman <chrisn+@cmu.edu>
 * Start Date: 3/25/93
 *
 * Topical Notes:
 * 
 * Memory usage
 * ------------
 * On a low memory machine, it might be useful to try to free cache entries
 * before giving up with a memory error.
 *
 * Setting values
 * --------------
 * sdb_set is currently quite inefficient, since it writes the entire file out
 * then reads it back in.  A tradeoff of code complexity for efficiency here
 * is probably in order.
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/param.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include "util.h"
#include "syncdb.h"
#include "glob.h"

/* from OS: */
extern char *malloc(), *realloc();
extern int strcasecmp(), strncasecmp();

/* prefixes for database files */
#define PREFIX		"/var/imsp"
#define PREFIXLEN	(sizeof (PREFIX) - 1)
#define PRIVPREFIX	"user"
#define PRIVPREFIXLEN	(sizeof (PRIVPREFIX) - 1)

/* maximum accepted length for a database specifier or path */
#define MAXDBLEN	256
#define MAXDBPATHLEN	(PREFIXLEN + MAXDBLEN)

/* a file cache */
typedef struct cache {
    unsigned long mtime;	/* last modified time */
    int loaded;			/* 0 = unloaded, 1 = loaded */
    int fd;			/* fd if already opened */
    int locks;			/* number of locks on db */
    int icase;			/* case insensitive flag for cache */
    char *data;			/* pointer to file in memory */
    unsigned long size;		/* size of file in memory */
    sdb_keyvalue *kv;		/* array of key/value pairs from file */
    int count;			/* number of key/value pairs in file */
    char db[MAXDBPATHLEN];	/* path to database */
} cache;

/* valid databases and caches */
static char *globdbstr[] = {
    "options", "mailboxes", "new", "changed", "abooks", NULL
};
static char *privdbstr[] = {
    "options", "mailboxes", "subs", "alock",
    "abooks", "abook", "abookacl", NULL
};
#define NUMGDBSTR	(sizeof (globdbstr) / sizeof (char *) - 1)
#define NUMPDBSTR	(sizeof (privdbstr) / sizeof (char *) - 1)
#define PDBPREFIXPOS    (NUMPDBSTR - 2)
static cache globdb[NUMGDBSTR];
static cache privdb[NUMPDBSTR];

/* lock file extension */
static char newext[] = "%s..";

/* free a cache (assume it's unlocked)
 */
static void freecache(c)
    cache *c;
{
    if (c->count) {
	free((char *)c->kv);
	c->count = 0;
	c->kv = NULL;
    }
    if (c->size) {
	free(c->data);
	c->size = 0;
	c->data = NULL;
    }
    c->loaded = 0;
}

/* find a cache
 */
static cache *findcache(db)
    char *db;
{
    char *scan;
    int i;
    
    if (!strncmp(db, PRIVPREFIX, PRIVPREFIXLEN) && db[PRIVPREFIXLEN] == '/') {
	scan = strchr(db + PRIVPREFIXLEN + 1, '/');
	if (scan == NULL) return (NULL);
	++scan;
	for (i = 0; privdbstr[i]; ++i) {
	    if (!strcmp(scan, privdbstr[i]) ||
		(i >= PDBPREFIXPOS
		 && !strncmp(scan, privdbstr[i], strlen(privdbstr[i])))) {
		if (strcmp(privdb[i].db + PREFIXLEN + 1, db)) {
		    if (privdb[i].locks) return (NULL);
		    freecache(&privdb[i]);
		    sprintf(privdb[i].db, "%s/%s", PREFIX, db);
		}
		return (&privdb[i]);
	    }
	}
	return (NULL);
    }
    for (i = 0; globdbstr[i]; ++i) {
	if (!strcmp(db, globdbstr[i])) {
	    sprintf(globdb[i].db, "%s/%s", PREFIX, db);
	    return (&globdb[i]);
	}
    }
    return (NULL);
}

/* key compare functions for the sdb_keyvalue structure
 */
static int keycmp(kv1, kv2)
    sdb_keyvalue *kv1, *kv2;
{
    return (strcmp(kv1->key, kv2->key));
}
static int ikeycmp(kv1, kv2)
    sdb_keyvalue *kv1, *kv2;
{
    return (strcasecmp(kv1->key, kv2->key));
}

/* parse the data in a cache
 * returns -1 on failure, 0 on success
 */
static int parsecache(c, flags)
    cache *c;
    int flags;
{
    int sorted = 1, lines = 0;
    char *scan, *dst;
    sdb_keyvalue *kv;
    int (*cmpf)() = flags & SDB_ICASE ? strcasecmp : strcmp;

    /* set a sentinel */
    c->data[c->size] = '\0';

    /* count lines */
    for (scan = c->data; *scan; ++scan) {
	if (*scan == '\n') ++lines;
    }
    if (scan[-1] != '\n') ++lines;
    *scan = '\n';

    /* make space */
    c->kv = (sdb_keyvalue *) malloc(lines * sizeof (sdb_keyvalue));
    if (c->kv == NULL) return (-1);
    c->count = lines;

    /* parse it */
    scan = c->data;
    for (kv = c->kv; lines; --lines, ++kv) {
	/* get the key, handling quoted characters */
	kv->key = scan;
	for (dst = scan; *scan != ' ' && *scan != '\n'; ++scan) {
	    if (*scan == '\\') {
		if (*++scan == 'n') {
		    *dst++ = '\n';
		    continue;
		} else if (*scan == 's') {
		    *dst++ = ' ';
		    continue;
		}
	    }
	    *dst++ = *scan;
	}

	/* get the value */
	if (*scan == '\n') {
	    *dst = '\0';
	    kv->value = NULL;
	} else {
	    *dst = '\0';
	    kv->value = ++scan;
	    for (dst = scan; *scan != '\n'; ++scan) {
		if (*scan == '\\') {
		    if (*++scan == 'n') {
			*dst++ = '\n';
			continue;
		    } else if (*scan == 's') {
			*dst++ = ' ';
			continue;
		    }
		}
		*dst++ = *scan;
	    }
	    *dst = '\0';
	}
	++scan;
	if (sorted && kv != c->kv && (*cmpf)(kv[-1].key, kv->key) > 0) {
	    sorted = 0;
	}
    }

    /* sort it, only if necessary */
    if (!sorted) {
	qsort(c->kv, c->count, sizeof (sdb_keyvalue),
	      flags & SDB_ICASE ? ikeycmp : keycmp);
    }
    c->icase = flags & SDB_ICASE;

    return (0);
}

/* refresh a cache
 * returns -1 on error, 0 on success
 */
static int refreshcache(c, flags)
    cache *c;
    int flags;
{
    int fd, rtval, count;
    struct stat stbuf;

    if (c->loaded && flags & SDB_QUICK) return (0);
    rtval = 0;
    if (!c->loaded || flags & SDB_ICASE != c->icase ||
	(stat(c->db, &stbuf) >= 0 && stbuf.st_mtime != c->mtime)) {
	freecache(c);

	/* open file */
	if ((!c->locks || (fd = c->fd) < 0)
	    && (fd = open(c->db, c->locks ? O_RDWR|O_CREAT : O_RDONLY,
			  0600)) < 0) {
	    rtval = -1;
	} else {
	    /* lock file if necessary and stat it */
	    if (c->locks && c->fd < 0) {
		if (lock_reopen(c->fd = fd, c->db, &stbuf, NULL) < 0) {
		    c->locks = 0;
		    rtval = -1;
		}
	    } else if ((c->locks && lseek(fd, 0, SEEK_SET) < 0)
		       || fstat(fd, &stbuf) < 0) {
		rtval = -1;
	    }

	    /* catch zero-length files */
	    if (rtval == 0 && stbuf.st_size == 0) {
		c->loaded = 1;
		c->icase = flags & SDB_ICASE;
		c->size = 0;
		c->count = 0;
	    } else {

		/* make space for data */
		c->data = malloc(stbuf.st_size + 1);
		if (c->data == NULL) {
		    rtval = -1;
		} else {

		    /* read data from file */
		    count = read(fd, c->data, stbuf.st_size);
		    if (count != stbuf.st_size) {
			free(c->data);
			rtval = -1;
		    } else {
			c->size = stbuf.st_size;
		    }
		}
	    }
	    if (!c->locks) close(fd);
	}

	/* parse the cached file */
	if (c->size && rtval == 0) {
	    if (parsecache(c, flags) < 0) {
		freecache(c);
		rtval = -1;
	    } else {
		c->loaded = 1;
		c->mtime = stbuf.st_mtime;
	    }
	}
    }

    return (rtval);
}

/* rewrite the database file
 */
static int rewritedb(c)
    cache *c;
{
    FILE *out;
    int i;
    char *scan;
    char newname[MAXDBPATHLEN + 5];

    /* open new file */
    sprintf(newname, newext, c->db);
    if ((out = fopen(newname, "w")) == NULL) {
	freecache(c);
	return (-1);
    }

    /* write updated contents encoded */
    for (i = 0; i < c->count; ++i) {
	for (scan = c->kv[i].key; *scan; ++scan) {
	    if (*scan == '\n') {
		putc('\\', out);
		putc('n', out);
	    } else if (*scan == ' ') {
		putc('\\', out);
		putc('s', out);
	    } else if (*scan == '\\') {
		putc('\\', out);
		putc('\\', out);
	    } else {
		putc(*scan, out);
	    }
	}
	putc(' ', out);
	if (c->kv[i].value) for (scan = c->kv[i].value; *scan; ++scan) {
	    if (*scan == '\n') {
		putc('\\', out);
		putc('n', out);
	    } else if (*scan == '\\') {
		putc('\\', out);
		putc('\\', out);
	    } else {
		putc(*scan, out);
	    }
	}
	putc('\n', out);
    }

    /* make sure write & rename succeed */
    if (fclose(out) == EOF || rename(newname, c->db) < 0) {
	unlink(newname);
	freecache(c);
	return (-1);
    }

    freecache(c);
    if (c->locks) {
	lock_unlock(c->fd);
	close(c->fd);
	c->fd = -1;
    }
    refreshcache(c, c->icase);
    
    return (0);
}

/* initialize sdb module (add to synchronization)
 * returns -1 on failure, 0 on success
 */
int sdb_init()
{
    int i;
    struct stat stbuf;
    char path[MAXPATHLEN];

    /* initialize cache */
    for (i = 0; i < NUMGDBSTR; ++i) {
	memset(globdb[i].db, 0, sizeof (globdb[i].db));
	globdb[i].loaded = 0;
	globdb[i].locks = 0;
	globdb[i].size = 0;
	globdb[i].count = 0;
    }
    for (i = 0; i < NUMPDBSTR; ++i) {
	memset(privdb[i].db, 0, sizeof (privdb[i].db));
	privdb[i].loaded = 0;
	privdb[i].locks = 0;
	privdb[i].size = 0;
	privdb[i].count = 0;
    }

    /* initialize directories */
    sprintf(path, "%s/%s", PREFIX, PRIVPREFIX);
    if (stat(path, &stbuf) < 0) {
	mkdir(PREFIX, 0700);
	if (mkdir(path, 0700) < 0) {
	    return (-1);
	}
    }

    return (0);
}

/* release any resources used by sdb module (remove from synchronization)
 */
void sdb_done()
{
    int i;
    cache *c;

    /* free caches */
    for (i = 0; i < NUMGDBSTR; ++i) {
	c = globdb + i;
	if (c->locks) {
	    lock_unlock(c->fd);
	    close(c->fd);
	    c->locks = 0;
	}
	freecache(c);
    }
    for (i = 0; i < NUMPDBSTR; ++i) {
	c = privdb + i;
	if (c->locks) {
	    lock_unlock(c->fd);
	    close(c->fd);
	    c->locks = 0;
	}
	freecache(c);
    }
}

/* check if a database exists
 *  returns 0 if exists, -1 otherwise
 */
int sdb_check(db)
    char *db;
{
    cache *c;
    struct stat stbuf;
    
    if ((c = findcache(db)) == NULL) return (-1);
    
    return ((c->loaded || stat(c->db, &stbuf) >= 0) ? 0 : -1);
}

/* create a new database.  fails if database exists or isn't createable.
 * returns -1 on failure, 0 on success
 */
int sdb_create(db)
    char *db;
{
    int fd;
    char *scan;
    cache *c;
    struct stat stbuf;

    /* verify that db is valid & doesn't exist */
    if ((c = findcache(db)) == NULL || c->loaded || stat(c->db, &stbuf) >= 0) {
	return (-1);
    }

    /* make user's subdirectory if needed */
    if (!strncmp(db, PRIVPREFIX, PRIVPREFIXLEN)) {
	scan = strchr(c->db + PREFIXLEN + PRIVPREFIXLEN + 2, '/');
	if (scan == NULL) return (-1);
	*scan = '\0';
	mkdir(c->db, 0700);
	*scan = '/';
    }

    /* create datafile */
    if ((fd = open(c->db, O_WRONLY | O_CREAT, 0600)) < 0 || close(fd) < 0) {
	return (-1);
    }

    return (0);
}

/* delete a database.  fails if database isn't deletable.
 * returns -1 on failure, 0 on success
 */
int sdb_delete(db)
    char *db;
{
    char lname[MAXDBPATHLEN];
    cache *c;

    /* remove it */
    if ((c = findcache(db)) == NULL || c->locks || unlink(c->db) < 0) {
	return (-1);
    }
    freecache(c);

    /* try removing user directory for cleanliness sake */
    if (!strncmp(db, PRIVPREFIX, PRIVPREFIXLEN)) {
	strcpy(lname, c->db);
	*strrchr(lname, '/') = '\0';
	rmdir(lname);
    }

    return (0);
}


/* copy the contents of one database to another
 *  returns -1 on failure, 0 on success
 */
#ifdef __STDC__
int sdb_copy(char *dbsrc, char *dbdst, int flags)
#else
int sdb_copy(dbsrc, dbdst, flags)
    char *dbsrc, *dbdst;
    int flags;
#endif
{
    cache *csrc, *cdst;
    int fd, result;

    /* make sure files are valid and unlocked */
    if ((csrc = findcache(dbsrc)) == NULL || csrc->locks
	|| (cdst = findcache(dbdst)) == NULL || cdst->locks) {
	return (-1);
    }
    
    /* grab lock on destination */
    if ((fd = open(cdst->db, O_RDWR|O_CREAT, 0600)) < 0) return (-1);
    if (lock_reopen(fd, cdst->db, NULL, NULL) < 0) {
	close(fd);
	return (-1);
    }

    /* read in source */
    if (refreshcache(csrc, flags) < 0) {
	lock_unlock(fd);
	close(fd);
	return (-1);
    }

    /* change cache to other file & write it */
    strcpy(csrc->db, cdst->db);
    result = rewritedb(csrc);
    lock_unlock(fd);
    close(fd);

    return (result);
}

/* get value of a key
 * on return, value points to a string which shouldn't be modified and may
 * change on future sdb_* calls.
 * returns -1 on failure, 0 on success
 */
#ifdef __STDC__
int sdb_get(char *db, char *key, int flags, char **value)
#else
int sdb_get(db, key, flags, value)
    char *db, *key;
    int flags;
    char **value;
#endif
{
    cache *c;
    sdb_keyvalue *kv;

    /* get db in cache */
    if ((c = findcache(db)) == NULL || refreshcache(c, flags) < 0) {
	return (-1);
    }

    /* if db empty, return no value */
    if (!c->count) {
	*value = NULL;
	return (0);
    }

    /* do binary search */
    kv = kv_bsearch(key, c->kv, c->count,
		    flags & SDB_ICASE ? strcasecmp : strcmp);
    *value = kv ? kv->value : NULL;

    return (0);
}

/* count the number of keys in a database
 *  returns -1 on failure, number of keys on success
 */
#ifdef __STDC__
int sdb_count(char *db, int flags)
#else
int sdb_count(db, flags)
    char *db;
    int flags;
#endif
{
    cache *c;

    /* get db in cache */
    if ((c = findcache(db)) == NULL || refreshcache(c, flags) < 0) {
	return (-1);
    }

    return (c->count);
}

/* check if a value matches a value pattern
 *  return 0 for match, 1 for no match, -1 for error
 */
static int valuematch(vpat, value)
    char *vpat, *value;
{
    glob *vg;
    int result;
    
    if (vpat == NULL) return (0);
    /*XXX: ?need to do international string check here */
    if ((vg = glob_init(vpat, 0)) == NULL) return (-1);
    result = GLOB_TEST(vg, value);
    glob_free(&vg);

    return (result >= 0 ? 0 : 1);
}

/* get keys & values that match a key wildcard (using '*' and '%')
 *  kv is set to a key/value array, count is set to the number of items in kv
 *  Caller must call sdb_freematch with kv when done.
 *  If the copy flag is 1, all data returned will be copied and may be used
 *  indefinitely.  If the copy flag is 0, then only the kv array will be
 *  copied as necessary.  If copy flag is 0, the data may become invalid on any
 *  future sdb_* call.
 * returns -1 on failure, 0 on success
 */
#ifdef __STDC__
int sdb_match(char *db, char *key, int flags, char *vpat,
	      int copy, sdb_keyvalue **pkv, int *count)
#else
int sdb_match(db, key, flags, vpat, copy, pkv, count)
    char *db, *key;
    int flags, copy;
    char *vpat;
    sdb_keyvalue **pkv;
    int *count;
#endif
{
    char *scan, *value;
    cache *c;
    sdb_keyvalue *ksrc, *kdst;
    glob *g, *vg;
    int gcount, copysize;

    /* if vpat is "*", set it to NULL */
    if (vpat != NULL && vpat[0] == '*' && vpat[1] == '\0') vpat = NULL;

    /* initialize to empty */
    *pkv = NULL;
    *count = 0;

    /* get db in cache */
    if ((c = findcache(db)) == NULL
	|| refreshcache(c, flags & GLOB_ICASE) < 0) {
	return (-1);
    }

    /* if db empty, return no match */
    if (!c->count) return (0);
    
    /* special case for no wildcards */
    for (scan = key; *scan != '*' && *scan != '%' && *scan != '?' && *scan;
	 ++scan);
    if (!*scan) {
	/* do binary search */
	ksrc = kv_bsearch(key, c->kv, c->count,
			  flags & GLOB_ICASE ? strcasecmp : strcmp);
	if (ksrc && valuematch(vpat, value = ksrc->value) == 0) {
	    key = ksrc->key;
	    kdst = *pkv = (sdb_keyvalue *)
		malloc(sizeof (sdb_keyvalue)
		       + (copy ? strlen(key) + strlen(value) + 2 : 0));
	    if (kdst == NULL) return (-1);
	    kdst->key = key;
	    kdst->value = value;
	    *count = 1;
	    if (copy) {
		scan = (char *) (kdst + 1);
		kdst->key = scan;
		while ((*scan++ = *key++));
		kdst->value = scan;
		while ((*scan++ = *value++));
	    }
	}
	return (0);
    }

    /* set key to NULL if it's a "*" */
    if (key[0] == '*' && key[1] == '\0') key = NULL;

    /* make space for a complete match -- we can reduce usage later */
    kdst = *pkv = (sdb_keyvalue *)
	malloc(sizeof (sdb_keyvalue) * c->count + (copy ? c->size : 0));
    if (kdst == NULL) return (-1);
    ksrc = c->kv;
    copysize = 0;

    /* special case for full match */
    if (!key && !vpat) {
	memcpy((void *) kdst, (void *) ksrc, c->count * sizeof (sdb_keyvalue));
	kdst += c->count;
    } else {
	/* do globbing */
	if (key && (g = glob_init(key, flags)) == NULL) {
	    free((char *) kdst);
	    return (-1);
	}
	if (vpat && (vg = glob_init(vpat, flags)) == NULL) {
	    if (key) glob_free(&g);
	    free((char *) kdst);
	    return (-1);
	}
	if (key && !vpat) {
	    for (gcount = c->count; gcount; --gcount, ++ksrc) {
		if (GLOB_TEST(g, ksrc->key) >= 0) {
		    *kdst++ = *ksrc;
		}
	    }
	} else if (vpat && !key) {
	    for (gcount = c->count; gcount; --gcount, ++ksrc) {
		if (GLOB_TEST(vg, ksrc->value) >= 0) {
		    *kdst++ = *ksrc;
		}
	    }
	} else {
	    for (gcount = c->count; gcount; --gcount, ++ksrc) {
		if (GLOB_TEST(g, ksrc->key) >= 0
		    && GLOB_TEST(vg, ksrc->value) >= 0) {
		    *kdst++ = *ksrc;
		}
	    }
	}
	if (vpat) glob_free(&vg);
	if (key) glob_free(&g);

	/* check for no match */
	if (kdst == *pkv) {
	    free((char *) kdst);
	    *pkv = NULL;
	    *count = 0;
	    return (0);
	}
    }
    *count = kdst - *pkv;

    /* get size of data to copy if copy requested */
    if (copy) {
	for (ksrc = *pkv; ksrc < kdst; ++ksrc) {
	    copysize += strlen(ksrc->key) + 1;
	    if (ksrc->value) {
		copysize += strlen(ksrc->value) + 1;
	    }
	}
    }

    /* adjust down amount of space used */
    if (copy || *count < c->count) {
	*pkv = (sdb_keyvalue *)
	    realloc((char *) *pkv, *count * sizeof (sdb_keyvalue) + copysize);
    }

    /* copy the data */
    if (copy) {
	kdst = *pkv + *count;
	scan = (char *) kdst;
	for (ksrc = *pkv; ksrc < kdst; ++ksrc) {
	    value = ksrc->key;
	    ksrc->key = scan;
	    while ((*scan++ = *value++));
	    value = ksrc->value;
	    if (value) {
		ksrc->value = scan;
		while ((*scan++ = *value++));
	    }
	}
    }

    return (*pkv == NULL ? -1 : 0);
}

/* free keyvalue list returned by sdb_match
 */
void sdb_freematch(kv)
    sdb_keyvalue *kv;
{
    if (kv != NULL) free((char *) kv);
}

/* unlock a key
 * returns -1 on failure, 0 on success
 */
#ifdef __STDC__
int sdb_unlock(char *db, char *key, int flags)
#else
int sdb_unlock(db, key, flags)
    char *db, *key;
    int flags;
#endif
{
    cache *c;

    /* find the appropriate cache entry */
    if ((c = findcache(db)) == NULL) return (-1);

    if (c->locks && !--c->locks) {
	lock_unlock(c->fd, LOCK_UN);
	close(c->fd);
    }

    return (0);
}

/* lock a key to allow local modification -- this may lock a whole set of keys
 * or database as a side effect.  specific key need not exist.
 * returns -1 on error, 0 on success
 */
#ifdef __STDC__
int sdb_writelock(char *db, char *key, int flags)
#else
int sdb_writelock(db, key, flags)
    char *db, *key;
    int flags;
#endif
{
    cache *c;

    /* find the appropriate cache entry */
    if ((c = findcache(db)) == NULL) return (-1);
    if (c->locks) {
	++c->locks;
	return (0);
    }

    /* get exclusive lock on file */
    c->fd = -1;
    c->locks = 1;
    if (refreshcache(c, flags) < 0) {
	(void) sdb_unlock(db, key, flags);
	return (-1);
    }

    return (0);
}

/* set the value for a key -- key must be locked
 * returns -1 on failure, 0 on success
 */
#ifdef __STDC__
int sdb_set(char *db, char *key, int flags, char *value)
#else
int sdb_set(db, key, flags, value)
    char *db, *key, *value;
    int flags;
#endif
{
    cache *c;
    int top, bot, mid, cmp;
    sdb_keyvalue *kvtop, *kvmid;
    int (*cmpf)() = flags & SDB_ICASE ? strcasecmp : strcmp;

    /* find the appropriate cache entry & make sure it's locked */
    if ((c = findcache(db)) == NULL || !c->locks) return (-1);

    /* if db not empty, look for the key */
    mid = 0;
    if (c->count) {
	/* do binary search */
	top = c->count - 1;
	bot = 0;
	while (bot <= top
	       && (cmp = (*cmpf)(key, c->kv[mid=(bot+top)>>1].key))) {
	    if (cmp < 0) {
		top = mid - 1;
	    } else {
		bot = mid + 1;
	    }
	}
	if (!cmp) {
	    c->kv[mid].value = value;
	    return (rewritedb(c));
	}
	if (cmp > 0) ++mid;
    }

    /* make space for the new key */
    if (c->count) {
	c->kv = (sdb_keyvalue *)
	    realloc((char *) c->kv, sizeof (sdb_keyvalue) * ++c->count);
    } else {
	c->kv = (sdb_keyvalue *) malloc(sizeof (sdb_keyvalue));
	c->count = 1;
    }
    if (c->kv == NULL) {
	freecache(c);
	return (-1);
    }

    /* insert key */
    kvtop = c->kv + (c->count - 1);
    kvmid = c->kv + mid;
    while (kvtop > kvmid) {
	*kvtop = *(kvtop - 1);
	--kvtop;
    }
    kvmid->key = key;
    kvmid->value = value;

    return (rewritedb(c));
}

/* remove the entry for a key
 * returns -1 on failure, 0 on success
 */
#ifdef __STDC__
int sdb_remove(char *db, char *key, int flags)
#else
int sdb_remove(db, key, flags)
    char *db, *key;
    int flags;
#endif
{
    cache *c;
    sdb_keyvalue *kvtop, *kvmid;

    /* find the appropriate cache entry & make sure it's locked */
    if ((c = findcache(db)) == NULL || !c->locks) return (-1);

    /* if db not empty, look for the key */
    if (!c->count) return (-1);
    
    /* do binary search */
    kvmid = kv_bsearch(key, c->kv, c->count,
		       flags & SDB_ICASE ? strcasecmp : strcmp);
    if (!kvmid) return (-1);

    kvtop = c->kv + --c->count;
    while (kvmid < kvtop) {
	kvmid[0] = kvmid[1];
	++kvmid;
    }
    
    return (rewritedb(c));
}
