/* kerberos.c -- Kerberos authentication routines for IMAP.
 *
 *	(C) Copyright 1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */
#include <stdio.h>
#include <string.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <krb.h>

#include "kerberos.h"

#ifndef NIL
#define NIL 0
#define T 1
#endif

extern char *malloc();
extern char *lcase();
extern char *krb_get_phost(), *krb_realmofhost();

static char *srvtab = "";	/* Srvtab filename */

static des_cblock session;	/* Our session key */
static des_key_schedule schedule; /* Schedule for our session key */

/* Table for converting binary values to and from hexadecimal */
static char hex[] = "0123456789abcdef";
static char dec[256] = {
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,   /*   0 -  15 */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,   /*  16 -  37 */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,   /* ' ' - '/' */
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 0, 0, 0, 0, 0,   /* '0' - '?' */
    0,10,11,12,13,14,15, 0, 0, 0, 0, 0, 0, 0, 0, 0,   /* '@' - 'O' */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,   /* 'P' - '_' */
    0,10,11,12,13,14,15, 0, 0, 0, 0, 0, 0, 0, 0, 0,   /* '`' - 'o' */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,   /* 'p' - DEL */
};

static afs_string_to_key();

/*
 * Convert a KTEXT to an ascii string.
 * Accepts: ktext
 * Returnss: a pointer to the result, or null pointer on malloc failure
 *           The caller is responsible for freeing the returned value.
 */
static char *ktext_to_str(ktext)
KTEXT ktext;
{
    char *result, *p;
    int i;

    p = result = malloc(ktext->length*2+1);
    if (!result) return 0;

    for (i=0; i<ktext->length; i++) {
	*p++ = hex[(ktext->dat[i]>>4)&0xf];
	*p++ = hex[(ktext->dat[i])&0xf];
    }
    *p++ = '\0';
    return result;
}

/*
 * Convert string to a ktext.
 * Accepts: string to convert
 *          ktext to put result in
 * Returns: T on success, NIL on failure
 */
static int str_to_ktext(str, ktext)
char *str;
KTEXT ktext;
{
    int i, len;
    
    len = strlen(str);
    if (len&1) return NIL;
    len /= 2;
    if (len > MAX_KTXT_LEN) return NIL;

    ktext->length = len;
    ktext->mbz = 0;
    
    for (i=0; *str; i++, str += 2) {
	ktext->dat[i] = (dec[str[0]]<<4) + dec[str[1]];
    }
    return T;
}	

/*
 * Kerberos set srvtab filename
 * Accepts: name of srvtab file to use in reading authenticators
 */
int kerberos_set_srvtab(fname)
char *fname;
{
    srvtab = fname;
    return 0;
}

/*
 * Kerberos interpret authenticator
 * Accepts: password string
 *          pointer to buffer in which to place the authentication data
 *          pointer to char pointer.  The char pointer is set to the
 *               text we want returned in the reply message.
 * Returns: T if login ok, NIL otherwise
 */
int kerberos_read_authenticator(pass, kdata, reply)
char *pass;
AUTH_DAT *kdata;
char **reply;
{
    KTEXT_ST authent;
    char instance[INST_SZ];
    int code;
    static char replybuf[256];
    char *p;

    /* Convert pass to authent */
    if (str_to_ktext(pass, &authent) == NIL) {
	*reply = "Invalid Kerberos authenticator";
	return NIL;
    }

    /* Verify authenticator */
    strcpy(instance, "*");
    code = krb_rd_req(&authent, "imap", instance, 0L, kdata, srvtab);
    if (code) {
	*reply = krb_err_txt[code];
	return NIL;
    }

    /* Save the session key */
    memcpy(session, kdata->session, sizeof(des_cblock));
    des_key_sched(session, schedule);
    
    /* Construct the response for mutual authentication */
    authent.length = sizeof(des_cblock);
    memset(authent.dat, 0, sizeof(des_cblock));
    *((long *)authent.dat) = htonl(kdata->checksum + 1);
    des_ecb_encrypt(authent.dat, authent.dat, schedule, 1);

    /* Convert response to string and place in buffer */
    p = ktext_to_str(&authent);
    if (!p) {
	*reply = "Out of memory";
	return NIL;
    }
    *replybuf = '[';
    strcpy(replybuf+1, p);
    strcat(replybuf, "] ");
    free(p);

    *reply = replybuf;
    return T;
}

/*
 * Kerberos build password
 * Accepts: host name
 *	    buffer to store user name in
 *	    buffer to store password in
 *	    optional buffer to store expected return token in
 * Returns: T on success, NIL on failure
 */
int kerberos_build_password(host, user, pwd, token)
char *host;
char *user;
char *pwd;
char *token;
{
    struct hostent *host_name;
    char hostname[MAXHOSTNAMELEN+1];
    char phost[MAXHOSTNAMELEN+1];
    KTEXT_ST authent;
    int checksum;
    char *pass;
    int code;
    int i;
    CREDENTIALS cr;
    Key_schedule key_s;

    *pwd = '\0';
    
    if (krb_get_tf_fullname(TKT_FILE, user, (char *)0, (char *)0)) {
	return NIL;
    }

    /* Canonicalize hostname */
    /* The domain literal form is used (rather than simply the dotted decimal
     as with other Unix programs) because it has to be a valid "host name"
     in mailsystem terminology. */
				/* look like domain literal? */
    if (host[0] == '[' && host[i = (strlen (host))-1] == ']') {
	strcpy (hostname,host+1);	/* yes, copy without brackets */
	hostname[i-1] = '\0';
    }
				/* note that Unix requires lowercase! */
    else {
	if (host_name = gethostbyname (lcase (strcpy (hostname,host))))
	  strcpy (hostname,host_name->h_name);
	else return NIL;
    }

    /*
     * Build an authenticator.
     */
    checksum = time(0) ^ getpid();
    strcpy(phost, krb_get_phost(hostname));
    code = krb_mk_req(&authent, "imap", phost,
		      krb_realmofhost(hostname), checksum);
    if (code) return NIL;

    pass = ktext_to_str(&authent);
    if (!pass) return NIL;

    /* Send Kerberos-format LOGIN command */
    sprintf(pwd, "%s%s", KERBEROS_IDENT, pass);
    free(pass);
    if (!token) return T;
    *token = '\0';

    /*
     * Build expected mutual authentication reply.
     */
    if (code = krb_get_cred("imap", phost, krb_realmofhost(hostname),&cr)) {
	return T;
    }
    des_key_sched(cr.session,key_s);
    authent.length = sizeof(des_cblock);
    memset(authent.dat, 0, sizeof(des_cblock));
    *((long *)authent.dat) = htonl(checksum + 1);
    des_ecb_encrypt(authent.dat, authent.dat, key_s, 1);
    pass = ktext_to_str(&authent);
    if (pass) {
	sprintf(token, "[%s]", pass);
	free(pass);
    }
    return T;
}

static use_key(user, instance, realm, key, returned_key)
char *user;
char *instance;
char *realm;
des_cblock key;
des_cblock returned_key;
{
    memcpy (returned_key, key, sizeof(des_cblock));
    return 0;
}

int kerberos_verify_password(user, passwd)
char *user;
char *passwd;
{
    int result;
    des_cblock key;
    char realm[REALM_SZ];
    char cell[REALM_SZ];
    char userbuf[ANAME_SZ];
    char hostname[MAXHOSTNAMELEN+1];
    char passbuf[4096];
    AUTH_DAT kdata;
    char *reply;

    if (krb_get_lrealm(realm,1)) return NIL;

    sprintf(passbuf, "/tmp/tkt_imapd_%d", getpid());
    krb_set_tkt_string(passbuf);

    /* First try Kerberos string-to-key */
    des_string_to_key(passwd, key);
    
    result = krb_get_in_tkt(user, "", realm,
			    "krbtgt", realm, 1, use_key, NULL, key);

    if (result == INTK_BADPW) {
	/* Now try andrew string-to-key */
	strcpy(cell, realm);
	lcase(cell);
	afs_string_to_key(passwd, key, cell);
    
	result = krb_get_in_tkt(user, "", realm,
				"krbtgt", realm, 1, use_key, NULL, key);
    }

    memset(key, 0, sizeof(key));

    if (result != 0) return NIL;

    /* Check validity of returned ticket */
    gethostname(hostname, sizeof(hostname));
    result = kerberos_build_password(hostname, userbuf, passbuf, (char *)0);
    if (result) {
	result = kerberos_read_authenticator(passbuf+strlen(KERBEROS_IDENT),
					     &kdata, &reply);
    }
    if (result) {
	if (strcmp(kdata.pname, user) != 0 || kdata.pinst[0] ||
	    strcmp(kdata.prealm, krb_realmofhost(hostname)) != 0) {
	    result = NIL;
	}
    }

    memset(passbuf, 0, sizeof(passbuf));
    dest_tkt();
    return result;
}

/* andrewstk.c -- afs string to key function
 *
 * Code taken from AuthMan from University of Michigan
 */

/* forward declarations */
static afs_transarc_StringToKey();
static afs_cmu_StringToKey();

extern char *crypt();

/* This defines the Andrew string_to_key function.  It accepts a password
 * string as input and converts its via a one-way encryption algorithm to a DES
 * encryption key.  It is compatible with the original Andrew authentication
 * service password database.
 */

static
afs_cmu_StringToKey (str, cell, key)
  char          *str;
  char          *cell;                  /* cell for password */
  des_cblock key;
{   char  password[8+1];                /* crypt is limited to 8 chars anyway */
    int   i;
    int   passlen;

    memset(key, 0, sizeof(des_cblock));
    memset((void *)password, 0, sizeof(password));

    strncpy (password, cell, 8);
    passlen = strlen (str);
    if (passlen > 8) passlen = 8;

    for (i=0; i<passlen; i++)
        password[i] = str[i] ^ cell[i];

    for (i=0;i<8;i++)
        if (password[i] == '\0') password[i] = 'X';

    /* crypt only considers the first 8 characters of password but for some
       reason returns eleven characters of result (plus the two salt chars). */
    strncpy((void *)key, crypt(password, "p1") + 2, sizeof(des_cblock));

    /* parity is inserted into the LSB so leftshift each byte up one bit.  This
       allows ascii characters with a zero MSB to retain as much significance
       as possible. */
    {   char *keybytes = (char *)key;
        unsigned int temp;

        for (i = 0; i < 8; i++) {
            temp = (unsigned int) keybytes[i];
            keybytes[i] = (unsigned char) (temp << 1);
        }
    }
    des_fixup_key_parity (key);
}

static
afs_transarc_StringToKey (str, cell, key)
  char          *str;
  char          *cell;                  /* cell for password */
  des_cblock *key;
{   des_key_schedule schedule;
    char temp_key[8];
    char ivec[8];
    char password[BUFSIZ];
    int  passlen;

    strncpy (password, str, sizeof(password));
    if ((passlen = strlen (password)) < sizeof(password)-1)
        strncat (password, cell, sizeof(password)-passlen);
    if ((passlen = strlen(password)) > sizeof(password)) passlen = sizeof(password);

    memcpy (ivec, "kerberos", 8);
    memcpy (temp_key, "kerberos", 8);
    des_fixup_key_parity ((void *)temp_key);
    des_key_sched (temp_key, schedule);
    des_cbc_cksum (password, ivec, passlen, schedule, ivec);

    memcpy (temp_key, ivec, 8);
    des_fixup_key_parity ((void *)temp_key);
    des_key_sched (temp_key, schedule);
    des_cbc_cksum (password, (void *)key, passlen, schedule, ivec);

    des_fixup_key_parity (key);
}

static afs_string_to_key(str, key, cell)
  char          *str;
  des_cblock	*key;
  char          *cell;                  /* cell for password */
{
	if (strlen(str) > 8)
		afs_transarc_StringToKey (str, cell, key);
	else
		afs_cmu_StringToKey (str, cell, key);
}

