/* imsp_server.c -- Interactive Mail Support Protocol Server
 *
 *	(C) Copyright 1993-1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman <chrisn+@cmu.edu>
 * Start Date: 2/16/93
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <netinet/in.h>
#include "version.h"
#include "dispatch.h"
#include "util.h"
#include "syncdb.h"
#include "option.h"
#include "glob.h"
#include "authize.h"
#include "abook.h"
#include "bb.h"
#include "imsp_routines.h"
#include "im_util.h"
#include "imap_client.h"
#include "acl.h"
#include "alock.h"
#include "acte.h"

/* import from OS */
extern char *malloc(), *realloc();

/* structure used for command dispatch list */
typedef struct command_t {
    char *word;
    int id;
    void (*proc)();
} command_t;

/* structure used for imap -> imsp relay proc */
typedef struct relay_handler {
    im_handler handler;
    fbuf_t *fbuf;
} relay_handler;

#define MAX_IDLE_TIME (30*60)	/* 30 minutes */
#define MAX_WRITE_WAIT (30)	/* 30 seconds */

/* IMSP commands */
#define IMSP_LOGIN         0
#define IMSP_LOGOUT        1
#define IMSP_NOOP          2
#define IMSP_GET           3
#define IMSP_SET           4
#define IMSP_UNSET         5
#define IMSP_SUBSCRIBE     6
#define IMSP_UNSUBSCRIBE   7
#define IMSP_CREATE        8
#define IMSP_DELETE        9
#define IMSP_RENAME        10
#define IMSP_REPLACE	   11
#define IMSP_MOVE          12
#define IMSP_FETCHADDRESS  13
#define IMSP_SEARCHADDRESS 14
#define IMSP_STOREADDRESS  15
#define IMSP_DELETEADDRESS 16
#define IMSP_SETACL        17
#define IMSP_DELETEACL     18
#define IMSP_GETACL        19
#define IMSP_MYRIGHTS      20
#define IMSP_LOCK          21
#define IMSP_UNLOCK        22
#define IMSP_ADDRESSBOOK   23
#define IMSP_CREATEABOOK   24
#define IMSP_DELETEABOOK   25
#define IMSP_RENAMEABOOK   26
#define IMSP_CAPABILITY    27
#define IMSP_AUTHENTICATE  28
#define IMSP_LIST          29
#define IMSP_LSUB          30
#define IMSP_LMARKED	   31
#define IMSP_LAST	   32
#define IMSP_SEEN	   33

/* IMSP find options */
#define FIND_MAILBOXES        0
#define FIND_ALL_MAILBOXES    1
#define FIND_UNSEEN_MAILBOXES 2
#define FIND_BBOARDS          3
#define FIND_ALL_BBOARDS      4
#define FIND_UNSEEN_BBOARDS   5

/* IMSP ACL options */
#define ACL_ADDRESSBOOK  0
#define ACL_MAILBOX      1
static char *aclopt[] = {
    "addressbook", "mailbox", NULL
};

/* IMSP LOCK/UNLOCK options */
#define LOCK_OPTION	 0
#define LOCK_ADDRESSBOOK 1
static char *lockopt[] = {
    "option", "addressbook", NULL
};

/* predefined options */
static char opt_newuser[]   = "imsp.create.new.users";
static char opt_required[]  = "imsp.required.bbsubs";

/* user information */
static auth_id *imsp_id;

/* file buffer used by idle procedure */
static fbuf_t im_fbuf;

/* login & logout messages */
static char msg_greeting[] = "* OK Cyrus IMSP version %s ready\r\n";
static char msg_autologout[] = "* BYE user was idle for too long, closing\r\n";
static char msg_logout[] = "* BYE Logging user out\r\n";
static char msg_svrexit[] = "* BYE IMSP server exiting (probably out of memory)\r\n";
static char msg_capability[] = "* CAPABILITY\r\n";
static char txt_logoutuser[] = "Logging user out";
/* generic command parse errors */
static char msg_badtag[] = "* BAD 7-bit ASCII tag required\r\n";
static char rpl_badcommand[] = "BAD 7-bit ASCII command required\r\n";
static char rpl_invalcommand[] = "BAD command '%s' unknown\r\n";
static char rpl_wrongargs[] =
    "BAD command '%s' requires %d properly formed argument(s)\r\n";
static char rpl_wrongargopt[] =
    "BAD command '%s' requires a string and an optional %s\r\n";
static char rpl_noargs[] = "BAD command '%s' requires no arguments\r\n";
static char rpl_noauth[] = "NO User must LOGIN to execute command '%s'\r\n";
static char rpl_badauth[] = "NO User not authorized to execute that command\r\n";
/* generic errors */
static char err_nomem[] = "IMSP server out of memory";
static char err_quota[] = "operation failed: IMSP user quota exceeded";
/* generic replies */
static char rpl_ok[] = "OK %s\r\n";
static char rpl_complete[] = "OK %s completed\r\n";
static char rpl_generic[] = "%s\r\n";
static char rpl_no[] = "NO %s\r\n";
static char rpl_badopt[] = "BAD Invalid option '%s' to command '%s'\r\n";
static char rpl_internalerr[] = "BAD Internal error in routine '%s'\r\n";
static char rpl_notsupported[] = "NO %s not suppported at this site\r\n";
/* authorization messages */
static char msg_bbaccess[] = "* NO Unable to create subscription list: LIST/LSUB commands will fail\r\n";
static char err_nologin[] = "Login incorrect";
static char err_invaluser[] = "User does not have an account on this server";
static char rpl_bad64[] = "BAD Invalid base64 string\r\n";
/* GET responses, errors, strings */
static char msg_option[] = "* OPTION %a %s [READ-%a]\r\n";
static char err_optiondb[] = "options database unavailable";
static char txt_readwrite[] = "WRITE";
static char txt_readonly[] = "ONLY";
/* SET/UNSET errors */
static char rpl_noset[] = "%a NO User '%p' not authorized to change option '%p'\r\n";
static char rpl_isunset[] = "%a NO option '%p' was already unset\r\n";
/* LIST options, errors */
static char rpl_nosubs[] = "%a NO user '%p' is not subscribed to any bboards\r\n";
static char txt_mailbox[] = "MAILBOX";
static char txt_marked[] = "\\Marked";
static char txt_unmarked[] = "\\Unmarked";
static char txt_noinfer[] = "\\Noinferiors";
static char txt_noselect[] = "\\Noselect";
static char msg_list[] = "* LIST (%a%a%a) %a %s %.*a\r\n";
static char msg_lsub[] = "* LSUB (%a%a%a) %a %s %.*a\r\n";
static char msg_deletebb[] = "* NO The bboard '%p' has been deleted.\r\n";
static char msg_renamebb[] = "* NO The bboard '%p' has been renamed to '%p'.\r\n";
static char msg_mergebb[] = "* NO The bboard '%p' has been merged into '%p'.\r\n";
/* subscribe/unsubscribe errors */
static char rpl_notexists[] = "%a NO %a '%p' does not exist\r\n";
static char rpl_alreadydid[] = "%a NO Already %ad to %a '%p'\r\n";
static char rpl_required[] = "%a NO bboard '%p' is required; you may not unsubscribe.\r\n";
/* create text */
static char txt_splist[] = "server/partition list";
/* delete text */
static char txt_hostname[] = "hostname";

/* address book messages */
static char err_noabooksearch[] = "Unable to search address book list";
static char rpl_badfetchaddr[] = "BAD fetchaddress requires properly formatted address book name and entry\r\n";
static char rpl_badastr[] = "BAD address book entry names must be properly formed strings\r\n";
static char rpl_abookexists[] = "%a NO address book '%p' already exists\r\n";
static char rpl_noentry[] = "%a NO entry '%p' not found\r\n";
static char rpl_abookauth[] = "%a NO User '%p' not permitted to %a address book '%p'\r\n";
static char txt_create[] = "create";
static char txt_access[] = "access";
static char txt_modify[] = "modify";
static char txt_delete[] = "delete";
static char rpl_norename[] = "%a NO User '%p' not permitted to rename address book '%p' to '%p'\r\n";
static char rpl_badsearchaddr[] = "BAD searchaddress requires a properly formatted name string\r\n";
static char rpl_badpairs[] = "BAD %s %s must be valid atom/string pairs\r\n";
static char txt_lookupcrit[] = "lookup criteria";
static char txt_fielddata[] = "field data";
static char err_badsearch[] = "Address book search failed";
static char rpl_badstoreaddr[] = "BAD storeaddress requires a properly formatted name, alias and field data\r\n";
static char err_badstore[] = "Failed to modify address book";
static char err_badcreate[] = "Failed to create new address book";
static char err_baddelete[] = "Failed to delete address book";
static char err_badrename[] = "Failed to rename address book";
static char rpl_noabook[] = "%a NO ADDRESSBOOK '%p' does not exist\r\n";
static char msg_addressbook[] = "* ADDRESSBOOK () \".\" %s\r\n";
static char msg_searchaddr[] = "* SEARCHADDRESS %s\r\n";
static char msg_fetchaddr[] = "* FETCHADDRESS %s %s";
static char msg_fielddata[] = " %a %s";
static char txt_addressbook[] = "ADDRESSBOOK";
/* ACL messages */
static char txt_acls[] = "ACL command";
static char txt_setacl[] = "modify ACL for";
static char rpl_badacl[] = "%a NO Failed to %a access control list for %a '%p'\r\n";
static char rpl_noacl[] = "%a NO No ACL entry for identity '%p' in address book '%p'\r\n";
static char msg_acl[] = "* ACL %a %s %s %s\r\n";
static char msg_myrights[] = "* MYRIGHTS %a %s %s\r\n";
/* LOCK messages */
static char rpl_badlock[] = "BAD command '%s' requires an option and one or two valid arguments\r\n";
static char rpl_locked[] = "%a NO [LOCKED] %a%a '%p' already locked by %p\r\n";
static char txt_entry[] = " entry";
static char rpl_notlock[] = "%a NO %a%a '%p' not locked by current client\r\n";
static char rpl_lockfail[] = "%a NO failed to %a %a%a '%p'\r\n";
/* SEEN/LAST messages */
static char rpl_dbfail[] = "NO failed to update mailbox database\r\n";

/* macros to send messages */
#define SEND_STRING(fbuf, str) dispatch_write((fbuf), (str), sizeof (str) - 1)
#define SEND_RESPONSE(fbuf, tag, str) \
    sprintf((tag) + strlen(tag), " %s", (str)); \
    dispatch_write((fbuf), (tag), 0);
#define SEND_RESPONSE1(fbuf, tag, str, arg1) \
    strcat((tag), " "); \
    sprintf((tag) + strlen(tag), (str), (arg1)); \
    dispatch_write((fbuf), (tag), 0);
#define SEND_RESPONSE2(fbuf, tag, str, arg1, arg2) \
    strcat((tag), " "); \
    sprintf((tag) + strlen(tag), (str), (arg1), (arg2)); \
    dispatch_write((fbuf), (tag), 0);

/* clean abort procedure
 */
static void imsp_clean_abort()
{
    /* notify user */
    if (im_fbuf.fd >= 0) {
	SEND_STRING(&im_fbuf, msg_svrexit);
	dispatch_close(&im_fbuf);
    }
    
    /* close any open IMAP connections */
    imap_closeall();

    /* release all advisory locks */
    alock_unlock();

    /* release all database locks and resources */
    sdb_done();

    /* clean up authorization */
    auth_free(imsp_id);

    exit(0);
}

/* fatal abort (called from xmalloc.c)
 */
void fatal(s, type)
    char *s;
    int type;
{
    imsp_clean_abort();
}

/* signal manager which clears out passwords
 */
static void imsp_signal_handler(sig)
    int sig;
{
    signal(sig, SIG_DFL);
    auth_free(imsp_id);
    kill(getpid(), sig);
}

/* set signals to nuke password before core dumping
 */
static void imsp_set_signals()
{
    signal(SIGQUIT, imsp_signal_handler);
    signal(SIGILL, imsp_signal_handler);
    signal(SIGTRAP, imsp_signal_handler);
    signal(SIGIOT, imsp_signal_handler);
    signal(SIGEMT, imsp_signal_handler);
    signal(SIGFPE, imsp_signal_handler);
    signal(SIGBUS, imsp_signal_handler);
    signal(SIGSEGV, imsp_signal_handler);
    signal(SIGSYS, imsp_signal_handler);
    signal(SIGURG, imsp_signal_handler);
    signal(SIGCHLD, imsp_signal_handler);
    signal(SIGIO, imsp_signal_handler);
    signal(SIGWINCH, imsp_signal_handler);
}

/* err procedure for server
 */
int im_err(type)
    int type;
{
    if (type != DISPATCH_READ_ERR) {
	if (type == DISPATCH_READ_IDLE) {
	    dispatch_write(&im_fbuf, msg_autologout,
			   sizeof (msg_autologout) - 1);
	}
	dispatch_close(&im_fbuf);
	imsp_clean_abort();
    }
    
    return (1);
}

/* get the option number for an acl
 */
static int lookupopt(atom, optarray)
    char *atom;
    char *optarray[];
{
    int opt;

    for (opt = 0; optarray[opt] && strcasecmp(atom, optarray[opt]); ++opt);

    return (optarray[opt] ? opt : -1);
}

/* convert from base64 to a binary buffer
 */
static char index_64[128] = {
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,62, -1,-1,-1,63,
    52,53,54,55, 56,57,58,59, 60,61,-1,-1, -1,-1,-1,-1,
    -1, 0, 1, 2,  3, 4, 5, 6,  7, 8, 9,10, 11,12,13,14,
    15,16,17,18, 19,20,21,22, 23,24,25,-1, -1,-1,-1,-1,
    -1,26,27,28, 29,30,31,32, 33,34,35,36, 37,38,39,40,
    41,42,43,44, 45,46,47,48, 49,50,51,-1, -1,-1,-1,-1
};
#define CHAR64(c)  (((c) < 0 || (c) > 127) ? -1 : index_64[(c)])
int from64(out, in)
    char *out, *in;
{
    int len = 0;
    int c1, c2, c3, c4;

    if (*in == '\r') return (0);
    do {
	c1 = in[0];
	if (CHAR64(c1) == -1) return (-1);
	c2 = in[1];
	if (CHAR64(c2) == -1) return (-1);
	c3 = in[2];
	if (c3 != '=' && CHAR64(c3) == -1) return (-1); 
	c4 = in[3];
	if (c4 != '=' && CHAR64(c4) == -1) return (-1);
	in += 4;
	*out++ = (CHAR64(c1) << 2) | (CHAR64(c2) >> 4);
	++len;
	if (c3 != '=') {
	    *out++ = ((CHAR64(c2) << 4) & 0xf0) | (CHAR64(c3) >> 2);
	    ++len;
	    if (c4 != '=') {
		*out++ = ((CHAR64(c3) << 6) & 0xc0) | CHAR64(c4);
		++len;
	    }
	}
    } while (*in && c4 != '=');

    return (len);
}

extern struct sockaddr_in imspd_localaddr, imspd_remoteaddr;

/* authenticate the user
 */
static void imsp_authenticate(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *auth_type;
    char *reply = NULL;
    int result;
    int (*authproc)();
    struct acte_server *mech;
    void *state;
    char *output;
    int len, olen;
    char *user;
    int protlevel;
    char *(*encodefunc)();
    char *(*decodefunc)();
    int maxplain;

    /* parse command */
    if ((auth_type = get_atom(fbuf)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 1);
	return;
    }

    /* start authentication process */
    lcase(auth_type);
    result = login_authenticate(auth_type, &mech, &authproc);
    if (result == 0) {
	result = mech->start("imap", authproc, ACTE_PROT_ANY, MAX_BUF,
			     &imspd_localaddr, &imspd_remoteaddr,
			     &olen, &output, &state, &reply);
    }

    /* exchange authentication credentials */
    while (result == 0) {
	im_send(fbuf, NULL, "+ %b\r\n", olen, output);
	dispatch_flush(fbuf);
	if (dispatch_readline(fbuf) == NULL) {
	    result = ACTE_FAIL;
	} else if ((len = from64(fbuf->upos, fbuf->upos)) < 0) {
	    SEND_RESPONSE(fbuf, tag, rpl_bad64);
	} else {
	    result = mech->auth(state, len, fbuf->upos, &olen,
				&output, &reply);
	}
	if (result && result != ACTE_DONE) {
	    mech->free_state(state);
	}
    }

    /* check for auth errors */
    if (result != ACTE_DONE) {
	SEND_RESPONSE1(fbuf, tag, rpl_no, reply ? reply : err_nologin);
	return;
    }

    /* get auth state */
    mech->query_state(state, &user, &protlevel,
		      &encodefunc, &decodefunc, &maxplain);
    auth_login(&id, user, NULL, &reply);

    /* check for valid user */
    if (option_check(auth_username(imsp_id = id)) < 0 &&
	(!option_test("", opt_newuser, 1, 0)
	 || option_create(auth_username(id)) < 0)) {
	SEND_RESPONSE1(fbuf, tag, rpl_no, err_invaluser);
	return;
    }

    /* initialize user & auth */
    if (bb_subsinit(auth_username(id)) < 0) {
	SEND_STRING(fbuf, msg_bbaccess);
    }
    dispatch_telemetry(fbuf, auth_username(id));
    SEND_RESPONSE1(fbuf, tag, rpl_ok, reply);
    dispatch_flush(fbuf);
    if (encodefunc || decodefunc) {
	fbuf->efunc = encodefunc;
	fbuf->dfunc = decodefunc;
	fbuf->maxplain = maxplain;
	fbuf->state = state;
	fbuf->free_state = mech->free_state;
    } else {
	mech->free_state(state);
    }
    imsp_id = id;
}

/* login the user
 */
static void imsp_login(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *user, *pass = NULL, *reply;

    /* check if user can login */
    if ((user = copy_astring(fbuf, 1)) == NULL
	|| (pass = copy_astring(fbuf, 1)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 2);
    } else if (auth_login(&id, user, pass, &reply) < 0) {
	SEND_RESPONSE1(fbuf, tag, rpl_no, reply);
    } else if (option_check(auth_username(imsp_id = id)) < 0 &&
	       (!option_test("", opt_newuser, 1, 0)
		|| option_create(auth_username(id)) < 0)) {
	SEND_RESPONSE1(fbuf, tag, rpl_no, err_invaluser);
    } else {
	if (bb_subsinit(auth_username(id)) < 0) {
	    SEND_STRING(fbuf, msg_bbaccess);
	}
	dispatch_telemetry(fbuf, auth_username(id));
	SEND_RESPONSE1(fbuf, tag, rpl_ok, reply);
    }

    /* free any leftover strings */
    if (pass) {
	memset(pass, 0, strlen(pass));
	free(pass);
    }
    if (user) free(user);
}

/* logout the user
 */
static void imsp_logout(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    if (fbuf->upos != fbuf->lend) {
	SEND_RESPONSE1(fbuf, tag, rpl_noargs, cp->word);
    } else {
	SEND_STRING(fbuf, msg_logout);
	SEND_RESPONSE1(fbuf, tag, rpl_ok, txt_logoutuser);
	dispatch_close(fbuf);
	imsp_clean_abort();
    }
}

/* do a noop
 */
static void imsp_noop(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    if (fbuf->upos != fbuf->lend) {
	SEND_RESPONSE1(fbuf, tag, rpl_noargs, cp->word);
    } else {
	SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
    }
}

/* do the "GET" command
 */
static void imsp_get(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *opt, *name, *value, *user;
    int rwflag;
    option_state ostate;

    if ((opt = copy_astring(fbuf, 3)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 1);
    } else {
	user = auth_level(id) >= AUTH_USER ? auth_username(id) : "";
	if (option_matchstart(&ostate, user, opt) < 0) {
	    SEND_RESPONSE1(fbuf, tag, rpl_no, err_optiondb);
	} else {
	    while (option_match(&ostate, user, &name, &value, &rwflag,
				auth_level(id) == AUTH_ADMIN) != NULL) { 
		im_send(fbuf, NULL, msg_option, name, value,
			  rwflag ? txt_readwrite : txt_readonly);
	    }
	    option_matchdone(&ostate);
	    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
	}
    }
    if (opt != NULL) free(opt);
}

/* do the "SET" command
 */
static void imsp_set(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *opt, *value = NULL, *user;
    int auth, result;

    if ((opt = copy_astring(fbuf, 1)) == NULL
	|| (value = copy_astring(fbuf, 1)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 2);
    } else if (auth_level(id) < AUTH_USER) {
	SEND_RESPONSE1(fbuf, tag,
		       auth_level(id) ? rpl_badauth : rpl_noauth, cp->word);
    } else {
	auth = auth_level(id) == AUTH_ADMIN;
	user = auth_username(id);
	if ((result = option_set(user, opt, auth, value)) < 0) {
	    if (result == -2) {
		SEND_RESPONSE1(fbuf, tag, rpl_no, err_quota);
	    } else {
		im_send(fbuf, NULL, rpl_noset, tag, user, opt);
	    }
	} else {
	    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
	}
    }
    if (opt) free(opt);
    if (value) free(value);
}

/* do the "UNSET" command
 */
static void imsp_unset(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *opt, *newval, *user;
    int result, auth, rwflag;

    if ((opt = copy_astring(fbuf, 1)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 1);
    } else if (auth_level(id) < AUTH_USER) {
	SEND_RESPONSE1(fbuf, tag,
		       auth_level(id) ? rpl_badauth : rpl_noauth, cp->word);
    } else {
	auth = auth_level(id) == AUTH_ADMIN;
	user = auth_username(id);
	result = option_unset(user, opt, auth);
	if (result < 0) {
	    im_send(fbuf, NULL, rpl_noset, tag, user, opt);
	} else if (!result) {
	    im_send(fbuf, NULL, rpl_isunset, tag, opt);
	} else {
	    while (*opt == '*' || *opt == '%') ++opt;
	    newval = option_get(user, opt, auth, &rwflag);
	    if (newval) {
		im_send(fbuf, NULL, msg_option, opt, newval,
			  rwflag ? txt_readwrite : txt_readonly);
		free(newval);
	    }
	    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
	}
    }
    if (opt != NULL) free(opt);
}

/* do the "ADDRESSBOOK" command
 */
static void imsp_addressbook(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *pat, *abook;
    int attrs;
    abook_state astate;

    if ((pat = copy_astring(fbuf, 3)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 1);
    } else if (abook_findstart(&astate, id, pat) < 0) {
	SEND_RESPONSE1(fbuf, tag, rpl_no, err_noabooksearch);
    } else {
	while (abook_find(&astate, id, &abook, &attrs)) {
	    im_send(fbuf, NULL, msg_addressbook, abook);
	}
	abook_finddone(&astate);
	SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
    }

    if (pat) free(pat);
}

/* show all appropriate bboards -- used by LIST/LSUB/LMARKED command
 */
static int show_bbmatch(state, fbuf, msg, id, pat, lmark)
    bb_state *state;
    fbuf_t *fbuf;
    char *msg;
    auth_id *id;
    char *pat;
    int lmark;
{
    char *name, *loc, *lend, *oldname, *newname;
    int flags, result, autosub;
    char sep_buf[4];

    sep_buf[3] = '\0';
    do {
	result = bb_matchverify(state, id, pat, &oldname, &newname, &autosub);
	if (result < 0) {
	    return (-1);
	} else if (result) {
	    if (!newname) {
		im_send(fbuf, NULL, msg_deletebb, oldname);
	    } else {
		im_send(fbuf, NULL, autosub ? msg_renamebb : msg_mergebb,
			  oldname, newname);
	    }
	}
    } while (result);
    while (bb_match(state, id, &name, &loc,
		    sep_buf + 1, &flags, msg != msg_list)) {
	if (lmark && (flags & BB_UNMARKED)) continue;
	if (sep_buf[1] == '\r' || sep_buf[1] == '\0') {
	    strcpy(sep_buf, "NIL");
	} else {
	    sep_buf[0] = sep_buf[2] = '"';
	}
	lend = strchr(loc, ')');
	if (lend) {
	    im_send(fbuf, NULL, msg,
		    flags & BB_STATE
		    ? (flags & BB_NOSELECT ? txt_noselect : txt_noinfer)
		    : "",
		    (flags & BB_STATE) && (flags & BB_MARK) ? " " : "",
		    flags & BB_MARK
		    ? (flags & BB_MARKED ? txt_marked : txt_unmarked)
		    : "",
		    sep_buf, name, lend - loc + 1, loc);
	}
    }
    bb_matchdone(state);

    return (0);
}

/* do the "LIST", "LSUB" and "LMARKED" commands
 */
static void imsp_list(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *ref, *mbox = NULL, *tmp, *user, *scan, *msg;
    bb_state bbstate;

    if ((ref = copy_astring(fbuf, 1)) == NULL
	|| (mbox = copy_astring(fbuf, 3)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 2);
    } else {
	/* combine reference & mbox if relative */
	if (*mbox == '.' && *ref) {
	    tmp = malloc(strlen(ref) + strlen(mbox) + 1);
	    if (tmp) {
		strcpy(tmp, ref);
		scan = strrchr(tmp, '.');
		if (!scan) scan = tmp;
		strcpy(scan, mbox);
		free(mbox);
		mbox = tmp;
	    }
	}
	user = auth_username(id);
	msg = cp->id == IMSP_LIST ? msg_list : msg_lsub;
	if (bb_matchstart(&bbstate, user, mbox) < 0) {
	    im_send(fbuf, NULL, rpl_nosubs, tag, user);
	} else if (show_bbmatch(&bbstate, fbuf, msg, id, mbox,
				cp->id == IMSP_LMARKED) < 0) {
	    SEND_RESPONSE1(fbuf, tag, rpl_internalerr, "bb_matchverify");
	} else {
	    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
	}
    }
    if (mbox) free(mbox);
    if (ref) free(ref);
}

/* do the "SUBSCRIBE" and "UNSUBSCRIBE" commands
 */
static void imsp_subscribe(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name = NULL;
    int result;

    if ((name = copy_astring(fbuf, 0)) == NULL
	|| (fbuf->upos != fbuf->lend &&
	    (strcasecmp(name, txt_mailbox)
	     || (free(name), name = copy_astring(fbuf, 1)) == NULL))
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 2);
    } else if (auth_level(id) < AUTH_USER) {
	SEND_RESPONSE1(fbuf, tag,
		       auth_level(id) ? rpl_badauth : rpl_noauth, cp->word);
    } else if (option_lookup("", opt_required, 1, name) == 1) {
	im_send(fbuf, NULL, rpl_required, tag, name);
    } else {
	result = bb_subscribe(id, name, cp->id == IMSP_SUBSCRIBE);
	if (result < 0) {
	    im_send(fbuf, NULL, rpl_notexists, tag, txt_mailbox, name);
	} else if (result > 0) {
	    im_send(fbuf, NULL, rpl_alreadydid, tag, cp->word,
		    txt_mailbox, name);
	} else {
	    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
	}
    }
    if (name) free(name);
}

/* do the "CREATE" command
 */
static void imsp_create(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name, *splist = NULL, *chost = NULL, *user, *resp;
    int spcount = 0;

    user = auth_username(id);
    if ((name = copy_astring(fbuf, 1)) == NULL
	|| (fbuf->upos != fbuf->lend
	    && (spcount = copy_atom_list(fbuf, &splist)) < 0)
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargopt, cp->word, txt_splist);
    } else if (auth_level(id) < AUTH_USER
	       || (spcount && auth_level(id) < AUTH_BB)) {
	SEND_RESPONSE1(fbuf, tag,
		       auth_level(id) ? rpl_badauth : rpl_noauth, cp->word);
    } else {
	resp = bb_create(id, name, spcount, splist);
	if (!resp) {
	    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
	} else if (resp[0] == '%') {
	    im_send(fbuf, NULL, resp, tag, name);
	} else {
	    SEND_RESPONSE1(fbuf, tag, rpl_generic, resp);
	}
    }
    if (chost && chost != splist) free(chost);
    if (splist) free(splist);
    if (name) free(name);
}

/* do the "DELETE" command
 */
static void imsp_delete(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name, *dhost = NULL, *resp;

    if ((name = copy_astring(fbuf, 1)) == NULL
	|| (fbuf->upos != fbuf->lend
	    && (dhost = get_atom(fbuf)) == NULL)
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargopt, cp->word, txt_hostname);
    } else if (auth_level(id) < AUTH_USER
	       || (dhost && auth_level(id) < AUTH_BB)) {
	SEND_RESPONSE1(fbuf, tag,
		       auth_level(id) ? rpl_badauth : rpl_noauth, cp->word);
    } else {
	resp = bb_delete(id, name, dhost);
	if (resp[0] == '%') {
	    im_send(fbuf, NULL, resp, tag, name);
	} else {
	    SEND_RESPONSE1(fbuf, tag, rpl_generic, resp);
	}
    }
    if (name) free(name);
}

/* do the "RENAME" and "REPLACE" commands
 */
static void imsp_rename(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name, *newname = NULL, *resp;

    if ((name = copy_astring(fbuf, 1)) == NULL
	|| (newname = copy_astring(fbuf, 1)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 2);
    } else if (auth_level(id) < AUTH_USER) {
	SEND_RESPONSE1(fbuf, tag,
		       auth_level(id) ? rpl_badauth : rpl_noauth, cp->word);
    } else {
	resp = bb_rename(id, name, newname, cp->id == IMSP_REPLACE);
	if (resp[0] == '%') {
	    im_send(fbuf, NULL, resp, tag, name, newname);
	} else {
	    SEND_RESPONSE1(fbuf, tag, rpl_generic, resp);
	}
    }
    if (newname) free(newname);
    if (name) free(name);
}

/* do the "MOVE" command
 */
static void imsp_move(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name, *splist = NULL, *user;
    int spcount;

    user = auth_username(id);
    if ((name = copy_astring(fbuf, 1)) == NULL
	|| (spcount = copy_atom_list(fbuf, &splist)) <= 0
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 3);
    } else if (auth_level(id) < AUTH_BB) {
	SEND_RESPONSE1(fbuf, tag,
		       auth_level(id) ? rpl_badauth : rpl_noauth, cp->word);
    } else {
	SEND_RESPONSE1(fbuf, tag, rpl_notsupported, cp->word);
    }
    if (splist) free(splist);
    if (name) free(name);
}

/* do the "CREATEADDRESSBOOK" command
 */
static void imsp_createabook(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name, *user;

    user = auth_username(auth_level(id) >= AUTH_USER ? id : NULL);
    if ((name = copy_astring(fbuf, 1)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 1);
    } else {
	lcase(name);
	switch (abook_create(id, name)) {
	    case AB_EXIST:
		im_send(fbuf, NULL, rpl_abookexists, tag, name);
		break;
	    case AB_PERM:
		im_send(fbuf, NULL, rpl_abookauth, tag, user, txt_create,
			name);
		break;
	    case AB_FAIL:
		SEND_RESPONSE(fbuf, tag, err_badcreate);
		break;
	    default:
		SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
		break;
	}
    }
    if (name) free(name);
}

/* do the "DELETEADDRESSBOOK" command
 */
static void imsp_deleteabook(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name, *user;

    user = auth_username(auth_level(id) >= AUTH_USER ? id : NULL);
    if ((name = copy_astring(fbuf, 1)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 1);
    } else {
	lcase(name);
	switch (abook_delete(id, name)) {
	    case AB_NOEXIST:
		im_send(fbuf, NULL, rpl_noabook, tag, name);
		break;
	    case AB_PERM:
		im_send(fbuf, NULL, rpl_abookauth, tag, user, txt_delete,
			name);
		break;
	    case AB_FAIL:
		SEND_RESPONSE1(fbuf, tag, rpl_no, err_baddelete);
		break;
	    default:
		SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
		break;
	}
    }
    if (name) free(name);
}

/* do the "RENAMEADDRESSBOOK" command
 */
static void imsp_renameabook(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name, *newname, *user;

    user = auth_username(auth_level(id) >= AUTH_USER ? id : NULL);
    if ((name = copy_astring(fbuf, 1)) == NULL
	|| (newname = copy_astring(fbuf, 1)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 2);
    } else {
	lcase(name);
	lcase(newname);
	switch (abook_rename(id, name, newname)) {
	    case AB_NOEXIST:
		im_send(fbuf, NULL, rpl_noabook, tag, name);
		break;
	    case AB_EXIST:
		im_send(fbuf, NULL, rpl_abookexists, tag, newname);
		break;
	    case AB_QUOTA:
		SEND_RESPONSE1(fbuf, tag, rpl_no, err_quota);
		break;
	    case AB_PERM:
		im_send(fbuf, NULL, rpl_norename, tag, user, name,
			newname);
		break;
	    case AB_FAIL:
		SEND_RESPONSE1(fbuf, tag, rpl_no, err_badrename);
		break;
	    default:
		SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
		break;
	}
    }
    if (name) free(name);
    if (newname) free(newname);
}

/* display an address book entry
 */
static int show_address(fbuf, name, alias)
    fbuf_t *fbuf;
    char *name, *alias;
{
    abook_state astate;
    abook_fielddata *fetch;
    int count, i;
    
    if ((fetch = abook_fetch(&astate, name, alias, &count))) {
	im_send(fbuf, NULL, msg_fetchaddr, name, alias);
	for (i = 0; i < count; ++i) {
	    im_send(fbuf, NULL, msg_fielddata,
				fetch[i].field, fetch[i].data);
	}
	SEND_STRING(fbuf, "\r\n");
	abook_fetchdone(&astate, fetch);
    }

    return (fetch ? 0 : -1);
}


/* do the "FETCHADDRESS" command
 */
static void imsp_fetchaddress(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name = NULL, *alias = NULL, *user;
    int result;

    user = auth_username(auth_level(id) >= AUTH_USER ? id : NULL);
    if ((name = copy_astring(fbuf, 1)) == NULL
	|| (alias = copy_astring(fbuf, 1)) == NULL) {
	SEND_RESPONSE(fbuf, tag, rpl_badfetchaddr);
    } else if (!abook_canfetch(id, name)) {
	im_send(fbuf, NULL, rpl_abookauth, tag, user, txt_access, name);
    } else {
	lcase(name);
	do {
	    if ((result = show_address(fbuf, name, alias)) < 0) break;
	    free(alias);
	    alias = copy_astring(fbuf, 1);
	} while (alias);
	if (result == -1) {
	    im_send(fbuf, NULL, rpl_noentry, tag, alias);
	} else if (alias == NULL && fbuf->upos != fbuf->lend) {
	    SEND_RESPONSE(fbuf, tag, rpl_badastr);
	} else {
	    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
	}
    }
    if (alias) free(alias);
    if (name) free(name);
}


/* do the "SEARCHADDRESS" and "STOREADDRESS" commands
 */
static void imsp_searchaddress(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name = NULL, *alias = NULL, *myalias, *user;
    int fused = 0, fsize = 0, abortflag = 0, result;
    abook_fielddata *flist = NULL;
    abook_state astate;

    if ((name = copy_astring(fbuf, 1)) == NULL ||
	(cp->id == IMSP_STOREADDRESS
	 && ((alias = copy_astring(fbuf, 1)) == NULL
	     || fbuf->upos == fbuf->lend))) {
	SEND_RESPONSE(fbuf, tag,
		      cp->id == IMSP_STOREADDRESS ?
		      rpl_badstoreaddr : rpl_badsearchaddr);
    } else {
	lcase(name);
	while (fbuf->upos < fbuf->lend) {
	    if (fused == fsize) {
		if (!fsize) {
		    flist = (abook_fielddata *)
			malloc((fsize = 32) * sizeof (abook_fielddata));
		} else {
		    flist = (abook_fielddata *)
			realloc((char *) flist,
				(fsize *= 2) * sizeof (abook_fielddata));
		}
	    }
	    if (flist == NULL) {
		SEND_RESPONSE1(fbuf, tag, rpl_no, err_nomem);
		abortflag = 1;
		break;
	    }
	    if ((flist[fused].field = copy_atom(fbuf)) == NULL
		|| (flist[fused].data = copy_astring(fbuf, 1)) == NULL) {
		SEND_RESPONSE2(fbuf, tag, rpl_badpairs, cp->word,
			       alias ? txt_fielddata : txt_lookupcrit);
		if (flist[fused].field) free(flist[fused].field);
		abortflag = 1;
		break;
	    }
	    ++fused;
	}
	if (!abortflag) {
	    user = auth_username(auth_level(id) >= AUTH_USER ? id : NULL);
	    if (cp->id == IMSP_STOREADDRESS) {
		result = abook_store(id, name, alias, flist, fused);
		if (result == AB_SUCCESS && abook_canfetch(id, name)) {
		    show_address(fbuf, name, alias);
		}
	    } else {
		result = abook_searchstart(&astate, id, name, flist, fused);
		if (result == AB_SUCCESS) {
		    while ((myalias = abook_search(&astate))) {
			im_send(fbuf, NULL, msg_searchaddr, myalias);
		    }
		    abook_searchdone(&astate);
		}
	    }
	    switch (result) {
		case AB_NOEXIST:
		    im_send(fbuf, NULL, rpl_noabook, tag, name);
		    break;
		case AB_QUOTA:
		    SEND_RESPONSE1(fbuf, tag, rpl_no, err_quota);
		    break;
		case AB_PERM:
		    im_send(fbuf, NULL, rpl_abookauth, tag, user,
			    alias ? txt_modify : txt_access, name);
		    break;
		case AB_FAIL:
		    SEND_RESPONSE1(fbuf, tag, rpl_no,
				   alias ? err_badstore : err_badsearch);
		    break;
		default:
		    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
		    break;
	    }
	}
    }
    if (flist) {
	while (fused--) {
	    free(flist[fused].data);
	    free(flist[fused].field);
	}
	free((char *) flist);
    }
    if (alias) free(alias);
    if (name) free(name);
}

/* do the "DELETEADDRESS" command
 */
static void imsp_deleteaddress(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *name = NULL, *alias = NULL, *user;
    int result;

    if ((name = copy_astring(fbuf, 1)) == NULL
	|| (alias = copy_astring(fbuf, 1)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 2);
    } else {
	lcase(name);
	user = auth_username(auth_level(id) >= AUTH_USER ? id : NULL);
	result = abook_deleteent(id, name, alias);
	switch (result) {
	    case AB_PERM:
		im_send(fbuf, NULL, rpl_abookauth, tag,
			user, txt_modify, name);
		break;
	    case AB_FAIL:
		SEND_RESPONSE1(fbuf, tag, rpl_no, err_badstore);
		break;
	    case AB_NOEXIST:
		im_send(fbuf, NULL, rpl_noentry, tag, alias);
		break;
	    default:
		SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
		break;
	}
    }
    if (alias) free(alias);
    if (name) free(name);
}

/* do the "SETACL" and "DELETEACL" commands
 */
static void imsp_setacl(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *opt = NULL, *item = NULL, *ident = NULL, *rights = NULL;
    char *user, *ahost = NULL, *resp;
    int result = 3, optnum;

    if ((opt = copy_atom(fbuf)) == NULL
	|| (item = copy_astring(fbuf, 1)) == NULL
	|| (ident = copy_astring(fbuf, 1)) == NULL
	|| (cp->id == IMSP_SETACL && (rights = copy_astring(fbuf, 1)) == NULL)
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word,
		       cp->id == IMSP_SETACL ? 4 : 3);
    } else {
	user = auth_username(auth_level(id) >= AUTH_USER ? id : NULL);
	switch (optnum = lookupopt(opt, aclopt)) {
	    case ACL_ADDRESSBOOK:
		lcase(item);
		result = abook_setacl(id, item, ident, rights);
		switch (result) {
		    case -2:
			im_send(fbuf, NULL, rpl_abookauth, tag, user,
				txt_setacl, item);
			break;
		    case -1:
			im_send(fbuf, NULL, rpl_badacl, tag, txt_modify,
				opt, item);
			break;
		    case 0:
			SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
			break;
		    case 1:
			im_send(fbuf, NULL, rpl_noacl, tag, ident, item);
			break;
		}
		break;
	    case ACL_MAILBOX:
		resp = bb_setacl(id, item, ident, rights);
		if (!resp) {
		    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
		} else if (resp[0] == '%') {
		    im_send(fbuf, NULL, resp, tag, item, ident);
		} else {
		    SEND_RESPONSE1(fbuf, tag, rpl_generic, resp);
		}
		break;
	}
    }
    if (ahost) free(ahost);
    if (rights) free(rights);
    if (ident) free(ident);
    if (item) free(item);
    if (opt) free(opt);
}

/* do the "GETACL" and "MYRIGHTS" commands
 */
static void imsp_getacl(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *opt = NULL, *item = NULL, *ident, *rights, *user;
    char *acl, tmp, *defacl = NULL;
    char rbuf[ACL_MAXSTR];
    int result = 2, optnum;

    if ((opt = copy_atom(fbuf)) == NULL
	|| (item = copy_astring(fbuf, 1)) == NULL
	|| fbuf->upos != fbuf->lend) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 2);
    } else {
	user = auth_username(auth_level(id) >= AUTH_USER ? id : NULL);
	lcase(item);
	optnum = lookupopt(opt, aclopt);
	if (optnum == ACL_MAILBOX
	    && (bb_get(item, NULL, NULL, &acl, NULL) < 0 || !acl)) {
	    result = 3;
	} else if (cp->id == IMSP_GETACL) {
	    if (optnum == ACL_ADDRESSBOOK) {
		acl = abook_getacl(id, item);
		if (*acl == '\0') {
		    defacl = malloc(strlen(user) + 32);
		    if (defacl) {
			sprintf(acl = defacl, "%s\t%s\t", user,
				acl_masktostr(ACL_ALL, rbuf));
		    }
		}
	    }
	    if (acl != NULL) {
		result = 0;
		do {
		    ident = acl;
		    for (rights = ident; *rights && *rights != '\t'; ++rights);
		    if (*rights) ++rights;
		    for (acl = rights; *acl && *acl != '\t'; ++acl);
		    if (*rights) {
			rights[-1] = '\0';
			tmp = *acl;
			*acl = '\0';
			im_send(fbuf, NULL, msg_acl, optnum == ACL_MAILBOX
				? txt_mailbox : txt_addressbook,
				item, ident, rights);
			*acl = tmp;
			rights[-1] = '\t';
		    }
		    if (*acl == '\t') ++acl;
		} while (*acl != '\0');
	    } else {
		result = 3;
	    }
	} else {
	    if (optnum == ACL_ADDRESSBOOK) {
		result = abook_myrights(id, item, rbuf);
	    } else if (optnum == ACL_MAILBOX) {
		acl_masktostr(bb_rights(id, item, acl), rbuf);
		result = 0;
	    }
	    if (!result) {
		im_send(fbuf, NULL, msg_myrights, opt, item, rbuf);
	    }
	}
	switch (result) {
	    default:
		im_send(fbuf, NULL, rpl_badacl, tag, txt_access,
			opt, item);
		break;
	    case 0:
		SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
		break;
	    case 1:
		SEND_RESPONSE1(fbuf, tag, rpl_notsupported, txt_acls);
		break;
	    case 2:
		SEND_RESPONSE2(fbuf, tag, rpl_badopt, opt, cp->word);
		break;
	    case 3:
		im_send(fbuf, NULL, rpl_notexists, tag, opt, item);
		break;
	}
    }
    if (defacl) free(defacl);
    if (item) free(item);
    if (opt) free(opt);
}

/* do the "LOCK" and "UNLOCK" commands
 */
static void imsp_lock(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *opt, *item1 = NULL, *item2 = NULL, *value = NULL, *lstr, *user;
    int optnum, perm, rwflag;
    
    if ((opt = get_atom(fbuf)) == NULL) {
	SEND_RESPONSE1(fbuf, tag, rpl_badlock, cp->word);
    } else if ((optnum = lookupopt(opt, lockopt)) < 0) {
	SEND_RESPONSE2(fbuf, tag, rpl_badopt, opt, cp->word);
    } else if ((item1 = copy_astring(fbuf, 1)) == NULL
	       || (optnum == 1 && (item2 = copy_astring(fbuf, 1)) == NULL)
	       || fbuf->upos != fbuf->lend) {
	SEND_RESPONSE1(fbuf, tag, rpl_badlock, cp->word);
    } else {
	perm = 1;
	user = auth_username(id);
	if (optnum) {
	    lcase(item1);
	    if (!abook_canlock(id, item1)) {
		im_send(fbuf, NULL, rpl_abookauth, tag, user, txt_modify,
			item1);
		perm = 0;
	    }
	} else {
	    value = option_get(user, item1, 1, &rwflag);
	    if (auth_level(id) == AUTH_NONE || !rwflag) {
		im_send(fbuf, NULL, rpl_noset, tag, user, item1);
		perm = 0;
	    }
	}
	if (perm) {
	    lstr = host;
	    switch (alock_dolock(user, item1, item2,
				 cp->id == IMSP_LOCK, &lstr)) {
		case -1:	/* failure */
		    im_send(fbuf, NULL, rpl_lockfail, tag, cp->word,
			    lockopt[optnum], optnum ? txt_entry : "",
			    item2 ? item2 : item1);
		    break;
		case 0:		/* no error */
		    if (cp->id == IMSP_LOCK) {
			if (optnum) {
			    if (abook_canfetch(id, item1)) {
				show_address(fbuf, item1, item2);
			    }
			} else if (value) {
			    im_send(fbuf, NULL, msg_option, item1, value,
				    txt_readwrite);
			}
		    }
		    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
		    break;
		case 1:		/* already locked/unlocked */
		    im_send(fbuf, NULL,
			    cp->id == IMSP_LOCK ? rpl_locked : rpl_notlock,
			    tag, lockopt[optnum], optnum ? txt_entry : "",
			    item2 ? item2 : item1, lstr);
		    break;
	    }
	}
    }
    if (value) free(value);
    if (item2) free(item2);
    if (item1) free(item1);
}

/* do the "CAPABILITY" command
 */
static void imsp_capability(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    SEND_STRING(fbuf, msg_capability);
    SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
}

/* do the "LAST" command
 */
static void imsp_last(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *mbox, *uid, *user;

    user = auth_username(id);
    if ((mbox = copy_astring(fbuf, 0)) == NULL
	|| (uid = get_atom(fbuf)) == NULL) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 2);
    } else if (auth_level(id) < AUTH_BB &&
	  (strlen(user) < 5 || strncasecmp("imap.", user, 5))) {
	SEND_RESPONSE(fbuf, tag, rpl_badauth);
    } else if (bb_last(mbox, uid, host) < 0) {
	SEND_RESPONSE(fbuf, tag, rpl_dbfail);
    } else {
	SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
    }
    if (mbox) free(mbox);
}

/* do the "SEEN" command
 */
static void imsp_seen(fbuf, cp, tag, id, host)
    fbuf_t *fbuf;
    command_t *cp;
    char *tag, *host;
    auth_id *id;
{
    char *mbox, *uid = NULL, *uname = NULL, *user;

    user = auth_username(id);
    if ((mbox = copy_astring(fbuf, 0)) == NULL
	|| (uname = copy_astring(fbuf, 0)) == NULL
	|| (uid = copy_atom(fbuf)) == NULL) {
	SEND_RESPONSE2(fbuf, tag, rpl_wrongargs, cp->word, 3);
    } else if (auth_level(id) < AUTH_BB &&
	  (strlen(user) < 5 || strncasecmp("imap.", user, 5))) {
	SEND_RESPONSE(fbuf, tag, rpl_badauth);
    } else if (bb_seen(mbox, uid, uname) < 0) {
	SEND_RESPONSE(fbuf, tag, rpl_dbfail);
    } else {
	SEND_RESPONSE1(fbuf, tag, rpl_complete, cp->word);
    }
    if (uname) free(uname);
    if (uid) free(uid);
    if (mbox) free(mbox);
}

/* list of commands & procedures to manage them
 */
static command_t com_list[] = {
    {"login", IMSP_LOGIN, imsp_login},
    {"logout", IMSP_LOGOUT, imsp_logout},
    {"noop", IMSP_NOOP, imsp_noop},
    {"get", IMSP_GET, imsp_get},
    {"set", IMSP_SET, imsp_set},
    {"unset", IMSP_UNSET, imsp_unset},
    {"subscribe", IMSP_SUBSCRIBE, imsp_subscribe},
    {"unsubscribe", IMSP_UNSUBSCRIBE, imsp_subscribe},
    {"create", IMSP_CREATE, imsp_create},
    {"delete", IMSP_DELETE, imsp_delete},
    {"rename", IMSP_RENAME, imsp_rename},
    {"replace", IMSP_REPLACE, imsp_rename},
    {"move", IMSP_MOVE, imsp_move},
    {"fetchaddress", IMSP_FETCHADDRESS, imsp_fetchaddress},
    {"searchaddress", IMSP_SEARCHADDRESS, imsp_searchaddress},
    {"storeaddress", IMSP_STOREADDRESS, imsp_searchaddress},
    {"deleteaddress", IMSP_DELETEADDRESS, imsp_deleteaddress},
    {"setacl", IMSP_SETACL, imsp_setacl},
    {"deleteacl", IMSP_DELETEACL, imsp_setacl},
    {"getacl", IMSP_GETACL, imsp_getacl},
    {"myrights", IMSP_MYRIGHTS, imsp_getacl},
    {"lock", IMSP_LOCK, imsp_lock},
    {"unlock", IMSP_UNLOCK, imsp_lock},
    {"addressbook", IMSP_ADDRESSBOOK, imsp_addressbook},
    {"createaddressbook", IMSP_CREATEABOOK, imsp_createabook},
    {"deleteaddressbook", IMSP_DELETEABOOK, imsp_deleteabook},
    {"renameaddressbook", IMSP_RENAMEABOOK, imsp_renameabook},
    {"capability", IMSP_CAPABILITY, imsp_capability},
    {"authenticate", IMSP_AUTHENTICATE, imsp_authenticate},
    {"list", IMSP_LIST, imsp_list},
    {"lsub", IMSP_LSUB, imsp_list},
    {"lmarked", IMSP_LMARKED, imsp_list},
    {"last", IMSP_LAST, imsp_last},
    {"seen", IMSP_SEEN, imsp_seen},
    {NULL, 0, NULL}
};

/* start the protocol exchange
 */
void im_start(fd, host)
    int fd;
    char *host;
{
    char *tag, *command;
    command_t *cp;
    fbuf_t *fbuf = &im_fbuf;
    char tagbuf[MAX_BUF * 3];

    /* send greeting, setup idle autologout */
    sprintf(tagbuf, msg_greeting, VERSION);
    if (write(fd, tagbuf, strlen(tagbuf)) < 0) {
	perror("write greeting");
	close(fd);
	return;
    }
    (void) dispatch_err(MAX_IDLE_TIME, MAX_WRITE_WAIT, im_err);
    dispatch_initbuf(fbuf, fd);

    /* initialize user authentication information */
    imsp_id = NULL;

    /* initialize signal handlers to nuke password */
    imsp_set_signals();

    /* main protocol loop */
    while (dispatch_readline(fbuf) != NULL) {
	/* get the tag */
	tag = get_atom(fbuf);
	if (tag == (char *) NULL) {
	    SEND_STRING(fbuf, msg_badtag);
	} else {
	    /* copy tag into tag-response buffer */
	    strcpy(tagbuf, tag);
	    
	    /* get the command */
	    command = get_atom(fbuf);
	    if (command == (char *) NULL) {
		SEND_RESPONSE(fbuf, tagbuf, rpl_badcommand);
	    } else {
		/* look up the command */
		lcase(command);
		cp = com_list;
		while (cp->word != NULL && strcmp(cp->word, command)) {
		    ++cp;
		}
		if (cp->proc) {
		    (*cp->proc)(fbuf, cp, tagbuf, imsp_id, host);
		} else {
		    SEND_RESPONSE1(fbuf, tagbuf, rpl_invalcommand, command);
		}
	    }
	}
	dispatch_flush(fbuf);
    }
    dispatch_close(fbuf);
    imsp_clean_abort();
}
