/* dispatch.h -- dispatch functions
 *
 *	(C) Copyright 1993 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman <chrisn+@cmu.edu>
 * Start Date: 2/22/93
 */

#define MAX_BUF 4096

/* a file buffer structure
 */
typedef struct fbuf_t {
    char *upos;			/* user position in line */
    int fd;			/* file descriptor */
    char *lend;			/* end of line in buffer */
    char *uend;			/* end of user area */
    char *iptr;			/* position for new data */
    int ileft;			/* unused bytes in ibuf */
    int ocount;			/* amount of data in obuf */
    int nonblocking;		/* flag for non-blocking input */
    int eof;			/* hit an EOF on read */
    int telem;			/* telemetry log */

    char *(*efunc)();		/* protection encoding function */
    char *(*dfunc)();		/* protection decoding function */
    void (*free_state)();	/* free protection state function */
    void *state;		/* protection state */
    int maxplain;		/* protection max plaintext on write */
    int dcount;			/* amount of decoded data in pbuf */
    int ecount;			/* amount of encoded data in pbuf */
    char *eptr;			/* position of encoded data in pbuf */
    char *dptr;			/* position in decoded data in pbuf */

    char ibuf[MAX_BUF];		/* line buffered data */
    char obuf[MAX_BUF];		/* output buffered data */
    char pbuf[MAX_BUF+4];	/* protection buffered data */
} fbuf_t;

/* a dispatch structure
 *  int read_proc(fbuf, data)
 *   fbuf_t *fbuf    file buffer
 *   void *data      user data
 *  returns -1 if self-removed, 0 otherwise
 */
typedef struct dispatch_t {
    struct dispatch_t *next;	/* next pointer */
    fbuf_t *fbuf;		/* file buffer */
    int (*read_proc)();		/* call on read */
    int (*write_proc)();	/* call on write */
    void *data;			/* generic data pointer */
} dispatch_t;

/* types for an error proc */
#define DISPATCH_READ_IDLE  0
#define DISPATCH_WRITE_IDLE 1
#define DISPATCH_READ_ERR   2
#define DISPATCH_WRITE_ERR  3

#ifdef __STDC__
/* an err procedure -- return non-zero to flag an error
 */
typedef int (*err_proc_t)(int);

/* initialize dispatch system */
void dispatch_init(void);

/* initialize a file buffer */
void dispatch_initbuf(fbuf_t *, int);

/* set err function, returns old err function */
err_proc_t dispatch_err(int, int, err_proc_t);

/* add a file descriptor to the dispatch list (structure not copied) */
void dispatch_add(dispatch_t *);

/* remove a file descriptor from the dispatch system */
void dispatch_remove(fbuf_t *);

/* check if a file descriptor is in the dispatch list */
int dispatch_check(int);

/* set the dispatch procedures */
void dispatch_setproc( int, int (*)(), int (*)());

/* (blocking) dispatch loop: returns 0 on success, -1 on unix select error */
int dispatch_loop(int, int);

/* (blocking) read specified amount of data from a file */
int dispatch_read(fbuf_t *, char *, int);

/* (blocking) read a line of text (CRLF terminated) from a file */
char *dispatch_readline(fbuf_t *);

/* flush data from a file buffer */
int dispatch_flush(fbuf_t *);

/* (blocking) write data */
int dispatch_write(fbuf_t *, char *, int);

/* close a file buffer and remove it from dispatch system */
void dispatch_close(fbuf_t *);

/* activate telemetry for user */
void dispatch_telemetry(fbuf_t *, char *);
#else
typedef int (*err_proc_t)();
void dispatch_init(), dispatch_initbuf(), dispatch_add(), dispatch_remove();
void dispatch_setproc(), dispatch_close(), dispatch_telemetry();
err_proc_t dispatch_err();
int dispatch_check(), dispatch_loop(), dispatch_read(), dispatch_flush();
int dispatch_write();
char *dispatch_readline();
#endif
