/* imsp.c -- Interactive Mail Support Protocol routines
 *
 *	(C) Copyright 1993-1994 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include "mail.h"
#include "osdep.h"
#include "imap2.h"
#include "imutil.h"
#include "support.h"
#include "imsp.h"
#include "misc.h"

/* driver name for IMSP drivers */
static char drvrname[] = "IMSP";

/* flag indicating we're in an expand operation
 */
static int in_expand = 0;
static char *expand_names;

/* IMSP addressbook function prototypes
 */
long imsp_abook_create(SUPPORTSTREAM *, char *);
long imsp_abook_delete(SUPPORTSTREAM *, char *);
long imsp_abook_rename(SUPPORTSTREAM *, char *, char *);
long imsp_abook_find(SUPPORTSTREAM *, char *);
abook *imsp_abook_open(SUPPORTSTREAM *, char *);
void imsp_abook_close(abook *);
long imsp_abook_getlist(abook *);
long imsp_abook_search(abook *, char *, keyvalue *, long);
long imsp_abook_fetch(abook *, char *);
long imsp_abook_lock(abook *, char *);
void imsp_abook_unlock(abook *, char *);
long imsp_abook_store(abook *, char *, keyvalue *, long);
long imsp_abook_deleteent(abook *, char *);
char *imsp_abook_expand(abook *, char *);
long imsp_abook_getacl(abook *, int);
long imsp_abook_setacl(abook *, char *, char *);
void imsp_abook_pfree(struct abookdriver *);

/* IMSP option function prototypes
 */
long imsp_option_get(SUPPORTSTREAM *, char *);
long imsp_option_lock(SUPPORTSTREAM *, char *);
void imsp_option_unlock(SUPPORTSTREAM *, char *);
long imsp_option_set(SUPPORTSTREAM *, char *, char *);
void imsp_option_pfree(struct optiondriver *);


/* IMSP addressbook driver
 */
static abookdriver IMSP_abook_drvr = {
    drvrname,
    NULL, NULL,
    imsp_abook_create,
    imsp_abook_delete,
    imsp_abook_rename,
    imsp_abook_find,
    imsp_abook_open,
    imsp_abook_close,
    imsp_abook_getlist,
    imsp_abook_search,
    imsp_abook_fetch,
    imsp_abook_lock,
    imsp_abook_unlock,
    imsp_abook_store,
    imsp_abook_deleteent,
    imsp_abook_expand,
    imsp_abook_getacl,
    imsp_abook_setacl,
    imsp_abook_pfree
};

/* IMSP option driver
 */
static optiondriver IMSP_option_drvr = {
    drvrname,
    NULL, NULL,
    imsp_option_get,
    imsp_option_lock,
    imsp_option_unlock,
    imsp_option_set,
    imsp_option_pfree
};

/* open an IMSP connection and install IMSP option/address-book drivers
 *  returns: T on success, NIL on failure
 */
long imsp_open(SUPPORTSTREAM *s, char *host)
{
    /* take a peek at IMAP's login trials parameter:
     * no need to have the same parameter twice.
     */
    extern long map_maxlogintrials;
    char username[MAILTMPLEN], pwd[MAILTMPLEN];
    long i;
    IMPARSEDREPLY *reply = &s->imsp.reply;
    abookdriver *advr;
    optiondriver *odvr;
    
    /* open the connection */
    if (!s || (s->imsp.tcp = tcp_open(host, IMSP_PORT)) == NULL) return (NIL);

    /* get the reply */
    if (im_reply(&s->imsp, NULL)) {
	if (reply->keyval == IMKEY_PREAUTH) {
	    advr = (abookdriver *) fs_get(sizeof (abookdriver));
	    *advr = IMSP_abook_drvr;
	    support_add_adriver(s, advr);
	    odvr = (optiondriver *) fs_get(sizeof (optiondriver));
	    *odvr = IMSP_option_drvr;
	    support_add_odriver(s, odvr);
	    return (T);
	} else {
	    for (i = 0; i < map_maxlogintrials; ++i) {
		*pwd = 0;
		mm_login(tcp_host(s->imsp.tcp), username, pwd, i);
		if (!*pwd) {
		    mm_log("Login aborted", ERROR);
		    break;
		} else if (im_send(&s->imsp, "%t LOGIN %s %S\015\012",
				   username, pwd)) {
		    /*XXX: verify kerberos reply token */
		    memset(pwd, '\0', strlen(pwd));
		    s->imsp.username = cpystr(username);
		    advr = (abookdriver *) fs_get(sizeof (abookdriver));
		    *advr = IMSP_abook_drvr;
		    support_add_adriver(s, advr);
		    odvr = (optiondriver *) fs_get(sizeof (optiondriver));
		    *odvr = IMSP_option_drvr;
		    support_add_odriver(s, odvr);
		    return (T);
		} else {
		    mm_log(reply->text, WARN);
		    if (reply->keyval == IMKEY_BYE || !s->imsp.tcp) break;
		}
	    }
	    if (i >= map_maxlogintrials) {
		mm_log("Too many login failures", ERROR);
	    }
	}
    } else {
	mm_log(reply->text, ERROR);
    }

    /* clean up on error exit */
    if (s->imsp.tcp) {
	tcp_close(s->imsp.tcp);
	s->imsp.tcp = NULL;
    }
    
    return (NIL);
}

static long dolog(SUPPORTSTREAM *s, long result)
{
    if (result == NIL) {
	mm_log(s->imsp.reply.text, ERROR);
    }

    return (result);
}

long imsp_abook_create(SUPPORTSTREAM *s, char *name)
{
    return (dolog(s, im_send(&s->imsp,
			     "%t CREATEADDRESSBOOK %s\015\012", name)));
}

long imsp_abook_delete(SUPPORTSTREAM *s, char *name)
{
    return (dolog(s, im_send(&s->imsp,
			     "%t DELETEADDRESSBOOK %s\015\012", name)));
}

long imsp_abook_rename(SUPPORTSTREAM *s, char *oldname, char *newname)
{
    return (dolog(s, im_send(&s->imsp, "%t RENAMEADDRESSBOOK %s %s\015\012",
			     oldname, newname)));
}

long imsp_abook_find(SUPPORTSTREAM *s, char *pattern)
{
    return (dolog(s, im_send(&s->imsp,
			     "%t FIND ALL.ADDRESSBOOKS %s\015\012", pattern)));
}

abook *imsp_abook_open(SUPPORTSTREAM *s, char *name)
{
    abook *ab;

    if (!name && !(name = s->imsp.username)) return (NULL);
    ab = (abook *) fs_get(sizeof (abook) + strlen(name) + 1);
    ab->name = (char *) (ab + 1);
    strcpy(ab->name, name);

    return (ab);
}

void imsp_abook_close(abook *ab)
{
    fs_give((void **) &ab);
}

long imsp_abook_getlist(abook *ab)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent,
		  im_send(&ab->parent->imsp, "%t SEARCHADDRESS %s\015\012",
			  ab->name)));
}

long imsp_abook_search(abook *ab, char *pattern, keyvalue *kv, long count)
{
    ab->parent->activeab = ab;
    if (!strcmp("*", pattern)) {
	return (dolog(ab->parent, im_send(&ab->parent->imsp,
					  "%t SEARCHADDRESS %s%k\015\012",
					  ab->name, count, kv)));
    }
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t SEARCHADDRESS %s name %s%k\015\012",
				      ab->name, pattern, count, kv)));
}

long imsp_abook_lock(abook *ab, char *name)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t LOCK ADDRESSBOOK %s %s\015\012",
				      ab->name, name)));
}

void imsp_abook_unlock(abook *ab, char *name)
{
    (void) dolog(ab->parent, im_send(&ab->parent->imsp,
				     "%t UNLOCK ADDRESSBOOK %s %s\015\012",
				     ab->name, name));
}

long imsp_abook_store(abook *ab, char *name, keyvalue *kv, long count)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t STOREADDRESS %s %s%k\015\012",
				      ab->name, name, count, kv)));
}

long imsp_abook_deleteent(abook *ab, char *name)
{
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t DELETEADDRESS %s %s\015\012",
				      ab->name, name)));
}

long imsp_abook_fetch(abook *ab, char *name)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t FETCHADDRESS %s %s\015\012",
				      ab->name, name)));

}

char *imsp_abook_expand(abook *ab, char *name)
{
    int result1, result2;
    
    in_expand = 1;
    expand_names = NULL;
    dolog(ab->parent,
	  im_send(&ab->parent->imsp, "%t FETCHADDRESS %s %s\015\012",
		  ab->name, name));
    in_expand = 0;
    
    return (expand_names);
}

/* fetchaddress callback
 */
void imsp_fetchaddress(SUPPORTSTREAM *s, char *abname, char *name,
		       keyvalue *kv, long kvcount)
{
    int i;
    
    if (kvcount) {
	if (in_expand) {
	    /* look for "email" field */
	    for (i = 0; i < kvcount; ++i) {
		lcase(kv[i].key);
		if (!strcmp(kv[i].key, "email")) {
		    expand_names = cpystr(kv[i].value);
		    break;
		}
	    }
	} else {
	    mm_address(s->activeab, name, kv, kvcount);
	}
    }
}

long imsp_abook_getacl(abook *ab, int myrights)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t %s ADDRESSBOOK %s\015\012",
				      myrights ? "MYRIGHTS" : "GETACL",
				      ab->name)));
}

long imsp_abook_setacl(abook *ab, char *identifier, char *rights)
{
    ab->parent->activeab = ab;
    return (dolog(ab->parent, im_send(&ab->parent->imsp,
				      "%t SETACL ADDRESSBOOK %s %s %s\015\012",
				      ab->name, identifier, rights)));
}

/* abookacl callback
 */
void imsp_abookacl(SUPPORTSTREAM *s, char *abname, char *identifier,
		   char *rights)
{
    mm_abookacl(s->activeab, identifier, rights);
}

void imsp_abook_pfree(struct abookdriver *adrvr)
{
    fs_give((void **) &adrvr);
    return;
}

long imsp_option_get(SUPPORTSTREAM *s, char *pattern)
{
    return (dolog(s, im_send(&s->imsp, "%t GET %s\015\012", pattern)));
}

long imsp_option_lock(SUPPORTSTREAM *s, char *name)
{
    return (dolog(s, im_send(&s->imsp, "%t LOCK OPTION %s\015\012", name)));
}

void imsp_option_unlock(SUPPORTSTREAM *s, char *name)
{
    dolog(s, im_send(&s->imsp, "%t UNLOCK OPTION %s\015\012", name));
}

long imsp_option_set(SUPPORTSTREAM *s, char *name, char *value)
{
    if (value) {
	return (dolog(s, im_send(&s->imsp, "%t SET %s %s\015\012",
				 name, value)));
    }
    return (dolog(s, im_send(&s->imsp, "%t UNSET %s\015\012", name)));
}

void imsp_option_pfree(struct optiondriver *odrvr)
{
    fs_give((void **) &odrvr);
    return;
}
