/* support.h -- Support stream routines for c-client
 *
 *	(C) Copyright 1993 by Carnegie Mellon University
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Author: Chris Newman
 */

/* a key-value pair for an address book
 */
typedef struct keyvalue {
    char *key, *value;
} keyvalue;

/* support stream structure
 */
typedef struct supportstream {
    struct supportstream *next;	/* in case someone wants more than one */
    struct abookdriver *adrvr;	/* list of addressbook drivers */
    struct optiondriver *odrvr;	/* list of options drivers */
    struct abook *ab;		/* list of open address books */
    struct abook *activeab;	/* "active" address book -- used as hint */
    MAILSTREAM *mstream;	/* list of open mailstreams
				   (next pointer kept in local data) */
    struct abookdriver *current_adrvr;
    struct optiondriver *current_odrvr;
    void *generic;              /* generic pointer for client context */
    IMSTREAMDATA imsp;		/* storage for IMSP stream info */
} SUPPORTSTREAM;

/* driver for address book functions
 */
typedef struct abookdriver {
    char *name;
    struct abookdriver *next;
    void *private;		/* storage for driver private data */
    long   (*create)(SUPPORTSTREAM *, char *);
    long   (*delete)(SUPPORTSTREAM *, char *);
    long   (*rename)(SUPPORTSTREAM *, char *, char *);
    long   (*find)(SUPPORTSTREAM *, char *);
    struct abook *(*open)(SUPPORTSTREAM *, char *);
    void   (*close)(struct abook *);
    long   (*getlist)(struct abook *);
    long   (*search)(struct abook *, char *, keyvalue *, long);
    long   (*fetch)(struct abook *, char *);
    long   (*lock)(struct abook *, char *);
    void   (*unlock)(struct abook *, char *);
    long   (*store)(struct abook *, char *, keyvalue *, long);
    long   (*deleteent)(struct abook *, char *);
    char  *(*expand)(struct abook *, char *);
    long   (*getacl)(struct abook *, int);
    long   (*setacl)(struct abook *, char *, char *);
    void   (*pfree)(struct abookdriver *); /* free private storage */
} abookdriver;

/* driver for option functions
 */
typedef struct optiondriver {
    char *name;
    struct optiondriver *next;
    void *private;		/* storage for driver private data */
    long (*get)(SUPPORTSTREAM *, char *);
    long (*lock)(SUPPORTSTREAM *, char *);
    void (*unlock)(SUPPORTSTREAM *, char *);
    long (*set)(SUPPORTSTREAM *, char *, char *);
    void (*pfree)(struct optiondriver *); /* free private storage */
} optiondriver;

/* address book file
 */
typedef struct abook {
    struct abook *next;		/* next abook in list of open ones */
    SUPPORTSTREAM *parent;	/* parent supportstream */
    abookdriver *adrvr;		/* driver for this abook */
    void *generic;		/* generic pointer for caller to use */

    /* driver abook_open needs to set the following: */
    char *name;			/* name of abook */
    long id;			/* address book numeric id, if meaningful */
    /* specific drivers may add more fields */
} abook;

/* local-part of SUPPORTed MAILSTREAM
 */
typedef struct imsp_local {
    IMAPLOCAL imap;
    MAILSTREAM *next;
    SUPPORTSTREAM *parent;
    int ownsupport;		/* if non-zero, then SUPPORTSTREAM should be
				 * closed when MAILSTREAM is closed.
				 */
} SUPPORTLOCAL;

/* function prototypes: */

/* support level functions */
SUPPORTSTREAM *support_new(void);
void support_done(SUPPORTSTREAM *);
void support_add_adriver(SUPPORTSTREAM *, abookdriver *);
void support_add_odriver(SUPPORTSTREAM *, optiondriver *);

/* macros to access supportstream generic pointer */
#define support_generic(s) ((s)->generic)
#define support_set_generic(s, g) ((s)->generic = (g))

/* option driver functions: */
long option_get(SUPPORTSTREAM *, char *);
long option_lock(SUPPORTSTREAM *, char *);
void option_unlock(SUPPORTSTREAM *, char *);
long option_set(SUPPORTSTREAM *, char *, char *);

/* address book driver front-end functions: */
long abook_create(SUPPORTSTREAM *, char *);
long abook_delete(SUPPORTSTREAM *, char *);
long abook_rename(SUPPORTSTREAM *, char *, char *);
long abook_find(SUPPORTSTREAM *, char *);
abook *abook_open(SUPPORTSTREAM *, char *, void *);
void abook_close(abook *);
long abook_getlist(abook *);
long abook_search(abook *, char *, keyvalue *, long);
long abook_fetch(abook *, char *);
long abook_lock(abook *, char *);
void abook_unlock(abook *, char *);
long abook_store(abook *, char *, keyvalue *, long);
long abook_deleteent(abook *, char *);
char *abook_expand(abook *, char *);

/* address book driver ACL front-end functions: */
long abook_getacl(abook *, int);
long abook_setacl(abook *, char *, char *);

/* macros to access data associated with address book */
#define abook_generic(abook) ((abook)->generic)
#define abook_name(abook) ((abook)->name)

/* new callback routines */
extern void mm_option(SUPPORTSTREAM *, char *, char *, int);
extern void mm_addressbook(SUPPORTSTREAM *, char *, char);
extern void mm_address(abook *, char *, keyvalue *, int);
extern void mm_abookacl(abook *, char *, char *);
