/* acap.h - ACAP Library Interface
 * Rob Earhart
 * $Id: acap.h,v 1.11 1997/11/17 22:46:51 rob Exp $
 */
/***********************************************************
        Copyright 1997 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#ifndef ACAP_H
#define ACAP_H

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/types.h>

/* Most functions, unless otherwise noted, return zero on success
 * and return -1 and set errno on failure. */

/* Types exported by the ACAP library */

typedef enum {
  ACAP_ACAP,
  ACAP_BAD,
  ACAP_BYE,
  ACAP_OK,
  ACAP_NO,
  ACAP_CONTINUATION,
  ACAP_RESPONSE_ENTRY,
  ACAP_RESPONSE_MODTIME
} acap_Status;

typedef struct _acap_Connection *acap_Connection;

/* ACAP uses binary data - null characters are allowed at all times.
 * Thus, we have this structure, which provides a string interface. */
typedef struct {
  size_t length;
  unsigned char str[1];		/* dummy length */
} acap_String;

/* Note: it's perfectly all right to have your callback longjmp
 * somewhere, but if you do it, don't use the synchronous interface
 * at *all*, because the synchronous interface *does* use longjmp(),
 * and if multiple things are longjmp()ing to multiple destinations,
 * your program will crash so hard you will never figure out what
 * went wrong (your stack will be toast). */

/* The "data" parameter is allocated with malloc(); it is the callee's
 * responsibility to free it after use. */

typedef void (acap_CmdCallback)(acap_Connection conn,
				void *clientData,
				acap_Status status,
				char *data);

typedef void (acap_OpenCallback)(acap_Connection conn,
				 void *clientData);

typedef void (acap_DataCallback)(acap_Connection conn,
				 void *clientData,
				 acap_Status status,
				 acap_String **data);

/* This is... a little twisted.
 * acap_ConstructCriteria takes a variable number of arguments and
 * constructs an acap_Criteria object for you.  The entire structure
 * will be allocated in one malloc(), which should be free'd after use
 * (none of the library procedures will free it for you).
 *
 * The first argument is the first tag; the second argument is
 * the first parameter to the first tag (and it may be a tag,
 * if the first tag's first argument is a criteria...)
 *
 * The whole thing looks like a prefix-encoded representation
 * of an acap search-criteria structure.
 *
 * Each comment after a tag specifies what's expected to follow after
 * the tag.  attribute, comparator, and value are strings
 * (const char *'s); criteria means some sub-criteria, and "criteria
 * pointer" (for ACAP_CRITERIA) should be an acap_Criteria object
 * returned by a previous call to acap_ConstructCriteria.
 *
 * This is probably pretty confusing, but the examples below ought
 * to clear things up a little... */

typedef enum {
  ACAP_CRITERIAPTR,		/* criteria pointer */
  ACAP_ALL,			/* no arguments */
  ACAP_AND,			/* two criteria */
  ACAP_COMPARE,			/* attribute, comparator, value */
  ACAP_COMPARESTRICT,		/* attribute, comparator, value */
  ACAP_EQUAL,			/* attribute, comparator, value */
  ACAP_NOT,			/* criteria */
  ACAP_OR,			/* two criteria */
  ACAP_PREFIX,			/* attribute, comparator, value */
  ACAP_RANGE,			/* low, high (as unsigned longs) */
  ACAP_SUBSTRING		/* attribute, comparator, value */
} acap_CriteriaTag;

typedef struct _acap_Criteria *acap_Criteria;

acap_Criteria acap_ConstructCriteria(acap_CriteriaTag tag, ...);

/* Sample use: to construct a criteria matching all records
 * whose "foo" attribute equals "bar" and whose "baz"
 * attribute contains "qux", use something like
 *
 * acap_ConstructCriteria(ACAP_AND, ACAP_EQUAL, "foo", "i;octet", "bar",
 *				    ACAP_SUBSTRING, "baz", "i;octet", "qux");
 *
 * Alternatively, you could construct it with several calls:
 *
 * acap_Criteria first = acap_ConstructCriteria(ACAP_EQUAL, "foo", "i;octet", "bar");
 * acap_Criteria second = acap_ConstructCriteria(ACAP_SUBSTRING, "baz", "i;octet", "qux");
 * acap_Criteria result = acap_ConstructCriteria(ACAP_AND, ACAP_CRITERIA, first,
 * 							   ACAP_CRITERIA, second);
 * ... except then it'd be your responsibility to free up
 * first, second, *and* result when you were done.
 */

typedef enum {
  ACAP_METADATADONE,
  ACAP_METADATAPTR,		/* acap_Metadata; terminates */
  ACAP_METAACL,			/* no parameters */
  ACAP_METAATTRIBUTE,		/* no parameters */
  ACAP_METAMYRIGHTS,		/* no parameters */
  ACAP_METASIZE,		/* no parameters */
  ACAP_METASUBVALUE,		/* start, end offsets (unsigned) */
  ACAP_METAVALUE		/* no parameters */
} acap_MetadataTag;

typedef struct _acap_Metadata *acap_Metadata;

acap_Metadata acap_ConstructMetadata(acap_MetadataTag tag, ...);

/* Example:
 *
 * acap_ConstructMetadata(ACAP_ATTRIBUTE, ACAP_VALUE, ACAP_SUBVALUE, 1, 2, ACAP_METADATADONE);
 */

typedef enum {
  ACAP_RETURNDONE,
  ACAP_RETURNPTR,		/* acap_Return; terminates */
  ACAP_RETURNATTR		/* attribute, metadata */
} acap_ReturnTag;
typedef struct _acap_Return *acap_Return;
acap_Return acap_ConstructReturn(acap_ReturnTag tag, ...);

/* Examples:
 *
 * acap_ConstructReturn(ACAP_RETURNDONE);
 * acap_ConstructReturn(ACAP_RETURNATTR, "entry", ACAP_METADATADONE, ACAP_RETURNDONE);
 * acap_ConstructReturn(ACAP_RETURNATTR, "entry", ACAP_ATTRIBUTE, ACAP_VALUE, ACAP_METADATADONE,
 *			ACAP_RETURNATTR, "foo", ACAP_MYRIGHTS, ACAP_METADATADONE,
 *			ACAP_RETURNDONE);
 * acap_Metadata foo = acap_ConstructMetadata(ACAP_MYRIGHTS, ACAP_METADATADONE);
 * acap_Return bar = acap_ConstructReturn(ACAP_RETURNATTR, "bar", ACAP_METADATAPTR, foo,
 * 					  ACAP_RETURNDONE);
 * acap_ConstructReturn(ACAP_RETURNATTR, "qux", ACAP_ACL, ACAP_METADATA, foo,
 *			ACAP_RETURNPTR, bar);
 */

typedef enum {
  ACAP_SORTDONE,
  ACAP_SORTPTR,			/* acap_Sort; terminates */
  ACAP_SORTATTR			/* attribute, ordering */
} acap_SortTag;
typedef struct _acap_Sort *acap_Sort;
acap_Sort acap_ConstructSort(acap_SortTag tag, ...);

/* Examples:
 * acap_ConstructSort(ACAP_SORTATTR, "entry", "+i;octet", ACAP_SORTDONE);
 * acap_ConstructSort(ACAP_SORTATTR, "entry", "+i;octet",
 *                    ACAP_SORTATTR, "foo.firstname", "-i;octet",
 * 		      ACAP_SORTDONE);
 */

typedef enum {
  ACAP_MODIFIERSDONE,
  ACAP_MODIFIERSPTR,		/* acap_Modifiers; terminates */
  ACAP_DEPTH,			/* one parameter - depth */
  ACAP_HARDLIMIT,		/* one parameter - the hard limit */
  ACAP_LIMIT,			/* two unsigned numbers */
  ACAP_MAKECONTEXT,		/* context name */
  ACAP_ENUMERATE,		/* enable context enumeration */
  ACAP_NOTIFY,			/* enable context notification */
  ACAP_NOINHERIT,		/* disable inheritance */
  ACAP_RETURN,			/* begins return list */
  ACAP_SORT			/* begins sort list */
} acap_ModifiersTag;

typedef struct _acap_Modifiers *acap_Modifiers;
acap_Modifiers acap_ConstructModifiers(acap_ModifiersTag tag, ...);

/* For a value, you either need to use VALUENIL, VALUEDEFAULT, SIMPLEVALUE,
 * or a sequence of one or more METAVALUE and VALUELISTs followed by a
 * VALUEDONE. */
typedef enum {
  ACAP_VALUEDONE,
  ACAP_VALUENIL,
  ACAP_VALUEDEFAULT,
  ACAP_VALUE,			/* followed by the actual value */
  ACAP_VALUEMETA,		/* followed by metadata (an acap_MetadataTag),
				 * and the value. */
  ACAP_VALUELIST		/* followed by metadata (an acap_MetadataTag),
				 * and the values (null-terminated char ** array). */
} acap_ValueTag;

typedef enum {
  ACAP_ATTRIBUTEDONE,
  ACAP_ATTRIBUTE		/* followed by attrname, then values */
} acap_AttributeTag;

typedef enum {
  ACAP_STOREFLAGSDONE,
  ACAP_NOCREATE,
  ACAP_UNCHANGEDSINCE		/* followed by the unchangedsince time */
} acap_StoreFlagTag;

typedef enum {
  ACAP_ENTRYDONE,
  ACAP_ENTRYPTR,		/* acap_Entry; terminates */
  ACAP_ENTRY			/* entry name, followed by flags, followed by
				 * attribute. */
} acap_EntryTag;

typedef struct _acap_Entry *acap_Entry;
acap_Entry acap_ConstructEntry(acap_EntryTag tag, ...);

/* An example using the above store tags:
 * acap_CmdStore(conn, &status, &info,
 *		      ACAP_ENTRY, "/byowner/usr/rob/options/atest",
 *		        ACAP_STOREFLAGSDONE,
 *		        ACAP_ATTRIBUTE, "atest.foo", ACAP_VALUE, "bar",
 *		        ACAP_ATTRIBUTEDONE,
 *		      ACAP_ENTRYDONE));
 */

/* ACAP authentication mechanism support.
 *
 * An authentication mechanism will initially be called with
 * status=ACAP_AUTH_MECHANISM, and should use the response function to
 * send the SASL mechanism name.  The mechanism will then be called
 * with status=ACAP_AUTH_INITIAL; the mechanism should send whatever
 * initial data is appropriate (if the mechanism does not have any
 * initial data to send, it should simply return).  Each server
 * challenge will be presented to the mechanism (with
 * status=ACAP_AUTH_INPROGRESS), which should use the response
 * function to send its response.  Upon command completion, the auth
 * mechanism will be called one last time, with
 * status=ACAP_AUTH_SUCCESS or ACAP_AUTH_FAILURE, depending on the
 * command completion code. */

/* For either the initial or any of the intermediate challenges, an
 * authentication mechanism may return 0 for success, or anything else
 * for failure; in this case, the authentication sequence will be
 * aborted (and the mechanism will be called once more, with
 * ACAP_AUTH_FAILURE). */

typedef enum {
  ACAP_AUTH_MECHANISM,
  ACAP_AUTH_INITIAL,
  ACAP_AUTH_INPROGRESS,
  ACAP_AUTH_SUCCESS,
  ACAP_AUTH_FAILURE
} acap_AuthStatus;

typedef int (*acap_AuthMech)(acap_Connection conn,
			     void *clientdata,
			     acap_AuthStatus status,
			     acap_String *data,
			     void (*response)(acap_Connection conn,
					      acap_String *data));

/* SASL mechanisms can optionally insert security layers.
 * Here're the APIs they should implement and use: */

/* This should always succeed */
typedef void (*acap_SecEncryptProc)(void *clientdata,
				    void *buffer,
				    size_t buffer_size);

/* This should return zero on success (if it returns anything
 * else, we WILL assume that the connection has been hijacked). */
typedef int (*acap_SecDecryptProc)(void *clientdata,
				   void *buffer,
				   size_t buffer_size);

/* This will be called when the connection is closed. */
typedef void (*acap_SecFreeProc)(void *clientdata);

/* This should only be called by the AuthMech upon
 * receiving ACAP_AUTH_SUCCESS. */
/* XXX For now, actually using a security layer will cause
 * things to break horribly, since the library isn't actually
 * smart enough to do decryption yet.  TODO.  XXX.  */
void acap_InsertSecurityLayer(acap_Connection conn,
			      void *clientdata,
			      acap_SecEncryptProc encrypt,
			      size_t encrypt_buffer_size,
			      acap_SecDecryptProc decrypt,
			      size_t decrypt_buffer_size,
			      acap_SecFreeProc free);

/* This is totally arbitrary number... but there needs
 * to be some maximum.  Note that increasing this
 * increases the size of an acap_Connection... */
#define ACAP_MAX_SEC_BUFFER_SIZE 512

/* These are for opening a connection to an ACAP server.
 * The address is specified using a struct sockaddr_in, because that's
 * the right datastructure for passing an internet address/port combo.
 * Far be it from us to suggest how you should do name resolution...
 */

/* This opens an acap connection to a server, using whatever the default
 * system mechanisms are for looking up the server, and sending
 * and receiving data. */
extern acap_Connection acap_Open(const char *server);

/* For those who want more control, acap_OpenConnection
 * allows you to specify a readproc, writeproc, and closeproc,
 * as follows: */

/* acap_ReadProc will be called by the library when it needs
 * more data.  It may longjmp out (in which case, acap_Parse()
 * should be called to resume parsing), or place some number of octets
 * (up to buffer_size) into the buffer, and return the number of
 * octets read, or 0 if the remove side closes the connection
 * (or something else which will prevent the read from ever succeeding).
 */
typedef size_t (*acap_ReadProc)(void *clientdata,
				void *buffer,
				size_t buffer_size);

/* acap_WriteProc will be called when the library needs to
 * send data.  It should write or buffer the data, and
 * return normally (the library assumes that data will
 * always be written if the connection's alive; if it's not,
 * you might as well throw the data away anyway... :-)
 */
typedef void (*acap_WriteProc)(void *clientdata,
			       const void *buffer,
			       size_t buffer_size);

/* This will be called when the library's written a complete
 * command; it should flush the output channel (if needed). */
typedef void (*acap_FlushProc)(void *clientdata);

/* This will be called by acap_Close(). */
typedef void (*acap_CloseProc)(void *clientdata);

/* And finally, acap_OpenConnection allows the creation of an acap
 * connection with a specified readproc, writeproc, and closeproc.
 * There's no openproc; it's assumed that you've setup clientData
 * before calling acap_OpenConnection such that the read, write, and
 * closeprocs can do their thing.*/

extern acap_Connection acap_OpenConnection(void *clientData,
					   acap_ReadProc readproc,
					   acap_WriteProc writeproc,
					   acap_FlushProc flushproc,
					   acap_CloseProc closeproc);

/* All good things must come to an end... */
extern void acap_Close(acap_Connection conn);

/* acap_Loop parses responses from the acap stream and applies
 * callbacks until either a callback longjmps, the acap_readproc
 * longjmps, or the acap_readproc returns zero (connection closed). */
extern int acap_Parse(acap_Connection conn);

/* These commands are turned into requests which're
 * written to the connection.  The asynchronous interfaces
 * register callbacks; the regular versions wait for command
 * completion, and return.
 *
 * Note that only one thread should be sending a command
 * at a time.  (As long as the connection's writeproc doesn't
 * block, this should be completely acceptable...)
 *
 * The synchronous interfaces may be given NULLs for
 * either their status_ret or info_ret parameters if the
 * caller doesn't care about the results.
 */

/* Authenticate */
extern int acap_CmdAuthenticate(acap_Connection conn,
				acap_Status *status_ret,
				char **info_ret,
				acap_AuthMech authmech,
				void *mechdata);

/* DeletedSince */
extern int acap_CmdDeletedsince(acap_Connection conn,
				acap_Status *status_ret,
				char **info_ret,
				const char *dataset,
				const char *modtime);

extern int acap_ACmdDeletedsince(acap_Connection conn,
				 acap_CmdCallback *callback,
				 void *callback_data,
				 const char *dataset,
				 const char *modtime);

/* FreeContext */
extern int acap_CmdFreeContext(acap_Connection conn,
			       acap_Status *status_ret,
			       char **info_ret,
			       const char *context);

extern int acap_ACmdFreeContext(acap_Connection conn,
				acap_CmdCallback *callback,
				void *callback_data,
				const char *context);

/* Lang - terminate langauge list with NULL.
 * (should these take arrays instead?) */
extern int acap_CmdLang(acap_Connection conn,
			acap_Status *status_ret,
			char **info_ret, ...);

extern int acap_ACmdLang(acap_Connection conn,
			 acap_CmdCallback *callback,
			 void *callback_data, ...);

/* Logout */
extern int acap_CmdLogout(acap_Connection conn,
			  acap_Status *status_ret,
			  char **info_ret);

extern int acap_ACmdLogout(acap_Connection conn,
			   acap_CmdCallback *callback,
			   void *callback_data);

/* Noop */
extern int acap_CmdNoop(acap_Connection conn,
			acap_Status *status_ret,
			char **info_ret);

extern int acap_ACmdNoop(acap_Connection conn,
			 acap_CmdCallback *callback,
			 void *callback_data);

/* Search
 * After the modifiers (however they're specified) should be the
 * criteria (however it's specified). */
extern int acap_CmdSearch(acap_Connection conn,
			  acap_Status *status_ret,
			  char **info_ret,
			  acap_DataCallback *data,
			  void *dataData,
			  const char *dataset,
			  acap_ModifiersTag tag, ...);
extern int acap_ACmdSearch(acap_Connection conn,
			   acap_CmdCallback *callback,
			   void *callback_data,
			   acap_DataCallback *data,
			   void *dataData,
			   const char *dataset,
			   acap_ModifiersTag tag, ...);

/* Store - takes acap_Entry constructor tags */
extern int acap_CmdStore(acap_Connection conn,
			 acap_Status *status_ret,
			 char **info_ret,
			 acap_EntryTag tag,
			 ...);
extern int acap_ACmdStore(acap_Connection conn,
			  acap_CmdCallback *callback,
			  void *callback_data,
			  acap_EntryTag tag,
			  ...);

/* UpdateContext - terminate context list with NULL.
 * (should these take arrays instead?) */
extern int acap_CmdUpdateContext(acap_Connection conn,
				 acap_Status *status_ret,
				 char **info_ret,
				 const char *first, ...);

extern int acap_ACmdUpdateContext(acap_Connection conn,
				  acap_CmdCallback *callback,
				  void *callback_data,
				  const char *first, ...);

#ifdef __cplusplus
}
#endif

#endif /* ACAP_H */
