/* acapint.h - internal acap library definitions
 * Rob Earhart
 * $Id: acapint.h,v 1.8 1997/11/17 22:46:52 rob Exp $
 */
/***********************************************************
        Copyright 1997 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#ifndef ACAPINT_H
#define ACAPINT_H

#include <stdio.h>
#include <sys/types.h>
#include <stdarg.h>
#include <setjmp.h>
#include "acap.h"

typedef struct {
  acap_CmdCallback *final;
  void *finalData;
  acap_DataCallback *data;
  void *dataData;
} _acap_callback;

#define ACAP_DATA_BUF_SIZE (ACAP_MAX_SEC_BUFFER_SIZE + 4)
#define ACAP_MAX_QSTRING_SIZE 1024

struct _acap_Connection {
  int isopen;			/* whether the connection's considered open */
  acap_ReadProc readproc;
  acap_WriteProc writeproc;
  acap_FlushProc flushproc;
  acap_CloseProc closeproc;
  void *clientdata;
  acap_SecEncryptProc encryptproc;
  acap_SecDecryptProc decryptproc;
  size_t enc_buf_sz, dec_buf_sz;
  acap_SecFreeProc secfreeproc;
  void *secdata;
  int needsp;			/* when sending; whether we need a space now */
  size_t lit_size;		/* size of literal we're reading */
  size_t lit_off;		/* offset in literal we're reading */
  int state;			/* current state */
  acap_Status status;		/* current status */
  int datan;			/* current data number */
  acap_String **data;		/* current data buffer */
  _acap_callback *cb;		/* current tag's callback, for parsing */
  _acap_callback *continuation;	/* callback to use for continuations */
  size_t in_len;		/* length of valid data in incoming */
  size_t in_pos;		/* current position in incoming data */
  size_t sec_len;		/* total length of data in incoming */
  size_t sec_pos;		/* current starting decryption position in incoming data */
  size_t out_len;		/* amount of data in outgoing */
  char incoming[ACAP_DATA_BUF_SIZE]; /* buffer of incoming data */
  char outgoing[ACAP_DATA_BUF_SIZE]; /* buffer of outgoing data */
};

/* Prototype for functions that send complex objects */
typedef int (_acap_ObjectSender)(acap_Connection, void *object);

typedef enum {
  ACAP_DONE,
  ACAP_ATOM,
  ACAP_CRLF,
  ACAP_STRING,
  ACAP_CSTRING,
  ACAP_UNSPACE,
  ACAP_LPAREN,
  ACAP_RPAREN,
  ACAP_INT,
  ACAP_UINT,
  ACAP_OBJECT			/* argument is an _acap_ObjectSender */
} _acap_token;

typedef struct {
  jmp_buf env;
  acap_Status *status_ret;
  char **info_ret;
} _acap_syndata;

extern acap_CmdCallback _acap_SynFinal;

/* This is a fairly evil little macro used by synchronous
 * commands that don't use _acap_SCmd. */
#define SYNDATA					\
  _acap_syndata syndata;			\
  if (setjmp(syndata.env)) return 0;		\
  syndata.status_ret = status_ret;		\
  syndata.info_ret = info_ret;

/* This is a pretty twisted macro, used to create data constructors. */
#define _ACAP_CONSTRUCT(name)						\
acap_ ## name acap_Construct ## name (acap_ ## name ## Tag tag, ...) {	\
  acap_ ## name buffer, b;						\
  va_list ap;								\
  jmp_buf bad;								\
  size_t size;								\
									\
  if (setjmp(bad)) return NULL;						\
  va_start(ap, tag);							\
  size = _acap_ ## name ## Size(tag, &ap, &bad);			\
  va_end(ap);								\
  buffer = (acap_ ## name)malloc(size);					\
  if (! buffer) return NULL;						\
  b = buffer;								\
  va_start(ap, tag);							\
  _acap_ ## name ## Construct(tag, &ap, &b);				\
  va_end(ap);								\
  return buffer;							\
}

/* Some object functions */
extern size_t _acap_CriteriaSize(acap_CriteriaTag tag,
				 va_list *ap,
				 jmp_buf *bad);
extern void _acap_CriteriaConstruct(acap_CriteriaTag tag,
				    va_list *ap,
				    acap_Criteria *buffer);
extern int _acap_SendCriteria(acap_Connection conn,
			      acap_Criteria c);

extern size_t _acap_ModifiersSize(acap_ModifiersTag tag,
				  va_list *ap,
				  jmp_buf *bad);
extern void _acap_ModifiersConstruct(acap_ModifiersTag tag,
				     va_list *ap,
				     acap_Modifiers *buffer);
extern int _acap_SendModifiers(acap_Connection conn,
			       acap_Modifiers m);

/* This is the general function for sending a stream of tokens... */
extern int _acap_SendTokens(acap_Connection conn, ...);

/* Functions for sending commands */
extern int _acap_VACmd(acap_Connection conn,
		       acap_CmdCallback *final,
		       void *finalData,
		       acap_DataCallback *data,
		       void *dataData,
		       va_list ap);

extern int _acap_ACmd(acap_Connection conn,
		      acap_CmdCallback *final,
		      void *finalData,
		      acap_DataCallback *data,
		      void *dataData,
		      ...);

extern int _acap_SCmd(acap_Connection conn,
		      acap_Status *status_ret,
		      char **info_ret,
		      ...);

#endif /* ACAPINT_H */
