/* acmd.c - asynchronous command interface
 * Rob Earhart
 * $Id: acmd.c,v 1.6 1997/11/17 22:46:53 rob Exp $
 */
/***********************************************************
        Copyright 1997 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#include <stdarg.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "acapint.h"

static inline void writedata(acap_Connection conn,
			     const void *buffer,
			     size_t buffer_size) {
  /* XXX Eventually, we need to do encryption here... XXX */
  (conn->writeproc)(conn->clientdata, buffer, buffer_size);
}

static inline void flushdata(acap_Connection conn) {
  (conn->flushproc)(conn->clientdata);
}

static inline int canquote(char *s) {
  /* I'd write this recursively, but I don't trust
   * C compilers to implement tail-recursion.
   * Since it'd take more work to support string with embedded
   * quoted-specials, we just consider them to be non-quoteable. */
  unsigned n;
  for (n = 1024; n; n--, s++)
    if (! *s) return 1;
    else if ((unsigned char)*s & 0x80
	     || *s == '\r'
	     || *s == '\n'
	     || *s == '\\'
	     || *s == '"') return 0;
  return 0;
}

static inline int canquotelen(unsigned char *s, size_t len) {
  if (len > 1024) return 0;
  for (; len; len--, s++)
    if ((unsigned char)*s & 0x80
	|| *s == '\r'
	|| *s == '\n'
	|| *s == '\\'
	|| *s == '"') return 0;
  return 1;
}

static inline void sendchar(acap_Connection conn, char c) {
  writedata(conn, &c, 1);
}

static int vsendacap(acap_Connection conn,
		     va_list ap) {
  char buf[64];			/* big enough for intermediate strings */
  while (1) {
    switch(va_arg(ap, _acap_token)) {
    case ACAP_DONE:
      return 0;
    case ACAP_ATOM:
      if (conn->needsp) sendchar(conn, ' ');
      {
	char *atom = va_arg(ap, char *);
	writedata(conn, atom, strlen(atom));
      }
      conn->needsp = 1;
      break;
    case ACAP_CRLF:
      {
	writedata(conn, "\r\n", 2);
      }
      conn->needsp = 0;
      break;
    case ACAP_STRING:
      {
	acap_String *str;
	if (conn->needsp) sendchar(conn, ' ');
	str = va_arg(ap, acap_String *);
	if (canquotelen(str->str, str->length)) {
	  sendchar(conn, '\"');
	  writedata(conn, str->str, str->length);
	  sendchar(conn, '\"');
	} else {
	  sprintf(buf, "{%d+}\r\n", str->length);
	  writedata(conn, buf, strlen(buf));
	  writedata(conn, str->str, str->length);
	}
	conn->needsp = 1;
	break;
      }
    case ACAP_CSTRING:
      {
	char *str;
	if (conn->needsp) sendchar(conn, ' ');
	str = va_arg(ap, char *);
	if (canquote(str)) {
	  sendchar(conn, '\"');
	  writedata(conn, str, strlen(str));
	  sendchar(conn, '\"');
	} else {
	  size_t sz = strlen(str);
	  sprintf(buf, "{%d+}\r\n", sz);
	  writedata(conn, buf, strlen(buf));
	  writedata(conn, str, sz);
	}
	conn->needsp = 1;
	break;
      }
    case ACAP_UNSPACE:
      /* GRN.  Couldn't John&Chris have used the same style everywhere?  Grn. */
      conn->needsp = 0;		/* force an antispace. */
      break;
    case ACAP_LPAREN:
      if (conn->needsp) sendchar(conn, ' ');
      sendchar(conn, '(');
      conn->needsp = 0;
      break;
    case ACAP_RPAREN:
      sendchar(conn, ')');
      conn->needsp = 1;
      break;
    case ACAP_INT:
      if (conn->needsp) sendchar(conn, ' ');
      sprintf(buf, "%d", va_arg(ap, int));
      writedata(conn, buf, strlen(buf));
      conn->needsp = 1;
      break;
    case ACAP_UINT:
      if (conn->needsp) sendchar(conn, ' ');
      sprintf(buf, "%u", va_arg(ap, unsigned));
      writedata(conn, buf, strlen(buf));
      conn->needsp = 1;
      break;
    case ACAP_OBJECT:
      {
	_acap_ObjectSender *sender = va_arg(ap, _acap_ObjectSender *);
	if (sender(conn, va_arg(ap, void *)))
	  return -1;
	break;
      }
    default:
      errno = EINVAL;
      return -1;
    } /* switch(next_token) */
  } /* while(1) */
}

int _acap_SendTokens(acap_Connection conn, ...) {
  int ret;
  va_list ap;
  va_start(ap, conn);
  ret = vsendacap(conn, ap);
  va_end(ap);
  return ret;
}

int _acap_VACmd(acap_Connection conn,
		acap_CmdCallback *final,
		void *finalData,
		acap_DataCallback *data,
		void *dataData,
		va_list ap) {
  _acap_callback *cb;
  int res;

  /* Okay!  Time to send a command down the pipe... */

  if (! (cb = malloc(sizeof(_acap_callback))))
    return -1;
  cb->final = final;
  cb->finalData = finalData;
  cb->data = data;
  cb->dataData = dataData;

  /* XXX This Sucks - but it's fast to write.  So, uh...
   * optimize this later.  :-) */
  {
    char buf[64];
    sprintf(buf, "%p", cb);
    writedata(conn, buf, strlen(buf));
  }
  conn->needsp = 1;
  res = vsendacap(conn, ap);
  va_end(ap);

  sendchar(conn, '\r');
  sendchar(conn, '\n');

  flushdata(conn);
  
  return res;
}

int _acap_ACmd(acap_Connection conn,
	       acap_CmdCallback *final,
	       void *finalData,
	       acap_DataCallback *data,
	       void *dataData,
	       ...) {
  va_list ap;
  int res;

  va_start(ap, dataData);
  res = _acap_VACmd(conn, final, finalData, data, dataData, ap);
  va_end(ap);
  return res;
}
