/* atest.c - simple libacap test program
 * Rob Earhart
 * $Id: atest.c,v 1.9 1997/11/17 22:53:40 rob Exp $
 */
/***********************************************************
        Copyright 1997 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <setjmp.h>
#include "acap.h"
#include "acapkrb.h"

static const char rcsid[] = "$Id: atest.c,v 1.9 1997/11/17 22:53:40 rob Exp $";

static jmp_buf jb;

static void logged_in(acap_Connection conn,
		      void *clientData,
		      acap_Status status,
		      char *info) {
  printf("Logged in async; status is %d, info is %s\n", status, info);
  free(info);
  longjmp(jb, 1);
}

static void untagged_info(acap_Connection conn,
			  void *clientData,
			  acap_Status status,
			  char *info) {
  printf("Untagged info: status %d, info %s\n", status, info);
  free(info);
}

static void finished_search(acap_Connection conn,
			    void *clientData,
			    acap_Status status,
			    char *info) {
  printf("Finished search: %d: %s\n", status, info);
  free(info);
  acap_CmdLogout(conn, NULL, NULL);
}

static void search_data(acap_Connection conn,
			void *clientData,
			acap_Status status,
			acap_String **data) {
  acap_String **s;
  switch(status) {
  case ACAP_RESPONSE_ENTRY:
    puts("Got a search result:");
    break;
  case ACAP_RESPONSE_MODTIME:
    puts("Got a search modtime:");
    break;
  default:
    puts("Got something unexpected from the search:");
  }
  for (s = data; *s; s++) {
    puts((char *)(*s)->str);
    free(*s);
  }
  puts("(end of search results)");
  free(data);
}

#define CHECK(test) {				\
  if (test) {					\
    perror(argv[0]);				\
    if (info) free(info);			\
    return 1;					\
  }						\
}

int main(int argc, char *argv[]) {
  acap_Connection conn;
  acap_Status status;
  char *info = NULL;
  acap_Criteria crit;
  void *authdat;

  if (argc != 2) {
    fprintf(stderr, "Usage: %s <acap server>\n", argv[0]);
    return 1;
  }

  conn = acap_Open(argv[1]);
  if (! conn) { perror(argv[0]); exit(1); }
  if (setjmp(jb)) {
    acap_Close(conn);
    return 0;
  }

  authdat = acapkrb_MakeAuthData(argv[1], NULL);
  CHECK(! authdat);
  CHECK(acap_CmdAuthenticate(conn, &status, &info, &acapkrb_Auth, authdat));
  free(authdat);
  printf("Logged in sync; status is %d, info is %s\n", status, info);
  if (status != ACAP_OK) {
    puts("Authentication failed - aborting");
    return 1;
  }
  free(info);
  info = NULL;
  
  CHECK(acap_CmdStore(conn, &status, &info,
		      ACAP_ENTRY, "/byowner/usr/rob/options/atest",
		        ACAP_STOREFLAGSDONE,
		        ACAP_ATTRIBUTE, "atest.foo", ACAP_VALUEMETA, "value", "bar", ACAP_VALUEDONE,
		        ACAP_ATTRIBUTEDONE,
		      ACAP_ENTRYDONE));
  printf("Store completed; status is %d, info is %s\n", status, info);
  free(info);
  info = NULL;
		
  crit = acap_ConstructCriteria(ACAP_AND, ACAP_COMPARE, "entry", "+i;octet", "aardvark",
					  ACAP_COMPARESTRICT, "entry", "-i;octet", "atrocious");
  if (! crit) {
    puts("NULL criteria constructed - aborting");
    return 1;
  }

  if (acap_ACmdSearch(conn,
		      &finished_search, NULL,
		      &search_data, NULL,
		      "/byowner/usr/rob/options",
		      ACAP_RETURN, ACAP_RETURNATTR, "entry", ACAP_METADATADONE,
		                   ACAP_RETURNATTR, "atest.foo",
		                     ACAP_METAATTRIBUTE, ACAP_METAVALUE, ACAP_METADATADONE,
		      ACAP_RETURNDONE,
		      ACAP_MODIFIERSDONE,
		      ACAP_CRITERIAPTR, crit)) {
    puts("acap_ACmdSearch failed - aborting");
    return 1;
  }
  free(crit);
  CHECK(acap_Parse(conn));
  acap_Close(conn);
  return 0;
}
