/* authenticate.c - authenticate command implementation
 * Rob Earhart
 * $Id: authenticate.c,v 1.2 1997/11/17 03:59:09 rob Exp $
 */
/***********************************************************
        Copyright 1997 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#include <setjmp.h>
#include <stdlib.h>
#include "acapint.h"

typedef struct {
  acap_AuthMech mech;
  void *mechdata;
  jmp_buf respenv;
  jmp_buf cmdenv;
  acap_Status *status_ret;
  char **info_ret;
} authdat;

static void sendstr(acap_Connection conn,
		    acap_String *data) {
  _acap_SendTokens(conn,
		   ACAP_STRING, data,
		   ACAP_DONE);
}

static void auth_respcallback(acap_Connection conn,
			      void *clientdata,
			      acap_Status status,
			      acap_String **data) {
  authdat *dat = clientdata;
  if (data) {
    if (! data[0]
	|| (dat->mech)(conn, dat->mechdata, ACAP_AUTH_INPROGRESS, data[0], &sendstr))
      _acap_SendTokens(conn, ACAP_ATOM, "*", ACAP_DONE);
    if (data[0]) free(data[0]);
    free(data);
  }
  longjmp(dat->respenv, 1);
}

static void auth_cmdcallback(acap_Connection conn,
			     void *clientdata,
			     acap_Status status,
			     char *info) {
  authdat *dat = clientdata;
  (dat->mech)(conn, dat->mechdata,
	      status == ACAP_OK ? ACAP_AUTH_SUCCESS : ACAP_AUTH_FAILURE,
	      NULL, &sendstr);
  if (dat->status_ret) *dat->status_ret = status;
  if (dat->info_ret) *dat->info_ret = info;
  else free(info);
  conn->continuation = NULL;
  longjmp(dat->cmdenv, 1);
}

static int sendauth(acap_Connection conn,
		    authdat *dat) {
  _acap_callback cb;
  cb.final = NULL;
  cb.finalData = NULL;
  cb.data = &auth_respcallback;
  cb.dataData = dat;
  conn->continuation = &cb;

  (dat->mech)(conn, dat->mechdata, ACAP_AUTH_MECHANISM, NULL, &sendstr);
  (dat->mech)(conn, dat->mechdata, ACAP_AUTH_INITIAL, NULL, &sendstr);

  while(1) {
    _acap_SendTokens(conn,
		     ACAP_CRLF,
		     ACAP_DONE);
    (conn->flushproc)(conn->clientdata);
    if (! setjmp(dat->respenv))
      acap_Parse(conn);
  }
}

int acap_CmdAuthenticate(acap_Connection conn,
			 acap_Status *status_ret,
			 char **info_ret,
			 acap_AuthMech authmech,
			 void *mechdata) {
  authdat dat;
  dat.mech = authmech;
  dat.mechdata = mechdata;
  dat.status_ret = status_ret;
  dat.info_ret = info_ret;
  if (setjmp(dat.cmdenv)) {
    conn->continuation = NULL;
    return 0;
  }

  return _acap_ACmd(conn,
		    &auth_cmdcallback,
		    &dat,
		    NULL,
		    NULL,
		    ACAP_ATOM, "Authenticate",
		    ACAP_OBJECT, &sendauth, &dat,
		    ACAP_DONE);
}
