/* criteria.c - criteria constructors
 * Rob Earhart
 * $Id: criteria.c,v 1.5 1997/11/17 22:46:59 rob Exp $
 */
/***********************************************************
        Copyright 1997 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#include <errno.h>
#include <stdlib.h>
#include "acapint.h"

struct _acap_Criteria {
  acap_CriteriaTag tag;
  union {
    struct {			/* for sub-components */
      struct _acap_Criteria *a, *b;
    } v;
    struct {			/* for comparisons */
      const char *attribute;	/* XXX should be length-strs, not char *s */
      const char *ordering;
      const char *value;
    } c;
    struct {
      unsigned low, high;
    } r;
    acap_Criteria criteria;
  } v;
};

size_t _acap_CriteriaSize(acap_CriteriaTag tag, va_list *ap, jmp_buf *bad) {
  char *foo;
  size_t size = sizeof(struct _acap_Criteria);
  switch(tag) {
  case ACAP_AND:
  case ACAP_OR:
    tag = va_arg((*ap), acap_CriteriaTag);
    size += _acap_CriteriaSize(tag, ap, bad);
    /* fallthrough */
  case ACAP_NOT:
    tag = va_arg((*ap), acap_CriteriaTag);
    size += _acap_CriteriaSize(tag, ap, bad);
    /* and keep falling through... */
  case ACAP_COMPARE:
  case ACAP_COMPARESTRICT:
  case ACAP_CRITERIAPTR:
  case ACAP_EQUAL:
  case ACAP_PREFIX:
  case ACAP_SUBSTRING:
    foo = va_arg((*ap), char *);
  case ACAP_RANGE:
    foo = va_arg((*ap), char *);
    foo = va_arg((*ap), char *);
  case ACAP_ALL:
    break;
  default:
    longjmp(*bad, 1);
  }
  return size;
}

void _acap_CriteriaConstruct(acap_CriteriaTag tag,
			     va_list *ap,
			     acap_Criteria *crit) {
  acap_Criteria c = (*crit)++;
  c->tag = tag;
  switch (tag) {
  case ACAP_AND:
  case ACAP_OR:
    c->v.v.a = *crit;
    tag = va_arg((*ap), acap_CriteriaTag);
    _acap_CriteriaConstruct(tag, ap, crit);
    /* fallthrough */
  case ACAP_NOT:
    c->v.v.b = *crit;
    tag = va_arg((*ap), acap_CriteriaTag);
    _acap_CriteriaConstruct(tag, ap, crit);
    break;
  case ACAP_ALL:
    break;
  case ACAP_CRITERIAPTR:
    c->v.criteria = va_arg((*ap), acap_Criteria);
    break;
  case ACAP_COMPARE:
  case ACAP_COMPARESTRICT:
  case ACAP_EQUAL:
  case ACAP_PREFIX:
  case ACAP_SUBSTRING:
    c->v.c.attribute = va_arg((*ap), const char *);
    c->v.c.ordering = va_arg((*ap), const char *);
    c->v.c.value = va_arg((*ap), const char *);
    break;
  case ACAP_RANGE:
    c->v.r.low = va_arg((*ap), unsigned);
    c->v.r.high = va_arg((*ap), unsigned);
    break;
  }
}

_ACAP_CONSTRUCT(Criteria)

int _acap_SendCriteria(acap_Connection conn, acap_Criteria c) {
  int res;
  switch(c->tag) {
  case ACAP_ALL:
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "All",
			   ACAP_DONE);
    break;
  case ACAP_AND:
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "And",
			   ACAP_OBJECT, &_acap_SendCriteria, c->v.v.a,
			   ACAP_OBJECT, &_acap_SendCriteria, c->v.v.b,
			   ACAP_DONE);
    break;
  case ACAP_COMPARE:
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "Compare",
			   ACAP_CSTRING, c->v.c.attribute,
			   ACAP_CSTRING, c->v.c.ordering,
			   ACAP_CSTRING, c->v.c.value,
			   ACAP_DONE);
    break;
  case ACAP_COMPARESTRICT:
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "CompareStrict",
			   ACAP_CSTRING, c->v.c.attribute,
			   ACAP_CSTRING, c->v.c.ordering,
			   ACAP_CSTRING, c->v.c.value,
			   ACAP_DONE);
    break;
  case ACAP_CRITERIAPTR:
    res = _acap_SendCriteria(conn, c->v.criteria);
    break;
  case ACAP_EQUAL:
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "Equal",
			   ACAP_CSTRING, c->v.c.attribute,
			   ACAP_CSTRING, c->v.c.ordering,
			   ACAP_CSTRING, c->v.c.value,
			   ACAP_DONE);
    break;
  case ACAP_NOT:
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "Not",
			   ACAP_OBJECT, &_acap_SendCriteria, c->v.v.b,
			   ACAP_DONE);
    break;
  case ACAP_OR:
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "Or",
			   ACAP_OBJECT, &_acap_SendCriteria, c->v.v.a,
			   ACAP_OBJECT, &_acap_SendCriteria, c->v.v.b,
			   ACAP_DONE);
    break;
  case ACAP_PREFIX:
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "Prefix",
			   ACAP_CSTRING, c->v.c.attribute,
			   ACAP_CSTRING, c->v.c.ordering,
			   ACAP_CSTRING, c->v.c.value,
			   ACAP_DONE);
    break;
  case ACAP_RANGE:
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "Range",
			   ACAP_UINT, c->v.r.low,
			   ACAP_UINT, c->v.r.high,
			   ACAP_DONE);
    break;
  case ACAP_SUBSTRING:
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "Substring",
			   ACAP_CSTRING, c->v.c.attribute,
			   ACAP_CSTRING, c->v.c.ordering,
			   ACAP_CSTRING, c->v.c.value,
			   ACAP_DONE);
    break;
  default:
    errno = EINVAL;
    res = -1;
  }
  return res;
}
