/* krb.c - ACAP kerberos authentication implementation
 * Rob Earhart
 * $Id: krb.c,v 1.2 1997/11/17 22:47:00 rob Exp $
 */
/***********************************************************
        Copyright 1997 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#include "acap.h"
#include "acapkrb.h"
/* We're going to let des and krb define our types, if needed... */
#ifdef int32_t
#undef int32_t
#endif
#ifdef uint32_t
#undef uint32_t
#endif
#include <krb.h>
#include <des.h>
#include <stdlib.h>
#include <string.h>

typedef struct {
  int challenge_number;
  int err;
  char *hostname;
  char *realm;
  char *authid;
  int32_t challenge;		/* in host byte order */
  CREDENTIALS credentials;
} krbdata;

void *acapkrb_MakeAuthData(const char *hostname,
			   const char *authid) {
  krbdata *k;
  const char *hostend, *realm = krb_realmofhost(hostname);
  if (! (hostend = strchr(hostname, '.')))
    hostend = hostname + strlen(hostname);
  k = malloc(sizeof(krbdata) + hostend - hostname + 1
	     + strlen(realm) + 1
	     + (authid ? strlen(authid) + 1 : 0));
  if (k) {
    k->challenge_number = 0;
    k->err = 0;
    k->hostname = (char *)k + sizeof(krbdata);
    strncpy(k->hostname, hostname, hostend - hostname);
    k->hostname[hostend - hostname] = '\0';
    k->realm = k->hostname + (hostend - hostname) + 1;
    strcpy(k->realm, realm);
    if (authid) {
      k->authid = k->realm + strlen(k->realm) + 1;
      strcpy(k->authid, authid);
    } else
      k->authid = NULL;
  }
  return k;
}

const char *acapkrb_AuthErr(void *clientdata) {
  krbdata *k = (krbdata *)clientdata;
  if (! k) return NULL;
  return krb_err_txt[k->err];
}

int acapkrb_Auth(acap_Connection conn,
		 void *clientdata,
		 acap_AuthStatus status,
		 acap_String *data,
		 void (*response)(acap_Connection conn,
				  acap_String *data)) {
  krbdata *k = (krbdata *)clientdata;
  if (! k) return -1;
  switch(status) {
  case ACAP_AUTH_MECHANISM:
    {
      union {
	char space[sizeof(acap_String) + 11];
	acap_String str;
      } b;
      b.str.length = 11;
      strcpy((char *)b.str.str, "Kerberos_V4");
      (*response)(conn, &b.str);
    }
    break;
  case ACAP_AUTH_INITIAL:
    break;
  case ACAP_AUTH_INPROGRESS:
    switch(k->challenge_number) {
    case 0:
      {
	/* We should've just recieved a 32-bit number in network byte order.
	 * We want to reply with an authenticator. */
	KTEXT_ST authent;
	authent.length = MAX_KTXT_LEN;
	k->challenge_number++;
	if (data->length != 4) return -1;
	k->challenge = ntohl(*((int32_t *)data->str));
	if ((k->err = krb_mk_req(&authent, "acap",
				 (char *)k->hostname,
				 (char *)k->realm,
				 k->challenge)))
	  return -1;
	if ((k->err = krb_get_cred("acap",
				   (char *)k->hostname,
				   (char *)k->realm,
				   &k->credentials)))
	  return -1;
	/* We take advantage of the fact that a KTEXT_ST
	 * has the same layout as our acap_String type... */
	(*response)(conn, (acap_String *)&authent);
	break;
      }
    case 1:
      {
	/* We should've just recieved eight octets of ECB-encrypted data,
	 * containing the incremented checksum (in network byte order),
	 * a security octet (which we ignore, because we don't support
	 * any security layers yet, XXX), and a cipher-text buffer size
	 * (which, again, we ignore for now).  We want to reply with
	 * the initial checksum, the security layer, a ciphertext buffer length,
	 * and our authorization identity, all wrapped up with PCBC.
	 * Whew! */
	des_key_schedule keysched;
	union {
	  char buf[sizeof(acap_String)		/* base string size, plus a bonus word */
		  + 4				/* for the checksum */
		  + ANAME_SZ + INST_SZ + 2	/* auth id, with . and trailing NULL */
		  + 8				/* null padding */];
	  acap_String str;
	} retbuf;
	acap_String *retdat = &retbuf.str;
	int len;
	k->challenge_number++;
	if (data->length != 8) return -1;
	des_key_sched(k->credentials.session, keysched);
	des_ecb_encrypt(data->str, data->str, keysched, DES_DECRYPT);
	if (k->challenge + 1 != ntohl(*(int32_t *)data->str))
	  return -1;
	if (! (data->str[4] & 1))
	  return -1;		/* other side requires a security layer */
	*((int32_t *)retdat->str) = htonl(k->challenge);
	retdat->str[4] = 1;	/* no security layer - XXX */
	retdat->str[5] = retdat->str[6] = retdat->str[7] = 0;
	if (k->authid) {
	  len = strlen(k->authid);
	  if (len > ANAME_SZ + INST_SZ + 1)
	    return -1;
	  strcpy((char *)retdat->str + 8, k->authid);
	} else {
	  len = strlen(k->credentials.pname);
	  strcpy((char *)retdat->str + 8, k->credentials.pname);
	  if (k->credentials.pinst[0]) {
	    retdat->str[8 + len] = '.';
	    len += strlen(k->credentials.pinst);
	    strcpy((char *)retdat->str + 8 + len + 1, k->credentials.pinst);
	    len++;		/* for the . */
	  }
	}
	len += 9;		/* 8 for the initial bit, 1 for the \0 at the end */
	/* Fill in the rest with zeros... */
	while(len % 8) {
	  retdat->str[len] = '\0';
	  len++;
	}
	des_key_sched(k->credentials.session, keysched);
	des_pcbc_encrypt((unsigned char *)retdat->str,
			 (unsigned char *)retdat->str,
			 len, keysched, k->credentials.session, DES_ENCRYPT);
	retdat->length = len;
	(*response)(conn, retdat);
	break;
      }
    default:
      return -1;
    }
    break;
  case ACAP_AUTH_SUCCESS:
    /* Yee-ha!  Establish security layer... */
    break;
  case ACAP_AUTH_FAILURE:
    break;
  default:
    return -1;
  }
  return 0;
}
