/* modifiers.c - modifier construction
 * Rob Earhart
 * $Id: modifiers.c,v 1.6 1997/11/17 22:47:01 rob Exp $
 */
/***********************************************************
        Copyright 1997 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#include <stdlib.h>
#include <errno.h>
#include "acapint.h"

struct _acap_Metadata {
  acap_Metadata next;
  acap_MetadataTag tag;
  unsigned start, finish;	/* for ACAP_SUBVALUE */
};

static size_t _acap_MetadataSize(acap_MetadataTag tag, va_list *ap, jmp_buf *bad) {
  char *foo;
  size_t size = 0;
  do {
    size += sizeof(struct _acap_Metadata);
    switch(tag) {
    case ACAP_METADATADONE:
      return size;
    case ACAP_METADATAPTR:
      foo = va_arg((*ap), char *);
      return size;
    case ACAP_METAACL:
    case ACAP_METAATTRIBUTE:
    case ACAP_METAMYRIGHTS:
    case ACAP_METASIZE:
    case ACAP_METAVALUE:
      break;
    case ACAP_METASUBVALUE:
      foo = va_arg((*ap), char *);
      foo = va_arg((*ap), char *);
      break;
    default:
      longjmp(*bad, 1);
    }
    tag = va_arg((*ap), acap_MetadataTag);
  } while(1);
}

static void _acap_MetadataConstruct(acap_MetadataTag tag, va_list *ap, acap_Metadata *buffer) {
  do {
    acap_Metadata m = (*buffer)++;
    m->tag = tag;
    if (tag == ACAP_METASUBVALUE) {
      m->start = va_arg((*ap), unsigned);
      m->finish = va_arg((*ap), unsigned);
    }
    if (tag == ACAP_METADATADONE) {
      m->next = NULL;
      return;
    }
    if (tag == ACAP_METADATAPTR) {
      m->next = va_arg((*ap), acap_Metadata);
      return;
    }
    m->next = *buffer;
    tag = va_arg((*ap), acap_MetadataTag);
  } while(1);
}

_ACAP_CONSTRUCT(Metadata)

static int SendMetadata(acap_Connection conn,
			acap_Metadata m) {
  int res = 0;
  while(m && ! res) {
    switch(m->tag) {
    case ACAP_METADATADONE:
      return res;
    case ACAP_METADATAPTR:
      break;
    case ACAP_METAACL:
      res |= _acap_SendTokens(conn,
			      ACAP_CSTRING, "acl",
			      ACAP_DONE);
      break;
    case ACAP_METAATTRIBUTE:
      res |= _acap_SendTokens(conn,
			      ACAP_CSTRING, "attribute",
			      ACAP_DONE);
      break;
    case ACAP_METAMYRIGHTS:
      res |= _acap_SendTokens(conn,
			      ACAP_CSTRING, "myrights",
			      ACAP_DONE);
      break;
    case ACAP_METASIZE:
      res |= _acap_SendTokens(conn,
			      ACAP_CSTRING, "size",
			      ACAP_DONE);
      break;
    case ACAP_METASUBVALUE: {
      char buf[50];
      sprintf(buf, "value<%u,%u>", m->start, m->finish);
      res |= _acap_SendTokens(conn,
			      ACAP_CSTRING, buf,
			      ACAP_DONE);
      break;
    }
    case ACAP_METAVALUE:
      res |= _acap_SendTokens(conn,
			      ACAP_CSTRING, "value",
			      ACAP_DONE);
      break;
    default:
      errno = EINVAL;
      res = -1;
    }
    m = m->next;
  }
  return res;
}

struct _acap_Return {
  acap_Return next;
  acap_ReturnTag tag;
  const char *attribute;
  acap_Metadata metadata;
};

static size_t _acap_ReturnSize(acap_ReturnTag tag, va_list *ap, jmp_buf *bad) {
  acap_MetadataTag mtag;
  char *foo;
  size_t size = 0;
  do {
    size += sizeof(struct _acap_Return);
    switch (tag) {
    case ACAP_RETURNDONE:
      return size;
    case ACAP_RETURNPTR:
      foo = va_arg((*ap), char *);
      return size;
    case ACAP_RETURNATTR:
      foo = va_arg((*ap), char *);
      mtag = va_arg((*ap), acap_MetadataTag);
      size += _acap_MetadataSize(mtag, ap, bad);
      break;
    default:
      longjmp(*bad, 1);
    }
    tag = va_arg((*ap), acap_ReturnTag);
  } while(1);
}

static void _acap_ReturnConstruct(acap_ReturnTag tag, va_list *ap, acap_Return *buffer) {
  acap_MetadataTag mtag;
  do {
    acap_Return r = (*buffer)++;
    r->tag = tag;
    switch(tag) {
    case ACAP_RETURNDONE:
      r->next = NULL;
      return;
    case ACAP_RETURNPTR:
      r->next = va_arg((*ap), acap_Return);
      return;
    case ACAP_RETURNATTR:
      r->attribute = va_arg((*ap), char *);
      r->metadata = *(acap_Metadata *)buffer;
      mtag = va_arg((*ap), acap_MetadataTag);
      _acap_MetadataConstruct(mtag, ap, (acap_Metadata *)buffer);
      r->next = *buffer;
      break;
    }
    tag = va_arg((*ap), acap_ReturnTag);
  } while(1);
}

_ACAP_CONSTRUCT(Return)

static int SendReturn(acap_Connection conn,
		      acap_Return r) {
  int res = 0;
  while(r && ! res) {
    switch(r->tag) {
    case ACAP_RETURNDONE:
      return res;
    case ACAP_RETURNPTR:
      break;
    case ACAP_RETURNATTR:
      /* God, I hate special casing this.
       * I don't know why metadata's done like this - it doesn't make sense,
       * it goes against the style of the rest of the spec... you *always* should
       * have a space between a string and an lparen.
       * Anyway, this is the ONLY REASON why we need ACAP_UNSPACE.  Grn.  Hate. */
      if (r->metadata && r->metadata->tag != ACAP_METADATADONE)
	res |= _acap_SendTokens(conn,
				ACAP_CSTRING, r->attribute,
				ACAP_UNSPACE,
				ACAP_LPAREN,
				ACAP_OBJECT, SendMetadata, r->metadata,
				ACAP_RPAREN,
				ACAP_DONE);
      else
	res |= _acap_SendTokens(conn,
				ACAP_CSTRING, r->attribute,
				ACAP_DONE);
    }
    r = r->next;
  }
  return res;
}

struct _acap_Sort {
  acap_Sort next;
  acap_SortTag tag;
  const char *attribute;
  const char *comparator;
};

static size_t _acap_SortSize(acap_SortTag tag, va_list *ap, jmp_buf *bad) {
  char *foo;
  size_t size = 0;
  do {
    size += sizeof(struct _acap_Sort);
    switch(tag) {
    case ACAP_SORTDONE:
      return size;
    case ACAP_SORTPTR:
      foo = va_arg((*ap), char *);
      return size;
    case ACAP_SORTATTR:
      foo = va_arg((*ap), char *);
      foo = va_arg((*ap), char *);
      break;
    default:
      longjmp(*bad, 1);
    }
    tag = va_arg((*ap), acap_SortTag);
  } while(1);
}

static void _acap_SortConstruct(acap_SortTag tag, va_list *ap, acap_Sort *buffer) {
  do {
    acap_Sort s = (*buffer)++;
    s->tag = tag;
    switch(tag) {
    case ACAP_SORTDONE:
      s->next = NULL;
      return;
    case ACAP_SORTPTR:
      s->next = va_arg((*ap), acap_Sort);
      return;
    case ACAP_SORTATTR:
      s->attribute = va_arg((*ap), const char *);
      s->comparator = va_arg((*ap), const char *);
      s->next = *buffer;
      break;
    }
    tag = va_arg((*ap), acap_SortTag);
  } while(1);
}

_ACAP_CONSTRUCT(Sort)

static int SendSort(acap_Connection conn,
		    acap_Sort s) {
  int res = 0;
  while(s && !res) {
    switch(s->tag) {
    case ACAP_SORTDONE:
      return res;
    case ACAP_SORTPTR:
      break;
    case ACAP_SORTATTR:
      res |= _acap_SendTokens(conn,
			      ACAP_CSTRING, s->attribute,
			      ACAP_CSTRING, s->comparator,
			      ACAP_DONE);
      break;
    }
    s = s->next;
  }
  return res;
}

struct _acap_Modifiers {
  int ptr;
  union {
    struct {
      int depth;
      int lowlimit, highlimit;
      int hardlimit;
      int inherit;
      const char *context;
      int enumerate, notify;
      acap_Return ret;
      acap_Sort sort;
    } m;
    acap_Modifiers p;
  } u;
};

size_t _acap_ModifiersSize(acap_ModifiersTag tag,
			   va_list *ap,
			   jmp_buf *bad) {
  char *foo;
  int have[9] = {0};
#define DEPTH 0
#define LIMIT 1
#define INHERIT 2
#define ENUMERATE 3
#define NOTIFY 4
#define HARD 5
#define CTX 6
#define RET 7
#define SORT 8
  size_t size = sizeof(struct _acap_Modifiers);
    
  if (tag == ACAP_MODIFIERSPTR)
    return size;
  do {
    switch(tag) {
    case ACAP_MODIFIERSDONE:
      return size;
    case ACAP_LIMIT:
      {
	int lowlimit, highlimit;
	if (have[LIMIT]++) longjmp(*bad, 1);
	lowlimit = va_arg((*ap), int);
	highlimit = va_arg((*ap), int);
	if (lowlimit < 0 || highlimit < 0)
	  longjmp(*bad, 1);
	break;
      }
    case ACAP_DEPTH:
      {
	int depth;
	if (have[DEPTH]++) longjmp(*bad, 1);
	depth = va_arg((*ap), int);
	if (depth < 0)
	  longjmp(*bad, 1);
	break;
      }
    case ACAP_HARDLIMIT:
      {
	int hardlimit;
	if (have[HARD]++) longjmp(*bad, 1);
	hardlimit = va_arg((*ap), int);
	if (hardlimit < 0)
	  longjmp(*bad, 1);
	break;
      }
    case ACAP_MAKECONTEXT:
      if (have[CTX]++) longjmp(*bad, 1);
      foo = va_arg((*ap), char *);
      break;
    case ACAP_ENUMERATE:
      if (have[ENUMERATE]++) longjmp(*bad, 1);
      foo = va_arg((*ap), char *);
      break;
    case ACAP_NOTIFY:
      if (have[NOTIFY]++) longjmp(*bad, 1);
      foo = va_arg((*ap), char *);
      break;
    case ACAP_NOINHERIT:
      if (have[INHERIT]++) longjmp(*bad, 1);
      break;
    case ACAP_RETURN:
      {
	acap_ReturnTag rtag;
	if (have[RET]++) longjmp(*bad, 1);
	rtag = va_arg((*ap), acap_ReturnTag);
	size += _acap_ReturnSize(rtag, ap, bad);
	break;
      }
    case ACAP_SORT:
      {
	acap_SortTag stag;
	if (have[SORT]++) longjmp(*bad, 1);
	stag = va_arg((*ap), acap_SortTag);
	size += _acap_SortSize(stag, ap, bad);
	break;
      }
    default:
      longjmp(*bad, 1);
    }
    tag = va_arg((*ap), acap_ModifiersTag);
  } while(1);
}

void _acap_ModifiersConstruct(acap_ModifiersTag tag,
			      va_list *ap,
			      acap_Modifiers *buffer) {
  acap_Modifiers m = (*buffer)++;
  if (tag == ACAP_MODIFIERSPTR) {
    m->ptr = 1;
    m->u.p = va_arg((*ap), acap_Modifiers);
    return;
  }
  m->ptr = 0;
  m->u.m.depth = -1;
  m->u.m.inherit = 1;
  m->u.m.enumerate = 0;
  m->u.m.notify = 0;
  m->u.m.lowlimit = m->u.m.highlimit = m->u.m.hardlimit = -1;
  m->u.m.context = NULL;
  m->u.m.ret = NULL;
  m->u.m.sort = NULL;
  do {
    switch(tag) {
    case ACAP_MODIFIERSDONE:
      return;
    case ACAP_LIMIT:
      m->u.m.lowlimit = va_arg((*ap), int);
      m->u.m.highlimit = va_arg((*ap), int);
      break;
    case ACAP_DEPTH:
      m->u.m.depth = va_arg((*ap), int);
      break;
    case ACAP_HARDLIMIT:
      m->u.m.hardlimit = va_arg((*ap), int);
      break;
    case ACAP_MAKECONTEXT:
      m->u.m.context = va_arg((*ap), const char *);
      break;
    case ACAP_ENUMERATE:
      m->u.m.enumerate = 1;
      break;
    case ACAP_NOTIFY:
      m->u.m.notify = 1;
      break;
    case ACAP_NOINHERIT:
      m->u.m.inherit = 0;
      break;
    case ACAP_RETURN:
      {
	acap_ReturnTag rtag = va_arg((*ap), acap_ReturnTag);
	m->u.m.ret = *(acap_Return *)buffer;
	_acap_ReturnConstruct(rtag, ap, (acap_Return *)buffer);
	break;
      }
    case ACAP_SORT:
      {
	acap_SortTag stag = va_arg((*ap), acap_SortTag);
	m->u.m.sort = *(acap_Sort *)buffer;
	_acap_SortConstruct(stag, ap, (acap_Sort *)buffer);
	break;
      }
    }
    tag = va_arg((*ap), acap_ModifiersTag);
  } while(1);
}

_ACAP_CONSTRUCT(Modifiers)

int _acap_SendModifiers(acap_Connection conn,
			acap_Modifiers m) {
  int res = 0;
  while (m->ptr)
    m = m->u.p;

  if (m->u.m.lowlimit >= 0 && m->u.m.highlimit >= 0) {
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "Limit",
			   ACAP_INT, m->u.m.lowlimit,
			   ACAP_INT, m->u.m.highlimit,
			   ACAP_DONE);
  }
  if (m->u.m.depth >= 0) {
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "Depth",
			   ACAP_INT, m->u.m.depth,
			   ACAP_DONE);
  }
  if (m->u.m.hardlimit >= 0) {
    res = _acap_SendTokens(conn,
			   ACAP_ATOM, "HardLimit",
			   ACAP_INT, m->u.m.hardlimit,
			   ACAP_DONE);
  }
  if (m->u.m.context) {
    res |= _acap_SendTokens(conn,
			    ACAP_ATOM, "MakeContext",
			    ACAP_CSTRING, m->u.m.context,
			    ACAP_DONE);
  }
  if (m->u.m.enumerate) {
    res |= _acap_SendTokens(conn,
			    ACAP_ATOM, "Enumerate",
			    ACAP_DONE);
  }
  if (m->u.m.notify) {
    res |= _acap_SendTokens(conn,
			    ACAP_ATOM, "Notify",
			    ACAP_DONE);
  }
  if (! m->u.m.inherit) {
    res |= _acap_SendTokens(conn,
			    ACAP_ATOM, "NoInherit",
			    ACAP_DONE);
  }
  if (m->u.m.ret) {
    res |= _acap_SendTokens(conn,
			    ACAP_ATOM, "Return",
			    ACAP_LPAREN,
			    ACAP_OBJECT, SendReturn, m->u.m.ret,
			    ACAP_RPAREN,
			    ACAP_DONE);
  }
  if (m->u.m.sort) {
    res |= _acap_SendTokens(conn,
			    ACAP_ATOM, "Sort",
			    ACAP_LPAREN,
			    ACAP_OBJECT, SendSort, m->u.m.sort,
			    ACAP_RPAREN,
			    ACAP_DONE);
  }
  return res;
}
