/* store.c - store command implementation
 * Rob Earhart
 * $Id: store.c,v 1.4 1997/11/17 22:47:04 rob Exp $
 */
/***********************************************************
        Copyright 1997 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "acapint.h"

typedef struct _acap_Value *acap_Value;
struct _acap_Value {
  acap_ValueTag tag;
  acap_Value next;
  const char *metadata;
  union {
    const char *value;
    const char **values;
  } u;
};

static size_t _acap_ValueSize(acap_ValueTag tag, va_list *ap, jmp_buf *bad) {
  size_t size = 0;
  char *foo;
  switch(tag) {
  case ACAP_VALUEDONE:
    longjmp(*bad, 1);
  case ACAP_VALUENIL:
  case ACAP_VALUEDEFAULT:
  case ACAP_VALUE:
    return sizeof(struct _acap_Value);
  }

  do {
    size += sizeof(struct _acap_Value);
    switch(tag) {
    case ACAP_VALUEDONE:
      return size;
    case ACAP_VALUEMETA:
    case ACAP_VALUELIST:
      foo = va_arg((*ap), char *);
      foo = va_arg((*ap), char *);
      break;
    default:
      longjmp(*bad, 1);
    }
    tag = va_arg((*ap), acap_ValueTag);
  } while(1);
}

static void _acap_ValueConstruct(acap_ValueTag tag,
				 va_list *ap,
				 acap_Value *buffer) {
  static const char value[] = "value";
  do {
    acap_Value v = (*buffer)++;
    v->tag = tag;
    switch(tag) {
    case ACAP_VALUEDONE:
    case ACAP_VALUENIL:
    case ACAP_VALUEDEFAULT:
      v->next = NULL;
      return;
    case ACAP_VALUE:
      v->metadata = value;
      v->u.value = va_arg((*ap), const char *);
      v->next = NULL;
      return;
    case ACAP_VALUEMETA:
      v->metadata = va_arg((*ap), const char *);
      v->u.value = va_arg((*ap), const char *);
      break;
    case ACAP_VALUELIST:
      v->metadata = va_arg((*ap), const char *);
      v->u.values = va_arg((*ap), const char **);
      break;
    }
    v->next = *buffer;
    tag = va_arg((*ap), acap_ValueTag);
  } while(1);
}

typedef struct _acap_Attribute *acap_Attribute;
struct _acap_Attribute {
  acap_Attribute next;
  const char *attr;
  acap_Value value;
};

static size_t _acap_AttributeSize(acap_AttributeTag tag,
				  va_list *ap,
				  jmp_buf *bad) {
  char *foo;
  size_t size = 0;
  acap_ValueTag vtag;
  do {
    size += sizeof(struct _acap_Attribute);
    switch(tag) {
    case ACAP_ATTRIBUTEDONE:
      return size;
    case ACAP_ATTRIBUTE:
      foo = va_arg((*ap), char *);
      vtag = va_arg((*ap), acap_ValueTag);
      size += _acap_ValueSize(vtag, ap, bad);
      break;
    default:
      longjmp(*bad, 1);
    }
    tag = va_arg((*ap), acap_AttributeTag);
  } while(1);
}

static void _acap_AttributeConstruct(acap_AttributeTag tag,
				     va_list *ap,
				     acap_Attribute *buffer) {
  acap_ValueTag vtag;
  do {
    acap_Attribute a = (*buffer)++;
    switch(tag) {
    case ACAP_ATTRIBUTEDONE:
      a->next = NULL;
      a->attr = NULL;
      a->value = NULL;
      return;
    case ACAP_ATTRIBUTE:
      a->attr = va_arg((*ap), const char *);
      a->value = (acap_Value)*buffer;
      vtag = va_arg((*ap), acap_ValueTag);
      _acap_ValueConstruct(vtag, ap, (acap_Value *)buffer);
      a->next = *buffer;
    }
    tag = va_arg((*ap), acap_AttributeTag);
  } while(1);
}

struct _acap_Entry {
  acap_EntryTag tag;
  acap_Entry next;
  const char *entry;
  int nocreate;
  const char *unchangedsince;
  acap_Attribute attribute;
};

static size_t _acap_EntrySize(acap_EntryTag tag, va_list *ap, jmp_buf *bad) {
  char *foo;
  size_t size = 0;
  acap_StoreFlagTag ftag;
  acap_AttributeTag atag;
  int haveunchanged, havenocreate;
  do {
    size += sizeof(struct _acap_Entry);
    switch(tag) {
    case ACAP_ENTRYDONE:
      return size;
    case ACAP_ENTRYPTR:
      foo = va_arg((*ap), char *);
      return size;
    case ACAP_ENTRY:
      foo = va_arg((*ap), char *); /* name */
      haveunchanged = havenocreate = 0;
      do {
	ftag = va_arg((*ap), acap_StoreFlagTag);
	if (ftag == ACAP_STOREFLAGSDONE) break;
	if (ftag != ACAP_NOCREATE
	    && ftag != ACAP_UNCHANGEDSINCE)
	  longjmp(*bad, 1);
	if (ftag == ACAP_NOCREATE && havenocreate++
	    || ftag == ACAP_UNCHANGEDSINCE && haveunchanged++)
	  longjmp(*bad, 1);
      } while(1);
      atag = va_arg((*ap), acap_AttributeTag);
      size += _acap_AttributeSize(atag, ap, bad);
      break;
    default:
      longjmp(*bad, 1);
    }
    tag = va_arg((*ap), acap_EntryTag);
  } while(1);
}

static void _acap_EntryConstruct(acap_EntryTag tag,
				 va_list *ap,
				 acap_Entry *buffer) {
  acap_StoreFlagTag ftag;
  acap_AttributeTag atag;
  do {
    acap_Entry e = (*buffer)++;
    e->tag = tag;
    switch(tag) {
    case ACAP_ENTRYDONE:
      e->next = NULL;
      return;
    case ACAP_ENTRYPTR:
      e->next = va_arg((*ap), acap_Entry);
      return;
    case ACAP_ENTRY:
      e->entry = va_arg((*ap), const char *);
      e->nocreate = 0;
      e->unchangedsince = NULL;
      do {
	ftag = va_arg((*ap), acap_StoreFlagTag);
	if (ftag == ACAP_NOCREATE)
	  e->nocreate = 1;
	else if (ftag == ACAP_UNCHANGEDSINCE)
	  e->unchangedsince = va_arg((*ap), const char *);
      } while (ftag != ACAP_STOREFLAGSDONE);
      e->attribute = (acap_Attribute)*buffer;
      atag = va_arg((*ap), acap_AttributeTag);
      _acap_AttributeConstruct(atag, ap, (acap_Attribute *)buffer);
      e->next = *buffer;
    }
    tag = va_arg((*ap), acap_EntryTag);
  } while(1);
}

_ACAP_CONSTRUCT(Entry)

int _acap_SendEntries(acap_Connection conn,
		      acap_Entry e) {
  int res;
  int have_parens;
  acap_Value v;
  acap_Attribute a;
  for(res = 0; !res && e; e = e->next) {
    if (e->tag != ACAP_ENTRY)
      continue;

    res = _acap_SendTokens(conn,
			   ACAP_LPAREN,
			   ACAP_CSTRING, e->entry,
			   ACAP_DONE);
    if (! res && e->nocreate)
      res = _acap_SendTokens(conn,
			     ACAP_ATOM, "NoCreate",
			     ACAP_DONE);
    if (! res && e->unchangedsince)
      res = _acap_SendTokens(conn,
			     ACAP_ATOM, "UnchangedSince",
			     ACAP_CSTRING, e->unchangedsince,
			     ACAP_DONE);
    for (a = e->attribute; !res && a && a->value; a = a->next) {
      res = _acap_SendTokens(conn,
			     ACAP_CSTRING, a->attr,
			     ACAP_DONE);
      if (! res)
	for (have_parens = 0, v = a->value; !res && v; v = v->next) {
	  switch(v->tag) {
	  case ACAP_VALUEDONE:
	    break;
	  case ACAP_VALUENIL:
	    res = _acap_SendTokens(conn,
				   ACAP_ATOM, "Nil",
				   ACAP_DONE);
	    break;
	  case ACAP_VALUEDEFAULT:
	    res = _acap_SendTokens(conn,
				   ACAP_ATOM, "Default",
				   ACAP_DONE);
	    break;
	  case ACAP_VALUE:
	    res = _acap_SendTokens(conn,
				   ACAP_CSTRING, v->u.value,
				   ACAP_DONE);
	    break;
	  case ACAP_VALUEMETA:
	    if (! have_parens) {
	      res = _acap_SendTokens(conn,
				     ACAP_LPAREN,
				     ACAP_DONE);
	      have_parens = 1;
	    }
	    if (! res)
	      res = _acap_SendTokens(conn,
				     ACAP_CSTRING, v->metadata,
				     ACAP_CSTRING, v->u.value,
				     ACAP_DONE);
	    break;
	  case ACAP_VALUELIST:
	    if (! have_parens) {
	      res = _acap_SendTokens(conn,
				     ACAP_LPAREN,
				     ACAP_DONE);
	      have_parens = 1;
	    }
	    if (! res)
	      res = _acap_SendTokens(conn,
				     ACAP_CSTRING, v->metadata,
				     ACAP_LPAREN,
				     ACAP_DONE);
	    {
	      const char **val;
	      for (val = v->u.values; !res && *val; val++)
		res = _acap_SendTokens(conn,
				       ACAP_CSTRING, *val,
				       ACAP_DONE);
	    }
	    if (! res)
	      res = _acap_SendTokens(conn,
				     ACAP_RPAREN,
				     ACAP_DONE);
	    break;
	  default:
	    res = 1;
	  }
	}
      if (!res && have_parens)
	res = _acap_SendTokens(conn,
			       ACAP_RPAREN,
			       ACAP_DONE);
    }
    if (!res)
      res = _acap_SendTokens(conn,
			     ACAP_RPAREN,
			     ACAP_DONE);
  }
  return res;
}

int acap_CmdStore(acap_Connection conn,
		  acap_Status *status_ret,
		  char **info_ret,
		  acap_EntryTag tag,
		  ...) {
  size_t size;
  va_list ap;
  jmp_buf env;
  acap_Entry entries, e;
  int res;

  if (setjmp(env)) return -1;

  va_start(ap, tag);
  size = _acap_EntrySize(tag, &ap, &env);
  va_end(ap);

  e = entries = (acap_Entry)malloc(size);
  if (! entries) return -1;
  va_start(ap, tag);
  _acap_EntryConstruct(tag, &ap, &e);
  va_end(ap);

  res = _acap_SCmd(conn,
		   status_ret,
		   info_ret,
		   ACAP_ATOM, "Store",
		   ACAP_OBJECT, &_acap_SendEntries, entries,
		   ACAP_DONE);
  free(entries);
  return res;
}

int acap_ACmdStore(acap_Connection conn,
		   acap_CmdCallback *callback,
		   void *callback_data,
		   acap_EntryTag tag,
		   ...) {
  size_t size;
  va_list ap;
  jmp_buf env;
  acap_Entry entries, e;
  int res;

  if (setjmp(env)) return -1;

  va_start(ap, tag);
  size = _acap_EntrySize(tag, &ap, &env);
  va_end(ap);

  e = entries = (acap_Entry)malloc(size);
  if (! entries) return -1;
  va_start(ap, tag);
  _acap_EntryConstruct(tag, &ap, &e);
  va_end(ap);

  res = _acap_ACmd(conn,
		   callback,
		   callback_data,
		   NULL,
		   NULL,
		   ACAP_ATOM, "Store",
		   ACAP_OBJECT, &_acap_SendEntries, entries,
		   ACAP_DONE);
  free(entries);
  return res;
}
