/* abook_ldap.c  --  address books implemented via LDAP lookups
 *
 * Copyright (c) 1998-2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *      Office of Technology Transfer
 *      Carnegie Mellon University
 *      5000 Forbes Avenue
 *      Pittsburgh, PA  15213-3890
 *      (412) 268-4387, fax: (412) 268-7395
 *      tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <config.h>
#include <stdlib.h>
#include <syslog.h>
#include <errno.h>
#include <ldap.h>
#include <strings.h>
#include "xmalloc.h"
#include "util.h"
#include "syncdb.h"
#include "option.h"
#include "authize.h"
#include "abook.h"
#include "abook_ldap.h"

#define ATTRMAPSIZE 20

struct map_pair {
    char *field;
    char *attr;
};

struct ldap_config {
    int configured;
    char *searchbase;
    int scope;
    char *ldaphost;
    int ldapport;
    char *fullnameattr;
    char *uniqueattr;
    char *defaultfilter;
    struct map_pair map[ATTRMAPSIZE];
};

static struct ldap_config config = {0};

static char opt_ldap_searchbase[]	= "imsp.ldap.searchbase";
static char opt_ldap_scope[]		= "imsp.ldap.scope";
static char opt_ldap_ldaphost[]		= "imsp.ldap.host";
static char opt_ldap_ldapport[]		= "imsp.ldap.port";
static char opt_ldap_fullnameattr[]	= "imsp.ldap.fullnameattr";
static char opt_ldap_uniqueattr[]	= "imsp.ldap.uniqueattr";
static char opt_ldap_defaultfilter[]	= "imsp.ldap.defaultfilter";
static char opt_ldap_attrmap[]		= "imsp.ldap.attrmap";
static char err_ldap_missing[] = 
    "Missing LDAP setting in global options file: %s";
static char err_ldap_badvalue[] =
    "Illegal value for LDAP option in global options file: %s";

static void
config_error(char *reason, char *optname)
{
    syslog(LOG_ERR, reason, optname);
}

static int
config_ldap(void)
{
    int i = 0;
    char *value;
    option_list *mapping;

    if (!config.configured) {
	/*
	syslog(LOG_NOTICE, "Configuring the LDAP settings for the first time");
	*/

	/* Search Base */
	value = option_get("", opt_ldap_searchbase, 1, NULL);
	if (value) {
	    config.searchbase = strdup(value);
	} else {
	    config_error(err_ldap_missing, opt_ldap_searchbase);
	    return -1;
	}

	/* Scope */
	value = option_get("", opt_ldap_scope, 1, NULL);
	if (value) {
	    if (strcasecmp(value, "subtree") == 0) {
		config.scope = LDAP_SCOPE_SUBTREE;
	    } else if (strcasecmp(value, "base") == 0) {
		config.scope = LDAP_SCOPE_BASE;
	    } else if (strcasecmp(value, "onelevel") == 0) {
		config.scope = LDAP_SCOPE_ONELEVEL;
	    } else {
		config_error(err_ldap_badvalue, opt_ldap_scope);
		return -1;
	    }
	} else {
	    config_error(err_ldap_missing, opt_ldap_scope);
	    return -1;
	}

	/* LDAP Server Hostname */
	value = option_get("", opt_ldap_ldaphost, 1, NULL);
	if (value) {
	    config.ldaphost = strdup(value);
	} else {
	    config_error(err_ldap_missing, opt_ldap_ldaphost);
	    return -1;
	}

	/* LDAP Server Port Number */
	value = option_get("", opt_ldap_ldapport, 1);
	if (value) {
	    config.ldapport = atoi(value);
	} else {
	    config.ldapport = LDAP_PORT;
	}

	/* Fullname Attribute */
	value = option_get("", opt_ldap_fullnameattr, 1, NULL);
	if (value) {
	    config.fullnameattr = strdup(value);
	} else {
	    config_error(err_ldap_missing, opt_ldap_fullnameattr);
	    return -1;
	}

	/* Unique Attribute */
	value = option_get("", opt_ldap_uniqueattr, 1, NULL);
	if (value) {
	    config.uniqueattr = strdup(value);
	} else {
	    config_error(err_ldap_missing, opt_ldap_uniqueattr);
	    return -1;
	}

	/* Default Search Filter */
	value = option_get("", opt_ldap_defaultfilter, 1, NULL);
	if (value) {
	    config.defaultfilter = strdup(value);
	} else {
	    config_error(err_ldap_missing, opt_ldap_defaultfilter);
	    return -1;
	}

	/* Mapping from IMSP fields to LDAP attributes */
	mapping = option_getlist("", opt_ldap_attrmap, 1);
	if (mapping == NULL) {
	    config_error(err_ldap_missing, opt_ldap_attrmap);
	    return -1;
	}
	/* there must be an even number of items to form pairs */
	if (mapping->count == 0 || 
	    (mapping->count % 2) != 0 ||
	    (mapping->count / 2) > ATTRMAPSIZE) {
	    config_error(err_ldap_badvalue, opt_ldap_attrmap);
	    return -1;
	}

	/* step through the items in the map option,
	   assigning them alternatively to the "field" and "value"
	   halfs of the map structure.
	*/
	for (i = 0; i < mapping->count; i++) {
	    if (strcasecmp(mapping->item[i], "null") == 0)
		value = NULL;
	    else
		value = strdup(mapping->item[i]);
	    if (i % 2 == 0)
		config.map[i / 2].field = value;
	    else
		config.map[i / 2].attr  = value;
	}
	/* add a null terminator pair at the end */
	config.map[i / 2].field = NULL;
	config.map[i / 2].attr  = NULL;

        config.configured = 1;
    }

    return 0;
}


/* Convert an IMSP search specification to an LDAP search filter.
 * Returns 0 on success, setting "filter" to the resulting filter.
 * Returns -1 if none of the IMSP fields could be converted to an
 * LDAP attribute.
 */
static int
imsp_to_ldap_filter(abook_fielddata *flist, int fcount, char **filter)
{
    int i, j;
    static char filt[2048];
    int filter_is_empty = 1;

    strlcpy(filt, "(&", sizeof(filt));
    strlcat(filt, config.defaultfilter, sizeof(filt));

    for (i = 0; i < fcount; i++) {
	for (j = 0; config.map[j].field != NULL; j++) {
	    if ((strcasecmp(flist[i].field, config.map[j].field) == 0)) {
		if (config.map[j].attr == NULL) {
		    syslog(LOG_ERR, "imsp_to_ldap_filter: skipping unmapped"
			   " field '%s'", flist[i].field);
		} else {
		    filter_is_empty = 0;
		    strlcat(filt, "(", sizeof(filt));
		    strlcat(filt, config.map[j].attr, sizeof(filt));
		    strlcat(filt, "=", sizeof(filt));
		    strlcat(filt, flist[i].data, sizeof(filt));
		    strlcat(filt, ")", sizeof(filt));
		}
		break;
	    }
	}
	if (config.map[j].field == NULL) {
	    syslog(LOG_ERR, "imsp_to_ldap_filter: skipping unknown"
		   " field '%s'", flist[i].field);
	}
    }

    strlcat(filt, ")", sizeof(filt));
    /* syslog(LOG_NOTICE, "Filter: %s", filt); */

    if (filter_is_empty) {
	return -1;
    } else {
	*filter = filt;
	return 0;
    }
}


int
abook_ldap_searchstart(abook_ldap_state **ldap_state, 
		       abook_fielddata *flist, int fcount)
{
    abook_ldap_state *mystate;
    int msgid, rc;
    int sizelimit;
    char *msg;
    char *attrs[20];
    LDAP *ld;
    LDAPMessage *result;
    char *filter, *searchbase, *ldaphost;
    int scope, ldapport;

    if (config_ldap() < 0) {
	syslog(LOG_ERR, "abook_ldap_searchstart: failed to configure LDAP");
	return -1;
    }
    
    if (imsp_to_ldap_filter(flist, fcount, &filter) < 0) {
	syslog(LOG_ERR, "abook_ldap_searchstart: failed to convert filter");
	return -1;
    }

    ld = ldap_init(config.ldaphost, config.ldapport);
    if (ld == NULL) {
	syslog(LOG_ERR, "abook_ldap_searchstart: LDAP init failed: %s",
	       strerror(errno));
	return -1;
    }

    rc = ldap_simple_bind_s(ld, NULL, NULL);
    if (rc != LDAP_SUCCESS) {
	syslog(LOG_ERR, "abook_ldap_searchstart: simple bind failed: %s",
	       ldap_err2string(rc));
	return -1;
    }

    /* For testing the error handlers...
      sizelimit = 4;
      ldap_set_option(ld, LDAP_OPT_SIZELIMIT, &sizelimit);
    */
    attrs[0] = config.fullnameattr;
    attrs[1] = config.uniqueattr;
    attrs[2] = NULL;

    msgid = ldap_search(ld, config.searchbase, config.scope, 
			filter, attrs, 0/*attrs-only*/);
    if (msgid == -1) {
#if 0
	rc = ldap_get_lderrno(ld, NULL, &msg);
	syslog(LOG_ERR, "abook_ldap_searchstart: LDAP search failed: %s (%s)",
	       ldap_err2string(rc), (msg == NULL) ? "" : msg);
#else
	syslog(LOG_ERR, "abook_ldap_searchstart: LDAP search failed");
#endif
	ldap_unbind(ld);
	return -1;
    }

    rc = ldap_result(ld, msgid, 0, NULL, &result);

    switch (rc) {
    case LDAP_RES_SEARCH_ENTRY:
	/* Do nothing here. The abook_search function will pull out this
	 * entry and send it back for display to the user.
	 * The result is freed later.
	 */
	break;

    case LDAP_RES_SEARCH_RESULT:
	rc = ldap_result2error(ld, result, 1 /* free result */);
	if (rc == LDAP_SUCCESS) {
	    /* Special case: search returned successfully, but with no
	     * matching entries. Send a null "prevresult" to the abook_search
	     * function.
	     */
	    result = NULL;

	} else {
	    syslog(LOG_ERR,"abook_ldap_searchstart: search returned error: %s",
		   ldap_err2string(rc));
	    ldap_unbind(ld);
	    return -1;
	}
	break;

    default:
	syslog(LOG_ERR, "abook_ldap_searchstart: ldap_result failed: %s",
	       ldap_err2string(rc));
	(void) ldap_msgfree(result);  /* ignore message type return value */
	ldap_unbind(ld);
	return -1;
    }

    mystate = (abook_ldap_state *) malloc (sizeof (abook_ldap_state));
    *ldap_state = mystate;

    if (mystate == NULL) {
	syslog(LOG_ERR, "abook_ldap_searchstart: Out of memory");
	(void) ldap_msgfree(result);  /* ignore message type return value */
	ldap_unbind(ld);
	return -1;
    }

    mystate->ld = ld;
    mystate->msgid = msgid;
    mystate->prevresult = result;

    return 0;
}


static int
count_identical_fullnames(abook_ldap_state *ldap_state, char *alias)
{
    int rc, count = 0;
    char filter[1024];
    LDAPMessage *results;

    /* 
     * To limit the work done for this search, look for some bogus attribute 
     * that's probably not in the entry and don't return any values.
     */
    char *attrs[] = {"c", NULL};

    snprintf(filter, sizeof(filter), "(&%s(%s=%s))", config.defaultfilter, 
	    config.fullnameattr, alias);
    rc = ldap_search_s(ldap_state->ld, config.searchbase, config.scope,
		       filter, attrs, 1 /*attrs-only*/, &results);
    if (rc != LDAP_SUCCESS) {
	syslog(LOG_ERR, "count_identical_fullnames: search failed: %s",
	       ldap_err2string(rc));
	count = -1;
    } else {
	count = ldap_count_entries(ldap_state->ld, results);
	/* Returns -1 on error, so just pass that back to the caller */
	(void) ldap_msgfree(results);  /* ignore message type return value */
    }

    return count;
}


char *
abook_ldap_search(abook_ldap_state *ldap_state)
{
    int rc, count;
    LDAP *ld;
    int msgid;
    LDAPMessage *result, *entry;
    char *dn;
    static char alias[1024];
    char **values;

    if (ldap_state->prevresult == NULL) {
	/* prevresult is set to NULL when the prior call to ldap_result 
	 * indicated that the search ended successfully.
	 */
	return NULL;

    } else {
	ld = ldap_state->ld;
	msgid = ldap_state->msgid;
	result = ldap_state->prevresult;

	/* Find the full name associated with this matching entry so we
	 * can return a pointer to it.
	 */

	entry = ldap_first_entry(ld, result);
	if (entry == NULL) {
#if 0
	    syslog(LOG_ERR, "abook_ldap_search: ldap_first_entry: %s", 
		   ldap_err2string(ldap_get_lderrno(ld, NULL, NULL)));
#else
	    syslog(LOG_ERR, "abook_ldap_search: ldap_first_entry failed");
#endif
	    return NULL;
	}
	values = ldap_get_values(ld, entry, config.fullnameattr);
	if (values == NULL || values[0] == NULL) {
#if 0
	    rc = ldap_get_lderrno(ld, NULL, NULL);
	    if (rc == LDAP_SUCCESS) {
		syslog(LOG_ERR, "abook_ldap_search: ldap_get_values didn't find the %s attribute",
		       config.fullnameattr);
	    } else {
		/* Seems to land here with LDAP_DECODING_ERROR if the entry
		 * doesn't have the attribute named by "fullnameattr".
		 */
		syslog(LOG_ERR, "abook_ldap_search: ldap_get_values failed: %s", 
		       ldap_err2string(rc));
	    }
#else
	    syslog(LOG_ERR, "abook_ldap_search: ldap_get_values failed");
#endif
	    return NULL;
	}

	strlcpy(alias, values[0], sizeof(alias));

	ldap_value_free(values);

	count = count_identical_fullnames(ldap_state, alias);
	if (count > 1) {
	    /* Find the uid for this entry */
	    values = ldap_get_values(ld, entry, config.uniqueattr);
	    if (values == NULL || values[0] == NULL) {
#if 0
		rc = ldap_get_lderrno(ld, NULL, NULL);
		if (rc == LDAP_SUCCESS) {
		    syslog(LOG_ERR, "abook_ldap_search: ldap_get_values didn't find the %s attr",
			   config.uniqueattr);
		} else {
		    /* Seems to land here with LDAP_DECODING_ERROR if the entry
		     * doesn't have the attribute.
		     */
		    syslog(LOG_ERR, "abook_ldap_search: ldap_get_values failed for attr '%s': %s",
			   config.uniqueattr, ldap_err2string(rc));
		}
#else
		syslog(LOG_ERR, "abook_ldap_search: ldap_get_values failed for attr '%s'", config.uniqueattr);
#endif
		return NULL;
	    }
	    strlcat(alias, "[", sizeof(alias));
	    strlcat(alias, config.uniqueattr, sizeof(alias));
	    strlcat(alias, ":", sizeof(alias));
	    strlcat(alias, values[0], sizeof(alias));
	    strlcat(alias, "]", sizeof(alias));
	    ldap_value_free(values);
	}

	ldap_msgfree(result);

	/* Now fetch the next result to get ready for the next iteration
	 * of this function.
	 */

	rc = ldap_result(ld, msgid, 0, NULL, &result);

	switch (rc) {
	case LDAP_RES_SEARCH_ENTRY:
	    ldap_state->prevresult = result;
	    break;
	    
	case LDAP_RES_SEARCH_RESULT:
	    rc = ldap_result2error(ld, result, 1 /* free result */);
	    /* This result had no entries, but indicated success or failure.
	     * Return the alias corresponding to the previous entry,
	     * but set "prevresult" to NULL to indicate to the next 
	     * iteration that searching is completed.
	     */
	    if (rc != LDAP_SUCCESS) {
		syslog(LOG_ERR,"abook_ldap_search: search completed with"
		       " error: %s", ldap_err2string(rc));
	    }
	    ldap_state->prevresult = NULL;
	    break;
	    
	default:
	    syslog(LOG_ERR, "abook_ldap_search: ldap_result failed: %s",
		   ldap_err2string(rc));
	    (void) ldap_msgfree(result); /* ignore message type return value */
	    ldap_state->prevresult = NULL;
	}

	return alias;
    }
}


void
abook_ldap_searchdone(abook_ldap_state *ldap_state)
{
    ldap_unbind(ldap_state->ld);
    free(ldap_state);
}


abook_fielddata *
abook_ldap_fetch(char *alias, int *count)
{
    int i, rc, ldapcount, mappedfieldcount;
    char *ptr;
    char prefix[1024];
    char filter[1024];
    abook_fielddata *fdata, *fptr;
    char *searchattr;
    char *searchkey;
    LDAP *ld;
    LDAPMessage *results, *entry;
    char **values;

    if (config_ldap() < 0) {
	syslog(LOG_ERR, "abook_ldap_fetch: failed to configure LDAP");
	return NULL;
    }

    /*
     * Decide how to search for the user.
     */

    snprintf(prefix, sizeof(prefix), "[%s:", config.uniqueattr);
    ptr = strstr(alias, prefix);
    if (ptr != NULL) {
	*ptr = '\0';
	ptr += 1 /*[*/ + strlen(config.uniqueattr) + 1 /*:*/;
	searchkey = ptr;
	ptr += strlen(ptr) - 1 /*]*/;
	*ptr = '\0';
	searchattr = config.uniqueattr;
    } else {
	searchkey  = alias;
	searchattr = config.fullnameattr;
    }
    snprintf(filter, sizeof(filter), "(&%s(%s=%s))", config.defaultfilter, 
	    searchattr, searchkey);

    ld = ldap_init(config.ldaphost, config.ldapport);
    if (ld == NULL) {
	syslog(LOG_ERR, "abook_ldap_fetch: LDAP init failed: %s",
	       strerror(errno));
	return NULL;
    }

    rc = ldap_simple_bind_s(ld, NULL, NULL);
    if (rc != LDAP_SUCCESS) {
	syslog(LOG_ERR, "abook_ldap_fetch: simple bind failed: %s",
	       ldap_err2string(rc));
	return NULL;
    }

    rc = ldap_search_s(ld, config.searchbase, config.scope, filter, 
		       NULL/*get all attrs*/, 0/*attrs-only*/, &results);
    if (rc != LDAP_SUCCESS) {
	syslog(LOG_ERR, "abook_ldap_fetch: LDAP search failed: %s",
	       ldap_err2string(rc));
	ldap_unbind(ld);
	return NULL;
    }

    ldapcount = ldap_count_entries(ld, results);
    if (ldapcount != 1) {
	syslog(LOG_ERR, "abook_ldap_fetch: unexpected count of search"
	       " hits: %d", ldapcount);
	(void) ldap_msgfree(results);  /* ignore message type return value */
	ldap_unbind(ld);
	return NULL;
    }	       

    entry = ldap_first_entry(ld, results);
    if (entry == NULL) {
#if 0
	syslog(LOG_ERR, "abook_ldap_fetch: ldap_first_entry: %s", 
	       ldap_err2string(ldap_get_lderrno(ld, NULL, NULL)));
#else
	syslog(LOG_ERR, "abook_ldap_fetch: ldap_first_entry failed");
#endif
	(void) ldap_msgfree(results);  /* ignore message type return value */
	ldap_unbind(ld);
	return NULL;
    }

    /* This memory is freed by abook_fetchdone() which is called by 
     * show_address() after it's finished sending the field/data pairs 
     * back to the IMSP client
     */
    
    mappedfieldcount = 0;
    for (i = 0; config.map[i].field != NULL; i++) {
	if (config.map[i].attr != NULL)
	    mappedfieldcount++;
    }

    fdata = (abook_fielddata *) 
	malloc(sizeof (abook_fielddata) * mappedfieldcount);
    if (fdata == NULL) {
	syslog(LOG_ERR, "abook_ldap_fetch: Out of memory");
	(void) ldap_msgfree(results);  /* ignore message type return value */
	ldap_unbind(ld);
	return NULL;
    }

    *count = 0;
    fptr = fdata;
    
    for (i = 0; config.map[i].field != NULL; i++) {
	if ((config.map[i].attr != NULL) &&
	    (strcmp(config.map[i].attr, config.fullnameattr) != 0)) {
	    values = ldap_get_values(ld, entry, config.map[i].attr);
	    if (values != NULL && values[0] != NULL) {
		fptr->field = strdup(config.map[i].field);
		fptr->data  = strdup(values[0]);
		(*count)++;
		fptr++;
	    }
	    if (values != NULL)
		ldap_value_free(values);
	}
    }

    (void) ldap_msgfree(results);  /* ignore message type return value */
    ldap_unbind(ld);

    return (fdata);
}
