(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

structure Throttle :> THROTTLE =
struct
  datatype hardresult = H_WAYTOOMANY | H_OK
  datatype softresult = S_TOOMANY of int | S_OK
  datatype result = OK

  datatype 'a throttle = 
    PassThrough of ('a -> unit) * (result -> unit)
  | SoftThrough of ('a -> unit) * (softresult -> unit) * 
    ('a Fifo.fifo * int * int * int) ref
  | HardThrough of ('a -> unit) * (hardresult -> unit) * 
    ('a Fifo.fifo * int) ref
  | SortThrough of ('a -> unit) * (result -> unit) * (('a * 'a) -> order) *
    'a Fifo.fifo ref

  (* create a hard throttle, plus what to call when done *)
  (* val createhard : int -> ('a -> unit) -> (unit -> unit) -> 'a throttle *)
  fun createhard lim f g =
    HardThrough(f, g, ref (Fifo.empty, lim))

  (* create a soft throttle; the first number is the limit *)
  (* val createsoft : (int * int) -> ('a -> unit) -> (unit -> unit)
    -> 'a throttle *)
  fun createsoft (lim, pass) f g =
    SoftThrough(f, g, ref (Fifo.empty, lim, pass, 0))

  (* create a sorting throttle *)
  (* val createsort : (('a * 'a) -> order) -> ('a -> unit) -> 
                   (result -> unit) -> 'a throttle *)
  fun createsort c f g = SortThrough (f, g, c, ref Fifo.empty)

  (* a throttle that just passes through data *)
  (* val createpass : ('a -> unit) -> (unit -> unit) -> 'a throttle *)
  fun createpass f g = PassThrough(f, g)

  (* send data into the throttle *)
  (* val call : 'a throttle -> 'a -> unit *)
  fun call (HardThrough(f, g, cell as ref (q, lim))) a =
    if lim > 0 then cell := (Fifo.enqueue (q, a), lim - 1)
    else if lim = 0 then (cell := (Fifo.empty, ~1); g H_WAYTOOMANY)
	 else ()
    | call (SoftThrough(f, g, cell as ref (q, lim, pass, tot))) a =
    if pass > 0 then (cell := (q, lim, pass - 1, tot + 1); f a)
    else if lim > 0 then cell := (Fifo.enqueue (q, a), lim - 1, pass, tot + 1)
	 else if lim = 0 then (cell := (Fifo.empty, ~1, ~1, tot +1))
	      else (cell := (q, ~1, ~1, tot + 1))
    | call (PassThrough(f, g)) a = f a
    | call (SortThrough(f, g, c, cell as ref q)) a = 
		cell := (Fifo.enqueue (q, a))


  fun quickSort (comp : 'a * 'a -> order) [] = []
    | quickSort (comp : 'a * 'a -> order) [x] = [x]
    | quickSort (comp : 'a * 'a -> order) unsorted =
    let
      fun quick' [] l = l
	| quick' [x] l = x::l
	| quick' (a::bs) l = (* pivot by "a" *)
	let
	  fun partition (left, right, []) =
	    quick' left (a::quick' right l)
	    | partition (left, right, x::xs) =
	    case comp(x, a) of
	      LESS => partition (x::left, right, xs)
	    | EQUAL => partition (x::left, right, xs)
	    | GREATER => partition (left, x::right, xs)
	in
	  partition([], [], bs)
	end
    in
      quick' unsorted nil
    end


  (* indicates that no more data will be coming through the throttle;
     if the limit is exceeded it just throws out the data *)
  (* val done : 'a throttle -> unit *)
  fun done (HardThrough(f, g, ref (q, lim))) = 
    if lim >= 0 then (Fifo.app f q; g H_OK)
    else ()
    | done (SoftThrough(f, g, ref (q, lim, pass, tot))) =
      if lim >= 0 then (Fifo.app f q; g S_OK)
      else (g (S_TOOMANY tot))
    | done (PassThrough(f, g)) = g OK
    | done (SortThrough(f, g, c, ref q)) = 
	    (List.app f (quickSort c (Fifo.contents q));
	     g OK)

end
