(* $Id: comparator.sml,v 1.3 2000/05/09 22:14:00 leg Exp $ *)

(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)

structure Comparator :> COMPARATOR =
struct
  structure V = Value
  
  type comparator = (Value.value * Value.value) -> order

  exception NoSuchComparator of string

  exception ImproperComparison

  datatype comparatortype = Equal | Order | Prefix | Substring

  fun flip LESS = GREATER
    | flip EQUAL = EQUAL
    | flip GREATER = LESS

  fun ascii_numeric (x, y) =
    let
      val x = (Int.fromString x handle General.Overflow => Int.maxInt)
      val y = (Int.fromString y handle General.Overflow => Int.maxInt)
    in
      case (x, y) of
	(NONE, NONE) => EQUAL
      | (SOME _, NONE) => LESS
      | (NONE, SOME _) => GREATER
      | (SOME x, SOME y) => Int.compare (x,y)
    end

  fun char_casemap (x, y) = Char.compare (Char.toUpper x, Char.toUpper y)

  val ascii_casemap = String.collate char_casemap

  fun octet_prefix (x, y) = String.isPrefix x y

  (* not extremely efficient! *)
  fun ascii_casemap_prefix (x, y) = 
    String.isPrefix (String.map Char.toUpper x) (String.map Char.toUpper y)

  fun boolcomp true = EQUAL
    | boolcomp false = LESS

  (* returns true if x is a substring of y *)
  fun octet_substring (x, y) =
    let
      val prefix = Substring.isPrefix x

      fun sub s = prefix s orelse case Substring.getc s of
	NONE => false
      | (SOME (_, s)) => sub s
    in
      sub (Substring.all y)
    end

  fun ascii_casemap_substring (x, y) =
    octet_substring (String.map Char.toUpper x, String.map Char.toUpper y)

  (* val lookup' : comparatortype -> string -> ((string * string) -> order) *)
  fun lookup' Order ("i;octet" | "+i;octet") = String.compare
    | lookup' Order ("-i;octet") = flip o String.compare
    | lookup' Order ("i;ascii-numeric" | "+i;ascii-numeric") =
    ascii_numeric
    | lookup' Order ("-i;ascii-numeric") = flip o ascii_numeric
    | lookup' Order ("i;ascii-casemap" | "+i;ascii-casemap") =
    ascii_casemap
    | lookup' Order ("-i;ascii-casemap") = flip o ascii_casemap
    | lookup' Equal ("i;octet") = String.compare
    | lookup' Equal ("i;ascii-numeric") = ascii_numeric
    | lookup' Equal ("i;ascii-casemap") = ascii_casemap
    | lookup' Prefix ("i;octet") = boolcomp o octet_prefix
    | lookup' Prefix ("i;ascii-casemap") = boolcomp o ascii_casemap_prefix
    | lookup' Substring ("i;octet") = boolcomp o octet_substring
    | lookup' Substring ("i;ascii-casemap") = 
            boolcomp o ascii_casemap_substring
    | lookup' _ s = raise (NoSuchComparator s)

  (* yuck, lots of ways to compare things *)
  fun valcomp f (V.Single s1, V.Single s2) = f (s1, s2)
    | valcomp f (V.Single s1, V.List l) = 
    if List.exists (fn x => EQUAL = f (s1, x)) l then EQUAL
    else LESS
    | valcomp f (V.List l, _) = raise ImproperComparison
    | valcomp f (V.Nil, V.Nil) = EQUAL
    | valcomp f (V.Nil, _) = GREATER
    | valcomp f (V.Single _, V.Nil) = LESS
    | valcomp f (V.Default, _) = raise ImproperComparison
    | valcomp f (_, V.Default) = raise ImproperComparison

  fun lookup comptype s = valcomp (lookup' comptype s)

end
