(* enum.sml - implementation of enumerated set
 * Larry Greenfield
 *
 * 1-based sets
 *)
(* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *)
(*
 * BASED ON CODE BY:
 *
 * COPYRIGHT (c) 1993 by AT&T Bell Laboratories.  See COPYRIGHT file for details.
 *
 * This code was adapted from Stephen Adams' binary tree implementation
 * of applicative integer sets.
 *
 *    Copyright 1992 Stephen Adams.
 *
 *    This software may be used freely provided that:
 *      1. This copyright notice is attached to any copy, derived work,
 *         or work including all or part of this software.
 *      2. Any derived work must contain a prominent notice stating that
 *         it has been altered from the original.
 *
 *   Name(s): Stephen Adams.
 *   Department, Institution: Electronics & Computer Science,
 *      University of Southampton
 *   Address:  Electronics & Computer Science
 *             University of Southampton
 *         Southampton  SO9 5NH
 *         Great Britian
 *   E-mail:   sra@ecs.soton.ac.uk
 *
 *   Comments:
 *
 *     1.  The implementation is based on Binary search trees of Bounded
 *         Balance, similar to Nievergelt & Reingold, SIAM J. Computing
 *         2(1), March 1973.  The main advantage of these trees is that
 *         they keep the size of the tree in the node, giving a constant
 *         time size operation.
 *
 *     2.  The bounded balance criterion is simpler than N&R's alpha.
 *         Simply, one subtree must not have more than `weight' times as
 *         many elements as the opposite subtree.  Rebalancing is
 *         guaranteed to reinstate the criterion for weight>2.23, but
 *         the occasional incorrect behaviour for weight=2 is not
 *         detrimental to performance.
 *
 *  ... comments deleted ...
 *)

structure Enum :> ENUM =
  struct
    type 'a sort = 'a * 'a -> order

    datatype 'a enum =
      Enum of 'a sort * 'a item

    and 'a item =
      T of {
	  elt : 'a, 
          cnt : int, 
          left : 'a item,
          right : 'a item
	}
    | E

    fun empty sort = Enum (sort, E)

    fun numItems (Enum (_, E)) = 0
      | numItems (Enum(_, T{cnt,...})) = cnt
        
    fun isEmpty (Enum (_, E)) = true
      | isEmpty _ = false

    fun sz E = 0
      | sz (T{cnt,...}) = cnt

    fun mkT(v,n,l,r) = T{elt=v,cnt=n,left=l,right=r}

      (* N(v,l,r) = T(v,1+numItems(l)+numItems(r),l,r) *)
    fun N(v,E,E) = mkT(v,1,E,E)
      | N(v,E,r as T{cnt=n,...}) = mkT(v,n+1,E,r)
      | N(v,l as T{cnt=n,...}, E) = mkT(v,n+1,l,E)
      | N(v,l as T{cnt=n,...}, r as T{cnt=m,...}) = mkT(v,n+m+1,l,r)

    fun single_L (a,x,T{elt=b,left=y,right=z,...}) = N(b,N(a,x,y),z)
      | single_L _ = raise Match
    fun single_R (b,T{elt=a,left=x,right=y,...},z) = N(a,x,N(b,y,z))
      | single_R _ = raise Match
    fun double_L (a,w,T{elt=c,left=T{elt=b,left=x,right=y,...},right=z,...}) =
          N(b,N(a,w,x),N(c,y,z))
      | double_L _ = raise Match
    fun double_R (c,T{elt=a,left=w,right=T{elt=b,left=x,right=y,...},...},z) =
          N(b,N(a,w,x),N(c,y,z))
      | double_R _ = raise Match

    (*
    **  val weight = 3
    **  fun wt i = weight * i
    *)
    fun wt (i : int) = i + i + i

    fun T' (v,E,E) = mkT(v,1,E,E)
      | T' (v,E,r as T{left=E,right=E,...}) = mkT(v,2,E,r)
      | T' (v,l as T{left=E,right=E,...},E) = mkT(v,2,l,E)

      | T' (p as (_,E,T{left=T _,right=E,...})) = double_L p
      | T' (p as (_,T{left=E,right=T _,...},E)) = double_R p

        (* these cases almost never happen with small weight*)
      | T' (p as (_,E,T{left=T{cnt=ln,...},right=T{cnt=rn,...},...})) =
            if ln<rn then single_L p else double_L p
      | T' (p as (_,T{left=T{cnt=ln,...},right=T{cnt=rn,...},...},E)) =
            if ln>rn then single_R p else double_R p

      | T' (p as (_,E,T{left=E,...})) = single_L p
      | T' (p as (_,T{right=E,...},E)) = single_R p

      | T' (p as (v,l as T{elt=lv,cnt=ln,left=ll,right=lr},
              r as T{elt=rv,cnt=rn,left=rl,right=rr})) =
          if rn >= wt ln (*right is too big*)
            then
              let val rln = sz rl
                  val rrn = sz rr
              in
                if rln < rrn then single_L p else double_L p
              end
          else if ln >= wt rn (*left is too big*)
            then
              let val lln = sz ll
                  val lrn = sz lr
              in
                if lrn < lln then single_R p else double_R p
              end
          else mkT(v,ln+rn+1,l,r)

    fun insert (Enum (compare, set), x) = 
      let
	val pos = ref 1
	fun insert' (E, x) = mkT(x, 1, E, E)
	  | insert' (T{elt=v,left=l,right=r,cnt}, x) =
	  case compare(x,v) of
	    LESS => T'(v, insert'(l, x), r)
	  | GREATER => (pos := !pos + (sz l) + 1;
			T'(v,l,insert'(r, x)))
	  | EQUAL => (pos := !pos + (sz l);
		      mkT(x, cnt, l, r))
      in (Enum (compare, insert' (set, x)), !pos) end

    fun min (T{elt=v,left=E,...}) = v
      | min (T{left=l,...}) = min l
      | min _ = raise Match
        
    fun delmin (T{left=E,right=r,...}) = r
      | delmin (T{elt=v,left=l,right=r,...}) = T'(v,delmin l,r)
      | delmin _ = raise Match

    fun deleteN (E,r) = r
      | deleteN (l,E) = l
      | deleteN (l,r) = T'(min r,l,delmin r)

    fun remove (Enum (compare, set), x) =
      let
	val pos = ref 1
	fun remove' (E, x) = raise LibBase.NotFound
	  | remove' (T{elt=v,left=l,right=r,cnt}, x) =
	  case compare(x, v) of
	    LESS => T'(v, remove'(l, x), r)
	  | GREATER => (pos := !pos + (sz l) + 1;
			T'(v, l, remove'(r, x)))
	  | EQUAL => (pos := !pos + (sz l);
		      deleteN(l, r))
      in (Enum (compare, remove' (set,x)), !pos) end

(*
    fun map f set = let
	  fun map'(acc, E) = acc
	    | map'(acc, T{elt,left,right,...}) =
		map' (add (map' (acc, left), f elt), right)
	  in 
	    map' (E, set)
	  end *)

    fun app apf (Enum (_, set))=
      let 
	val pos = ref 1
	fun apply (E) = ()
	  | apply (T{elt,left,right,...}) = 
	  (apply left; 
	   apf (elt, !pos); 
	   pos := !pos + 1; 
	   apply right)
      in
	apply set
      end

    fun foldl f b set = let
	  fun foldf (E, b) = b
	    | foldf (T{elt,left,right,...}, b) = 
		foldf (right, f(elt, foldf (left, b)))
          in
            foldf (set, b)
          end

    fun foldr f b set = let
	  fun foldf (E, b) = b
	    | foldf (T{elt,left,right,...}, b) = 
		foldf (left, f(elt, foldf (right, b)))
          in
            foldf (set, b)
          end

    fun listItems set = foldr (op::) [] set

(*    fun filter pred set =
	  foldl (fn (item, s) => if (pred item) then add(s, item) else s)
	    empty set *)

    fun find p E = NONE
      | find p (T{elt,left,right,...}) = (case find p left
	   of NONE => if (p elt)
		then SOME elt
		else find p right
	    | a => a
	  (* end case *))

    fun exists p E = false
      | exists p (T{elt, left, right,...}) =
	  (exists p left) orelse (p elt) orelse (exists p right)

  end (* BinarySetFn *)
