/* frontend.c -- Frontend for SML acap server
 * $Id: frontend.c,v 1.8 2000/06/05 20:23:23 leg Exp $
 */
/* 
 * Copyright (c) 2000 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any other legal
 *    details, please contact  
 *	Office of Technology Transfer
 *	Carnegie Mellon University
 *	5000 Forbes Avenue
 *	Pittsburgh, PA  15213-3890
 *	(412) 268-4387, fax: (412) 268-7395
 *	tech-transfer@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <syslog.h>
#include <com_err.h>
#include <netdb.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <sasl.h>

#include "../lib/xmalloc.h"
#include "../lib/prot.h"

#define CONFIGDIR "/var/acap/"
#define LOGDIR "log/"
#define ADDRESS "socket"

static sasl_conn_t *saslconn;
static struct protstream *in, *out;
static char *userid;
static char clienthost[250] = "[local]";
static time_t logtime;

#define ACAP_OK 0
#define ACAP_BADTAG 1
#define ACAP_EOF -1

struct buf {
    char *s;
    int alloc;
    int len;
};

void fatal(const char *s, int code)
{
    prot_printf(out, "* Bye \"Fatal error: %s\"\r\n", s);
    prot_flush(out);
    exit(code);
}

/* we pipe between the network and the ACAP server */
void bitpipe(void)
{
    struct sockaddr_un saun;
    int sock, c, d;
    int len;
    FILE *f;
    char buf[4096];
    fd_set read_set, rset;
    int nfds;

    /* connect to Unix socket */
    if ((sock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
	fatal("can't create socket for backend", 1);
    }
    
    saun.sun_family = AF_UNIX;
    strcpy(saun.sun_path, CONFIGDIR ADDRESS);
    len = sizeof(saun.sun_family) + strlen(saun.sun_path);

    if (connect(sock, (struct sockaddr *) &saun, len) < 0) {
	fatal("couldn't connect to backend", 1);
    }

    /* send authentication information, and discard opener */
    c = read(sock, buf, sizeof(buf));
    if (c < 0) {
	fatal("where's the opener?", 1);
    }
    snprintf(buf, sizeof(buf), "A01 AUTHENTICATE {%d+}\r\n%s\r\n", 
	     strlen(userid), userid);
    c = write(sock, buf, strlen(buf));
    if (c < 0) {
	exit(0);
    }
    c = read(sock, buf, sizeof(buf));
    if (c <= 0) {
	fatal("no auth response?", 1);
	exit(0);
    }

    /* throw the bits back 'n' forth */
    FD_ZERO(&read_set);
    FD_SET(0, &read_set);
    FD_SET(sock, &read_set);
    nfds = sock + 1;

    for (;;) {
	rset = read_set;
	select(nfds, &rset, NULL, NULL, NULL);
	/* input from client */
	if (FD_ISSET(0, &rset)) {
	    do {
		c = prot_read(in, buf, sizeof(buf));
		if (c == 0 || c < 0) {
		    close(sock);
		    exit(0);
		}
		d = 0;
		while (c > d) {
		    int a = write(sock, buf + d, c - d);
		    if (a == -1) {
			fatal("can't write to server\n", 1);
		    }
		    d += a;
		}
	    } while (in->cnt > 0);
	}

	/* input from server */
	if (FD_ISSET(sock, &rset)) {
	    c = read(sock, buf, sizeof(buf));
	    if (c < 0 || c == 0) {
		exit(0);
	    } else {
		prot_write(out, buf, c);
		prot_flush(out);
	    }
	}
    }
}

/* get a TAG or a COMMAND */
#define BUFGROWSIZE 100
int getword(struct buf *buf)
{
    int c;
    int len = 0;

    for (;;) {
	if (len == buf->alloc) {
	    buf->alloc += BUFGROWSIZE;
	    buf->s = xrealloc(buf->s, buf->alloc+1);
	}
	    
	c = prot_getc(in);
	if (c == EOF || isspace(c) || c == '(' || c == ')' || c == '\"') {
	    buf->s[len] = '\0';
	    buf->len = len;
	    return c;
	}
	buf->s[len++] = c;
    }
}

/* get a quoted or literal string */
int getstring(struct buf *buf)
{
    int c;
    int i, len = 0;
    int sawdigit = 0;
    int isnowait;

    if (buf->alloc == 0) {
	buf->alloc = BUFGROWSIZE;
	buf->s = xmalloc(buf->alloc+1);
    }

    c = prot_getc(in);
    switch (c) {
    case '\"':
	/* quoted string; max size is 1024 */
	for (;;) {
	    c = prot_getc(in);
	    if (c == '\\') {
		c = prot_getc(in);
	    } else if (c == '\"') {
		buf->s[len] = '\0';
		buf->len = len;
		return prot_getc(in);
	    } else if (c == EOF || c == '\r' || c == '\n') {
		buf->s[len] = '\0';
		if (c != EOF) prot_ungetc(c, in);
		return EOF;
	    }
	    if (len == buf->alloc) {
		buf->alloc += BUFGROWSIZE;
		buf->s = xrealloc(buf->s, buf->alloc+1);
	    }
	    buf->s[len++] = c;
	    if (len > 1024) {
		buf->s[len] = '\0';
		return EOF;
	    }
	}
    case '{':
	/* literal */
	isnowait = 0;
	buf->s[0] = '\0';
	while ((c = prot_getc(in)) != EOF && isdigit(c)) {
	    sawdigit = 1;
	    len = len * 10 + c - '0';
	}
	if (c == '+') {
	    isnowait++;
	    c = prot_getc(in);
	}
	if (!sawdigit || c != '}') {
	    if (c != EOF) prot_ungetc(c, in);
	    return EOF;
	}
	c = prot_getc(in);
	if (c != '\r') {
	    if (c != EOF) prot_ungetc(c, in);
	    return EOF;
	}
	c = prot_getc(in);
	if (c != '\n') {
	    if (c != EOF) prot_ungetc(c, in);
	    return EOF;
	}
	if (len >= buf->alloc) {
	    buf->alloc = len+1;
	    buf->s = xrealloc(buf->s, buf->alloc + 1);
	}
	if (!isnowait) {
	    prot_printf(out, "+ \r\n");
	    prot_flush(out);
	}
	for (i = 0; i < len; i++) {
	    c = prot_getc(in);
	    if (c == EOF) {
		buf->s[len] = '\0';
		return EOF;
	    }
	    buf->s[i] = c;
	}
	buf->s[len] = '\0';
	buf->len = len;
	return prot_getc(in);

    default:
	/* not a string! */
	buf->s[0] = '\0';
	if (c != EOF) prot_ungetc(c, in);
	return EOF;
    }
}

/*
 * Eat characters up to and including the next newline
 * Also look for and eat non-synchronizing literals.
 */
void eatline(int c)
{
    int state = 0;
    char *statediagram = " {+}\r";
    int size = -1;

    for (;;) {
	if (c == '\n') return;
	if (c == statediagram[state+1]) {
	    state++;
	    if (state == 1) size = 0;
	    else if (c == '\r') {
		/* Got a non-synchronizing literal */
		c = prot_getc(in);/* Eat newline */
		while (size) {
		    c = prot_getc(in); /* Eat contents */
		}
		state = 0;	/* Go back to scanning for eol */
	    }
	}
	else if (state == 1 && isdigit(c)) {
	    size = size * 10 + c - '0';
	}
	else state = 0;

	c = prot_getc(in);
	if (c == EOF) return;
    }
}

int tagchar(char ch)
{
    if (ch == 0x21) return 1;
    if (0x23 <= ch && ch <= 0x27) return 1;
    if (0x2C <= ch && ch <= 0x5B) return 1;
    if (0x5D <= ch && ch <= 0x7A) return 1;
    if (0x7C <= ch && ch <= 0x7E) return 1;
    return 0;
}

/* verify that b is a legal TAG */
int istag(struct buf *b)
{
    char *p = b->s;
    int cnt = 0;

    if (!*p) return 0;
    for (; *p; p++) { /* loop through the tag */
	cnt++;
	if (cnt == 33) return 0;
	if (!tagchar(*p)) return 0;
    }
    
    return 1;
}

/* do a SASL Authentication; command has been completely parsed;
   returns zero if the authentication is successful */
int cmd_authenticate(char *tag, char *mech, struct buf *initial)
{
    int sasl_result, ch;
    static struct buf clientin;
    int clientinlen = 0;

    char *serverout;
    unsigned int serveroutlen;
    const char *errstr;
    FILE *logfile;
    char buf[1024];

    sasl_result = sasl_server_start(saslconn, mech,
				    initial ? initial->s : NULL, 
				    initial ? initial->len : 0,
				    &serverout, &serveroutlen,
				    &errstr);

    while (sasl_result == SASL_CONTINUE) {
	prot_printf(out, "+ {%d}\r\n", serveroutlen);
	prot_write(out, serverout, serveroutlen);
	prot_printf(out, "\r\n");
	free(serverout);

	ch = getstring(&clientin);
	if (ch == '\r') ch = prot_getc(in);
	if (ch != '\n') {
	    prot_printf(out, "%s Bad \"Authentication cancelled\"\r\n", 
			tag);
	    eatline(ch);
	    return SASL_FAIL;
	}
	sasl_result = sasl_server_step(saslconn, clientin.s, clientin.len,
				       &serverout, &serveroutlen, &errstr);
    }

    if (sasl_result != SASL_OK) {
	/* failed authentication */
	if (!errstr) {
	    errstr = sasl_errstring(sasl_result, NULL, NULL);
	}
	if (errstr) {
	    prot_printf(out, "%s No \"%s\"\r\n", tag, errstr);
	} else {
	    prot_printf(out, "%s No \"Error authenticating\"\r\n", tag);
	}

	return sasl_result;
    }

    /* authentication successful! */
    sasl_result = sasl_getprop(saslconn, SASL_USERNAME,
			       (void **) &userid);

    if (sasl_result != SASL_OK) {
	prot_printf(out, "%s No \"Internal error\"", tag);
	/* what should i do now? */
	return sasl_result;
    }
    if (serveroutlen > 0) {
	prot_printf(out, "%s Ok (SASL {%d}\r\n", tag, serveroutlen);
	prot_write(out, serverout, serveroutlen);
	prot_printf(out, ") \"Welcome\"\r\n");
    } else {
	prot_printf(out, "%s Ok \"Welcome\"\r\n", tag);
    }
    prot_flush(out);

    prot_setsasl(in,saslconn);
    prot_setsasl(out,saslconn);

    syslog(LOG_NOTICE, "login: %s %s mech %s", clienthost, userid, mech);

    /* Create telemetry log */
    sprintf(buf, "%s%s%u", CONFIGDIR, LOGDIR, getpid());
    logfile = fopen(buf, "w");
    if (logfile) {
	/* we want to log everybody */
	prot_setlog(in, fileno(logfile));
	prot_setlog(out, fileno(logfile));

	prot_setlogtime(in, &logtime);
	prot_setlogtime(out, &logtime);

    }

    return 0;
}

/* perform a NOOP; command has been completely parsed */
void cmd_noop(char *tag)
{
    prot_printf(out, "%s Ok \"Noop completed\"\r\n", tag);
}

void cmd_logout(char *tag)
{
    prot_printf(out, "* Bye \"Whoever you are\"\r\n");
    prot_printf(out, "%s Ok \"Logout completed\"\r\n", tag);
    prot_flush(out);
    exit(0);
}

/* we implement the ACAP unauthenticated state:
   AUTHENTICATE
   NOOP
   LANG
   LOGOUT */
void cmdloop(void)
{
    static struct buf tag, cmd, arg1, arg2;
    int ch, *p;
    int args;

    for (;;) {
	ch = getword(&tag);

	if (ch == EOF) {
	    exit(0);
	}

	if (ch != ' ' || !istag(&tag)) {
	    prot_printf(out, "* BAD Invalid tag\r\n");
	    eatline(ch);
	    continue;
	}

	ch = getword(&cmd);
	if (!cmd.s[0]) {
	    prot_printf(out, "%s BAD Null command\r\n", tag.s);
	    eatline(ch);
	    continue;
	}

	switch (cmd.s[0]) {
	case 'A': case 'a':
	    /* authenticate */
	    if (!strcasecmp(cmd.s, "Authenticate")) {
		if (ch != ' ') goto missingargs;
		ch = getstring(&arg1);
		args = 1;
		if (ch == ' ') {
		    ch = getstring(&arg2);
		    args = 2;
		}
		if (ch == '\r') ch = prot_getc(in);
		if (ch != '\n') goto extraargs;
		if (!cmd_authenticate(tag.s, arg1.s, 
				      (args == 2 ? &arg2 : NULL))) {
		    bitpipe();
		}
		continue;
	    }
	    goto badcmd;

	case 'N': case 'n':
	    /* noop */
	    if (!strcasecmp(cmd.s, "Noop")) {
		if (ch == '\r') ch = prot_getc(in);
		if (ch != '\n') goto extraargs;
		cmd_noop(tag.s);
		continue;
	    }
	    goto badcmd;

	case 'L': case 'l':
	    /* logout */
	    if (!strcasecmp(cmd.s, "Logout")) {
		if (ch == '\r') ch = prot_getc(in);
		if (ch != '\n') goto extraargs;
		cmd_logout(tag.s);
		continue;
	    }

	    /* lang */
	    if (!strcasecmp(cmd.s, "Lang")) {
		if (ch != ' ') goto missingargs;
		while (ch != '\n') {
		    ch = getstring(&arg1);
		    if (ch == '\r') ch = prot_getc(in);
		    if (ch != '\n' && ch != ' ') {
			goto malformedcmd;
		    }
		}
		prot_printf(out, 
			    "%s No \"No alternate languages available\"\r\n",
			    tag.s);
		continue;
	    }
	    goto badcmd;
	    
	default:
	badcmd:
	    prot_printf(out, "%s Bad \"Unrecognized command\"\r\n", tag.s);
	    eatline(ch);
	    continue;
	}	    
	
    missingargs:
	prot_printf(out, "%s Bad \"Missing required argument\"\r\n", 
		    tag.s);
	eatline(ch);
	continue;

    extraargs:
	prot_printf(out, "%s Bad \"Unexpected extra arguments\"\r\n", 
		    tag.s);
	eatline(ch);
	continue;

    malformedcmd:
	prot_printf(out, "%s Bad \"Malformed command\"\r\n", tag.s);
	eatline(ch);
	continue;
    }

}

/* This creates a structure that defines the allowable
 *   security properties 
 */
static sasl_security_properties_t *make_secprops(int min,int max)
{
  sasl_security_properties_t *ret=
    (sasl_security_properties_t *) xmalloc(sizeof(sasl_security_properties_t));

  ret->maxbufsize = 4000;
  ret->min_ssf = min;		/* minimum allowable security strength */
  ret->max_ssf = max;		/* maximum allowable security strength */

  ret->security_flags = 0;
#if 0
  if (!config_getswitch("allowplaintext", 1)) {
      ret->security_flags |= SASL_SEC_NOPLAINTEXT;
  }
  if (!config_getswitch("allowanonymouslogin", 0)) {
      ret->security_flags |= SASL_SEC_NOANONYMOUS;
  }
#endif
  ret->property_names = NULL;
  ret->property_values = NULL;

  return ret;
}

int main(int argc, char **argv, char **envp)
{
    int salen;
    struct hostent *hp;
    int timeout;
    char hostname[MAXHOSTNAMELEN+1];
    sasl_security_properties_t *secprops = NULL;
    struct sockaddr_in localaddr, remoteaddr;
    int haveaddr = 0;
    int count;
    char *str;

    openlog("acapd", LOG_PID, LOG_LOCAL6);

    if (gethostname(hostname, MAXHOSTNAMELEN) != 0) {
	fatal("gethostname failed\n", 1);
    }

    in = prot_new(0, 0);
    out = prot_new(1, 1);

    signal(SIGPIPE, SIG_IGN);

    
    /* Find out name of client host */
    salen = sizeof(remoteaddr);
    if (getpeername(0, (struct sockaddr *)&remoteaddr, &salen) == 0 &&
	remoteaddr.sin_family == AF_INET) {
	if (hp = gethostbyaddr((char *)&remoteaddr.sin_addr,
			       sizeof(remoteaddr.sin_addr), AF_INET)) {
	    strncpy(clienthost, hp->h_name, sizeof(clienthost)-30);
	    clienthost[sizeof(clienthost)-30] = '\0';
	}
	else {
	    clienthost[0] = '\0';
	}
	strcat(clienthost, "[");
	strcat(clienthost, inet_ntoa(remoteaddr.sin_addr));
	strcat(clienthost, "]");
	salen = sizeof(localaddr);
	if (getsockname(0, (struct sockaddr *)&localaddr, &salen) == 0) {
	    haveaddr = 1;
	}
    }

    syslog(LOG_NOTICE, "connection from %s", clienthost);

    /* Make a SASL connection and setup some properties for it */
    if (sasl_server_init(NULL, "Cyrus") != SASL_OK) {
	fatal("SASL failed initializing: sasl_server_init()", 1); 
    }

    /* other params should be filled in */
    if (sasl_server_new("acap", hostname, NULL, NULL, SASL_SECURITY_LAYER, 
			&saslconn) != SASL_OK) {
	fatal("SASL failed initializing: sasl_server_new()", 1); 
    }

    secprops = make_secprops(0, 128);
    sasl_setprop(saslconn, SASL_SEC_PROPS, secprops);

    sasl_setprop(saslconn, SASL_IP_REMOTE, &remoteaddr);
    sasl_setprop(saslconn, SASL_IP_LOCAL, &localaddr);

    /* Set inactivity timer */
    if (timeout < 30) timeout = 30;
    prot_settimeout(in, timeout*60);
    prot_setflushonread(in, out);

    sasl_listmech(saslconn, NULL, 
		  " (Sasl \"",
		  "\" \"", "\")",
		  &str, NULL, &count);
    if (count == 0) str = "";
    prot_printf(out, "* Acap "
		"(Implementation "
                      "\"SML Frontend, Carnegie Mellon Project Cyrus\") "
		"(ContextLimit \"100\")%s\r\n", str);

    cmdloop();
}
