/* pgp4pine keyrings.c
 *
 * See common.c for the version and license.
 * Copyright (C) 1998 by Chris Wiegand
 */

#include <fcntl.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <time.h>
#include "defines.h"
#include "includes.h"
#include "structs.h"
#include "declares.h"
#include "externs.h"

/* ------------------------------------------------------------ */
/* getItem is a parser like 'cut -f<fieldNum> -d<delimiter>     */
/* ------------------------------------------------------------ */

char *getItem(const char *incoming, const char delimiter, unsigned int fieldNum)
{
	/* I support a string as long as an "unsigned long" is big,
	       and as many fields as an "unsigned int" is big.
	   I return a pointer that must be "free"ed! */

	char *tmpString = NULL, *retVal=NULL, *tmpStr = NULL;
	unsigned long l;
	unsigned int thisItem=1;

	tmpString = (char *) myMalloc(strlen(incoming)+2);
	retVal = (char *) myMalloc(strlen(incoming)+2);    /* 2 just to be safe. */
	strcpy(tmpString,incoming); /* STRCPY SAFE */
	strcpy(retVal,"");          /* STRCPY SAFE */

	/* I only set startChar and endChar on a delimiter so that if the
	 * requested item is the 1st item, it starts and ends on item #1.
	 */

	for (l=0;l<=strlen(tmpString);l++) {
		if (tmpString[l] == delimiter) {
			thisItem++;
		} else {
			if (thisItem == fieldNum) {
				tmpStr = tmpString+l; /* we want to set the "tmpStr" to a particular starting place within tmpString; */
				strncat(retVal,tmpStr,1); /* now, copy 1 ONE character... */
			}
		}
	}

	if (tmpString != NULL) free(tmpString);
	return retVal;
}

/* ------------------------------------------------------------ */
/* Little helper, used in pkiKey()                              */
/* ------------------------------------------------------------ */

void cpString(FILE *inFile, char *buf, int maxLen)
{
	int len;
	
	fgets(buf,maxLen,inFile);
	
	len = strlen(buf);
	if ( buf[len-1] == '\n' )
           buf[len-1]='\0';
}

/* ------------------------------------------------------------ */
/* Deallocate Keyring						*/
/* ------------------------------------------------------------ */

void deallocAllKeys(struct pkiKey *killKey)
{
	struct pkiKey *thisKey = NULL;
	struct pkiKey *nextKey2 = NULL;
	thisKey = killKey; /* copy the pointer... */
	if (thisKey != NULL) {
		do {
			snprintf(debugLine,DEBUG_LINE_LENGTH,"%#x, ",thisKey);
			documentStatus(debugLine);
			nextKey2 = thisKey->nextKey;
			myFree(thisKey);
			thisKey = nextKey2;
		} while (thisKey != NULL);
	}
	documentStatus("Done with this keyring.\n");
}

/* ------------------------------------------------------------ */
/* Fetch a Keyring, using a program. 0: Public, 1: Secret 	*/
/* ------------------------------------------------------------ */

struct pkiKey *getKeyring_program(int getSecretOnly)  /* Get ring from program, instead of file */
{
	/* I return a pointer that must be "free"ed! */
	struct pkiKey *myNewKey = NULL;
	struct pkiKey *firstKey = NULL;
	struct pkiKey *oldKey = NULL;

	FILE *inFile, *outFile; /* DO NOT FREE ME ( I don't think so, that is. ) */
	char buf[65530], buf2[CONSOLE_IO_LINE_LENGTH], keyType[8], tmpString[CONSOLE_IO_LINE_LENGTH], appString[MAX_COMMAND_LINE_LENGTH];
	char *tmpString2;
	char *homeDir;
	int inSec=0, lineType=0, needSubkey=0, knowpgpversion;
	
	homeDir = getenv("HOME");
	switch (myUserPrefs->pgpVersion) {
	case 5:
/* Parses: 
Type Bits KeyID      Created    Expires    Algorithm       Use
sec+ 1024 0xEF2769D3 1999-11-30 2001-07-22 DSS             Sign & Encrypt 
sub  2048 0x11F70D15 1999-11-30 2001-07-22 Diffie-Hellman                 
uid  Holger Lamm <holger@eit.uni-kl.de> */

		/* PGP 5 only needs one pass, as it gives us ALL the keys... */
		sprintf(appString,"%s +language=us -l",myUserPrefs->pgp5pgpk);
			
		pipe_pgp(appString,&inFile,&outFile);
		if (outFile == NULL) return NULL;
		while (!feof(outFile)) {
			fgets(buf,sizeof(buf),outFile);
			if (strlen(buf) < 1) break; /* we're done..... */
			documentStatus(buf);
			strncpy(keyType,strtok(buf," "),8);
			lineType=0; /* unknown default... */
			if (strcmp(keyType,"sec") == 0 || strcmp(keyType,"sec+") == 0) lineType=1; /* secret line... */
			if (strcmp(keyType,"pub") == 0) lineType=2; /* public key */
			if (strcmp(keyType,"sub") == 0) lineType=3; /* subkey...  */
			if (strcmp(keyType,"uid") == 0) lineType=4; /* userid..   */
			if (strcmp(keyType,"sig") == 0 || strcmp(keyType,"SIG") == 0 
			                               || strcmp(keyType,"sig?") == 0) lineType=5; /* signature... */

			if (lineType == 1) inSec = 1;
			if (lineType == 2) inSec = 0;

			switch (lineType) {
			case 1:
			case 2:
				/* If we don't want public keys, skip this look... */
				if (lineType == 2 && getSecretOnly) break;
				stillAlive();
                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
                                if (firstKey == NULL) firstKey = myNewKey;
                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
                                oldKey = myNewKey;
                                                                                                                                
				/* next, key size... */
				strncpy(tmpString,strtok('\0'," "),KEY_SIZE_LENGTH);
				strncpy(myNewKey->keySize,tmpString,KEY_SIZE_LENGTH);

				/* next, keyID */
				strncpy(tmpString,strtok('\0'," "),KEY_ID_LENGTH);
				if ((tmpString[0] == '0') && (tmpString[1] == 'x')) {  /* normal PGP5: 0x12345678 */
				  strncpy(myNewKey->keyID,tmpString,KEY_ID_LENGTH);
				} else {				/* Martin Mokrejs owns a strange version without 0x */
				  sprintf(myNewKey->keyID,"0x%s",tmpString); /* SPRINTF SAFE */	
				}
				
				/* next, 2 dates */
				strncpy(tmpString,strtok('\0'," "),5);
				strncpy(tmpString,strtok('\0'," "),5);
				/* I don't use them... */
				
				/* next, key type... */
				strncpy(tmpString,strtok('\0'," "),KEY_TYPE_LENGTH);
				strncpy(myNewKey->keyType,tmpString,KEY_TYPE_LENGTH);
				break;
			case 4:
				if (!getSecretOnly || inSec) {
					/* If we're loading only secret keys, I only need one name
					   If we're loading all keys, I need all names... */
					if (getSecretOnly) {
						/* If loading only secret keys, and already have a name, break. */
						if (strlen(myNewKey->displayName) > 2) break;
					} else {
						/* If loading all keys, need all names, if last key has a name already, make new key... */
						if (strlen(myNewKey->displayName) > 2) {
			                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
			                                if (firstKey == NULL) firstKey = myNewKey;
			                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
							strcpy(myNewKey->keyID,oldKey->keyID);     /* STRCPY SAFE */
							strcpy(myNewKey->keySize,oldKey->keySize); /* STRCPY SAFE */
							strcpy(myNewKey->keyType,oldKey->keyType); /* STRCPY SAFE */
			                                oldKey = myNewKey;
						}
					}

					strncpy(tmpString,strtok('\0',"\n"),EMAIL_ADDRESS_MAX_LENGTH);
					UnescapeString(tmpString);	/* remove escaped Chars */
					/* Has a leading space, so copy from tmpString+1 */
					strncpy(myNewKey->displayName,extractDisplayName(tmpString+1),EMAIL_ADDRESS_MAX_LENGTH);
					strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString+1),EMAIL_ADDRESS_MAX_LENGTH);
				}
				break;
			}
		}
		fclose(outFile);
		break;
	case 2:
	case 6:
		/* PGP 6.5.2 has the same output than PGP 2.6.2, like this:
/*
Type bits/keyID    Date       User ID
pub  1024/29F7DDE1 1998/09/22 Holger Lamm <lamm@rhrk.uni-kl.de>
                              Holger Lamm <lamm@student.uni-kl.de>
                              Holger Lamm <holger@e-technik.uni-kl.de>
*/			      
                /* PGP 6.5.1 is totally different: */
/*
Type bits      keyID      Date       User ID
DSS  1024      0x8E9D00CB 1999/05/01 Cognite.net <cognition@bigfoot.com>		// Main key with UID
DSS  4096/1024 0x45FAC283 1999/06/17 expires 2000/05/15                                 // subkey with expire date
                                      cogNiTioN, cognite.net <cognition@bigfoot.com>	// UID belonging to previous key
                                     David Webster <cognition@bigfoot.com> 
DSS  1024      0xA4679654 1999/01/17 *** KEY REVOKED ***                                // revoked		   
*/
	    /* Try to distinguish here */

	    
	    if (getSecretOnly) {
		if (myUserPrefs->pgpVersion==2) /* different keyring name for 2 and 6 */
		   sprintf(appString,"%s +LANGUAGE=en -kv \"*\" %s/.pgp/secring.pgp",myUserPrefs->pgp2pgp,homeDir);
	        else
		   sprintf(appString,"%s +LANGUAGE=en -kv \"*\" %s/.pgp/secring.skr",myUserPrefs->pgp2pgp,homeDir);
	    } else {
	    	    sprintf(appString,"%s +LANGUAGE=en -kv", myUserPrefs->pgp2pgp);
	    }
	    pipe_pgp(appString,&inFile,&outFile);
	    if (outFile == NULL) return NULL;

	    knowpgpversion=0;
	    while (!knowpgpversion) {
		if (!fgets(buf,sizeof(buf),outFile)) knowpgpversion=-1;	/* EOF. Something wrong! */
		documentStatus(buf);
		if (!strncmp(buf,"Type",4))                  knowpgpversion=262;
		if (!strncmp(buf,"Type bits      keyID",20)) knowpgpversion=651;
		/* If a 'Type' line comes, the loop terminates. If the format happens to change 
		   to something neither of these two, this is the default (probably malfunctioning,
			  but better than loop over the whole keyring here */
	    }			
	    if (knowpgpversion==262) {
		documentStatus("Recognized PGP version 2.6.2 or 6.5.2\n");
		while (!feof(outFile)) {
			stillAlive();
			fgets(buf,sizeof(buf),outFile);
			if (strlen(buf) < 1) break; /* we're done..... */
			documentStatus(buf);
			strncpy(keyType,buf,3);
			lineType = 0;                                 /* unknown yet... */
			if (strcmp(keyType,"sec") == 0) lineType = 1; /* secret key	*/
			if (strcmp(keyType,"pub") == 0) lineType = 2; /* public key	*/
			if (strcmp(keyType,"sig") == 0) lineType = 5; /* signature	*/
			if (strcmp(keyType,"   ") == 0) lineType = 4; /* name		*/

			switch (lineType) {
			case 1:
			case 2:
				if (lineType == 2 && getSecretOnly) break;
                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
                                if (firstKey == NULL) firstKey = myNewKey;
                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
                                oldKey = myNewKey;

				/* eat leading "sec" or "pub" */
				strtok(buf, " ");
				
				/* next, key size... */
				strncpy(tmpString,strtok('\0',"/"),KEY_SIZE_LENGTH);
				strncpy(myNewKey->keySize,tmpString,KEY_SIZE_LENGTH);

				/* next, keyID */
				strncpy(tmpString,strtok('\0'," "),KEY_ID_LENGTH);
				sprintf(myNewKey->keyID,"0x%s",tmpString); /* SPRINTF SAFE */
				
				/* next, 1 date */
				strncpy(tmpString,strtok('\0'," "),5);
				/* I don't use it... */
					
				/* next, key type... */
				strcpy(myNewKey->keyType,"RSA"); /* STRCPY SAFE */
				
				strncpy(tmpString,strtok('\0',"\n"),EMAIL_ADDRESS_MAX_LENGTH);
				strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
				strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
				break;
			case 4:
				if (!getSecretOnly || inSec) {
					/* If we're loading only secret keys, I only need one name
					   If we're loading all keys, I need all names... */
					if (getSecretOnly) {
						/* If loading only secret keys, and already have a name, break. */
						if (strlen(myNewKey->displayName) > 2) break;
					} else {
						/* If loading all keys, need all names, if last key has a name already, make new key... */
						if (strlen(myNewKey->displayName) > 2) {
			                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
			                                if (firstKey == NULL) firstKey = myNewKey;
			                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
							strcpy(myNewKey->keyID,oldKey->keyID);     /* STRCPY SAFE */
							strcpy(myNewKey->keySize,oldKey->keySize); /* STRCPY SAFE */
							strcpy(myNewKey->keyType,oldKey->keyType); /* STRCPY SAFE */
			                                oldKey = myNewKey;
						}
					}

					/* We have to use sscanf with a space in the first position
					   of the format string to ignore the leading spaces before the name */
					strncpy(buf2,buf,CONSOLE_IO_LINE_LENGTH); /* Makes sscanf safe... */
					sscanf(buf2," %[^\n]",tmpString); /* SSCANF SAFE */
					strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
				}
				break;
			}
		}
		fclose(outFile);
		break;
	    } else if (knowpgpversion == 651) {		/* PGP 6.5.1 style keyring */
		documentStatus("Recognized PGP version 6.5.1\n");
		while (!feof(outFile)) {
			stillAlive();
			fgets(buf,sizeof(buf),outFile);
			if (strlen(buf) < 1) break; /* we're done..... */
			documentStatus(buf);
			strncpy(keyType,buf,3);
			lineType = 0; /* unknown yet... */
			if ((strcmp(keyType,"DSS") == 0) || (strcmp(keyType,"RSA") == 0))
			  lineType = (getSecretOnly ? 1 : 2);         /* secret or public key */
			if (needSubkey && (strcmp(keyType," DH") == 0))
			  lineType = 6;	/* Subkey which carries UID we need */
			if (strcmp(keyType,"sig") == 0) lineType = 5; /* signature */
			if (strcmp(keyType,"   ") == 0) lineType = 4; /* name      */

			switch (lineType) {
			case 1:
			case 2:
                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
                                if (firstKey == NULL) firstKey = myNewKey;
                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
                                oldKey = myNewKey;

				/* get key type (still in 'keyType') */
			        if (strcmp(keyType,"DSS") == 0) strcpy(myNewKey->keyType,"DSS"); else
			        if (strcmp(keyType,"RSA") == 0) strcpy(myNewKey->keyType,"RSA");
			        else strcpy(myNewKey->keyType,"???");
			
				/* eat leading "sec" or "pub" */
				strtok(buf, " ");
				
				/* next, key size... */
				strncpy(tmpString,strtok('\0'," "),KEY_SIZE_LENGTH);
				strncpy(myNewKey->keySize,tmpString,KEY_SIZE_LENGTH);

				/* next, keyID */
				strncpy(myNewKey->keyID,strtok('\0'," "),KEY_ID_LENGTH);
				
				/* next, 1 date */
				strncpy(tmpString,strtok('\0'," "),5);
				/* I don't use it... */
				
				/* Now either a UID follows, or nothing, or "expires..."
				   In the last two cases, a " DH" subkey with UID follows */
				
				needSubkey = 1;
				if ((tmpString2=strtok('\0',"\n")) != NULL) /* otherwise nothing follows */
				 if (strncmp(tmpString2,"expires",6))
				 {			
				   strncpy(myNewKey->displayName,extractDisplayName(tmpString2),EMAIL_ADDRESS_MAX_LENGTH);
				   strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString2),EMAIL_ADDRESS_MAX_LENGTH);
				   needSubkey=0;
				 }
				break;
			case 6: /* Get UID from Subkey to current 'myNewKey'
				   Skip a number of tokens */
				strtok(buf ," "); /* " DH" */
				strtok('\0'," "); /* Size  */
				strtok('\0'," "); /* KeyID */
				strtok('\0'," "); /* Creation Date */
				tmpString2 = strtok('\0',"\n");
				if (tmpString2 != NULL) /* It also happens the subkey to be empty */
				{
				  strncpy(myNewKey->displayName,extractDisplayName(tmpString2),EMAIL_ADDRESS_MAX_LENGTH);
				  strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString2),EMAIL_ADDRESS_MAX_LENGTH);
				  needSubkey=0; /* We got it !!! */
				}
				break;
			case 4:
				if (!getSecretOnly || inSec) {
					/* If we're loading only secret keys, I only need one name
				 	   If we're loading all keys, I need all names... */
					if (getSecretOnly) {
						/* If loading only secret keys, and already have a name, break. */
						if (strlen(myNewKey->displayName) > 2) break;
					} else {
						/* If loading all keys, need all names, if last key has a name already, make new key... */
						if (strlen(myNewKey->displayName) > 2) {
			                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
			                                if (firstKey == NULL) firstKey = myNewKey;
			                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
							strcpy(myNewKey->keyID,oldKey->keyID);     /* STRCPY SAFE */
							strcpy(myNewKey->keySize,oldKey->keySize); /* STRCPY SAFE */
							strcpy(myNewKey->keyType,oldKey->keyType); /* STRCPY SAFE */
			                                oldKey = myNewKey;
						}
					}

					/* We have to use sscanf with a space in the first position
					   of the format string to ignore the leading spaces before the name */
					strncpy(buf2,buf,CONSOLE_IO_LINE_LENGTH); /* Makes sscanf safe... */
					sscanf(buf2," %[^\n]",tmpString); /* SSCANF SAFE */
					strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
				}
				break;
			} /* switch */
		}
		fclose(outFile);
		break;		
	} else {	/* no known PGP version. Abort. */
		fprintf(stderr,"PGP version unknown. Please contact the author, holger@flatline.de\n");
		fclose(outFile);
		break;
	}
	case 1:
/* Parses:
pub:q:1024:17:E32BD3F1BE07D996:1999-11-03::77:-:Holger Lamm <holger@pgp4pine.flatline.de>:
uid:q::::::::Holger Lamm <holger@flatline.de>:
sub:q:2048:16:6360A0A5D47D03BA:1999-11-03::77::
*/
		/* GPG also requires two passes... */
		if (getSecretOnly) {
			/* Because of a bug in GnuPG 0.4x and 0.9x (thru 0.95, at least), we can only tell it ONE keyring file
			   when we're loading keyrings, or it displays it as many times as keyring files we give it. */
			sprintf(appString,"%s --list-secret-keys --with-colons --no-greeting 2> /dev/null",myUserPrefs->pgp1gpg);
		} else {
			sprintf(appString,"%s --list-keys --with-colons --no-greeting 2> /dev/null",myUserPrefs->pgp1gpg);
		}

		pipe_pgp(appString,&inFile,&outFile);
		if (outFile == NULL) return NULL;
		while (!feof(outFile)) {
			stillAlive();
			/* using fgets() causes problems... */
			cpString(outFile,buf,65530);
			if (strlen(buf) < 1) break; /* we're done..... */
			documentStatus(buf);
			if (strchr(buf,':') != NULL) {
				strncpy(keyType,getItem(buf,':',1),3);
				lineType = 0;
				if (strcmp(keyType,"sec") == 0) lineType = 1; /* secret line... */
				if (strcmp(keyType,"pub") == 0) lineType = 2; /* public key     */
				if (strcmp(keyType,"uid") == 0) lineType = 4; /* user id        */
	
				if (lineType == 1) inSec = 1;
				if (lineType == 2) inSec = 0;

				switch (lineType) {
				case 1:
				case 2:
					if (lineType == 2 && getSecretOnly) break;
	                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
	                                if (firstKey == NULL) firstKey = myNewKey;
	                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
	                                oldKey = myNewKey;
		
					/* next, key size... */
					strncpy(tmpString,getItem(buf,':',3),KEY_SIZE_LENGTH);
					strncpy(myNewKey->keySize,tmpString,KEY_SIZE_LENGTH);
	
					/* next, GPG key type... */
					strncpy(tmpString,getItem(buf,':',4),KEY_TYPE_LENGTH);
					strcpy(myNewKey->keyType,"Unknown"); /* STRCPY SAFE */
					if (strcmp(tmpString,"1") == 0) strcpy(myNewKey->keyType,"RSA");      /* STRCPY SAFE */
					if (strcmp(tmpString,"16") == 0) strcpy(myNewKey->keyType,"ElGamal"); /* STRCPY SAFE */
					if (strcmp(tmpString,"17") == 0) strcpy(myNewKey->keyType,"DSA");     /* STRCPY SAFE */
					if (strcmp(tmpString,"20") == 0) strcpy(myNewKey->keyType,"ElGamal"); /* STRCPY SAFE */
				
					/* keyID */
					strncpy(tmpString,getItem(buf,':',5),KEY_ID_LENGTH);
					/*	Any GPG keyid starting w/ a letter	      */
				        /*	must be prefixed with '0' to work.	      */
				        /*						      */
				        /*	As the additional '0' does no harm	      */
				        /*	in case of the keyid starting w/ a	      */
				        /*	digit, add it anywhere. 		      */
				        /*						      */
				        /*	Keyid len is 64bits, which makes for	      */
				        /*	16 digits, size of array is big enough.       */
				        /*						      */
					sprintf(myNewKey->keyID,"0%s",tmpString); /* SPRINTF SAFE */

					/* finally, user name */
					strncpy(tmpString,getItem(buf,':',10),EMAIL_ADDRESS_MAX_LENGTH);
					UnescapeString(tmpString);	/* The ':' is quoted with '\x3a', so unescape */
					strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					break;
				case 4:
					if (!getSecretOnly || inSec) {
						/* If we're loading only secret keys, I only need one name
						   If we're loading all keys, I need all names... */
						if (getSecretOnly) {
							/* If loading only secret keys, and already have a name, break. */
							if (strlen(myNewKey->displayName) > 2) break;
						} else {
							/* If loading all keys, need all names, if last key has a name already, make new key... */
							if (strlen(myNewKey->displayName) > 2) {
				                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
				                                if (firstKey == NULL) firstKey = myNewKey;
				                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
								strcpy(myNewKey->keyID,oldKey->keyID);     /* STRCPY SAFE */
								strcpy(myNewKey->keySize,oldKey->keySize); /* STRCPY SAFE */
								strcpy(myNewKey->keyType,oldKey->keyType); /* STRCPY SAFE */
				                                oldKey = myNewKey;
							}
						}

						/* finally, user name */
						strncpy(tmpString,getItem(buf,':',10),EMAIL_ADDRESS_MAX_LENGTH);
						strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
						strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					}
					break;
				}
			}
		}
		fclose(outFile);
		break;
	}
	return firstKey; /* Return pointer to keyring */
}

/* ------------------------------------------------------------ */
/* Prompt to select one of different secret keys, if ambiguous  */
/* ------------------------------------------------------------ */

void chooseSecretKey()
{
	char retValue[EMAIL_ADDRESS_MAX_LENGTH];
	strcpy(retValue,""); /* STRCPY SAFE */

	snprintf(debugLine,DEBUG_LINE_LENGTH,"keyrings.c: chooseSecretKey()\n");
	documentStatus(debugLine);
	
	/* now, let's ask which one they want to use... */
	if (secretKeyring == NULL) {
		/* no secret keys found!!! */
		printf("\nWarning! It appears that you don't have a secret key defined! Please see\n"
		       "your PGP/GPG documentation for assistance in making a key pair!\n");
	} else {
		if (secretKeyring->nextKey != NULL) {
			/* multiple keys... */
			int i=0, sel = 0, found=0;
			struct pkiKey *currentKey = NULL;
	
			currentKey = secretKeyring;
			do {
				if (strcmp(myUserPrefs->myAddress,currentKey->emailAddress)==0)
				{
				  found=1;
				  documentStatus("chooseSecretKey(): Using ");
				  documentStatus(myUserPrefs->myAddress);
				}
				else
				  currentKey = currentKey->nextKey;
			} while (currentKey!=NULL && !found);
			
			if (!found)
			{
			/* got no default key defined... */
			  printf("\nYou have multiple secret keys. Listed below are the keys. Please select one.\n"
			         "If you always use the same key, set it in your .pgp4pinerc!\n");
			  currentKey = secretKeyring;
			  do {
				i++;
				printKeyInfo((char) (i+'a'-1),currentKey);
				currentKey = currentKey->nextKey;
			  } while (currentKey != NULL);
			  printf("\n");
			  sel = askAlphaRange("Please enter a valid selection ",(i+'a'-1),'a',0)-'a';
			  currentKey = secretKeyring; /* set back to beginning... */

			  snprintf(debugLine,DEBUG_LINE_LENGTH,"Selected: %i\n",sel+'a');
			  documentStatus(debugLine);
			  for (i=0;i<sel;i++) currentKey = currentKey->nextKey; /* skip to key we want */
			  snprintf(debugLine,DEBUG_LINE_LENGTH,"CurrentKey: %s\n",currentKey->keyID);	
			  documentStatus(debugLine);
			}
#ifdef USE_DISPLAY_NAME
			strcpy(retValue,currentKey->displayName);
#else
			strcpy(retValue,currentKey->keyID);
#endif
	
			strcpy(mySecretKey,retValue);
			printf("Please wait ...\n");
		} else {
			/* only one secret key... */
#ifdef USE_DISPLAY_NAME
			strcpy(retValue,secretKeyring->displayName);
#else
			strcpy(retValue,secretKeyring->keyID);
#endif
			strcpy(mySecretKey,retValue);
		}
	}
	documentStatus("Leaving secret area!\n");
}

/* ------------------------------------------------------------ */
/* Throw out Keyrings belonging to Profile from cache		*/
/* ------------------------------------------------------------ */

void flushKeyringcache(char *profilename)
{
   int cache,flush;
   struct stat cachestats;
   off_t  size,newsize;
   u_int8_t *cacheBuf, *cacheBufstart;
   u_int32_t entrysize;
   
   if ((cache=open(DecryptCacheName,O_RDWR))<0) return;		/* Can't open -> done */
   if (fstat(cache,&cachestats)<0) return;			/* stat for size */
   newsize = size = cachestats.st_size;				/* copy length and use as counter */
   if ((cacheBuf = cacheBufstart = (u_int8_t *)mmap(0,cachestats.st_size, 
         PROT_READ|PROT_WRITE,MAP_SHARED,cache,0)) == (u_int8_t *) -1) return;
   
   while(size>0) {
      flush=0;
      entrysize = (cacheBuf[2]<<24)|(cacheBuf[3]<<16)|(cacheBuf[4]<<8)|cacheBuf[5];
      if ((cacheBuf[0]==0x12) && (cacheBuf[1]==0x34) && (cacheBuf[6]==CACHE_VERSION) &&
          ((cacheBuf[7]==1) || (cacheBuf[7]==2)) &&
           (strncmp(cacheBuf+8,profilename,16)==0))
	 flush=1;
      
      size -= entrysize;
      if (size>=0) /* if size<0, size integrity was violated */
      {
	if (flush==1)  /* do a flush: Copy from next msg until EOF */
	{
	  if (size>0) 
	  {
	     memmove(cacheBuf,cacheBuf + entrysize, size);
	     memset (cacheBuf+size,0,entrysize);  /* Zero rest of file */
          }
	  newsize -= entrysize;	        
	} else cacheBuf += entrysize;
      }
    }
    munmap(cacheBufstart, cachestats.st_size);
	
    if ((newsize < cachestats.st_size) && newsize>0) 
	  /* If we flushed, truncate file */
	ftruncate(cache, newsize);
    close(cache);
    if (newsize<=0) remove(DecryptCacheName);
          /* <0 must be an error. Kill either way .. */
}

/* ------------------------------------------------------------ */
/* Write keyrings to cache file.		 		*/
/* Assumes already flushed, so just append the entries.		*/
/* ------------------------------------------------------------ */

void buildKeyringcache(char *profilename) 
{
   int cache;
   u_int32_t entrysize,date;
   u_int8_t  b;
   u_int16_t numentries;
   struct pkiKey *Key;  

   if (prefsCacheKeyrings <= 0) return;  /* No not cache at all. */
   if ((publicKeyring == NULL) || (secretKeyring == NULL)) return; 
   /* No keys to flush, what are we doing here ?? */
   /* Addition: If no secret keys are found, secretKeyring is NULL. 
      It is convenient to not cache then, as this (hopefully) changes soon. */
   if ((cache = open(DecryptCacheName,O_WRONLY|O_CREAT|O_APPEND,0600))<0) return;
	   
   /* First Part: Public Keyring */
   write(cache,"\x12\x34",2); /* magic 0x1234 */

   /* Now, stat size */
   entrysize=30;
   numentries = 0;
   for (Key=publicKeyring; Key != NULL; Key = Key->nextKey) 
   {
     entrysize += strlen(Key->emailAddress)+1;  /* +1 because of trailing newline */
     entrysize += strlen(Key->displayName)+1;  
     entrysize += KEY_ID_LENGTH+KEY_SIZE_LENGTH+KEY_TYPE_LENGTH;
     numentries++;
   }
   entrysize = to_bigend_u32(entrysize);
   write(cache, &entrysize,4);
   b=CACHE_VERSION; write(cache,&b,1);	/* version */
   b=1;             write(cache,&b,1);	/* Characterize as Public Keyring */
   write(cache,profilename,16); 	/* 16 characters profile name, either short name,\0,garbage
   					   or only the first few */
   date = (time(0)/3600);               /* hour resolution */
   date = to_bigend_u32(date);
   write(cache,&date,4);		/* Date field not really used */
   numentries = to_bigend_u16(numentries);
   write(cache,&numentries,2);		/* Number of records */
   
   b=0;					/* End-of-String delimiter */
   for (Key=publicKeyring; Key != NULL; Key = Key->nextKey) 
   {
      write(cache,Key->emailAddress,strlen(Key->emailAddress)); write(cache,&b,1);
      write(cache,Key->displayName, strlen(Key->displayName));  write(cache,&b,1);
      write(cache,Key->keyID,KEY_ID_LENGTH);  /* Fields could be aligned, so write part by part */
      write(cache,Key->keySize,KEY_SIZE_LENGTH);
      write(cache,Key->keyType,KEY_TYPE_LENGTH);
   }
   /* Second Part: Secret Keyring */
   write(cache,"\x12\x34",2); /* magic 0x1234 */

   /* Now, stat size */
   entrysize=30;
   numentries = 0;
   for (Key=secretKeyring; Key != NULL; Key = Key->nextKey) 
   {
     entrysize += strlen(Key->emailAddress)+1;  /* +1 because of trailing newline */
     entrysize += strlen(Key->displayName)+1;  
     entrysize += KEY_ID_LENGTH+KEY_SIZE_LENGTH+KEY_TYPE_LENGTH;
     numentries++;
   }
   entrysize = to_bigend_u32(entrysize);
   write(cache, &entrysize,4);
   b=CACHE_VERSION; write(cache,&b,1);	/* version */
   b=2;             write(cache,&b,1);	/* Characterize as Secret Keyring */
   write(cache,profilename,16); 	/* 16 characters profile name, either short name,\0,garbage
   					   or only the first few */
   date = (time(0)/3600);               /* hour resolution */
   date = to_bigend_u32(date);
   write(cache,&date,4);		/* Date field not really used */
   numentries = to_bigend_u16(numentries);
   write(cache,&numentries,2);
   
   b=0;					/* End-of-String delimiter */
   for (Key=secretKeyring; Key != NULL; Key = Key->nextKey) 
   {
      write(cache,Key->emailAddress,strlen(Key->emailAddress)); write(cache,&b,1);
      write(cache,Key->displayName, strlen(Key->displayName));  write(cache,&b,1);
      write(cache,Key->keyID,KEY_ID_LENGTH);  /* Fields could be aligned, so write part by part */
      write(cache,Key->keySize,KEY_SIZE_LENGTH);
      write(cache,Key->keyType,KEY_TYPE_LENGTH);
   }
   close(cache);
}

/* ------------------------------------------------------------ */
/* Reget Public Keys after having asked a Keyserver. 		*/
/* Actualize the Cache.						*/
/* ------------------------------------------------------------ */

void regetPublicKeyring()
{
	printf("Okay, getting public keys again ...\n");
	deallocAllKeys(publicKeyring);
	flushKeyringcache(myUserPrefs->profileName);  /* Flush and reget Pubring cache */
	publicKeyring = getKeyring_program(0);
	buildKeyringcache(myUserPrefs->profileName);  
	printf("\n");
}

/* ------------------------------------------------------------ */
/* Helper for getthekeys(). Parse the cache file.		*/
/* ------------------------------------------------------------ */

struct pkiKey *parsecacheentry(u_int16_t num_entries, u_int8_t *cacheBuf)
{
   struct pkiKey *myNewKey = NULL;
   struct pkiKey *firstKey = NULL;
   struct pkiKey *oldKey = NULL;
   char   *name,c;
   int    count;
   
   while (num_entries--) {              
     myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
     if (firstKey == NULL) firstKey = myNewKey;
     if (oldKey != NULL) oldKey->nextKey = myNewKey;
     oldKey = myNewKey;
   
     /* Email */
     name   = myNewKey->emailAddress;
     count=EMAIL_ADDRESS_MAX_LENGTH;
       do{ *name++ = c = *cacheBuf++; } while (c && count--);  /* strncpy, but increments pointers */
   
     /* displayname */
     name   = myNewKey->displayName;
     count=EMAIL_ADDRESS_MAX_LENGTH;
       do{ *name++ = c = *cacheBuf++; } while (c && count--);  /* strncpy, but increments pointers */
   
     /* keyID   */
     memcpy(myNewKey->keyID,   cacheBuf, KEY_ID_LENGTH);   cacheBuf+=KEY_ID_LENGTH;
     /* keySize */
     memcpy(myNewKey->keySize, cacheBuf, KEY_SIZE_LENGTH); cacheBuf+=KEY_SIZE_LENGTH;
     /* keyType */
     memcpy(myNewKey->keyType, cacheBuf, KEY_TYPE_LENGTH); cacheBuf+=KEY_TYPE_LENGTH;
     
     snprintf(debugLine,DEBUG_LINE_LENGTH,"Mail:%s, Name:%s, ID:%s, Size:%s, Type:%s\n",
        myNewKey->emailAddress,myNewKey->displayName,myNewKey->keyID,myNewKey->keySize,myNewKey->keyType);
     documentStatus(debugLine);
   }
   return firstKey;
}

/* -------------------------------------------------------------------- */
/* Get Both Keyrings. Try cache first, return to file if things failed.	*/
/* -------------------------------------------------------------------- */

void parseKeyringcache(char *profilename)
{
   int cache,found=0,strange=0;
   struct stat cachestats;
   off_t  size;
   u_int8_t *cacheBuf, *cacheBufstart;
   u_int32_t entrysize;
   u_int16_t numentries;
   
   if (publicKeyring && secretKeyring) return; 
   /* What are we doing here ?? */
   
   if ((cache=open(DecryptCacheName,O_RDONLY))<0) return;  /* Can't open -> done */
   if (fstat(cache,&cachestats)<0) return;                 /* stat for size */
   size = cachestats.st_size;                              /* copy length and use as counter */
   if ((cacheBuf = cacheBufstart = (u_int8_t *)mmap(0,cachestats.st_size, 
         PROT_READ,MAP_SHARED,cache,0)) == (u_int8_t *) -1) return;
   
   do {  
      entrysize = (cacheBuf[2]<<24)|(cacheBuf[3]<<16)|(cacheBuf[4]<<8)|cacheBuf[5];
      if ((cacheBuf[0]==0x12) && (cacheBuf[1]==0x34) && (cacheBuf[6]==CACHE_VERSION) &&
          ((cacheBuf[7]==1) || (cacheBuf[7]==2)) &&
           (strncmp(cacheBuf+8,profilename,16)==0))
      { 
	 found |= cacheBuf[7];
	 numentries = (cacheBuf[28]<<8) | cacheBuf[29];
	 if (cacheBuf[7]==1) {  /* Public */	   
	   snprintf(debugLine,DEBUG_LINE_LENGTH,"Found Public Keyring in cache file with %u entries:\n",numentries);
	   documentStatus(debugLine);
	   if (publicKeyring) { 
	     documentStatus("Oops.. found two public keyrings in cache. Dropping first one.\n");
	     deallocAllKeys(publicKeyring);
	     strange = 1;
	   }
	   publicKeyring = parsecacheentry(numentries,cacheBuf+30); /* Buffer start minus header is start of data */
	   documentStatus("Done.\n");
	 } else {		/* Secret */
	   snprintf(debugLine,DEBUG_LINE_LENGTH,"Found Secret Keyring in cache file with %u entries:\n",numentries);
	   documentStatus(debugLine);
	   if (secretKeyring) { 
	     documentStatus("Oops.. found two secret keyrings in cache. Dropping first one.\n");
	     deallocAllKeys(secretKeyring);
	     strange = 1;
	   }
	   secretKeyring = parsecacheentry(numentries,cacheBuf+30); /* Buffer start minus header is start of data */	   
	   documentStatus("Done.\n");
	 }
      }
      cacheBuf += entrysize;
   } while ((found != 3) && (cacheBuf<cacheBufstart+cachestats.st_size));
   munmap(cacheBufstart,cachestats.st_size);
   close(cache);   
}

/* -------------------------------------------------------------------- */
/* Get Both Keyrings. Try cache first, return to file if things failed.	*/
/* -------------------------------------------------------------------- */

void gettheKeys()
{
   int mustrebuild=0;
   
   parseKeyringcache(myUserPrefs->profileName);
   
   if (!publicKeyring) { /* Found no Public Keyring */
      printf("Loading public keyring (this may take a couple of seconds)...\n");
      publicKeyring = getKeyring_program(0);
      printf("\n");
      mustrebuild = 1;
   }
   if (!secretKeyring) { /* Found no Secret Keyring */
      printf("Loading secret keyring (this may take a couple of seconds)...\n");
      secretKeyring = getKeyring_program(1);
      printf("\n");
      mustrebuild = 1;
   }
   
   if (mustrebuild) {
     flushKeyringcache(myUserPrefs->profileName);
     buildKeyringcache(myUserPrefs->profileName);
   } 
   /* Is my Adress given in pgp4pine.conf ? */
   if ((myUserPrefs->myAddress[0]=='0')
      && ((myUserPrefs->myAddress[1]=='x')
      || ((myUserPrefs->myAddress[1]=='X'))))
     strcpy(mySecretKey,myUserPrefs->myAddress);
   else
     chooseSecretKey();
}

/* ------------------------------------------------------------------------ */
/* 'DisplayName' is the <Email> if found. If not, whatever the UID field is */
/* ------------------------------------------------------------------------ */

char *extractDisplayName(const char *stringIn)
{
	/* I return a pointer that needs to be "free"ed! */
	char tmpString[EMAIL_ADDRESS_MAX_LENGTH], *newStr;
	strcpy(tmpString,stringIn);

	newStr = (char *) myMalloc(EMAIL_ADDRESS_MAX_LENGTH);
	if (tmpString[0] == '<') {
		/* this looks like a <email> only line ... */
		strcpy(newStr,tmpString); /* or what ??? */
	} else if (strchr(tmpString,'<') && strchr(tmpString,'>')) {
		/* okay, we've got a realName <email> line... */
		strcpy(newStr,strtok(tmpString,"<"));
	} else 
		strcpy(newStr,tmpString);
	return newStr;
}

/* ------------------------------------------------------------ */
/* The same as Displayname, but without <>	 		*/
/* ------------------------------------------------------------ */

char *extractEmailAddress(const char *stringIn)
{
	/* I return a pointer that needs to be "free"ed! */
	char tmpString[EMAIL_ADDRESS_MAX_LENGTH], *newStr;
	strcpy(tmpString,stringIn);

	newStr = (char *) myMalloc(EMAIL_ADDRESS_MAX_LENGTH);
	if (tmpString[0] == '<') {
		/* this looks like a <email> only line ... */
		strcpy(newStr,strtok(tmpString+1,">"));
	} else if (strchr(tmpString,'<') && strchr(tmpString,'>')) {
		/* okay, we've got a realName <email> line... */
		strcpy(newStr,strtok(tmpString,"<"));
		strcpy(newStr,strtok('\0',">"));
	} else 
		strcpy(newStr,tmpString);
	return newStr;
}

