/***************************************************************************
                          fish.cpp  -  a FISH kioslave
                             -------------------
    begin                : Thu Oct  4 17:09:14 CEST 2001
    copyright            : (C) 2001 by Jrg Walter
    email                : trouble@garni.ch
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
  This code contains fragments and ideas from the ftp kioslave
  done by David Faure <faure@kde.org>.

  Structure is a bit complicated, since I made the mistake to use
  KProcess... now there is a lightweight homebrew async IO system
  inside, but if signals/slots become available for ioslaves, switching
  back to KProcess should be easy.
*/

#include "../config.h"

#include <qcstring.h>
#include <qsocket.h>
#include <qdatetime.h>
#include <qbitarray.h>

#include <stdlib.h>
#ifdef HAVE_PTY_H
#include <pty.h>
#endif
#ifdef HAVE_TERMIOS_H
#include <termios.h>
#endif
#include <math.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <kapp.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kinstance.h>
#include <kglobal.h>
#include <kstddirs.h>
#include <klocale.h>
#include <kurl.h>
#include <ksock.h>
#include <stdarg.h>
#include <time.h>
#include <sys/stat.h>
#include <kmimemagic.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <errno.h>
#include <sys/resource.h>

#include "fish.h"

QDateTime fishProtocol::epoch;

using namespace KIO;
extern "C" {

int kdemain( int argc, char **argv )
{
	KInstance instance("fish");

	kdDebug() << "*** Starting fish " << endl;
	if (argc != 4) {
		kdDebug() << "Usage: fish  protocol domain-socket1 domain-socket2" << endl;
		exit(-1);
	}

	fishProtocol slave(argv[2], argv[3]);
	slave.dispatchLoop();

	kdDebug() << "*** fish Done" << endl;
	return 0;
}

}

const struct fishProtocol::fish_info fishProtocol::fishInfo[] = {
	{ new QString("FISH"), 0,
	  new QString("echo; /bin/sh start_fish_server > /dev/null 2>/dev/null;"),
	  1 },
	{ new QString("VER 0.0.2 copy append lscount lslinks lsmime"), 0,
	  new QString("echo 'VER 0.0.2 copy append lscount lslinks lsmime'"),
	  1 },
	{ new QString("PWD"), 0,
	  new QString("pwd"),
	  1 },
	{ new QString("LIST"), 1,
	  new QString("echo `ls -la %1 2> /dev/null | grep '^[-dsplcb]' | wc -l`; ls -la %1 2>/dev/null | grep '^[-dspl]' | ( while read p x u g s m d y n; do file -b -m /etc/apache/magic $n 2>&1 | sed -e 's,.* text$,text/plain,;\\,^[^/]*$,d;s/^/M/;s,/.*[ \t],/,'; echo \":$n\" | sed -e 's/ -> /\\\nL/'; echo \"P$p $u.$g\nS$s\nd$m $d $y\n\"; done; );"
	            "ls -la %1 2>/dev/null | grep '^[cb]' | ( while read p x u g a i m d y n; do echo \"P$p $u.$g\nE$a$i\nd$m $d $y\n:$n\n\"; done; )"),
	  0 },
	{ new QString("RETR"), 1,
	  new QString("ls -l %1 2>&1 | ( read a b c d x e; echo $x ) 2>&1; echo '### 001'; cat %1"),
	  1 },
	{ new QString("STOR"), 2,
	  new QString("> %2; echo '### 001'; ( [ \"`expr %1 / 4096`\" -gt 0 ] && dd bs=4096 count=`expr %1 / 4096` 2>/dev/null;"
		      "[ \"`expr %1 % 4096`\" -gt 0 ] && dd bs=`expr %1 % 4096` count=1 2>/dev/null; ) | ( cat > %2 || echo Error $?; cat > /dev/null )"),
	  0 },
	{ new QString("CWD"), 1,
	  new QString("cd %1"),
	  0 },
	{ new QString("CHMOD"), 2,
	  new QString("chmod %1 %2"),
	  0 },
	{ new QString("DELE"), 1,
	  new QString("rm -f %1"),
	  0 },
	{ new QString("MKD"), 1,
	  new QString("mkdir %1"),
	  0 },
	{ new QString("RMD"), 1,
	  new QString("rmdir %1"),
	  0 },
	{ new QString("RENAME"), 2,
	  new QString("mv -f %1 %2"),
	  0 },
	{ new QString("LINK"), 2,
	  new QString("ln -f %1 %2"),
	  0 },
	{ new QString("SYMLINK"), 2,
	  new QString("ln -sf %1 %2"),
	  0 },
	{ new QString("CHOWN"), 2,
	  new QString("chown %1 %2"),
	  0 },
	{ new QString("CHGRP"), 2,
	  new QString("chgrp %1 %2"),
	  0 },
	{ new QString("READ"), 3,
	  new QString("cat %3 | ( [ \"`expr %1 / 4096`\" -gt 0 ] && dd bs=4096 count=`expr %1 / 4096` >/dev/null;"
		      "[ \"`expr %1 % 4096`\" -gt 0 ] && dd bs=`expr %1 % 4096` count=1 >/dev/null;"
		      "dd bs=%2 count=1; ) 2>/dev/null;"),
	  0 },
	// Yes, this is "ibs=1", since dd "count" is input blocks.
	// On network connections, read() may not fill the buffer
	// completely (no more data immediately available), but dd
	// does ignore that fact. Sorry, writes are slow.
	{ new QString("WRITE"), 3,
	  new QString(">> %3; echo '### 001'; ( [ %2 -gt 0 ] && dd ibs=1 obs=%2 count=%2 2>/dev/null ) | "
		      "( dd ibs=32768 obs=%1 seek=1 of=%3 2>/dev/null || echo Error $?; cat >/dev/null; )"),
	  0 },
	{ new QString("COPY"), 2,
	  new QString("cp -f %1 %2"),
	  0 },
	{ new QString("APPEND"), 2,
	  new QString(">> %2; echo '### 001'; ( [ %1 -gt 0 ] && dd ibs=1 obs=%1 count=%1 2> /dev/null; ) | ( cat >> %2 || echo Error $?; cat >/dev/null; )"),
	  0 }
};

fishProtocol::fishProtocol(const QCString &pool_socket, const QCString &app_socket)
  : SlaveBase("fish", pool_socket, app_socket), mimeBuffer(1024)
{
	kdDebug() << "fishProtocol::fishProtocol()" << endl;
	childPid = 0;
	connectionPort = 0;
	isLoggedIn = false;
	writeReady = true;
	isRunning = false;
	errorCount = 0;
	rawRead = 0;
	rawWrite = -1;
	recvLen = -1;
	sendLen = -1;
	connectionAuth.keepPassword = true;
	connectionAuth.url.setProtocol("fish");
	epoch.setTime_t(0);
	outBufPos = -1;
	outBuf = NULL;
	outBufLen = 0;
}
/* ---------------------------------------------------------------------------------- */


fishProtocol::~fishProtocol()
{
  kdDebug() << "fishProtocol::~fishProtocol()" << endl;
  closeConnection();
}

/* --------------------------------------------------------------------------- */

/**
Connects to a server and logs us in via SSH. Then starts FISH protocol.
*/
void fishProtocol::openConnection() {
	if (childPid) return;
	infoMessage("Connecting...");

	kdDebug() << "connecting to: " << connectionUser << "@" << connectionHost << ":" << connectionPort << endl;
	sendCommand(FISH_FISH);
	sendCommand(FISH_VER);
	if (connectionStart()) {
		error(ERR_SERVICE_NOT_AVAILABLE,thisFn);
		closeConnection();
		return;
	};
	kdDebug() << "subprocess is running" << endl;
}

static int open_pty_pair(int fd[2])
{
#if defined(HAVE_GETPT) && defined(HAVE_TERMIOS_H)
/** with kind regards to The GNU C Library
Reference Manual for Version 2.2.x of the GNU C Library */
	int master, slave;
	char *name;
	struct ::termios ti;
	memset(&ti,0,sizeof(ti));

	ti.c_cflag = CLOCAL|CREAD|CS8;
	ti.c_cc[VMIN] = 1;

	master = getpt();
	if (master < 0) return 0;

	if (grantpt(master) < 0 || unlockpt(master) < 0) goto close_master;

	name = ptsname(master);
	if (name == NULL) goto close_master;

	slave = open(name, O_RDWR);
	if (slave == -1) goto close_master;

#if (defined(HAVE_ISASTREAM) || defined(isastream)) && defined(I_PUSH)
	if (isastream(slave) &&
		(ioctl(slave, I_PUSH, "ptem") < 0 ||
		 ioctl(slave, I_PUSH, "ldterm") < 0))
			goto close_slave;
#endif

	tcsetattr(slave, TCSANOW, &ti);
	fd[0] = master;
	fd[1] = slave;
	return 0;

close_slave:
	close(slave);

close_master:
	close(master);
	return -1;
#else
#ifdef HAVE_OPENPTY
	struct ::termios ti;
	memset(&ti,0,sizeof(ti));

	ti.c_cflag = CLOCAL|CREAD|CS8;
	ti.c_cc[VMIN] = 1;

	return openpty(fd,fd+1,NULL,&ti,NULL);
#else
#warning "No tty support available. Password dialog won't work."
	return socketpair(PF_UNIX,SOCK_STREAM,0,fd);
#endif
#endif
}
/**
creates the subprocess
*/
bool fishProtocol::connectionStart() {
	int fd[2];
	int rc, flags;
	thisFn = QString::null;

	rc = open_pty_pair(fd);
	if (rc == -1) {
		kdDebug() << "socketpair failed, error: " << strerror(errno) << endl;
		return true;
	}

	childPid = fork();
	if (childPid == -1) {
		kdDebug() << "fork failed, error: " << strerror(errno) << endl;
		close(fd[0]);
		close(fd[1]);
		return true;
	}
	if (childPid == 0) {
		// taken from konsole, see TEPty.C for details
		// note: if we're running on socket pairs,
		// this will fail, but thats what we expect

		for (int sig = 1; sig < NSIG; sig++) signal(sig,SIG_DFL);

		struct rlimit rlp;
		getrlimit(RLIMIT_NOFILE, &rlp);
		for (int i = 0; i < (int)rlp.rlim_cur; i++)
			if (i != fd[1]) close(i);

		dup2(fd[1],0);
		dup2(fd[1],1);
		dup2(fd[1],2);
		if (fd[1] > 2) close(fd[1]);

		setsid();

#if defined(TIOCSCTTY)
		ioctl(0, TIOCSCTTY, 0);
#endif

		int pgrp = getpid();
#ifdef _AIX
		tcsetpgrp(0, pgrp);
#else
		ioctl(0, TIOCSPGRP, (char *)&pgrp);
#endif

		const char *dev = ttyname(0);
		setpgid(0,0);
		if (dev) close(open(dev, O_WRONLY, 0));
		setpgid(0,0);


		QString ex("exec ssh -l '%u' -x -T -C -e none -p %p -q %h '%c'");
		ex.replace(QRegExp("%u"),connectionUser);
		ex.replace(QRegExp("%p"),QString::number(connectionPort));
		ex.replace(QRegExp("%h"),connectionHost);
		ex.replace(QRegExp("%c"),"echo FISH:;env TZ=UTC LANG=C LC_ALL=C LOCALE=C /bin/sh");
		execl("/bin/sh", "sh", "-c", ex.latin1(), NULL);
		kdDebug() << "could not exec " << ex << endl;
		exit(-1);
	}
	close(fd[1]);
	rc = fcntl(fd[0],F_GETFL,&flags);
	rc = fcntl(fd[0],F_SETFL,flags|O_NONBLOCK);
	childFd = fd[0];

	fd_set rfds, wfds;
	FD_ZERO(&rfds);
	FD_ZERO(&wfds);
	char buf[32768];
	int offset = 0;
	while (!isLoggedIn) {
		FD_SET(childFd,&rfds);
		FD_ZERO(&wfds);
		if (outBufPos >= 0) FD_SET(childFd,&wfds);
		rc = select(childFd+1, &rfds, &wfds, NULL, NULL);
		if (rc < 0) {
			kdDebug() << "select failed, rc: " << rc << ", error: " << strerror(errno) << endl;
			return true;
		}
		if (FD_ISSET(childFd,&wfds) && outBufPos >= 0) {
			if (outBuf > 0) rc = write(childFd,outBuf+outBufPos,outBufLen-outBufPos);
			else rc = 0;
			if (rc >= 0) outBufPos += rc;
			else {
				kdDebug() << "write failed, rc: " << rc << ", error: " << strerror(errno) << endl;
				outBufPos = -1;
				//return true;
			}
			if (outBufPos >= outBufLen) {
				outBufPos = -1;
				outBuf = NULL;
				outBufLen = 0;
			}
		}
		if (FD_ISSET(childFd,&rfds)) {
			rc = read(childFd,buf+offset,32768-offset);
			if (rc > 0) {
				int noff = establishConnection(buf,rc+offset);
				if (noff > 0) memmove(buf,buf+offset+rc-noff,noff);
				offset = noff;
			} else {
				kdDebug() << "read failed, rc: " << rc << ", error: " << strerror(errno) << endl;
				return true;
			}
		}
	}
	return false;
}

/**
writes one chunk of data to stdin of child process
*/
void fishProtocol::writeChild(const char *buf, int len) {
	if (outBufPos >= 0) {
		if (len > 0) kdDebug() << "write request while old one is pending, throwing away input" << endl;
		return;
	}
	outBuf = buf;
	outBufPos = 0;
	outBufLen = len;
}

/**
manages initial communication setup including password queries
*/
int fishProtocol::establishConnection(char *buffer, int len) {
	QString buf;
	buf.setLatin1(buffer,len);
	int result;
	int pos;
	kdDebug() << buf << endl;
	while (childPid && ((pos = buf.find('\n')) >= 0 ||
			buf.right(2) == ": " || buf.right(2) == "? ")) {
		pos++;
		QString str = buf.left(pos);
		buf = buf.mid(pos);
		if (str == "\n") continue;
	  	if (str == "FISH:\n") {
			thisFn = QString::null;
  			infoMessage("Initiating protocol...");
			if (!connectionAuth.password.isEmpty()) {
				connectionAuth.password = connectionAuth.password.left(connectionAuth.password.length()-1);
	  			cacheAuthentication(connectionAuth);
			}
	  		isLoggedIn = true;
	  		return 0;
		} else if (!str.isEmpty()) {
			thisFn += str;
	  	} else if (buf.right(2) == ": ") {
			if (!connectionPassword.isEmpty()) {
				//kdDebug() << "sending cpass" << endl;
				connectionAuth.password = connectionPassword+"\n";
				connectionPassword = QString::null;
				writeChild(connectionAuth.password.latin1(),connectionAuth.password.length());
			} else {
				//kdDebug() << "sending mpass" << endl;
				connectionAuth.prompt = thisFn+buf;
				connectionAuth.caption = "SSH Authorization";
				if (!checkCachedAuthentication(connectionAuth) && !openPassDlg(connectionAuth)) {
					error(ERR_USER_CANCELED,connectionHost);
					closeConnection();
					return 0;
				}
				connectionAuth.password += "\n";
				writeChild(connectionAuth.password.latin1(),connectionAuth.password.length());
			}
			thisFn = QString::null;
			return 0;
		} else if (buf.right(2) == "? ") {
			int rc = messageBox(QuestionYesNo,thisFn+buf);
			if (rc == KMessageBox::Yes) {
				writeChild("yes\n",4);
			} else {
				writeChild("no\n",3);
			}
			thisFn = QString::null;
			return 0;
		} else {
			kdDebug() << "unmatched case in initial handling! shouldn't happen!" << endl;
	  	}
	}
	return buf.length();
}
/**
sets connection information for subsequent commands
*/
void fishProtocol::setHost(const QString & host, int port, const QString & u, const QString & pass){
	QString user(u);
	kdDebug() << "setHost " << host << endl;
	if (host.isEmpty()) {
		error(ERR_UNKNOWN_HOST,QString::null);
		return;
	}
	if (port <= 0) port = 22;
	if (user.isEmpty()) user = getenv("LOGNAME");

	if (host == connectionHost && port == connectionPort && user == connectionUser)
		return;

	if (childPid) closeConnection();

	connectionHost = host;
	connectionAuth.url.setHost(host);

	connectionUser = user;
	connectionAuth.username = user;
	connectionAuth.url.setUser(user);

	connectionPort = port;
	connectionPassword = pass;
}
/**
Closes the connection
 */
void fishProtocol::closeConnection(){
	if (childPid) {
		kill(childPid,SIGTERM);
		childPid = 0;
		close(childFd);
		childFd = -1;
	}
	outBufPos = -1;
	outBuf = NULL;
	outBufLen = 0;
	qlist.clear();
	commandList.clear();
	commandCodes.clear();
	isLoggedIn = false;
	writeReady = true;
	isRunning = false;
	infoMessage("Disconnected.");
	rawRead = 0;
	rawWrite = -1;
	recvLen = -1;
	sendLen = -1;
}
/**
builds each FISH request and sets the error counter
*/
bool fishProtocol::sendCommand(fish_command_type cmd, ...) {
	const fish_info &info = fishInfo[cmd];
	//kdDebug() << "queueing: cmd="<< cmd << "['" << *info.command << "'](" << info.params <<"), alt=['" << *info.alt << "'], lines=" << info.lines << endl;

	va_list list;
	va_start(list, cmd);
	QString realCmd = *info.command;
	QString realAlt = *info.alt;
	static QRegExp rx("[][\\\\\n $`#!()*?{}~&<>;'\"|\t]");
	for (int i = 0; i < info.params; i++) {
		QString arg(va_arg(list, const char *));
		int pos = -2;
		while ((pos = rx.find(arg,pos+2)) >= 0) {
			arg.replace(pos,0,"\\");
		}
		//kdDebug() << "arg " << i << ": " << arg << endl;
		realCmd.append(" ").append(arg);
		realAlt.replace(QRegExp("%"+QString::number(i+1)),arg);
	}
	QString s("#");
	s.append(realCmd).append("\n").append(realAlt).append(" 2>&1;echo '### 000'\n");
	commandList.append(s);
	commandCodes.append(cmd);
}

/**
checks response string for result code, converting 000 and 001 appropriately
*/
int fishProtocol::handleResponse(const QString &str){
	//kdDebug() << "handling: " << str << endl;
	if (str.startsWith("### ")) {
		bool isOk = false;
		int result = str.mid(4,3).toInt(&isOk);
		if (!isOk) result = 500;
		if (result == 0) result = (errorCount != 0?500:200);
		if (result == 1) result = (errorCount != 0?500:100);
		kdDebug() << "result: " << result << ", errorCount: " << errorCount << endl;
		return result;
	} else {
		errorCount++;
		return 0;
	}
}

int fishProtocol::makeTimeFromLs(const QString &monthStr, const QString &dayStr, const QString &timeyearStr)
{
	QDateTime dt;
	dt.setTime_t(time(0));
	dt.setTime(QTime());
	int year = dt.date().year();
	int month = dt.date().month();
	int currentMonth = month;
	int day = dayStr.toInt();

	static const char *monthNames[12] = {
  		"Jan", "Feb", "Mar", "Apr", "May", "Jun",
		"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
	};

	for (int i=0; i < 12; i++) if (monthStr.startsWith(monthNames[i])) {
		month = i+1;
		break;
	}

	int pos = timeyearStr.find(':');
	if (timeyearStr.length() == 4 && pos == -1) {
		year = timeyearStr.toInt();
	} else if (pos == -1) {
		return 0;
	} else {
		if (month > currentMonth + 1) year--;
		dt.time().setHMS(timeyearStr.left(pos).toInt(),timeyearStr.mid(pos+1).toInt(),0);
	}
	dt.date().setYMD(year,month,day);

	return epoch.secsTo(dt);
}

/**
parses response from server and acts accordingly
*/
void fishProtocol::manageConnection(const QString &l) {
	QString line(l);
	int rc = handleResponse(line);
	UDSAtom atom;
	QDateTime dt;
	int pos, pos2, pos3;
	bool isOk = false;
	if (!rc) {
		switch (fishCommand) {
		case FISH_VER:
			if (line.startsWith("VER 0.0.2")) {
				line.append(" ");
				hasCopy = line.contains(" copy ");
				hasRsync = line.contains(" rsync ");
				hasAppend = line.contains(" append ");
			} else {
				error(ERR_UNSUPPORTED_PROTOCOL,line);
				closeConnection();
			}
			break;
		case FISH_PWD:
			url.setPath(line);
			redirection(url);
			break;
		case FISH_LIST:
			if (line.length() > 0) {
				switch (line[0].cell()) {
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					pos = line.toInt(&isOk);
					if (pos > 0 && isOk) errorCount--;
					if (listReason == LIST) totalSize(line.toInt());
					break;

				case 'P':
					errorCount--;
					atom.m_uds = UDS_FILE_TYPE;
					atom.m_long = 0;
					if (line[1] == 'd') {
						atom.m_long = S_IFDIR;
					} else if (line[1] == '-') {
						atom.m_long = S_IFREG;
					} else if (line[1] == 'l') {
						atom.m_long = S_IFLNK;
					} else if (line[1] == 'c') {
						atom.m_long = S_IFCHR;
					} else if (line[1] == 'b') {
						atom.m_long = S_IFBLK;
					} else if (line[1] == 's') {
						atom.m_long = S_IFSOCK;
					} else if (line[1] == 'p') {
						atom.m_long = S_IFIFO;
					} else {
						kdDebug() << "unknown file type: " << line[1].cell() << endl;
						errorCount++;
						break;
					}
					//kdDebug() << "file type: " << atom.m_long << endl;
                                        udsEntry.append(atom);

					atom.m_uds = UDS_ACCESS;
					atom.m_long = 0;
					if (line[2] == 'r') atom.m_long |= S_IRUSR;
					if (line[3] == 'w') atom.m_long |= S_IWUSR;
					if (line[4] == 'x' || line[4] == 's') atom.m_long |= S_IXUSR;
					if (line[4] == 'S' || line[4] == 's') atom.m_long |= S_ISUID;
					if (line[5] == 'r') atom.m_long |= S_IRGRP;
					if (line[6] == 'w') atom.m_long |= S_IWGRP;
					if (line[7] == 'x' || line[7] == 's') atom.m_long |= S_IXGRP;
					if (line[7] == 'S' || line[7] == 's') atom.m_long |= S_ISGID;
					if (line[8] == 'r') atom.m_long |= S_IRUSR;
					if (line[9] == 'w') atom.m_long |= S_IWUSR;
					if (line[10] == 'x' || line[10] == 't') atom.m_long |= S_IXUSR;
					if (line[10] == 'T' || line[10] == 't') atom.m_long |= S_ISVTX;
					udsEntry.append(atom);

					atom.m_uds = UDS_USER;
					atom.m_long = 0;
					pos = line.find('.',12);
					if (pos < 0) {
						errorCount++;
						break;
					}
					atom.m_str = line.mid(12,pos-12);
					udsEntry.append(atom);

					atom.m_uds = UDS_GROUP;
					atom.m_long = 0;
					atom.m_str = line.mid(pos+1);
					udsEntry.append(atom);
					break;

				case 'd':
					atom.m_uds = UDS_MODIFICATION_TIME;
					pos = line.find(' ');
					pos2 = line.find(' ',pos+1);
					if (pos < 0 || pos2 < 0) break;
					errorCount--;
					atom.m_long = makeTimeFromLs(line.mid(1,pos-1), line.mid(pos+1,pos2-pos), line.mid(pos2+1));
					udsEntry.append(atom);
					break;

				case 'D':
					atom.m_uds = UDS_MODIFICATION_TIME;
					pos = line.find(' ');
					pos2 = line.find(' ',pos+1);
					pos3 = line.find(' ',pos2+1);
					if (pos < 0 || pos2 < 0 || pos3 < 0) break;
					dt.setDate(QDate(line.mid(1,pos-1).toInt(),line.mid(pos+1,pos2-pos-1).toInt(),line.mid(pos2+1,pos3-pos2-1).toInt()));
					pos = pos3;
					pos2 = line.find(' ',pos+1);
					pos3 = line.find(' ',pos2+1);
					if (pos < 0 || pos2 < 0 || pos3 < 0) break;
					dt.setTime(QTime(line.mid(pos+1,pos2-pos-1).toInt(),line.mid(pos2+1,pos3-pos2-1).toInt(),line.mid(pos3+1).toInt()));
					errorCount--;
					atom.m_long = epoch.secsTo(dt);
					udsEntry.append(atom);
					break;

				case 'S':
					atom.m_uds = UDS_SIZE;
					atom.m_long = line.mid(1).toInt(&isOk);
					if (listReason == SIZE && atom.m_long < recvLen) recvLen = atom.m_long;
					if (!isOk) break;
					errorCount--;
					udsEntry.append(atom);
					break;

				case 'E':
					errorCount--;
					break;

				case ':':
					atom.m_uds = UDS_NAME;
					atom.m_long = 0;
					pos = line.findRev('/');
					atom.m_str = thisFn = line.mid(pos < 0?1:pos+1);
					udsEntry.append(atom);
					errorCount--;
					break;

				case 'M':
					if (line.right(8) != "/unknown") {
						atom.m_uds = UDS_MIME_TYPE;
						atom.m_long = 0;
						atom.m_str = line.mid(1);
						udsEntry.append(atom);
					}
					errorCount--;
					break;

				case 'L':
					atom.m_uds = UDS_LINK_DEST;
					atom.m_long = 0;
					atom.m_str = line.mid(1);
					udsEntry.append(atom);
					errorCount--;
					break;
				}
			} else {
				if (listReason == STAT && (thisFn == "." || isFirst)) udsStatEntry = udsEntry;
				else if (listReason == LIST) listEntry(udsEntry, false);
				else if (listReason == CHECK) checkExist = true;
				else if (listReason == SIZE && !isFirst) {
					if (recvLen >= 0) mimeType("inode/directory");
					recvLen = -1;
				}
				isFirst = false;
				errorCount--;
				udsEntry.clear();
			}
			break;

		case FISH_RETR:
			if (line.length() == 0) {
				error(ERR_IS_DIRECTORY,url.url());
				recvLen = 0;
				break;
			}
			recvLen = line.toInt(&isOk);
			if (!isOk) {
				error(ERR_COULD_NOT_READ,url.url());
				closeConnection();
			}
			break;
		}

	} else if (rc == 100) {
		switch (fishCommand) {
		case FISH_RETR:
			if (recvLen == -1) {
				error(ERR_COULD_NOT_READ,url.url());
				closeConnection();
			} else {
				rawRead = recvLen;
				t_start = t_last = time(NULL);
				dataRead = 0;
			}
			break;
		case FISH_STOR:
			if (!checkExist) sendCommand(FISH_CHMOD,QString::number(putPerm,8).latin1(),url.path().latin1());
		case FISH_WRITE:
		case FISH_APPEND:
			rawWrite = sendLen;
			//kdDebug() << "sending " << sendLen << endl;
			writeChild(NULL,0);
		}
	} else if (rc/100 != 2) {
		switch (fishCommand) {
		case FISH_STOR:
		case FISH_WRITE:
			error(ERR_COULD_NOT_WRITE,url.url());
			closeConnection();
			break;
		case FISH_RETR:
			error(ERR_COULD_NOT_READ,url.url());
			closeConnection();
			break;
		case FISH_READ:
			mimeType("inode/directory");
			recvLen = 0;
			finished();
			break;
		case FISH_FISH:
		case FISH_VER:
			error(ERR_UNSUPPORTED_PROTOCOL,line);
			closeConnection();
			break;
		case FISH_PWD:
		case FISH_CWD:
			error(ERR_CANNOT_ENTER_DIRECTORY,url.url());
			break;
		case FISH_LIST:
			//kdDebug() << "list error. reason: " << listReason << endl;
			if (listReason == LIST || listReason == SIZE) error(ERR_CANNOT_ENTER_DIRECTORY,url.path());
			else if (listReason == STAT) {
				error(ERR_DOES_NOT_EXIST,url.path());
				udsStatEntry.clear();
			} else if (listReason == CHECK) {
				checkExist = false;
				finished();
			}
			break;
		case FISH_CHMOD:
			error(ERR_CANNOT_CHMOD,url.url());
			break;
		case FISH_CHOWN:
		case FISH_CHGRP:
			error(ERR_ACCESS_DENIED,url.url());
			break;
		case FISH_MKD:
			error(ERR_COULD_NOT_MKDIR,url.url());
			break;
		case FISH_RMD:
			error(ERR_COULD_NOT_RMDIR,url.url());
			break;
		case FISH_DELE:
		case FISH_RENAME:
		case FISH_COPY:
		case FISH_LINK:
		case FISH_SYMLINK:
			error(ERR_COULD_NOT_WRITE,url.url());
			break;
		}
	} else {
		bool wasSize = false;
		if (fishCommand == FISH_STOR) fishCommand = (hasAppend?FISH_APPEND:FISH_WRITE);
		if (fishCommand == FISH_FISH) {
			connected();
		} else if (fishCommand == FISH_LIST) {
			if (listReason == LIST) {
				listEntry(UDSEntry(),true);
			} else if (listReason == STAT) {
				statEntry(udsStatEntry);
				udsStatEntry.clear();
			} else if (listReason == CHECK) {
				if (!checkOverwrite && checkExist) error(ERR_FILE_ALREADY_EXIST,url.url());
			} else if (listReason == SIZE) {
				wasSize = true;
			}
		} else if (fishCommand == FISH_APPEND) {
			dataReq();
			if (readData(rawData) > 0) sendCommand(FISH_APPEND,QString::number(rawData.size()).latin1(),url.path().latin1());
			sendLen = rawData.size();
		} else if (fishCommand == FISH_WRITE) {
			dataReq();
			if (readData(rawData) > 0) sendCommand(FISH_WRITE,QString::number(putPos).latin1(),QString::number(rawData.size()).latin1(),url.path().latin1());
			putPos += rawData.size();
			sendLen = rawData.size();
		}
		finished();
		if (wasSize) {
			rawRead = recvLen;
			t_start = t_last = time(NULL);
			dataRead = 0;
		}
	}
}

void fishProtocol::writeStdin(const QString &line)
{
	qlist.append(line);

	if (writeReady) {
		writeReady = false;
		kdDebug() << "Writing: " << qlist.first().mid(0,qlist.first().find('\n')) << endl;
		//kdDebug() << "Writing: " << qlist.first();
		writeChild(qlist.first().latin1(), qlist.first().length());
	}
}

void fishProtocol::sent()
{
	if (rawWrite > 0) {
		//kdDebug() << "writing raw: " << rawData.size() << "/" << rawWrite << endl;
		writeChild(rawData.data(),(rawWrite > rawData.size()?rawData.size():rawWrite));
		rawWrite -= rawData.size();
		if (rawWrite > 0) {
			dataReq();
			if (readData(rawData) <= 0) {
				closeConnection();
			}
		}
		return;
	} else if (rawWrite == 0) {
		// workaround: some dd's insist in reading multiples of
		// 8 bytes, swallowing up to seven bytes. sending
		// newlines is safe even when a sane dd is used
		writeChild("\n\n\n\n\n\n\n",7);
		rawWrite = -1;
		return;
	}
	if (qlist.count() > 0) qlist.remove(qlist.begin());
	if (qlist.count() == 0) {
		writeReady = true;
	} else {
		kdDebug() << "Writing: " << qlist.first().mid(0,qlist.first().find('\n')) << endl;
		//kdDebug() << "Writing: " << qlist.first();
		writeChild(qlist.first().latin1(),qlist.first().length());
	}
}

int fishProtocol::received(const char *buffer, int buflen)
{
	QString buf;
	do {
		if (buflen <= 0) break;
		//kdDebug() << "processing raw: " << rawRead << endl;
		if (rawRead > 0) {
			//kdDebug() << "processedSize " << dataRead << ", len " << buflen << "/" << rawRead << endl;
			int dataSize = (rawRead > buflen?buflen:rawRead);
			if (dataRead < mimeBuffer.size()) {
				int mimeSize = (dataSize > mimeBuffer.size()-dataRead?mimeBuffer.size()-dataRead:dataSize);
				memcpy(mimeBuffer.data()+dataRead,buffer,mimeSize);
				if (dataSize >= mimeSize) {
					if (rawRead+dataRead < mimeBuffer.size())
						mimeBuffer.resize(rawRead+dataRead);
					mimeType(KMimeMagic::self()->findBufferFileType(mimeBuffer,url.path())->mimeType());
					if (fishCommand != FISH_READ) {
						data(mimeBuffer);
						totalSize(rawRead+dataRead);
					}
					mimeBuffer.resize(1024);
					if (dataSize > mimeSize) {
						QByteArray rest(dataSize-mimeSize);
						rest.duplicate(buffer+mimeSize,dataSize-mimeSize);
						data(rest);
					 }
				}
			} else {
				QByteArray bdata;
				bdata.duplicate(buffer,dataSize);
				data(bdata);
			}
			dataRead += dataSize;
			rawRead -= dataSize;
			time_t t = time(NULL);
			if (t-t_last >= 1) {
				processedSize(dataRead);
				speed(dataRead/(t-t_start));
				t_last = t;
			}
			if (rawRead <= 0) {
				buffer += dataSize;
				buflen -= dataSize;
			} else {
				return 0;
			}
		}

		if (buflen <= 0) break;
		buf.setLatin1(buffer,buflen);
		int pos;
		if ((pos = buf.find('\n')) >= 0) {
			buffer += pos+1;
			buflen -= pos+1;
			QString s = buf.left(pos);
			manageConnection(s);
			buf = buf.mid(pos+1);
		}
	} while (childPid && buflen && (rawRead > 0 || buf.find('\n') >= 0));
	return buflen;
}
/** get a file */
void fishProtocol::get(const KURL& u){
	setHost(u.host(),u.port(),u.user(),u.pass());
	openConnection();
	url = u;
	url.cleanPath();
	if (!url.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		recvLen = -1;
		sendCommand(FISH_RETR,url.path().latin1());
	}
	run();
}

/** put a file */
void fishProtocol::put(const KURL& u, int permissions, bool overwrite, bool resume){
	setHost(u.host(),u.port(),u.user(),u.pass());
	openConnection();
	url = u;
	url.cleanPath();
	if (!url.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		putPerm = permissions;
		checkOverwrite = overwrite;
		checkExist = false;
		putPos = 0;
		listReason = CHECK;
		sendCommand(FISH_LIST,url.path().latin1());
		sendCommand(FISH_STOR,"0",url.path().latin1());
	}
	run();
}
/** executes next command in sequence or calls finished() if all is done */
void fishProtocol::finished() {
	if (commandList.count() > 0) {
		fishCommand = (fish_command_type)commandCodes.first();
		errorCount = -fishInfo[fishCommand].lines;
		rawRead = 0;
		rawWrite = -1;
		udsEntry.clear();
		udsStatEntry.clear();
		isFirst = true;
		writeStdin(commandList.first());
		if (fishCommand != FISH_APPEND && fishCommand != FISH_WRITE) infoMessage("Sending "+(commandList.first().mid(1,commandList.first().find("\n")-1))+"...");
		commandList.remove(commandList.begin());
		commandCodes.remove(commandCodes.begin());
	} else {
		SlaveBase::finished();
		isRunning = false;
	}
}
/** aborts command sequence and calls error() */
void fishProtocol::error(int type, const QString &detail) {
	commandList.clear();
	commandCodes.clear();
	kdDebug() << "ERROR: " << type << " - " << detail << endl;
	SlaveBase::error(type,detail);
	isRunning = false;
}
/** executes a chain of commands */
void fishProtocol::run() {
	if (!isRunning) {
		int rc;
		isRunning = true;
		finished();
		fd_set rfds, wfds;
		FD_ZERO(&rfds);
		char buf[32768];
		int offset = 0;
		while (isRunning) {
			FD_SET(childFd,&rfds);
			FD_ZERO(&wfds);
			if (outBufPos >= 0) FD_SET(childFd,&wfds);
			rc = select(childFd+1, &rfds, &wfds, NULL, NULL);
			if (rc < 0) {
				kdDebug() << "select failed, rc: " << rc << ", error: " << strerror(errno) << endl;
				error(ERR_CONNECTION_BROKEN,connectionHost);
				closeConnection();
				return;
			}
			if (FD_ISSET(childFd,&wfds) && outBufPos >= 0) {
				//kdDebug() << "now writing "+(outBufLen-outBufPos) << endl;
				if (outBufLen-outBufPos > 0) rc = write(childFd,outBuf+outBufPos,outBufLen-outBufPos);
				else rc = 0;
				if (rc >= 0) outBufPos += rc;
				else {
					kdDebug() << "write failed, rc: " << rc << ", error: " << strerror(errno) << endl;
					error(ERR_CONNECTION_BROKEN,connectionHost);
					closeConnection();
					return;
				}
				if (outBufPos >= outBufLen) {
					outBufPos = -1;
					outBuf = NULL;
					sent();
				}
			}
			if (FD_ISSET(childFd,&rfds)) {
				rc = read(childFd,buf+offset,32768-offset);
				//kdDebug() << "read " << rc << " bytes" << endl;
				if (rc > 0) {
					int noff = received(buf,rc+offset);
					if (noff > 0) memmove(buf,buf+offset+rc-noff,noff);
					//kdDebug() << "left " << noff << " bytes: " << QString::fromLatin1(buf,offset) << endl;
					offset = noff;
				} else {
					kdDebug() << "read failed, rc: " << rc << ", error: " << strerror(errno) << endl;
					error(ERR_CONNECTION_BROKEN,connectionHost);
					closeConnection();
					return;
				}
			}
		}
	}
}
/** stat a file */
void fishProtocol::stat(const KURL& u){
	setHost(u.host(),u.port(),u.user(),u.pass());
	openConnection();
	if (childPid <= 0) return;
 	url = u;
	url.cleanPath();
	if (!url.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		listReason = STAT;
		sendCommand(FISH_LIST,url.path().latin1());
	}
	run();
}
/** find mimetype for a file */
void fishProtocol::mimetype(const KURL& u){
	setHost(u.host(),u.port(),u.user(),u.pass());
	openConnection();
	if (childPid <= 0) return;
 	url = u;
	url.cleanPath();
	if (!url.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		recvLen = 1024;
		listReason = SIZE;
		sendCommand(FISH_LIST,url.path().latin1());
		sendCommand(FISH_READ,"0","1024",url.path().latin1());
	}
	run();
}
/** list a directory */
void fishProtocol::listDir(const KURL& u){
	setHost(u.host(),u.port(),u.user(),u.pass());
	openConnection();
	if (childPid <= 0) return;
 	url = u;
	url.cleanPath();
	if (!url.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		listReason = LIST;
		sendCommand(FISH_LIST,url.path().latin1());
	}
	run();
}
/** create a directory */
void fishProtocol::mkdir(const KURL& u, int permissions) {
	setHost(u.host(),u.port(),u.user(),u.pass());
	openConnection();
 	url = u;
	url.cleanPath();
	if (!url.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		sendCommand(FISH_MKD,url.path().latin1());
		if (permissions != -1) sendCommand(FISH_CHMOD,QString::number(permissions,8).latin1(),url.path().latin1());
	}
	run();
}
/** rename a file */
void fishProtocol::rename(const KURL& s, const KURL& d, bool overwrite) {
	if (s.host() != d.host() || s.port() != d.port() || s.user() != d.user()) {
		error(ERR_UNSUPPORTED_ACTION,s.path().latin1());
		return;
	}
	setHost(s.host(),s.port(),s.user(),s.pass());
	openConnection();
	if (childPid <= 0) return;
	KURL src = s;
 	url = d;
	url.cleanPath();
	src.cleanPath();
	if (!url.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		if (!overwrite) {
			listReason = CHECK;
			checkOverwrite = false;
			sendCommand(FISH_LIST,url.path().latin1());
		}
		sendCommand(FISH_RENAME,src.path().latin1(),url.path().latin1());
	}
	run();
}
/** create a symlink */
void fishProtocol::symlink(const QString& target, const KURL& u, bool overwrite) {
	setHost(u.host(),u.port(),u.user(),u.pass());
	openConnection();
	if (childPid <= 0) return;
 	url = u;
	url.cleanPath();
	if (!url.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		if (!overwrite) {
			listReason = CHECK;
			checkOverwrite = false;
			sendCommand(FISH_LIST,url.path().latin1());
		}
		sendCommand(FISH_SYMLINK,target.latin1(),url.path().latin1());
	}
	run();
}
/** change file permissions */
void fishProtocol::chmod(const KURL& u, int permissions){
	setHost(u.host(),u.port(),u.user(),u.pass());
	openConnection();
	if (childPid <= 0) return;
 	url = u;
	url.cleanPath();
	if (!url.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		sendCommand(FISH_CHMOD,QString::number(permissions,8).latin1(),url.path().latin1());
	}
	run();
}
/** copies a file */
void fishProtocol::copy(const KURL &s, const KURL &d, int permissions, bool overwrite) {
	if (s.host() != d.host() || s.port() != d.port() || s.user() != d.user() || !hasCopy) {
		error(ERR_UNSUPPORTED_ACTION,s.path().latin1());
		return;
	}
	//kdDebug() << s.url() << endl << d.url() << endl;
	setHost(s.host(),s.port(),s.user(),s.pass());
	openConnection();
	if (childPid <= 0) return;
	KURL src = s;
 	url = d;
	url.cleanPath();
	src.cleanPath();
	if (!src.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		if (!overwrite) {
			listReason = CHECK;
			checkOverwrite = false;
			sendCommand(FISH_LIST,url.path().latin1());
		}
		sendCommand(FISH_COPY,src.path().latin1(),url.path().latin1());
		if (permissions != -1) sendCommand(FISH_CHMOD,QString::number(permissions,8).latin1(),url.path().latin1());
	}
	run();
}
/** removes a file or directory */
void fishProtocol::del(const KURL &u, bool isFile){
	setHost(u.host(),u.port(),u.user(),u.pass());
	openConnection();
	if (childPid <= 0) return;
	url = u;
	url.cleanPath();
	if (!url.hasPath()) {
		sendCommand(FISH_PWD);
	} else {
		sendCommand((isFile?FISH_DELE:FISH_RMD),url.path().latin1());
	}
	run();
}
/** report status */
void fishProtocol::slave_status() {
	if (childPid > 0)
		slaveStatus(connectionHost,isLoggedIn);
	else
		slaveStatus(QString::null,false);
}
