/*
 * main.cxx
 *
 * A simple H.323 MCU
 *
 * Copyright (c) 2000 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Portable Windows Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions are Copyright (C) 1993 Free Software Foundation, Inc.
 * All Rights Reserved.
 *
 * Contributor(s): Derek J Smithies (derek@indranet.co.nz)
 *                 ------------------------------
 *
 * $Log: main.cxx,v $
 * Revision 1.40  2001/08/27 03:47:39  robertj
 * Changed if statement with constant expression to be #ifdef.
 *
 * Revision 1.39  2001/08/24 13:48:01  rogerh
 * Delete the listener thread if StartListener() fails.
 *
 * Revision 1.38  2001/08/23 07:54:09  rogerh
 * Add code which allows a user to hear their own audio echo back from the MCU
 * This needs to be enabled at compile time with the HEAR_OWN_AUDIO define.
 *
 * Revision 1.37  2001/07/23 03:55:13  rogerh
 * Seperate out codec names for audio and video
 *
 * Revision 1.36  2001/07/23 03:28:03  rogerh
 * Display the codec name in the statistics page
 *
 * Revision 1.35  2001/07/17 11:49:38  rogerh
 * Change title, Mcu -> MCU
 *
 * Revision 1.34  2001/07/17 10:57:45  rogerh
 * Add LPC-10 codec support, from Santiago Garcia Mantinan <openh323@manty.net>
 *
 * Revision 1.33  2001/07/12 06:18:27  rogerh
 * Add G711 A-Law codec
 *
 * Revision 1.32  2001/06/28 06:08:17  rogerh
 * Fix compilation warning
 *
 * Revision 1.31  2001/06/13 06:01:35  rogerh
 * Add some comments
 *
 * Revision 1.30  2001/06/13 05:47:19  rogerh
 * Fix the Make Call menu option which was broken when multiple rooms were
 * added. Calls made from OpenMCU are added to the default room name.
 * Add the Microsoft GSM codec to the codecs list.
 *
 * Revision 1.29  2001/05/31 17:01:52  rogerh
 * Fixes from Dan Johansson <djn98006@student.mdh.se> to make video work.
 *  Go back to using 'closed' for the Video Classes. This is needed as
 *  the the video classes come from PVideoChannel which does not use os_handle
 *  in its IsOpen() method. Instead, we must define our own IsOpen() method.
 *  Also, back out the size of the image change.
 *  Finally, add a new feature. For the first 4 connections, video from an
 *  endpoint is displayed immediatly rather than waiting until that ep sends
 *  some audio. (handy for endpoints with video only and with no talking)
 *
 * Revision 1.28  2001/05/31 14:29:29  rogerh
 * Add --disable-menu to OpenMCU
 *
 * Revision 1.27  2001/05/17 22:07:51  dereks
 * fixed calculation of the number of bytes in the image.
 *
 * Revision 1.26  2001/05/08 13:43:11  rogerh
 * Connections without a room name will now join the default room (room101)
 * Handy for NetMeeting users who cannot specify the room to OpenMCU.
 * Add --defaultroom to change the default room name. Add --no-defaultroom to
 * prevent use of the default room and to reject calls without a room name.
 *
 * Revision 1.25  2001/05/08 11:30:39  rogerh
 * clean up display
 *
 * Revision 1.24  2001/05/08 10:47:07  rogerh
 * Handle the removal of members correctly.
 *
 * Revision 1.23  2001/05/04 08:07:49  rogerh
 * Improve stats when there are no connections and fix nested mutex bug
 *
 * Revision 1.22  2001/03/20 23:34:33  robertj
 * Added creating of room member lists on first use of room.
 * Used the new PTrace::Initialise function for starting trace code.
 *
 * Revision 1.21  2001/03/18 07:40:45  robertj
 * Fixed MSVC compatibility.
 *
 * Revision 1.20  2001/03/18 06:50:20  robertj
 * More changes for multiple conferences from Patrick Koorevaar.
 *
 * Revision 1.19  2001/03/06 04:32:50  robertj
 * Fixed another error in previous update, variable declared twice.
 *
 * Revision 1.18  2001/03/05 22:36:22  robertj
 * Changes for logging and multiple conferences from Patrick Koorevaar.
 *
 * Revision 1.17  2001/02/19 05:04:18  robertj
 * Added more fixes on shutdown, thanks Paul E. Zaremba.
 *
 * Revision 1.16  2001/02/09 06:09:42  robertj
 * Added fix for crashing on exit problem, thanks Dhomin.
 *
 * Revision 1.15  2001/02/08 07:06:37  robertj
 * Added 'm' command to make call, thanks Paul Zaremba.
 * Added ability to send CIF size images, thanks again Paul Zaremba.
 *
 * Revision 1.14  2001/01/04 01:35:43  dereks
 * Add user interface thread to openmcu.
 * tidy up the exiting process, but it is still in need of work.
 *
 * Revision 1.13  2001/01/03 03:59:26  dereks
 * Adjusted algorithm for selecting which corners contain which video stream.
 * Add gsmframes and g711frames option. Add documentation describing data flows.
 *
 * Revision 1.12  2000/12/22 08:28:23  dereks
 * Optimise video handling, and reduce load on mcu computer
 * Include noise detection routine, to determine which images are displayed when > 4 connections.
 *
 * Revision 1.11  2000/12/19 22:41:44  dereks
 * Add video conferencing - limited to 5 connected nodes.
 * Use the video channel facility now in openh323 and pwlib modules
 * Add simple interface to handle commands entered at the keyboard.
 *
 * Revision 1.10  2000/11/08 04:27:33  robertj
 * Fixed MSVC warnings.
 *
 * Revision 1.9  2000/11/03 06:38:07  craigs
 * Added conditional for windows until we decide what to do
 *
 * Revision 1.8  2000/11/02 03:33:41  craigs
 * Changed to provide some sort of software timeing loop
 *
 * Revision 1.7  2000/06/20 02:38:32  robertj
 * Changed H323TransportAddress to default to IP.
 *
 * Revision 1.6  2000/05/25 13:26:18  robertj
 * Fixed incorrect "save" parameter specification.
 *
 * Revision 1.5  2000/05/25 12:06:20  robertj
 * Added PConfigArgs class so can save program arguments to config files.
 *
 * Revision 1.4  2000/05/11 11:47:11  robertj
 * Fixed alpha linux GNU compiler problems.
 *
 * Revision 1.3  2000/05/11 09:54:02  robertj
 * Win32 compilation
 *
 * Revision 1.2  2000/05/10 08:11:57  craigs
 * Fixed copyrights and names
 *
 * Revision 1.1  2000/05/10 05:54:06  craigs
 * Initial version
 *
 */

#include <ptlib.h>
#include <ptlib/pipechan.h>

#include "version.h"
#include "mscodecs.h"
#include "lpc10codec.h"
#ifndef NO_VIDEO
#include "h261codec.h"
#include "videoio.h"
#endif
#include "main.h"

//#include "acmencr.h"


PCREATE_PROCESS(OpenMcu);

#define new PNEW

// size of a PCM data packet, in samples
#define PCM_PACKET_LEN          240

// size of a PCM data buffer, in bytes
#define PCM_BUFFER_LEN          (PCM_PACKET_LEN * 2)

// number of PCM buffers to keep
#define PCM_BUFFER_COUNT        2

#define PCM_BUFFER_SIZE         (PCM_BUFFER_LEN * PCM_BUFFER_COUNT)

// Setting HEAR_OWN_AUDIO to TRUE will echo a user's voice back to them.
// This can be handy for testing connections and audio levels.
//#define HEAR_OWN_AUDIO


#ifndef NO_VIDEO
#define VIDEO_BUFFER_SIZE ((352*288*3)>>1)
//#define VIDEO_BUFFER_SIZE (352*288*3)
#endif

///////////////////////////////////////////////////////////////
#ifdef LOGGING
static PString DEFAULT_CALL_LOG = "c:\\mcu_log.txt"; 
static PMutex logMutex;
static PTextFile logFile;
static PFilePath logFilename = DEFAULT_CALL_LOG;

static void LogMessage(const PString & str)
{
  PTime now;
  PString msg = now.AsString("dd/MM/yyyy") & str;
  logMutex.Wait();

  if (!logFile.IsOpen()) {
    logFile.Open(logFilename, PFile::ReadWrite);
    logFile.SetPosition(0, PFile::End);
  }

  logFile.WriteLine(msg);

  logFile.Close();
  
  logMutex.Signal();
}


static void LogCall(MyH323Connection& connection, 
					const BOOL accepted = TRUE)
{
  H323TransportAddress address = connection.GetControlChannel().GetRemoteAddress();
  PIPSocket::Address ip;
  WORD port;
  PStringStream stringStream, timeStream;
  address.GetIpAndPort(ip, port);
  timeStream << connection.GetConnectionStartTime().AsString("hh:mm:ss");
  stringStream << ' ' << "caller-ip:" << ip << ':' << port << ' '
	       << connection.GetRemotePartyName() 
	       << " room:" << connection.GetRoomID();

  if (accepted) {
    PStringStream connectionDuration;
    connectionDuration << setprecision(0) << setw(5) << (PTime() - connection.GetConnectionStartTime());
    LogMessage(timeStream + stringStream	+ " connection duration:" + connectionDuration);
  }
  else 
    LogMessage(timeStream + " Call denied:" + stringStream);		
}
#endif

///////////////////////////////////////////////////////////////


OpenMcu::OpenMcu()
  : PProcess("OpenH323 Project", "OpenMCU",
             MAJOR_VERSION, MINOR_VERSION, BUILD_TYPE, BUILD_NUMBER)
{
}


OpenMcu::~OpenMcu()
{
}


void OpenMcu::Main()
{
  cout << GetName()
       << " Version " << GetVersion(TRUE)
       << " by " << GetManufacturer()
       << " on " << GetOSClass() << ' ' << GetOSName()
       << " (" << GetOSVersion() << '-' << GetOSHardware() << ")\n";
  //PArgList & args = GetArguments();
  PConfigArgs args(GetArguments());

  args.Parse(
             "g-gatekeeper:"         "n-no-gatekeeper."
             "-require-gatekeeper."  "-no-require-gatekeeper."
             "h-help."
             "i-interface:"          "-no-interface."
             "-save."
             "j-jitter:"
             "-gsmframes:"           "-no-gsmframes."
             "-g711frames:"          "-no-g711frames."
#if PTRACING
             "t-trace."
             "o-output:"
#endif
             "-defaultroom:"         "-no-defaultroom."
             "u-username:"           "-no-username."
#ifndef NO_VIDEO
             "v-video."              "-no-video."
						 "-videolarge."
             "-videotxquality:"      "-no-videotxquality."
             "-videofill:"           "-no-videofill."
             "-videotxfps:"          "-no-videotxfps."
             "-disable-menu."
#endif
          , FALSE);

#if PTRACING
  PTrace::Initialise(args.GetOptionCount('t'),
                     args.HasOption('o') ? (const char *)args.GetOptionString('o') : NULL);
#endif

  if (args.HasOption('h')) {
    cout << "Usage : " << GetName() << " [options]\n"
            "Options:\n"
            "  -u --username str   : Set the local endpoint name to str\n"
            "  -g --gatekeeper host: Specify gatekeeper host.\n"
            "  -n --no-gatekeeper  : Disable gatekeeper discovery.\n"
            "  --require-gatekeeper: Exit if gatekeeper discovery fails.\n"
            "  -i --interface ip   : Bind to a specific interface\n"
            "  --g711frames count  : Set the number G.711 frames in capabilities (default 30)\n"
            "  --gsmframes count   : Set the number GSM frames in capabilities (default 4)\n"
#if PTRACING
            "  -t --trace          : Enable trace, use multiple times for more detail\n"
            "  -o --output         : File for trace output, default is stderr\n"
#endif
            "  --save              : Save arguments in configuration file\n"
#ifndef NO_VIDEO
            "  -v --video          : Enable H261 video handling\n"
            "  --videolarge        : Set the video size from normal (176x144) to large (352x288)."
            "  --videotxquality n  : Select sent video quality,(def 9). 1(good)<=n<=31\n" 
            "  --videofill n       : Select number of updated background blocks per frame 1<=n<=99 (2 def)\n"
      	    "  --videotxfps n      : Maximum number of transmitted video frames per sec 1<10(def)<30\n\n"
#endif
            "  --defaultroom name  : Connections without a room name will join this room\n"
            "                         (Default room is room101)\n"
            "  --no-defaultroom    : Reject connections with no room specified\n"
            "  --disable-menu      : Disable the command line menu\n"
            "  -h --help           : Display this help message\n";
    return;
  }

  args.Save("save");

  MyH323EndPoint endpoint;

  PString userName = "OpenH323 MCU v" + GetVersion();
  if (args.HasOption('u'))
    userName = args.GetOptionString('u');
  endpoint.SetLocalUserName(userName);

  int g711Frames = 30;
  if (args.HasOption("g711frames")) {
    g711Frames = args.GetOptionString("g711frames").AsInteger();
    if (g711Frames <= 10 || g711Frames > 240) {
      cerr << "error: G.711 frame size must be in range 10 to 240" << endl;
      g711Frames = 30;
    }
  }
  
  int gsmFrames = 4;
  if (args.HasOption("gsmframes")) {
    gsmFrames = args.GetOptionString("gsmframes").AsInteger();
    if (gsmFrames < 1 || gsmFrames > 7) {
      cerr << "error: GSM frame size must be in range 1 to 7" << endl;
      gsmFrames = 4;
    }
  }
  
  H323Capability * gsm    = new H323_GSM0610Capability;
  H323Capability * msgsm  = new MicrosoftGSMAudioCapability;
  H323Capability * ulaw64 = new H323_G711Capability(H323_G711Capability::muLaw, H323_G711Capability::At64k);
  H323Capability * alaw64 = new H323_G711Capability(H323_G711Capability::ALaw, H323_G711Capability::At64k);
   H323Capability * lpc10  = new H323_LPC10Capability(endpoint);

  endpoint.SetCapability(0, 0, gsm);
  gsm->SetTxFramesInPacket(gsmFrames);

  endpoint.SetCapability(0, 0, msgsm);
  msgsm->SetTxFramesInPacket(gsmFrames);

  endpoint.SetCapability(0, 0, ulaw64);
  ulaw64->SetTxFramesInPacket(g711Frames);

  endpoint.SetCapability(0, 0, alaw64);
  alaw64->SetTxFramesInPacket(g711Frames);

  endpoint.SetCapability(0, 0, lpc10);

  int jitter = 120;
  if (args.HasOption('j'))
    jitter = args.GetOptionString('j').AsInteger();
  endpoint.SetMaxAudioDelayJitter(jitter);

  // start the H.323 listener
  H323ListenerTCP * listener;
  if (args.GetOptionString('i').IsEmpty())
    listener  = new H323ListenerTCP(endpoint);
  else {
    PIPSocket::Address interfaceAddress(args.GetOptionString('i'));
    listener  = new H323ListenerTCP(endpoint, interfaceAddress);
  }
  if (!endpoint.StartListener(listener)) {
    cout <<  "Could not open H.323 listener port on "
         << listener->GetListenerPort() << endl;
    delete listener;
    return;
  }

#ifndef NO_VIDEO
  // If videoLarge was specified
  if (args.HasOption("videolarge")) {
    endpoint.videoLarge = TRUE;
    endpoint.SetVideoSize(352,288);
  } else {
    endpoint.videoLarge = FALSE;
  }
         
  if (args.HasOption('v')) {
    //Add capability to allow the reception of video.

    // this is the only way I know of at present to transmit
    // CIF size video - pez
    // Do not allow CIF video if size is medium
    if ( endpoint.videoLarge )
      endpoint.SetCapability(0, 1, new H323_H261Capability(0, 4, FALSE, FALSE, 6217)); // CIF
      endpoint.SetCapability(0, 1, new H323_H261Capability(2, 0, FALSE, FALSE, 6217)); // QCIF
  } 

  endpoint.EnableVideoReception(args.HasOption('v'));

  int videoTxQual = 10;
  if (args.HasOption("videotxquality")) 
      videoTxQual = args.GetOptionString("videotxquality").AsInteger();
  endpoint.videoTxQuality = PMAX(1, PMIN(31, videoTxQual));

  int videoF = 2;
  if (args.HasOption("videofill")) 
    videoF = args.GetOptionString("videofill").AsInteger();
  endpoint.videoFill = PMAX(1, PMIN(99, videoF));

  int videoFPS = 10;
  if (args.HasOption("videotxfps")) 
    videoFPS = args.GetOptionString("videotxfps").AsInteger();
  endpoint.videoFramesPS = PMAX(1,PMIN(30,videoFPS));
#endif

  if (args.HasOption('g')) {
    PString gkName = args.GetOptionString('g');
    if (endpoint.SetGatekeeper(gkName, new H323TransportUDP(endpoint)))
      cout << "Gatekeeper set: " << *endpoint.GetGatekeeper() << endl;
    else {
      cout << "Error registering with gatekeeper at \"" << gkName << '"' << endl;
      return;
    }
  }
  else if (!args.HasOption('n')) {
    cout << "Searching for gatekeeper..." << flush;
    if (endpoint.DiscoverGatekeeper(new H323TransportUDP(endpoint)))
      cout << "\nGatekeeper found: " << *endpoint.GetGatekeeper() << endl;
    else {
      cout << "\nNo gatekeeper found." << endl;
      if (args.HasOption("require-gatekeeper"))
        return;
    }
  }  

  PString roomName = "room101";
  if (args.HasOption("defaultroom")) {
    roomName = args.GetOptionString("defaultroom");
  }
  if (args.HasOption("no-defaultroom")) {
    roomName = "";
  }
  endpoint.SetDefaultRoomName(roomName);


  if (args.HasOption("disable-menu")) {
    endpoint.hasMenu = FALSE;
  } else {
    endpoint.hasMenu = TRUE;
  }

  cout <<  "Codecs (in preference order):\n" << setprecision(2) << endpoint.GetCapabilities() << endl << endl;

  endpoint.ListenForIncomingCalls();
  endpoint.AwaitTermination();
  cout << "GoodBye. " << endl;

  PThread::Current()->Sleep(2000);
  cout << "Finished 2 sec sleep" <<endl;
  
#if 0 //PTRACING
  if (output != NULL) 
    delete output;
#endif
}


///////////////////////////////////////////////////////////////

MyH323EndPoint::MyH323EndPoint()
{
	terminalType = e_MCUWithAudioMP;
#ifndef NO_VIDEO
	terminalType = e_MCUWithAVMP;
  PINDEX i;
  for(i=0;i<4;i++)
    videoPosn[i]="";
#endif
}

H323Connection * MyH323EndPoint::CreateConnection(unsigned callReference)
{
  return new MyH323Connection(*this, callReference);
}

void MyH323EndPoint::ListenForIncomingCalls()
{
  cout << "Waiting for incoming calls for \"" << GetLocalUserName() << '"' << endl;
}


void MyH323EndPoint::AwaitTermination()
{
  PThread * userInterfaceThread = NULL;
  if (hasMenu)
    userInterfaceThread = new UserInterfaceThread(*this);

  // poll the handset every 100ms looking for state changes
  while (!exitFlag.Wait(100)) {

    // lock the user interface state whilst we change it
  }

  if (userInterfaceThread != NULL) {
    userInterfaceThread->Terminate();
    userInterfaceThread->WaitForTermination();
    delete userInterfaceThread;

    cout << "User Interface thread has terminated successfully."<<endl;  
  }

}

void
MyH323EndPoint::MakeOutgoingCall(const PString& dest,
				 const PString& gateway)
{
  PString callToken;
  PString fullAddress = ( gateway != "" ? gateway : dest );

  if ( fullAddress.Find(':') == P_MAX_INDEX )
    fullAddress += psprintf(":%i", H323ListenerTCP::DefaultSignalPort);

  if ( gateway != "" )
    fullAddress = dest.Trim() + "@" + fullAddress;

  cerr << "Making call to \"" << fullAddress << '"' << endl;

  if ( !MakeCall(fullAddress, callToken) ) {
    cerr << "Error making call to \"" << fullAddress << '"' << endl;
  }
}



void MyH323EndPoint::HandleUserInterface()
{
  PConsoleChannel console(PConsoleChannel::StandardInput);
  PString user;
  PString str; 
  PINDEX i, idx;
 
  PTRACE(2, "OpenMCU\tUser interface thread started.");

  PStringStream help;
  help << "Select:\n"
          " ? : Print this help message\n"
#ifndef NO_VIDEO
          " v : report  connections are at which video corner\n"
#endif
					" m : make a call\n"
          " x : Exit immediately\n"
	        " s : Report statistics\n"
          " z : Put message in log file (for debugging)\n"
    ;
  
  for (;;) {
    // display the prompt
    cout << "Command ? " << flush;

    
    //terminate the menu loop if console finished
     char ch = (char)console.peek();
     if (console.eof()) {
        cerr << "\nConsole gone - menu disabled" << endl;
        cerr << "Finishing program"<<endl;        
        ch = 'x';
      } else
        console >> ch;

    switch (tolower(ch)) {
      case '?' :
        cout << help << endl;
        break;
      case 'm' :
      	console >> user;
 	MakeOutgoingCall(user.Trim(),"");
        break;
      case 's' :
        {
        memberMutex.Wait();

        PINDEX rooms = memberListDict.GetSize();

        if (rooms == 0) {
          cout << "No statistics available" << endl;
        } else {

          for (idx = 0; idx < rooms; idx++) {
            PStringList & memberList = memberListDict.GetDataAt(idx);
          
            cout << "Statistics for " << memberList.GetSize() << " connected parties in room "
                 << (idx+1) << " of "<< rooms << " with room id " 
                 << memberListDict.GetKeyAt(idx) << endl;

            PINDEX i;
            for (i = 0; i < memberList.GetSize(); i++) {
              PString token = memberList[i];
              MyH323Connection * conn = (MyH323Connection *)FindConnectionWithLock(token);
              if (conn != NULL) {
                PTime now;
                PTime callStart = conn->GetConnectionStartTime();
                cout << "Connected to : " << conn->GetRemotePartyName() << " for " 
                     << setw(5) << setprecision(0) << (now - callStart) << " mins " <<endl;

                RTP_Session * session = conn->GetSession(RTP_Session::DefaultAudioSessionID);
                if (session == NULL)
                  cout << "No RTP session statistics available.\n";
                else
                  cout << "             : "
                       << session->GetPacketsSent() << '/'
                       << session->GetOctetsSent() 
                       << " audio packets/bytes sent ("
		       << conn->GetAudioTransmitCodecName()
		       << ")" << endl;
                 cout  << "             : "
                       << session->GetPacketsReceived() << '/'
                       << session->GetOctetsReceived()
                       << " audio packets/bytes received ("
		       << conn->GetAudioReceiveCodecName()
		       << ")" << endl;
#ifndef NO_VIDEO
		  cout << "             : "
		       << "Sending video with "
		       << conn->GetVideoTransmitCodecName()
		       << endl; 
		  cout << "             : "
		       << "Receiving video with "
		       << conn->GetVideoReceiveCodecName()
		       << endl;
#endif
                conn->Unlock();
              }
            }
          }
        }
        memberMutex.Signal();
        }
        break;
#ifndef NO_VIDEO
    case 'v' : 
      for(i=0;i<4;i++) {
        str = videoPosn[i];
        str.Delete(0,3);
        cout << "   video for position "<<i<<" is \""<<
                  str.Left(str.Find(':'))  <<"\""<<endl;
      }      
      break;
#endif    
    case 'z' : 
      console >> str;
      PTRACE(0,"Log message=" << str);
      break;

    case 'q' :
    case 'x' :
      cout << "Exiting." << endl;
      memberMutex.Wait(); //Stops flow of data into/out of audio buffers.
			
			ClearAllCalls(H323Connection::EndedByLocalUser, FALSE);
      /*for (i = 0; i < memberList.GetSize(); i++) {
        PString token = memberList[i];
        MyH323Connection * conn = (MyH323Connection *)FindConnectionWithLock(token);
        if (conn != NULL)           
          conn->ClearCall();
      }*/
      
      cout << "All calls cleared"<<endl;      
      exitFlag.Signal();
      console.ignore(INT_MAX, '\n');
      memberMutex.Signal();
      
      return;

    default :
      cout << "The input character \""<<ch<<"\" is not defined"<<endl;
		}
  }
}

void MyH323EndPoint::SetDefaultRoomName(PString roomName)
{
  defaultRoomName = roomName;
}

PString MyH323EndPoint::GetDefaultRoomName(void)
{
  return defaultRoomName;
}

void MyH323EndPoint::AddMember(MyH323Connection * newMember)
{
  PWaitAndSignal mutex(memberMutex);
  PString newToken = newMember->GetCallToken();
  PString newRoomID = newMember->GetRoomID();

  if (!memberListDict.Contains(newRoomID))
    memberListDict.SetAt(newRoomID, new PStringList);

  PStringList & memberList = memberListDict[newRoomID];
  memberList.AppendString(newToken);

#ifndef NO_VIDEO
  if (!spokenListDict.Contains(newRoomID))
    spokenListDict.SetAt(newRoomID, new PStringList);
  PStringList & spokenList = spokenListDict[newRoomID];
  spokenList.AppendString(newToken);

  // Normally we display the video for the "active" users, ie
  // people who are currently speaking (based on noise detection)
  //
  // If there are currently less than 4 members then we can display
  // the video right from the start, before they begin talking.
  // This is also handy for connections with video, but no audio.
  //
  // Problem is finding out whether or not an ep sends video.
  // That is for a coming version.
 
  PINDEX freeVideo = spokenList.GetSize();
  if(freeVideo<4){
    videoPosn[freeVideo-1]=newToken;
  }

#endif

  // add the new member to every other member,
  // and add every other member to the new member.
  PINDEX i;
  for (i = 0; i < memberList.GetSize(); i++) {
    PString token = memberList[i];
    if (token != newToken) {      
      cout << "Adding member " << newToken << " to list of " << token << endl;
      MyH323Connection * conn = (MyH323Connection *)FindConnectionWithLock(token);
      if (conn != NULL) {
        conn->AddMember(newToken);
        newMember->AddMember(token);
        conn->Unlock();
      }
    } else {
#ifdef HEAR_OWN_AUDIO
        cout << "Adding " << newToken << "to it's own list" << endl;
        newMember->AddMember(newToken);
#else
        cout << "Not adding member " << newToken << " to it's own list" << endl;
#endif
    }
  }

}

void MyH323EndPoint::RemoveMember(MyH323Connection * oldConn)
{
  PWaitAndSignal mutex(memberMutex);
  PString oldToken = oldConn->GetCallToken();
  PString oldRoomID = oldConn->GetRoomID();
 
  // get the list of members of this room 
  PStringList & memberList = memberListDict[oldRoomID];

  PINDEX i;

#ifndef NO_VIDEO
  PStringList & spokenList = spokenListDict[oldRoomID];

  spokenList.RemoveAt(spokenList.GetStringsIndex(oldToken));

  if (spokenList.GetSize() == 0) 
    spokenListDict.RemoveAt(oldRoomID);

  //Clear the corner of the video window that the connection just
  //vacated.
  i = FindTokensVideoPosn(oldToken);
  if (i<4) {    
    videoBuffer.Clear(i);
    videoPosn[i] = "";  
  }
#endif

  // remove this member from the audio buffer lists
  for (i = 0; i < memberList.GetSize(); i++) {
    PString token = memberList[i];
    if (token != oldToken) {
      MyH323Connection * conn = (MyH323Connection *)FindConnectionWithLock(token);
      if (conn != NULL) {
	conn->RemoveMember(oldToken);
	oldConn->RemoveMember(token);
        conn->Unlock();
      }
    }
  }

  // remove this connection from the member list
  memberList.RemoveAt(memberList.GetStringsIndex(oldToken));

  // if there are no more members in this room, delete the room
  if (memberList.GetSize() == 0) {
    cout << "Room " << oldRoomID << " is now empty" << endl;
    memberListDict.RemoveAt(oldRoomID); // Note: memberList is now invalid
  }
}

#ifndef NO_VIDEO
PINDEX MyH323EndPoint::FindTokensVideoPosn(const PString & thisToken)
{
  PINDEX keyIndex;
  for (keyIndex = 0; keyIndex<4; keyIndex++)
    if (thisToken == videoPosn[keyIndex] )
      return keyIndex;
  
  return P_MAX_INDEX;
}

BOOL MyH323EndPoint::AddVideoPosnToken(const PString & thisToken)
{
  PINDEX keyIndex;
  for (keyIndex = 0; keyIndex<4; keyIndex++)
    if (videoPosn[keyIndex] == "" ) {
      videoPosn[keyIndex] = thisToken;
      return TRUE;
    }
  
  return FALSE;
}

void MyH323EndPoint::SetVideoSize( int x, int y )
{
  videoBuffer.SetSize( x, y );
}
#endif

BOOL MyH323EndPoint::WriteAudio(const PString & thisToken, const void * buffer, PINDEX amount, PString roomID)
{
  PWaitAndSignal mutex(memberMutex);

  //The spokenList construct is required to determine who were the last
  //to speak, which is needed for determining which four videoimages are displayed.
  //spokenList contains a sorted list of when the last packet was received from
  //which connection. Thus, when an audio packet is received from a connection,
  //the name of the connection is moved to the end of the list.

  // If someone new comes along, everyone moves down.
#ifndef NO_VIDEO
  if ( DetectNoise(buffer,amount) ) {
    PStringList & spokenList = spokenListDict[roomID];
    PINDEX keyIndex = spokenList.GetStringsIndex(thisToken);
    spokenList.RemoveAt(keyIndex);
    spokenList.AppendString(thisToken);  
    if (FindTokensVideoPosn(thisToken) < 4)
      goto processAudio;

    PINDEX numberConns = spokenList.GetSize();
    if (numberConns < 5) {      
      AddVideoPosnToken(thisToken);
      goto processAudio;
    }
    
    PString tokenToWipe = spokenList[spokenList.GetSize()-5];    
    keyIndex = FindTokensVideoPosn(tokenToWipe);
    if(keyIndex<4)
      videoPosn[keyIndex] = thisToken;
  }

    processAudio:
#endif

  PStringList & memberList = memberListDict[roomID];

  PINDEX i;
  for (i = 0; i < memberList.GetSize(); i++) {
    PString token = memberList[i];
    if (token != thisToken) {
      MyH323Connection * conn = (MyH323Connection *)FindConnectionWithLock(token);
      if (conn != NULL) { 
	conn->WriteAudio(thisToken, buffer, amount);
        conn->Unlock();
      }
    } else {
#ifdef HEAR_OWN_AUDIO
	// Do not lock as we are looking for our own connection.
        MyH323Connection * conn = (MyH323Connection *)FindConnectionWithoutLocks(thisToken);
        if (conn != NULL) { 
          conn->WriteAudio(thisToken, buffer, amount);
	}
#endif
    }
  }

  return TRUE;
}

BOOL MyH323EndPoint::ReadAudio(const PString & thisToken, void * buffer, PINDEX amount, PString roomID)
{
  PWaitAndSignal mutex(memberMutex);

  PStringList & memberList = memberListDict[roomID];

  PINDEX i;
  for (i = 0; i < memberList.GetSize(); i++) {
    PString token = memberList[i];
    if (token == thisToken) {
      MyH323Connection * conn = (MyH323Connection *)FindConnectionWithLock(token);
      if (conn != NULL) {
				conn->ReadAudio(thisToken, buffer, amount);
        conn->Unlock();
      }
    }
  }

  return TRUE;
}

#ifndef NO_VIDEO
BOOL MyH323EndPoint::DetectNoise(const void * buffer, PINDEX amount)
{
  short *start = (short *)buffer;
  short *end   = start + (amount/2);
  int sum;
  
  sum=0;
  while (start != end) 
    if(*start<0)
      sum -= *start++;
    else
      sum += *start++;

  return (sum/amount) > 50;
}



BOOL MyH323EndPoint::WriteVideo(const PString & thisToken, const void * buffer, PINDEX amount, PString roomID)
{
  PWaitAndSignal mutex(memberMutex);

  // The last four elements of spokenList indicate the last
  // four connections from which audio was received.
  PStringList & spokenList = spokenListDict[roomID];
  PINDEX keyIndex = spokenList.GetStringsIndex(thisToken);
  if ( (keyIndex+4) >= spokenList.GetSize() ) {    
    keyIndex = FindTokensVideoPosn(thisToken);
    if ( keyIndex < 4 )
      videoBuffer.Write((BYTE *)buffer, amount, keyIndex);
  }  

  return TRUE;
}

BOOL MyH323EndPoint::ReadVideo(const PString & /*thisToken*/, void * buffer, PINDEX amount)
{
  PWaitAndSignal mutex(memberMutex);

  videoBuffer.Read((BYTE *)buffer,amount);
  
  return TRUE;
}
#endif

///////////////////////////////////////////////////////////////

MyH323Connection::MyH323Connection(MyH323EndPoint & _ep, unsigned callReference)
  : H323Connection(_ep, callReference), ep(_ep), connected(FALSE)
{
  incomingAudio = NULL;
  outgoingAudio = NULL;
  audioReceiveCodecName = audioTransmitCodecName = "none";

#ifndef NO_VIDEO
  incomingVideo = NULL;
  outgoingVideo = NULL;
  videoReceiveCodecName = videoTransmitCodecName = "none";
#endif

	aborted = FALSE;

  cout << "Opening connection" << endl;
}

MyH323Connection::~MyH323Connection()
{
  cout << "Closing connection" << endl; 

#ifdef LOGGING
	LogCall(*this);
#endif

    delete incomingAudio;
    delete outgoingAudio;

#ifndef NO_VIDEO
    delete incomingVideo;
    delete outgoingVideo;
#endif
}

void MyH323Connection::CleanUpOnCallEnd()
{
  if (incomingAudio)
    incomingAudio->Close();
  if (outgoingAudio)
    outgoingAudio->Close();

#ifndef NO_VIDEO
  if (incomingVideo)
    incomingVideo->Close();
  if (outgoingVideo)
    outgoingVideo->Close();
#endif

  if (!aborted && connected)
    ep.RemoveMember(this);

  H323Connection::CleanUpOnCallEnd();
}

H323Connection::AnswerCallResponse
     MyH323Connection::OnAnswerCall(const PString & caller,
                                    const H323SignalPDU & setupPDU,
                                    H323SignalPDU & /*connectPDU*/)
{
  if (!setupPDU.GetQ931().GetCalledPartyNumber(roomID)) {
    const H225_Setup_UUIE & setup = setupPDU.m_h323_uu_pdu.m_h323_message_body;
    PINDEX i;
    for (i = 0; i < setup.m_destinationAddress.GetSize(); i++) {
      roomID = H323GetAliasAddressString(setup.m_destinationAddress[i]);
      if (!roomID)
        break;
    }

    // If a room was not specified in the connection then use the default
    // room name.
    PString defaultRoomName = ep.GetDefaultRoomName();

    if ((roomID.IsEmpty()) && (defaultRoomName.IsEmpty()) ) {
      cout << "Incoming H.323 call from " << caller << " has not selected a room." << endl;
      cout << "No default room specified." << endl;
#ifdef LOGGING
      LogCall(*this, FALSE);
#endif
      aborted = TRUE;
      return AnswerCallDenied;
    }

    if ((roomID.IsEmpty()) && (!defaultRoomName.IsEmpty()) ) {
      cout << "Incoming H.323 call from " << caller << " has not selected a room." << endl;
      cout << "Using room " << defaultRoomName << " as the default." << endl;
      roomID = defaultRoomName;
    }
  }
  
  PString product = "Unknown";

  const H225_Setup_UUIE & setup = setupPDU.m_h323_uu_pdu.m_h323_message_body;
  const H225_EndpointType & epInfo = setup.m_sourceInfo;

  if (epInfo.HasOptionalField(H225_EndpointType::e_vendor)) {
    const H225_VendorIdentifier & vendorInfo = epInfo.m_vendor;
    if (vendorInfo.HasOptionalField(H225_VendorIdentifier::e_productId))
      product = vendorInfo.m_productId.AsString();
    if (vendorInfo.HasOptionalField(H225_VendorIdentifier::e_versionId))
      product = product + "/" + vendorInfo.m_versionId.AsString();
  }
  
  cout << "Accepting call from " << caller << " using " << product << " with room id " << roomID << endl;

  connected = TRUE;
  ep.AddMember(this);

  return AnswerCallNow;
}

BOOL
MyH323Connection::OnSendSignalSetup( H323SignalPDU & callProceedingPDU )
{
  // We are making a connection to a remote EP so add this connection to
  // the list of rooms and members.
  // We will add them to the default room as we have no method of
  // specifying which room our connection should join.
  connected = TRUE;
  roomID = ep.GetDefaultRoomName();
  cout << "Adding connection to room " << roomID << endl;
  ep.AddMember(this);
  return H323Connection::OnSendSignalSetup( callProceedingPDU );
}

BOOL MyH323Connection::OpenAudioChannel(BOOL isEncoding, unsigned /* bufferSize */, H323AudioCodec & codec)
{
  PStringStream codecName;
  codecName << codec;

  codec.SetSilenceDetectionMode( H323AudioCodec::NoSilenceDetection );

//  if (!codec.IsDescendant(H323_GSM0610Codec::Class()) &&
//      - need to add MS-GSM here along with any other codecs
//      !codec.IsDescendant(H323_muLawCodec::Class())) {
//    cerr << "Unknown codec \"" << codecName << endl;
//    return FALSE;
//  }

  PWaitAndSignal mutex(audioMutex);
  if (incomingAudio == NULL) {
    incomingAudio = new IncomingAudio(ep, *this);
  }

  if (outgoingAudio == NULL) {
    outgoingAudio = new OutgoingAudio(ep, *this);
  }

  if (isEncoding) {
    audioTransmitCodecName = codecName;
    codec.AttachChannel(outgoingAudio, FALSE);
  } else {
    audioReceiveCodecName = codecName;
    codec.AttachChannel(incomingAudio, FALSE);
  }

  return TRUE;
}

#ifndef NO_VIDEO
BOOL MyH323Connection::OpenVideoChannel(BOOL isEncoding,
                                        H323VideoCodec & codec)
{
    PStringStream codecName;
    codecName << codec;

    PWaitAndSignal mutex(videoMutex);

     if (isEncoding) {      
      if (outgoingVideo == NULL) {
        outgoingVideo = new OutgoingVideo(ep, *this, ep.videoFramesPS, ep.videoLarge);
        codec.AttachChannel(outgoingVideo,FALSE);
//       outgoingVideo->SetFrameSize(352>>1,288>>1);
      }

      /*At last. Modularity. The video codec is told the parameters of video compresion/decompression.
        The only thing the video codec knows about the ouside world is how to acquire/render data, which
	      is via the video channel, provided by the OutgoingVideo class.
	      
	      The codec does provide a second interface, through which ethernet packets enter (or leave) */
      codec.SetTxQualityLevel(ep.videoTxQuality);
      codec.SetBackgroundFill(ep.videoFill);
     
      videoTransmitCodecName = codecName;
    } else {
      if (incomingVideo == NULL) 
        incomingVideo = new IncomingVideo(ep, *this);
        codec.AttachChannel(incomingVideo,FALSE);
        videoReceiveCodecName = codecName;
    }
  
    return TRUE;
}
#endif

BOOL MyH323Connection::OnStartLogicalChannel(H323Channel & channel)
{
  if (!H323Connection::OnStartLogicalChannel(channel))
    return FALSE;

  cout << "Started logical channel: ";

  switch (channel.GetDirection()) {
    case H323Channel::IsTransmitter :
      cout << "sending ";
      break;

    case H323Channel::IsReceiver :
      cout << "receiving ";
      break;

    default :
      break;
  }

  cout << channel.GetCapability() << endl;

  return TRUE;
}

BOOL MyH323Connection::OnOutgoingAudio(void * buffer, PINDEX amount)
{
  return ep.ReadAudio(GetCallToken(), buffer, amount, roomID);  
}

BOOL MyH323Connection::OnIncomingAudio(const void * buffer, PINDEX amount)
{
  return ep.WriteAudio(GetCallToken(), buffer, amount, roomID);
}

#ifndef NO_VIDEO
BOOL MyH323Connection::OnOutgoingVideo(void * buffer, PINDEX & amount)
{
  return ep.ReadVideo(GetCallToken(), buffer, amount);
} 

BOOL MyH323Connection::OnIncomingVideo(const void * buffer, PINDEX amount)
{
  return ep.WriteVideo(GetCallToken(), buffer, amount, roomID);
}
#endif

void MyH323Connection::AddMember(const PString & token)
{
   audioMutex.Wait();

   cout << "Adding audio buffer for " << token 
        << " to connection " << GetCallToken() << endl;

   // Create a new audio buffer to hold audio between the connection
   // called 'token' and this connection.
   audioBuffers.SetAt(token, new AudioBuffer);
   
   audioMutex.Signal();
}


void MyH323Connection::RemoveMember(const PString & token)
{
  PWaitAndSignal mutex(audioMutex);

  cout << "Removing audio buffer for " << token 
       << " from connection " << GetCallToken() << endl;

  audioBuffers.RemoveAt(token);  
}


BOOL MyH323Connection::WriteAudio(const PString & token, const void * buffer, PINDEX amount)
{
  PWaitAndSignal mutex(audioMutex);
  AudioBuffer * audioBuffer = audioBuffers.GetAt(token);
  if (audioBuffer != NULL)
    audioBuffer->Write((BYTE *)buffer, amount);
  return TRUE;
}

BOOL MyH323Connection::ReadAudio(const PString & /*token*/, void * buffer, PINDEX amount)
{  
  PWaitAndSignal mutex(audioMutex);
  
  // First, set the buffer to empty.
  memset(buffer, 0, amount);

  // get number of channels to mix
  PINDEX numChannels = audioBuffers.GetSize();

  if (numChannels== 0) 
    return TRUE;

  // scan through the audio buffers and mix the signals
  PINDEX i;
  for (i = 0; i < numChannels; i++) {
    PString key = audioBuffers.GetKeyAt(i);
    audioBuffers[key].Read((BYTE *)buffer, amount, numChannels);
  }

  return TRUE;
}

///////////////////////////////////////////////////////////////

OutgoingAudio::OutgoingAudio(MyH323EndPoint & _ep, MyH323Connection & _conn)
  : ep(_ep), conn(_conn)
{
  os_handle = 0;
}

void OutgoingAudio::CreateSilence(void * buffer, PINDEX amount)
{
  memset(buffer, 0, amount);
  lastReadCount = amount;
}

BOOL OutgoingAudio::Read(void * buffer, PINDEX amount)
{
  PWaitAndSignal mutexR(audioChanMutex);
  
  if (!IsOpen())
    return FALSE;

  if (!delay.Delay(amount / 16)) {

    // do the read call here, by calling conn.OnOutgoingAudio():
    BOOL doSilence = !conn.OnOutgoingAudio(buffer, amount);

    if (doSilence)
      CreateSilence(buffer, amount);
  }

  return TRUE;
}

BOOL OutgoingAudio::Close()
{
	if (!IsOpen()) 
		return FALSE;

	PWaitAndSignal mutexC(audioChanMutex);
	os_handle = -1;
  return TRUE;
}

///////////////////////////////////////////////////////////////////////////

IncomingAudio::IncomingAudio(MyH323EndPoint & _ep, MyH323Connection & _conn)
  : ep(_ep), conn(_conn)
{
  os_handle = 0;
}

BOOL IncomingAudio::Write(const void * buffer, PINDEX amount)
{
  PWaitAndSignal mutexW(audioChanMutex);
  
  if (!IsOpen())
    return FALSE;

  if (!delay.Delay(amount / 16))
    conn.OnIncomingAudio(buffer, amount);

  return TRUE;
}

BOOL IncomingAudio::Close()
{
	if (!IsOpen()) 
		return FALSE;

	PWaitAndSignal mutexA(audioChanMutex);
	os_handle = -1;
  return TRUE;
}

///////////////////////////////////////////////////////////////////////////

AudioBuffer::AudioBuffer()
  : bufferSize(PCM_BUFFER_SIZE)
{
  buffer = new BYTE[bufferSize];
  bufferStart = bufferLen = 0;
}

AudioBuffer::~AudioBuffer()
{
  delete[] buffer;
}

void AudioBuffer::Read(BYTE * data, PINDEX amount, PINDEX channels)
{
  if (amount == 0)
    return;

  PWaitAndSignal mutex(audioBufferMutex);
  
  if (bufferLen == 0) {
    memset(data, 0, amount);
    return;
  }

  // fill output data block  with silence if audiobuffer is
  //  almost empty.
  if (amount > bufferLen) 
    memset(data + bufferLen, 0, amount - bufferLen);

  // only copy up to the amount of data remaining
  PINDEX copyLeft = PMIN(amount, bufferLen);

  // if buffer is wrapping, get first part
  if ((bufferStart + copyLeft) > bufferSize) {
    PINDEX toCopy = bufferSize - bufferStart;

    Mix(data, buffer + bufferStart, toCopy, channels);

    data        += toCopy;
    bufferLen   -= toCopy;
    copyLeft    -= toCopy;
    bufferStart = 0;
  }

  // get the remainder of the buffer
  if (copyLeft > 0) {

    Mix(data, buffer + bufferStart, copyLeft, channels);

    bufferLen -= copyLeft;
    bufferStart = (bufferStart + copyLeft) % bufferSize;
  }
}

void AudioBuffer::Mix(BYTE * dst, const BYTE * src, PINDEX count, PINDEX /*channels*/)
{
#if 0
  memcpy(dst, src, count);
#else
  PINDEX i;
  for (i = 0; i < count; i += 2) {

    int srcVal = *(short *)src;
    int dstVal = *(short *)dst;

    int newVal = dstVal;

#if 1     //The loudest person gains the channel.
#define mix_abs(x) ((x) >= 0 ? (x) : -(x))
    if (mix_abs(newVal) > mix_abs(srcVal))
      dstVal = newVal;
    else
      dstVal = srcVal; 
#else   //Just add up all the channels.
    if ((newVal + srcVal) > 0x7fff)
      dstVal = 0x7fff;
    else
      dstVal += srcVal;
#endif
    *(short *)dst = (short)dstVal;

    dst += 2;
    src += 2;
  }
#endif
}

void AudioBuffer::Write(const BYTE * data, PINDEX amount)
{
  if (amount == 0)
    return;

  PWaitAndSignal mutex(audioBufferMutex);
  
  // if there is not enough room for the new data, make room
  PINDEX newLen = bufferLen + amount;
  if (newLen > bufferSize) {
    PINDEX toRemove = newLen - bufferSize;
    bufferStart = (bufferStart + toRemove) % bufferSize;
    bufferLen -= toRemove;
  }

  // copy data to the end of the new data, up to the end of the buffer
  PINDEX copyStart = (bufferStart + bufferLen) % bufferSize;
  if ((copyStart + amount) > bufferSize) {
    PINDEX toCopy = bufferSize - copyStart;
    memcpy(buffer + copyStart, data, toCopy);
    copyStart = 0;
    data      += toCopy;
    amount    -= toCopy;
    bufferLen += toCopy;
  }

  // copy the rest of the data
  if (amount > 0) {
    memcpy(buffer + copyStart, data, amount);
    bufferLen   += amount;
  }
}

///////////////////////////////////////////////////////////////////////////
#ifndef NO_VIDEO
IncomingVideo::IncomingVideo(MyH323EndPoint & _ep, MyH323Connection & _conn)
  : ep(_ep), conn(_conn), width(0), height(0), frameSize(0)
{
  closed = FALSE;
}

IncomingVideo::~IncomingVideo()
{
   IncomingVideo::Close();
   PVideoChannel::Close();
}
   
BOOL IncomingVideo::Write(const void * buffer, PINDEX amount)
{
  amount = (frameSize*3) >> 1;    // frameSize==width*height
  
  PWaitAndSignal mutex( videoChanMutex );

  if (closed){
    return FALSE;
  }

  conn.OnIncomingVideo(buffer, amount);

  return TRUE;
}

void IncomingVideo::SetRenderFrameSize(int _width, int _height) 
{ 
   PTRACE(3,"IncomingVideo Set size"); 
   width  = _width;
   height = _height;
   frameSize = width * height;
}
      
BOOL IncomingVideo::Close()
{
   PWaitAndSignal mutex(videoChanMutex);
   closed = TRUE;

   return TRUE;
  
}
 
///////////////////////////////////////////////////////////////////////////

OutgoingVideo::OutgoingVideo(MyH323EndPoint & _ep, MyH323Connection & _conn, int framesPerSec, BOOL _videoLarge)
  : ep(_ep), conn(_conn), videoLarge(_videoLarge)
{
  closed = FALSE;

  if ( ( framesPerSec>0 ) && ( framesPerSec < 31 ) ) {
    msBetweenFrames= 1000/framesPerSec;
  } else {
    cerr << "Invalid video transmit frame rate. Frame rate should be between 1 and 30 frames per second"<<endl;
    msBetweenFrames = 100;
  }
}

OutgoingVideo::~OutgoingVideo()
{
   OutgoingVideo::Close();
}

BOOL OutgoingVideo::Read(void *buffer, PINDEX  amount)
{
   PWaitAndSignal mutex1(videoChanMutex);

   amount = (( videoLarge ? 352*288*3 : 176*144*3 ) >> 1);
 
   if (!delay.Delay(msBetweenFrames)) 
      conn.OnOutgoingVideo(buffer, amount);
  
   return TRUE;
}


BOOL OutgoingVideo::Close()
{
//  PWaitAndSignal mutex(videoChanMutex);
  closed = TRUE;
  return TRUE;  
}


///////////////////////////////////////////////////////////////////////////

VideoBuffer::VideoBuffer()
  : videoBufferSize(VIDEO_BUFFER_SIZE), xSize(176), ySize(144)
{
  buffer = NULL;//new BYTE[videoBufferSize];
  
//  memset( buffer, 0x20, videoBufferSize);
  SetSize( xSize, ySize );
}

VideoBuffer::~VideoBuffer()
{
  delete[] buffer;
}


//Writes data into the specified posn of the buffer.
//0 == top left, 1 == top right
//2 == bot left, 3 == bot right
void VideoBuffer::Write(BYTE * data, PINDEX amount,  PINDEX posn)
{
   // It appears that a full frame is always written.
   // We can determine the size of the frame by how much
   // is written.

   if (amount == 0)
     return;

   PWaitAndSignal mutex(videoBufferMutex);
  
   BYTE *yFirst, *ySecond, *u, *v, *srcYFirst, *srcYSecond, *srcU, *srcV;
   int  srcFrameSize = (amount<<1)/3;
   int  srcXSize     = (srcFrameSize == (176 * 144) ? 176 : 352 );
   int  srcYSize     = (srcFrameSize == (176 * 144) ? 144 : 288 );

   yFirst  = buffer;
   u= buffer + bufferFrameSize;
   v= buffer + bufferFrameSize + (bufferFrameSize >> 2);

   srcYFirst  = data;
   srcYSecond = srcYFirst + (xSize == srcXSize ?
   			     srcXSize << 1 :
			     srcXSize);
   srcU  = data + srcFrameSize;
   srcV  = data + srcFrameSize + (srcFrameSize >> 2);

   switch (posn) {
      case 0:
         break;
      case 1:
         yFirst +=(xSize >> 1);
         u +=(xSize >> 2);
         v +=(xSize >> 2);
         break;
      case 2:
         yFirst += (bufferFrameSize >> 1);
         u += (bufferFrameSize >> 3);
         v += (bufferFrameSize >> 3);
         break;
      case 3:
         yFirst += (bufferFrameSize >> 1) + (xSize >> 1);
         u += (bufferFrameSize >> 3) + (xSize >> 2);
         v += (bufferFrameSize >> 3) + (xSize >> 2);
         break;
      default: 
         return;
   }
    
   ySecond = yFirst + xSize;    // added down here so that any changes to
   				// yFirst are taken into account - pez
   // Special case, can fit 'in' images perfectly inside 'out' images
   if ( xSize == 352 && srcXSize == 176 ) {
      for(int i=0; i < 144; i+=2) {
         memcpy(yFirst, srcYFirst, 176);
	 memcpy(ySecond, srcYSecond, 176);
	 memcpy(u, srcU, 88 );
	 memcpy(v, srcV, 88 );

	 srcYFirst += 352;
	 srcYSecond += 352;
	 yFirst += 704;
	 ySecond += 704;
	 srcU += 88;
	 srcV += 88;
	 u    += 176;
	 v    += 176;
      }
   } else {
      // This code handles the other 2 cases in a generic fashion
      int src_step   = 3 * srcXSize;
      int step       = xSize + (xSize >> 1);
      int srcuv_step = (srcXSize >> 1);
      int uv_step    = (xSize >> 2);

      for(int i=0; i<srcYSize; i+=4) {
         for(int j=0; j<srcXSize; j+=4) {
            *yFirst++  = *srcYFirst;
            srcYFirst += 2;
            *yFirst++  = *srcYFirst;
            srcYFirst += 2;
            *ySecond++ = *srcYSecond;
            srcYSecond += 2;
            *ySecond++ = *srcYSecond;
            srcYSecond += 2;
            *u++ = *srcU;
            *v++ = *srcV;
            srcU+= 2;
            srcV+= 2;
         }
         srcYFirst  += src_step;
         srcYSecond += src_step;
         yFirst     += step;
         ySecond    += step;
         srcU += srcuv_step;
         srcV += srcuv_step;
         u    += uv_step;
         v    += uv_step;
      }
   }

   return;
}

void VideoBuffer::Clear(PINDEX posn)
{
   PWaitAndSignal mutex(videoBufferMutex);
  
   BYTE *yFirst, *ySecond, *u, *v;

   yFirst  = buffer;
   u= buffer + bufferFrameSize;
   v= buffer + bufferFrameSize + (bufferFrameSize/4);

   switch (posn) {
      case 0:
         break;
      case 1:
         yFirst += (xSize >> 1);
         u += (xSize >> 2);
         v += (xSize >> 2);
	 break;
      case 2:
         yFirst += (bufferFrameSize >> 1);
         u += (bufferFrameSize >> 3);
         v += (bufferFrameSize >> 3);
         break;
      case 3:
         yFirst += (bufferFrameSize >>1) + (xSize >> 1);
         u += (bufferFrameSize >> 3) + (xSize >> 2);
	 v += (bufferFrameSize >> 3) + (xSize >> 2);
	 break;
      default:
         return;
   }
   ySecond = yFirst + xSize;

   int step = (xSize == 176 ? 4 : 2);
   int uv_step     = (xSize >> 2);
   for(int y=0; y<144; y+=step) {
      for(int x=0; x<176; x+=step) {
         *u++ = *v++ = 0x80;
      }
      u    += uv_step;
      v    += uv_step;
   }

   return;
}

void VideoBuffer::SetSize(int x, int y)
{
  PWaitAndSignal mutex(videoBufferMutex);
  if ( buffer != NULL )
    delete[] buffer;
  xSize = x;
  ySize = y;
  bufferFrameSize = xSize * ySize;
  videoBufferSize = bufferFrameSize * 3;
  buffer = new BYTE[ videoBufferSize ];
  memset( buffer, 0x80, videoBufferSize);
}


void VideoBuffer::Read(BYTE * data, PINDEX amount)
{
  if (amount == 0)
    return;
  PWaitAndSignal mutex(videoBufferMutex);
  memcpy(data,buffer,amount);
}
#endif //NO_VIDEO
///////////////////////////////////////////////////////////////////////////



#define	MIN_HEADROOM	30
#define	MAX_HEADROOM	60

AudioDelay::AudioDelay()
{
  firstTime = TRUE;
  error = 0;
}

void AudioDelay::Restart()
{
  firstTime = TRUE;
}

BOOL AudioDelay::Delay(int frameTime)
{
  if (firstTime) {
    firstTime = FALSE;
    previousTime = PTime();
    return TRUE;
  }

  error += frameTime;

  PTime now;
  PTimeInterval delay = now - previousTime;
  error -= (int)delay.GetMilliSeconds();
  previousTime = now;

  if (error > 0) 
#ifdef P_LINUX
    usleep(error * 1000);
#else
    PThread::Current()->Sleep(error);
#endif

  return error <= -frameTime;

  //if (headRoom > MAX_HEADROOM)
  //  PThread::Current()->Sleep(headRoom - MIN_HEADROOM);
}

// End of File ///////////////////////////////////////////////////////////////
