/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, astmail@yahoo.com
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* pingpong: ICMP echo request/reply                                          */
/*============================================================================*/

/*
 * icmphandler
 *
 * input:  anchor - pointer to vpnd global data
 *
 * return: 0 in case of creation success, 1 in case of creation error,
 *	   always 1 in case of deletion
 *
 * This procedure creates the icmp socket and presets the required
 * structures or it deletes the icmp socket.
 */

int icmphandler(VPN *anchor)
{
	int i;		/* used to set socket to nonblocking */


	/* debug message */

	ENTER("icmphandler");

	/* use error handling to close open icmp socket */

	if(anchor->ping!=-1)goto err2;

	/* create icmp socket, react to errors */

	if((anchor->ping=socket(AF_INET,SOCK_RAW,IPPROTO_ICMP))==-1)
		JUMP("socket",err1);

	/* set socket to nonblocking, handle errors */

	i=1;
	if(ioctl(anchor->ping,FIONBIO,&i))JUMP("ioctl(FIONBIO)",err2);

	/* fill icmp output structure with random data */

	getrandom(anchor->icmp_out,sizeof(anchor->icmp_out),anchor);

	/* preset icmp header */

#ifdef LINUX
	((struct icmphdr *)(anchor->icmp_out))->type=ICMP_ECHO;
	((struct icmphdr *)(anchor->icmp_out))->code=0;
	((struct icmphdr *)(anchor->icmp_out))->checksum=0;
	((struct icmphdr *)(anchor->icmp_out))->un.echo.id=getpid();
	((struct icmphdr *)(anchor->icmp_out))->un.echo.sequence=0;
#elif FreeBSD
	((struct icmp *)(anchor->icmp_out))->icmp_type=ICMP_ECHO;
	((struct icmp *)(anchor->icmp_out))->icmp_code=0;
	((struct icmp *)(anchor->icmp_out))->icmp_cksum=0;
	((struct icmp *)(anchor->icmp_out))->icmp_hun.ih_idseq.icd_id=getpid();
	((struct icmp *)(anchor->icmp_out))->icmp_hun.ih_idseq.icd_seq=0;
#endif

	/* prepare target address structure */

	anchor->icmp_target.sin_family=AF_INET;
	anchor->icmp_target.sin_addr.s_addr=getaddr(anchor->remote_ip);
	anchor->icmp_target.sin_port=0;

	/* debug message */

	LEAVE("icmphandler");

	/* signal success */

	return 0;

	/* error handling and icmp socket close */

err2:	if(close(anchor->ping))ERRNO("close");
	anchor->ping=-1;
err1:	return 1;
}

/*
 * recvicmp
 *
 * input:  anchor - pointer to vpnd global data
 *
 * return: then amount of icmp echo replies received (up to 100)
 *
 * This procedure reads up to 100 icmp echo replies from the icmp
 * socket and returns the amount of replies read.
 */

int recvicmp(VPN *anchor)
{
	int total;	/* amount of echo replys read		*/
	int size;	/* size of buffer for echo reply	*/


	/* debug message */

	ENTER("recvicmp");

	/* for up to 100 echo replies */

	for(total=0;total<100;total++)
	{
		/* set up icmp echo reply sender information */

		anchor->icmp_source=anchor->icmp_target;

		/* set up echo reply buffer size */

		size=sizeof(anchor->icmp_source);

		/* try to read echo reply, exit loop if there is none */

		if(recvfrom(anchor->ping,anchor->icmp_in,
			sizeof(anchor->icmp_in),0,
			(struct sockaddr *)(&(anchor->icmp_source)),
			&size)==-1)break;
	}

	/* debug message */

	LEAVE("recvicmp");

	/* return the amount of echo replies read */

	return total;
}

/*
 * sendicmp
 *
 * input: anchor - pointer to vpnd global data
 *
 * This procedure send an icmp echo request packet to the peer.
 */

void sendicmp(VPN *anchor)
{
	int sum;	/* used to create checksum for icmp */
	int i;		/* used to create checksum for icmp */


	/* debug message */

	ENTER("sendicmp");

	/* reset checksum in icmp header */

#ifdef LINUX
	((struct icmphdr *)(anchor->icmp_out))->checksum=0;
#elif FreeBSD
	((struct icmp *)(anchor->icmp_out))->icmp_cksum=0;
#endif

	/* calculate packet checksum */

	for(sum=i=0;i<sizeof(anchor->icmp_out);)
	{
		sum+=anchor->icmp_out[i++];
		sum+=((int)(anchor->icmp_out[i++]))<<8;
	}
	while(sum&0xffff0000)sum=(sum>>16)+(sum&0xffff);
	sum=~sum;

	/* write checksum complement to header */

#ifdef LINUX
	((WORD08 *)&(((struct icmphdr *)(anchor->icmp_out))->checksum))[0]=
		(WORD08)(sum);
	((WORD08 *)&(((struct icmphdr *)(anchor->icmp_out))->checksum))[1]=
		(WORD08)(sum>>8);
#elif FreeBSD
	((WORD08 *)&(((struct icmp *)(anchor->icmp_out))->icmp_cksum))[0]=
		(WORD08)(sum);
	((WORD08 *)&(((struct icmp *)(anchor->icmp_out))->icmp_cksum))[1]=
		(WORD08)(sum>>8);
#endif

	/* send echo request */

	sendto(anchor->ping,anchor->icmp_out,sizeof(anchor->icmp_out),0,
		(struct sockaddr *)(&(anchor->icmp_target)),
		sizeof(anchor->icmp_target));

	/* next sequence number */

#ifdef LINUX
	((struct icmphdr *)(anchor->icmp_out))->un.echo.sequence+=1;
#elif FreeBSD
	((struct icmp *)(anchor->icmp_out))->icmp_hun.ih_idseq.icd_seq+=1;
#endif

	/* debug message */

	LEAVE("sendicmp");
}
