#include <QDomDocument>
#include <QtGui>
#include "MainWindow.h"
#include "Graph.h"
#include "RenderArea.h"
#include "PreferencesDialog.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    mdiArea = new QMdiArea();
    createActionsAndMenus();
    setCentralWidget(mdiArea);
    connect(mdiArea, SIGNAL(subWindowActivated(QMdiSubWindow*)), this, SLOT(updateMenus()));
    windowMapper = new QSignalMapper(this);
    connect(windowMapper, SIGNAL(mapped(QWidget*)), this, SLOT(setActiveSubWindow(QWidget*)));
//    mdiArea->setViewMode(QMdiArea::SubWindowView);
    mdiArea->setViewMode(QMdiArea::TabbedView);
    mdiArea->setTabsClosable(true);
    mdiArea->setTabsMovable(true);

    emptyWebPage = new QWebPage(this);
    emptyWebPageMainFrame = emptyWebPage->mainFrame();

    settings = new QSettings("Free Software", "Lybniz");
    setWindowTitle(tr("Lybniz"));
    setUnifiedTitleAndToolBarOnMac(true);
    restoreState();

    Graph* nf = newFile();
    if (mdiArea->viewMode() == QMdiArea::TabbedView)
        nf->window->setWindowState(Qt::WindowMaximized);
    nf->y1LineEdit->setText("Math.sin(x)");
    nf->plot();
}

void MainWindow::createActionsAndMenus()
// Construct actions, menus and toolbars
{    
    plotAction = new QAction(QIcon::fromTheme("system-run", QIcon(":/Images/Apply.png")), tr("P&lot"), this );
    plotAction->setStatusTip(tr("Plot the graph"));
    plotAction->setShortcut(QKeySequence::Refresh);
    connect(plotAction, SIGNAL(triggered()), this, SLOT(plot()));

    newAction = new QAction(QIcon::fromTheme("document-new", QIcon(":/Images/FileNew.png")), tr("&New"), this );
    newAction->setStatusTip(tr("Create a new file"));
    newAction->setShortcut(QKeySequence::New);
    connect(newAction, SIGNAL(triggered()), this, SLOT(newFile()));

    openAction = new QAction(QIcon::fromTheme("document-open", QIcon(":/Images/FileOpen.png")), tr("&Open"), this );
    openAction->setStatusTip(tr("Open a file with function definitions"));
    openAction->setShortcut(QKeySequence::Open);
    connect(openAction, SIGNAL(triggered()), this, SLOT(openFile()));

    saveAsAction = new QAction(QIcon::fromTheme("document-save-as", QIcon(":/Images/FileSaveAs.png")), tr("Save &As"), this );
    saveAsAction->setStatusTip(tr("Save the function definitions under a new name"));
    saveAsAction->setShortcut(QKeySequence::SaveAs);
    connect(saveAsAction, SIGNAL(triggered()), this, SLOT(saveAsFile()));

    saveAction = new QAction(QIcon::fromTheme("document-save", QIcon(":/Images/FileSave.png")), tr("&Save"), this );
    saveAction->setStatusTip(tr("Save the function definitions"));
    saveAction->setShortcut(QKeySequence::Save);
    connect(saveAction, SIGNAL(triggered()), this, SLOT(saveFile()));

    zoomInAction = new QAction(QIcon::fromTheme("zoom-in", QIcon(":/Images/ZoomIn.png")), tr("Zoom &In"), this );
    zoomInAction->setStatusTip(tr("Zoom into a quarter of the area"));
    zoomInAction->setShortcut(QKeySequence::ZoomIn);
    connect(zoomInAction, SIGNAL(triggered()), this, SLOT(zoomIn()));

    zoomOutAction = new QAction(QIcon::fromTheme("zoom-out", QIcon(":/Images/ZoomOut.png")), tr("Zoom &Out"), this );
    zoomOutAction->setStatusTip(tr("Zoom out to see four times the area"));
    zoomOutAction->setShortcut(QKeySequence::ZoomOut);
    connect(zoomOutAction, SIGNAL(triggered()), this, SLOT(zoomOut()));

    zoomResetAction = new QAction(QIcon::fromTheme("zoom-original", QIcon(":/Images/ZoomBack.png")), tr("Zoom &Reset"), this );
    zoomResetAction->setStatusTip(tr("Reset zoom factor to original"));
    zoomResetAction->setShortcut(QKeySequence::MoveToStartOfDocument);
    connect(zoomResetAction, SIGNAL(triggered()), this, SLOT(zoomReset()));

    connectPointsAction = new QAction(tr("&Connect Points"), this );
    connectPointsAction->setCheckable(true);
    connectPointsAction->setChecked(true);
    connect(connectPointsAction, SIGNAL(triggered()), this, SLOT(connectPoints()));

    preferencesAction = new QAction(QIcon::fromTheme("preferences-other"), tr("&Preferences..."), this );
    preferencesAction->setStatusTip(tr("Set Preferences"));
    connect(preferencesAction, SIGNAL(triggered()), this, SLOT(showPreferences()));

    aboutAction = new QAction(QIcon::fromTheme("help-about"), tr("&About"), this );
    aboutAction->setStatusTip(tr("About Lybniz"));
    aboutAction->setMenuRole (QAction::AboutRole);
    connect(aboutAction, SIGNAL(triggered()), this, SLOT(showAboutBox()));

    closeAction = new QAction(QIcon::fromTheme("window-close"), tr("&Close"), this );
    closeAction->setStatusTip(tr("Close file"));
    closeAction->setShortcut(QKeySequence::Close);
    connect(closeAction, SIGNAL(triggered()), this, SLOT(closeFile()));

    exitAction = new QAction(QIcon::fromTheme("application-exit"), tr("E&xit"), this );
    exitAction->setStatusTip(tr("Exit Lybniz"));
    exitAction->setMenuRole(QAction::QuitRole);
    exitAction->setShortcut(QKeySequence::Quit);
    connect(exitAction, SIGNAL(triggered()), this, SLOT(close()));

    separatorAction = new QAction(this);
    separatorAction->setSeparator(true);

    menuBar = new QMenuBar(0);

    fileMenu = menuBar->addMenu(tr("&File"));
    fileMenu->addAction(newAction);
    fileMenu->addAction(openAction);
    fileMenu->addAction(saveAction);
    fileMenu->addAction(saveAsAction);
    fileMenu->addAction(closeAction);
    fileMenu->addSeparator();
    fileMenu->addAction(exitAction);

    graphMenu = menuBar->addMenu(tr("&Graph"));
    graphMenu->addAction(plotAction);
    graphMenu->addSeparator();
    graphMenu->addAction(zoomInAction);
    graphMenu->addAction(zoomOutAction);
    graphMenu->addAction(zoomResetAction);
    graphMenu->addSeparator();
    graphMenu->addAction(connectPointsAction);
    graphMenu->addAction(preferencesAction);

    windowMenu = menuBar->addMenu(tr("&Window"));
    updateWindowMenu();
    connect(windowMenu, SIGNAL(aboutToShow()), this, SLOT(updateWindowMenu()));

    helpMenu = menuBar->addMenu(tr("&Help"));
    helpMenu->addAction(aboutAction);

    toolBar = addToolBar(tr("Main"));
    toolBar->addAction(openAction);
    toolBar->addAction(saveAction);
    toolBar->addSeparator();
    toolBar->addAction(zoomInAction);
    toolBar->addAction(zoomOutAction);
    toolBar->addAction(zoomResetAction);
    toolBar->addSeparator();
    toolBar->addAction(plotAction);

    setMenuBar(menuBar);
}

Graph* MainWindow::newFile()
{
    static int newFileNumber = 1;
    Graph* child = new Graph(this);
    child->window = mdiArea->addSubWindow(child);
    child->window->setWindowIcon(QIcon(":/Images/Sheet.png"));
    child->window->setWindowTitle(tr("Graph%1.lyb[*]").arg(newFileNumber++));
    child->window->show();
    return child;
}

QVariant MainWindow::evaluate(qreal x, const QString& expression)
{
    QString jsExpression = QString("(function(){var x=%1; return %2})()").arg(x).arg(expression);
    return emptyWebPageMainFrame->evaluateJavaScript(jsExpression);
}

Graph* MainWindow::activeGraph()
{
    if (QMdiSubWindow *activeSubWindow = mdiArea->activeSubWindow())
        return qobject_cast<Graph*>(activeSubWindow->widget());
    return 0;
}

void MainWindow::plot()
{
    activeGraph()->plot();
}

void MainWindow::zoomIn()
{
    activeGraph()->zoomIn();
}

void MainWindow::zoomOut()
{
    activeGraph()->zoomOut();
}

void MainWindow::zoomReset()
{
    activeGraph()->zoomReset();
}

void MainWindow::openFile()
{
    QString fileName = QFileDialog::getOpenFileName(this, "Open Graph", "", "Lybniz (*.lyb)");
    if (fileName.isNull())
        return;

    Graph* nf = newFile();
    if (nf->openFile(fileName)) {
        nf->plot();
        if (mdiArea->viewMode() == QMdiArea::TabbedView)
            nf->window->setWindowState(Qt::WindowMaximized);
        nf->show();
    }else {
        nf->close();
    }
}

bool MainWindow::saveAsFile()
{
    return activeGraph()->saveAsFile();
}

bool MainWindow::saveFile()
{
    return activeGraph()->saveFile();
}

void MainWindow::closeFile()
{
    activeGraph()->close();
}

void MainWindow::connectPoints()
{
    activeGraph()->renderArea->connectPoints = !activeGraph()->renderArea->connectPoints;
    activeGraph()->plot();
}

void MainWindow::showPreferences()
{
    preferences = new PreferencesDialog(this);
    preferences->exec();
}

void MainWindow::saveState()
{
    settings->setValue("Geometry", saveGeometry());
    settings->setValue("MdiViewMode", mdiArea->viewMode());
}

void MainWindow::restoreState()
{
    restoreGeometry(settings->value("Geometry").toByteArray());
    mdiArea->setViewMode(static_cast<QMdiArea::ViewMode> (settings->value("MdiViewMode", QMdiArea::TabbedView).toInt()));
}

void MainWindow::showAboutBox()
{
    QMessageBox::about(this, tr("About Lybniz"),
                       tr("<h2>Lybniz</h2>"
                          "<p><em>Version 3.0 beta</em>"
                          "<p>Function graph plotter<br>"
                          "2012 by Thomas F&uuml;hringer<br>"
                          "<p>Note: function terms need to be <br>entered in JavaScript syntax.<br>"));
}

void MainWindow::updateMenus()
{
    bool hasMdiChild = (activeGraph() != 0);
    saveAction->setEnabled(hasMdiChild);
    saveAsAction->setEnabled(hasMdiChild);
    closeAction->setEnabled(hasMdiChild);
    plotAction->setEnabled(hasMdiChild);
    zoomInAction->setEnabled(hasMdiChild);
    zoomOutAction->setEnabled(hasMdiChild);
    zoomResetAction->setEnabled(hasMdiChild);
    connectPointsAction->setEnabled(hasMdiChild);
}

void MainWindow::updateWindowMenu()
{
    windowMenu->clear();
    windowMenu->addAction(closeAction);
//    windowMenu->addSeparator();
    windowMenu->addAction(separatorAction);
    QList<QMdiSubWindow *> windows = mdiArea->subWindowList();
    separatorAction->setVisible(!windows.isEmpty());

    for (int i = 0; i < windows.size(); ++i) {
        Graph* graph = qobject_cast<Graph*>(windows.at(i)->widget());

        QString text;
        if (i < 9) {
            text = tr("&%1 %2").arg(i + 1).arg(graph->window->windowTitle());
        } else {
            text = tr("%1 %2").arg(i + 1).arg(graph->window->windowTitle());
        }
        QAction *action  = windowMenu->addAction(text);
        action->setCheckable(true);
        action->setChecked(graph == activeGraph());
        connect(action, SIGNAL(triggered()), windowMapper, SLOT(map()));
        windowMapper->setMapping(action, windows.at(i));
    }
}

void MainWindow::setActiveSubWindow(QWidget *window)
{
    if (!window)
        return;
    mdiArea->setActiveSubWindow(qobject_cast<QMdiSubWindow*>(window));
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    mdiArea->closeAllSubWindows();
    if (mdiArea->currentSubWindow()) {
        event->ignore();
    } else {
        saveState();
        event->accept();
    }
}

MainWindow::~MainWindow()
{

}
