require File.expand_path(File.dirname(__FILE__) + "/spec_helper")
require 'mizuho/parser'

shared_examples_for "an Asciidoc document with 3 chapters" do
	it "extracts the chapter contents" do
		@chapter_one.contents.should =~ /Chapter 1 contents/
		@chapter_two.contents.should =~ /Chapter 2 contents/
	end

	specify "the extracted chapter contents include (sub)section contents as well" do
		@chapter_two.contents.should =~ /Subsection/
		@chapter_two.contents.should =~ /This is a subsection/
	end

	it "doesn't HTML-escape auto-generated HTML formatting tags in the title" do
		@html_chapter.title.should =~ %r(<strong>Look</strong>)
	end

	it "HTML-escapes non-autogenerated HTML tags in the title" do
		@html_chapter.title.should =~ %r(&lt;i&gt;hurray&lt;/i&gt;)
	end

	it "HTML-escapes non-HTML special characters in the title" do
		@html_chapter.title.should =~ %r(&lt;3)
	end
end

describe Mizuho::Parser do
	it "extracts the title" do
		parser = generate_and_parse(%Q{
			= Hello world
			A sample document.
		})
		parser.title.should == "Hello world"
	end
	
	specify "the extracted title is not HTML escaped" do
		parser = generate_and_parse(%Q{
			= *Look at this!* 2 < 1!
			A sample document.
		})
		parser.title.should == "<strong>Look at this!</strong> 2 < 1!"
	end
	
	describe "construction of table of contents" do
		before :each do
			@parser = generate_and_parse(%Q{
				= Hello world
				A sample document.
				
				== Scientists Erase Specific Memories In Mice
				It sounds like science fiction, but scientists say it might
				one day be possible to erase undesirable memories from the
				brain, selectively and safely.
				
				=== How it works
				After exposing mice to emotionally powerful stimuli, such
				as a mild shock to their paws, the scientists then observed
				how well or poorly the animals subsequently recalled the
				particular trauma as their brain's expression of CaMKII was
				manipulated up and down. When the brain was made to
				overproduce CaMKII at the exact moment the mouse was prodded
				to retrieve the traumatic memory, the memory wasn't just
				blocked, it appeared to be fully erased.
				
				== Researchers Developing Cancer-Fighting Beer
				Ever picked up a cold, frosty beer on a hot summer's day
				and thought that it simply couldn't get any better? Well,
				think again. A team of researchers at Rice University in
				Houston is working on helping Joe Six Pack fight aging and
				cancer with every swill of beer.
				
				=== Comment by samzenpus
				Thank you science! Now we just need cigarettes that cure baldness.
				
				== *Look*, non-HTML special characters <3, <i>hurray</i>
			})
			@toc = @parser.table_of_contents
		end
		
		it "contains all headers" do
			@toc.should have(3).items
			
			@toc[0].title.should =~ /Scientists Erase Specific Memories In Mice/
			@toc[0].children.should have(1).item
			@toc[0].children[0].title.should =~ /How it works/
			
			@toc[1].title.should =~ /Researchers Developing Cancer-Fighting Beer/
			@toc[1].children.should have(1).item
			@toc[1].children[0].title.should =~ /Comment by samzenpus/
			
			@toc[2].title.should =~ /Look/
			@toc[2].children.should be_empty
		end
		
		it "records the corresponding header levels" do
			@toc[0].level.should == 2
			@toc[0].children[0].level.should == 3
			@toc[1].level.should == 2
			@toc[1].children[0].level.should == 3
			@toc[2].level.should == 2
		end
		
		it "doesn't HTML-escape auto-generated HTML formatting tags in the title" do
			@toc[2].title.should =~ %r(<strong>Look</strong>)
		end
		
		it "HTML-escapes non-autogenerated HTML tags in the title" do
			@toc[2].title.should =~ %r(&lt;i&gt;hurray&lt;/i&gt;)
		end
		
		it "HTML-escapes non-HTML special characters in the title" do
			@toc[2].title.should =~ %r(&lt;3)
		end
		
		it "prepends heading titles with a corresponding section number" do
			@toc[0].title.should =~ /^1\. /
			@toc[0].children[0].title.should =~ /^1\.1\. /
			
			@toc[1].title.should =~ /^2\. /
			@toc[1].children[0].title.should =~ /^2\.1\. /
			
			@toc[2].title.should =~ /^3\. /
		end
	end
	
	describe "chapter extraction" do
		def test_input(with_preamble)
			return %Q{
				= Hello world
				
				#{"A preamble." if with_preamble}
			
				== Chapter 1
				Chapter 1 contents.
			
				== Chapter 2
				Chapter 2 contents.
			
				=== Subsection
				This is a subsection.
			
				== *Look*, non-HTML special characters <3, <i>hurray</i>
			}
		end
		
		
		describe "when there is a premable" do
			before :each do
				@parser = generate_and_parse(test_input(true))
				@chapters = @parser.chapters
				@chapter_one = @chapters[1]
				@chapter_two = @chapters[2]
				@html_chapter = @chapters[3]
			end
		
			it "consists of a preamble followed by level 2 headings" do
				@chapters.should have(4).items
			end
			
			specify "the preamble chapter has no title" do
				@chapters.first.title.should be_nil
			end
			
			it_should_behave_like "an Asciidoc document with 3 chapters"
		end
		
		describe "when there is no preamble" do
			def default_test_input
				return test_input(false)
			end
			
			before :each do
				@parser = generate_and_parse(test_input(false))
				@chapters = @parser.chapters
				@chapter_one = @chapters[0]
				@chapter_two = @chapters[1]
				@html_chapter = @chapters[2]
			end
			
			it "consists of only level 2 headings" do
				@chapters.should have(3).items
			end
			
			it_should_behave_like "an Asciidoc document with 3 chapters"
		end
	end
	
	describe "content extraction" do
		describe "when there is a preamble" do
			before :each do
				@parser = generate_and_parse(%Q{
					= Ruby Enterprise Edition Features Guide
					Today
					
					Hello world
					
					== Overview
					Ruby Enterprise Edition (REE) is a
					server-oriented distribution of the
					official Ruby interpreter, and includes
					various additional enhancements.
				})
			end
			
			it "extracts the contents" do
				@parser.contents.should include("various additional enhancements.")
			end
			
			it "strips away the irrelevant HTML that come before the actual content" do
				@parser.contents.should_not include("<title>")
			end
		end
		
		describe "when there is no preamble" do
			before :each do
				@parser = generate_and_parse(%Q{
					= Ruby Enterprise Edition Features Guide
					
					== Overview
					Ruby Enterprise Edition (REE) is a
					server-oriented distribution of the
					official Ruby interpreter, and includes
					various additional enhancements.
				})
			end
			
			it "extracts the contents" do
				@parser.contents.should include("various additional enhancements.")
			end
			
			it "strips away the irrelevant HTML that come before the actual content" do
				@parser.contents.should_not include("<title>")
			end
		end
	end
	
	it "fixes cross-chapter references when in multi-page output mode"
end
