/*
 * systrace-private.h
 *
 * Copyright (c) 2002 Marius Aamodt Eriksen <marius@umich.edu>
 * Copyright (c) 2002 Niels Provos <provos@citi.umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. The names of the copyright holders may not be used to endorse or
 *     promote products derived from this software without specific
 *     prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 *  THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef SYSTRACE_PRIVATE_H
#define SYSTRACE_PRIVATE_H

#define POLICY_VALID(x)	((x) == SYSTR_POLICY_PERMIT || \
			 (x) == SYSTR_POLICY_ASK ||    \
			 (x) == SYSTR_POLICY_NEVER)

#define DPRINTF(x) if (systrace_debug) printk x

struct str_policy {
	int                      nr;
	struct emul             *emul;	   /* XXX */
	int                      refcount;
	int                      nsysent;
	short                   *sysent;
	TAILQ_ENTRY(str_policy)  next;
};

#define STR_PROC_ONQUEUE	0x01
#define STR_PROC_WAITANSWER	0x02
#define STR_PROC_SYSCALLRES	0x04
#define STR_PROC_REPORT		0x08	/* Report emulation */
#define STR_PROC_FSCHANGE	0x10
#define STR_PROC_SETEUID        0x20    /* Elevate privileges */ 
#define STR_PROC_SETEGID        0x40

struct str_process {
	TAILQ_ENTRY(str_process)  next;
	TAILQ_ENTRY(str_process)  msg_next;
	struct semaphore          lock;	
	struct task_struct       *proc;
	pid_t pid;
	struct fsystrace         *parent;
	struct str_policy        *policy;
	wait_queue_head_t         wqh;	
	int                       flags;
	short                     answer;
	short                     error;
	u16                       seqnr; /* XXX: convert to u_int16_t  */
	struct str_message        msg;
	struct systrace_replace  *replace;
	int                       report;
	mm_segment_t              oldfs;
	int                       maycontrol;
	int                       code;
	register_t                args[8];
	uid_t                     oldeuid;
	gid_t                     oldegid;
	uid_t                     savedeuid;
	uid_t                     savedegid;
	uid_t                     seteuid;
	uid_t                     setegid;
	int                       issuser;
};

/* VFS interface */
int                systracef_ioctl(struct inode *, struct file *, unsigned int,
                       unsigned long);
ssize_t            systracef_read(struct file *, char *, size_t, loff_t *);
ssize_t            systracef_write(struct file *, const char *, size_t, loff_t *);
int                systracef_open(struct inode *, struct file *);
int                systracef_release(struct inode *, struct file *);
unsigned int       systracef_poll(struct file *, struct poll_table_struct *);

/* Policy handling */
struct str_policy *systrace_newpolicy(struct fsystrace *, int);
void               systrace_closepolicy(struct fsystrace *, struct str_policy *);
int                systrace_policy(struct fsystrace *, struct systrace_policy *);
struct str_policy *systrace_newpolicy(struct fsystrace *, int);

/* Message utility functions */
int                 systrace_msg_child(struct fsystrace *, struct str_process *, pid_t);
int                 systrace_msg_result(struct fsystrace *, struct str_process *, int, int,
                        size_t, register_t[]);
int                 systrace_msg_ask(struct fsystrace *, struct str_process *, int, size_t, register_t[]);
int                 systrace_msg_ugid(struct fsystrace *, struct str_process *);
int                 systrace_msg_execve(struct fsystrace *, struct str_process *, register_t);
int                 systrace_make_msg(struct str_process *, int);
int                 systrace_make_msg(struct str_process *, int);

int                 systrace_io(struct str_process *, struct systrace_io *);
int                 systrace_getcwd(struct fsystrace *, struct str_process *);
int                 systrace_rescwd(struct fsystrace *);
int                 systrace_attach(struct fsystrace *, pid_t);
int                 systrace_detach(struct str_process *);
int                 systrace_answer(struct str_process *, struct systrace_answer *);
int                 systrace_insert_process(struct fsystrace *, struct task_struct *);
int                 systrace_processready(struct str_process *);
struct str_process *systrace_findpid(struct fsystrace *, pid_t);
struct task_struct *systrace_find(struct str_process *);

int                 systrace_preprepl(struct str_process *, struct systrace_replace *);
int                 systrace_replace(struct str_process *, size_t, register_t[]);
uid_t               systrace_seteuid(struct task_struct *, uid_t);
gid_t               systrace_setegid(struct task_struct *, gid_t);

#if 0
void                systrace_lock(void);
void                systrace_unlock(void);
#endif /* 0 */
/*
 * Currently, disable the fine grained locking and use the big kernel
 * lock instead.  The only thing keeping me from using the fine
 * grained locking is in systrace_make_msg(); when fst->lock is
 * relinquished, there is a race condition until we sleep on the strp;
 * it could have been detached in the mean time, causing nasty things
 * to happen.  When using the kernel lock, it is automatically
 * relinquished when needed.
 */
#define systrace_lock(...) lock_kernel();
#define systrace_unlock(...) unlock_kernel();

#endif /* SYSTRACE_PRIVATE_H */
