#define UwsIoctl_c

// **********************************************************************
// * uwsioctl.c                                                         *
// *                                                                    *
// * Written by Matthew DeLoera (deloera@us.ibm.com)                    *
// * (C) Copyright IBM Corporation, 1998-2001                           *
// *                                                                    *
// *  This software may be used and distributed according to the terms  *
// *  of the GNU Public License, incorporated herein by reference.      *
// **********************************************************************

// **********************************************************************
// * uwsioctl.c - This source file provides all routines for event      *
// *              processing.                                           *
// *                                                                    *
// * Exported Functions                                                 *
// * ------------------                                                 *
// * sysSpProcessIoctl   - this processes all commands to be sent to    *
// *                       ASM, whether from external apps or from      *
// *                       internal driver processes                    *
// *                                                                    *
// * Changes                                                            *
// * -------                                                            *
// * 01/29/2001 - Cleaned up for open source release.                   *
// * ----------------------------------------------------------------   *
// * 06/27/2002 - Added multi node support       // PHR_MULTI_4         *
// **********************************************************************


// ************
// * INCLUDES *
// ************

#include "uwscmd.h"


// ***********************
// * EXTERNAL PROTOTYPES *
// ***********************
extern PWORK_BLOCK sysSpQueryQueue(void *pHead, PDD_IOCTL_INFO pDDIoctlInfo);
extern PWORK_BLOCK sysSpPeekQueue(void *pHead);
extern PWORK_BLOCK sysSpEnqueue(PDRIVER_INFO pDriverNode, void *pHead, PWORK_BLOCK pBlock);
extern PWORK_BLOCK sysSpDequeue(PDRIVER_INFO pDriverNode, void *pHead);
extern void        sysSpProcessPciRequest(PDRIVER_INFO pDriverNode, PCMD_BLOCK pCmdBlk, unsigned long ultimeoutValue);
extern void        sysSpRunSpCmd(PDRIVER_INFO pDriverNode);
extern UCHAR       sysSpSeekQueue(void *pHead, unsigned int PID);
extern void        sysSpDisplayQueue(void *pHead);
extern void        sysSpPurgeQueueDeregisterEvents(PDRIVER_INFO pDriverNode, void *pHead, unsigned int PID);
extern void        sysSpPurgeQueueByPID(PDRIVER_INFO pDriverNode, void *pHead, unsigned int PID);
extern void        sysSpRegisterEvent(PDRIVER_INFO pDriverNode, PEVENT_BLOCK pEventBlk, PUSHORT pIoctlMagicNumber);
extern void        sysSpRevHbTimeout(PDRIVER_INFO pDriverNode); 

#ifndef IA64  // PHR_180930
  extern void        sysSpProcessIsaRequest(PCMD_BLOCK pCmdBlk, unsigned long ultimeoutValue);
#endif

extern u32   READ_REGISTER_ULONG(void *x);               // PHR_180930 
extern void  WRITE_REGISTER_ULONG(unsigned long *x, unsigned long y);

// extern unsigned long READ_REGISTER_ULONG(void *x);    // PHR_180930

// ********************
// * LOCAL PROTOTYPES *
// ********************
int sysSpProcessIoctl(volatile int cmd, volatile PDD_IOCTL_INFO pDDIoctlInfo, volatile unsigned int PID, volatile int callerFlag, PDRIVER_INFO pDriverNode);

int MallocCount = 0;    // PHR_166576

// *****************************************************************************
// * sysSpProcessIoctl() - This function processes both the external IOCTLs    *
// *                       such as one that is invoked by an application; as   *
// *                       well as internal IOCTLs which are called by this    *
// *                       driver.                                             *
// *                                                                           *
// * Parameters:                                                               *
// *      cmd          - The IOCTL command code                                *
// *      pDDIoctlInfo - Pointer to the IOCTL parm structure                   *
// *      PID          - PID of associated application                         *
// *      callerFlag   - Flag indicating whether IOCTL is external or internal *
// *      PDRIVER_INFO - Pointer to the device driver node info block          *
// *                                                                           *
// * Return Codes:                                                             *
// *      0 = failed  - check parm structure for specific error code           *
// *      1 = success                                                          *
// *****************************************************************************
int sysSpProcessIoctl(volatile int cmd,
                      volatile PDD_IOCTL_INFO pDDIoctlInfo, 
                      volatile unsigned int PID, 
                      volatile int callerFlag,
                      PDRIVER_INFO pDriverNode)
{
   PEVENT_BLOCK pBlock;
   PCMD_BLOCK   pCmdBlk = NULL;

   PCMD_BUFFER pCmdBuffer;

   unsigned long CopyRC;
   unsigned long timeoutValue;

   //unsigned long Resolution;                   // PHR_188142
   u32 Resolution;                               // PHR_180930

   int i;
   int rc = 1;
   int RetryCount = 0;                           // PHR_181239
   unsigned long flags = 0;

   // IF SHARED LIBRARY IS REQUESTING PID, RETURN IT
   if (cmd == IOCTL_GETPID)
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - Request for PID received, returning PID %x.\n", PID);
      }

      return PID;
   }

   // IF SHARED LIBRARY IS REQUESTING NUMBER OF SPS, RETURN IT
   if (cmd == IOCTL_COUNTSP)
   {
      if (DBG_MULTI)
      {
         printk(KERN_CRIT "ibmasm: DBG_MULTI - Request for NumSpNodes received, returning %d.\n", NumSpNodes); 
      }

      return NumSpNodes;
   }

   if (callerFlag == EXTERNAL_IOCTL)
   {
      PID = pDDIoctlInfo->uiPID;

      // FIRST ENSURE PID IS REGISTERED
      for (i = 0; i < MAXPIDS; i++)
      {
         if((pDriverNode->PIDtable[i].valid)      &&
            (pDriverNode->PIDtable[i].PID == PID)  )
         {
             break;
         }
      }

      if (i == MAXPIDS)
      {
         // PID NOT REGISTERED. REJECT COMMAND.
         pDDIoctlInfo->returnCode = DDERR_DRIVER_CLOSED;

         if (DBG_IOCTL)
         {
            printk(KERN_CRIT "ibmasm: DBG_MULTI - PID %x not registered. Request rejected.\n", PID);
         }
         return 0;
      }

   }  // end if - DriverGeneratedPID

   // BRANCH ON IOCTL FUNCTION
   switch (cmd)
   {
      case IOCTL_DEREGISTER:
      {
          if (DBG_EVENTS)
          {
             printk(KERN_CRIT "ibmasm: DBG_EVENTS - received IOCTL_DEREGISTER request.\n");
          }

          // PURGE ALL EVENTS ASSOCIATED WITH THIS PID
          spin_lock_irqsave( &(pDriverNode->EventQueueLock), flags );
            
          if (sysSpSeekQueue(pDriverNode->EventQueue, PID))
          {
             sysSpPurgeQueueDeregisterEvents(pDriverNode, pDriverNode->EventQueue, PID);

             spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );
                
             pDDIoctlInfo->returnCode = DDERR_SUCCESS;

             if (DBG_EVENTS)
             {
                 printk(KERN_CRIT "ibmasm: DBG_EVENTS - Deregistering complete.\n");
             }
          }
          else
          {
             spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );

             pDDIoctlInfo->returnCode = DDERR_PID_NOT_REGISTERED;

             rc = 0;

             if (DBG_EVENTS)
             {
                printk(KERN_CRIT "ibmasm: DBG_EVENTS - PID not registered. IOCTL rejected.\n");
             }
          }

          if (DBG_EVENTS)
          {
             printk(KERN_CRIT "ibmasm: DBG_EVENTS - Completed processing this DeregisterForEvents call.\n");
          }

          break;
      }  // end case IOCTL_DEREGISTER
      case IOCTL_REGISTER:
      {
          if (DBG_EVENTS)
          {
             printk(KERN_CRIT "ibmasm: DBG_EVENTS - received IOCTL_REGISTER request.\n");
          }

          // VERIFY THAT THIS PID ISN'T ALREADY REGISTERED
          spin_lock_irqsave( &(pDriverNode->EventQueueLock), flags );
            
          if (sysSpSeekQueue(pDriverNode->EventQueue, PID))
          {
             spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );
                
             if (DBG_EVENTS)
             {
                printk(KERN_CRIT "ibmasm: DBG_EVENTS - PID previously registered, rejecting request.\n");
             }
        
             pDDIoctlInfo->returnCode = DDERR_EVENT_ALREADY_REGISTERED;
        
             rc = 0;
        
             break;
          }
          else  // PID WAS NOT PREVIOUSLY REGISTERED, SO REGISTER IT
          {
             spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );
          }

          // ALLOCATE & INITIALIZE AN EVENT BLOCK
          if (DBG_EVENTS)
          {
             printk(KERN_CRIT "ibmasm: DBG_EVENTS - Registering PID %x for an event.\n", PID);
          }

          pBlock = NULL;
          pBlock = (PWORK_BLOCK)kmalloc((sizeof(WORK_BLOCK)), GFP_KERNEL);

          // ENSURE MEMORY ALLOCATION WAS SUCCESSFUL
          if (!pBlock)
          {
             if (DBG_IOCTL)
             {
                printk(KERN_CRIT "ibmasm DBG_EVENTS - Failed to allocate an event block. Exiting call.\n");
             }

             // EVENT BLOCK WAS UNABLE TO BE ALLOCATED
             pDDIoctlInfo->returnCode = DDERR_ZALLOC_EVENT;

             rc = 0;

             break;
          }

          memset(pBlock, 0, sizeof(WORK_BLOCK));

          // PREPARE EVENT BLOCK
          pBlock->PID            = PID;
          pBlock->pBuffer        = pDDIoctlInfo->pBuffer;
          pBlock->bufferLength   = pDDIoctlInfo->bufferLength;
          pBlock->completionCode = DDERR_EV_INITIATED;

          // REGISTER AND BLOCK FOR NEW EVENT
          sysSpRegisterEvent(pDriverNode, pBlock, &(pDDIoctlInfo->magicNumber));

          // SET THE RETURN CODE IN THE IOCtl PARM STRUCT
          pDDIoctlInfo->returnCode = pBlock->completionCode;

          // IF FAILED, BUT DeregisterForEvents WASN'T CALLED, PURGE THE EVENT QUEUE OF THIS PID'S NODE
          if ((pDDIoctlInfo->returnCode != DDERR_EVENT_CANCELLED) &&
              (pDDIoctlInfo->returnCode != DDERR_SUCCESS)          )
          {
             // PURGE ALL EVENTS ASSOCIATED WITH THIS PID
             spin_lock_irqsave( &(pDriverNode->EventQueueLock), flags );
                  
             if ( sysSpSeekQueue(pDriverNode->EventQueue, PID) )
             {
                sysSpPurgeQueueDeregisterEvents(pDriverNode, pDriverNode->EventQueue, PID);
             }

             spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );
          }

          switch (pDDIoctlInfo->returnCode)
          {
            case DDERR_EVENT_CANCELLED:
            {
               // DeregisterForEvents HAS BEEN CALLED
               if (DBG_EVENTS)
               {
                  printk(KERN_CRIT "ibmasm: DBG_EVENTS - DeregisterForEvents has been called.\n");
               }

               rc = 0;

               break;
            }
            case DDERR_EV_PENDING:
            case DDERR_EV_INITIATED:
            {
               // MOST LIKELY, THE CLIENT APP HAS CLOSED WHILE EVENTS ARE REGISTERED AND BLOCKED
               if (DBG_EVENTS)
               {
                  printk(KERN_CRIT "ibmasm: DBG_EVENTS - Event state = DDERR_EV_PENDING, event cancelled.\n");
               }

               pDDIoctlInfo->returnCode = DDERR_EVENT_CANCELLED;

               rc = 0;

               break;
            }
            case DDERR_INTERRUPTED:
            {
               // A SIGNAL HAS INTERRUPTED US
               if (DBG_EVENTS)
               {
                   printk(KERN_CRIT "ibmasm: DBG_EVENTS - A signal has interrupted a blocked event!\n");
               }

               rc = 0; 

               break;
            }

          }  // end switch

          // DEALLOCATE THE EVENT BLOCK
          if (pBlock)
          {
             kfree(pBlock);
             pBlock = NULL;
          }
         
          break;
      }  // end case IOCTL_REGISTER
      case IOCTL_RW:
      {  
          if (DBG_IOCTL && DBG_PHR)
          {
              printk(KERN_CRIT "ibmasm: DBG_IOCTL - Received a IOCTL_RW request. \n");
          }

          // ALLOCATE AND INITIALIZE AN IOCTL WORK BLOCK 
          pCmdBlk = NULL;
          pCmdBlk = (PWORK_BLOCK)kmalloc((sizeof(WORK_BLOCK)), GFP_KERNEL);

          if ( !pCmdBlk )
          {
             printk(KERN_CRIT "ibmasm: ERROR - Could not allocate IOCTL work block, rc = -ENOMEM. \n");
             pDDIoctlInfo->returnCode =  DDERR_ZALLOC_CMD_BUFFER; 
             break;
          }

          memset(pCmdBlk, 0, sizeof(WORK_BLOCK));

          // PREPARE THE COMMAND BLOCK AND COPY THE SP COMMAND INTO IT
          pCmdBlk->PID            = PID;
          pCmdBlk->bufferLength   = pDDIoctlInfo->bufferLength;
          pCmdBlk->completionCode = DDERR_CMD_INITIATED;
          pCmdBlk->pDDIoctlInfo   = pDDIoctlInfo;                       

          if (callerFlag == EXTERNAL_IOCTL) 
          {
             // ALLOCATE AND INITIALIZE A DATA BUFFER 
             pCmdBlk->pBuffer = NULL;
             pCmdBlk->pBuffer = (unsigned char *)kmalloc( pDDIoctlInfo->bufferLength, GFP_KERNEL);

             if ( !pCmdBlk->pBuffer )
             {
                printk(KERN_CRIT "ibmasm: ERROR - Could not allocate a data buffer, rc = -ENOMEM. \n");

                if (pCmdBlk)
                {
                    kfree(pCmdBlk);
                    pCmdBlk = NULL;
                }

                pDDIoctlInfo->returnCode =  DDERR_ZALLOC_CMD_BUFFER; 
                break;
             }

             memset(pCmdBlk->pBuffer, 0, pDDIoctlInfo->bufferLength);

             CopyRC = copy_from_user(pCmdBlk->pBuffer, pDDIoctlInfo->pBuffer, pCmdBlk->bufferLength);

             if ( CopyRC )   
             {
                printk(KERN_CRIT "ibmasm: ERROR - Could not copy calling applications data to the send buffer, rc = -EFAULT. \n"); 
                pDDIoctlInfo->returnCode = DDERR_ZALLOC_CMD_BUFFER; 

                if (pCmdBlk)
                {
                   if (pCmdBlk->pBuffer)
                   {
                      kfree(pCmdBlk->pBuffer);
                      pCmdBlk->pBuffer = NULL;
                   }

                   kfree(pCmdBlk);
                   pCmdBlk = NULL;
                }

                break;
             }
          }
          else  // INTERNAL IOCTL CALLS HAVE THEIR OWN DATA BUFFERS SO USE THAT ONE
          {
              pCmdBlk->pBuffer = pDDIoctlInfo->pBuffer;
          }

          // ADD THIS COMMAND TO THE COMMAND QUEUE - IF THE QUEUE WAS NON NULL (INDICATING THAT THERE
          // IS A COMMAND IN FRONT OF THIS ONE), WAIT FOR THIS COMMAND'S TURN TO RUN BY BLOCKING THIS
          // ROUTINE AND WAITING FOR sysSpRunSpCmd() TO KICK OFF THIS ROUTINE AGAIN. OTHERWISE, SEND
          // THE COMMAND TO THE SP IMMEDIATELY

          spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );

          if ( (sysSpEnqueue(pDriverNode, pDriverNode->CmdQueue, pCmdBlk) != NULL))
          { 
              spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

              // INITIALIZE THE WAIT QUEUE AND WAIT FOR THIS COMMANDS TURN TO RUN
              init_waitqueue_head(&(pCmdBlk->pWaitQueueRun));

              sleep_on(&(pCmdBlk->pWaitQueueRun));                 

              // CHECK TO SEE IF WE WERE UNBLOCKED BY A SIGNAL. IF SO, RETURN TO THE CALLER WITH 
              // DDERR_INITIATED_CMD_INTERRUPTED SO THE APP CAN DECIDE WHAT ACTION TO TAKE. 
              if ( signal_pending(current) )
              {

                 pDDIoctlInfo->returnCode = DDERR_INITIATED_CMD_INTERRUPTED;

                 if( DBG_IOCTL || DBG_SIGNALS )
                 {
                     printk(KERN_CRIT "uwsioctl: initiated RW_IOCTL was interrupted by a signal.\n"); 
                 }

                 // DEQUEUE THE COMMAND FROM THE COMMAND QUEUE
                 spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );
                 sysSpDequeue(pDriverNode, pDriverNode->CmdQueue);
                 spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

                 // FREE MEMORY
                 if (pCmdBlk)
                 {
                    if (callerFlag == EXTERNAL_IOCTL)
                    {
                       if (pCmdBlk->pBuffer)
                       {
                          kfree(pCmdBlk->pBuffer);
                          pCmdBlk->pBuffer = NULL;
                       }
                    }

                    kfree(pCmdBlk);
                    pCmdBlk = NULL;
                 }

                 return 0;

              }
              else
              {
                 spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );
              }
          }

          // RETRIEVE THE COMMAND FROM THE COMMAND QUEUE
          pCmdBlk = sysSpPeekQueue(pDriverNode->CmdQueue);

          spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

          if (pCmdBlk->completionCode == DDERR_CMD_INITIATED )
          {
             // I'M USING pCmdBuffer BECAUSE IT MAKES IT EAISER TO PORT, THIS WAY I CAN STILL CALL
             // THE IS_EXTRA_TIMEOUT_NEEDED MACRO WHICH REQUIRES pCmdBuffer 
             pCmdBuffer = (PCMD_BUFFER)pCmdBlk->pBuffer;

             timeoutValue = CMD_TIMEOUT_45; 

             // FOR CERTAIN COMMANDS (6.3.1, 7.1, or 8.x) USE A LONGER TIMEOUT VALUE INSTEAD
             if (IS_EXTRA_TIMEOUT_NEEDED)
             {
                timeoutValue = (ULONG)CMD_TIMEOUT_240;
             }

             if (DBG_IOCTL && DBG_PHR)
             {
                 printk(KERN_CRIT "ibmasm: DBG_IOCTL - Sending IOCTL request to the SP. \n");
             }

             // SEND THE IOCTL COMMAND REQUEST TO THE SP (ISA/PCI) 
             if ( pDriverNode->AsmType == ASMTYPE_WISEMAN ||
                  pDriverNode->AsmType == ASMTYPE_EAGLE   )
             {
                 sysSpProcessPciRequest(pDriverNode, pCmdBlk, timeoutValue);
             }
             #ifndef IA64     // PHR_180930
             else
             {  
                 sysSpProcessIsaRequest(pCmdBlk, timeoutValue);
             }
             #endif           // PHR_180930 

             if ( pCmdBlk->completionCode == DDERR_CMD_PENDING ) 
             {
                 // INITIALIZE THE WAIT QUEUE AND WAIT FOR THE COMMAND RESPONSE FROM THE SP
                 init_waitqueue_head(&(pCmdBlk->pWaitQueueRun));

                 sleep_on(&(pCmdBlk->pWaitQueueRun));                  


                 if (DBG_IOCTL && DBG_PHR)
                 {
                    printk(KERN_CRIT "ibmasm: DBG_IOCTL -  IOCTL_RW unblocked after command set to SP.\n"); 
                 }
                 
                 // CHECK TO SEE IF WE WERE UNBLOCKED BY A SIGNAL. IF SO, RETURN TO THE CALLER WITH 
                 // DDERR_PENDING_CMD_INTERRUPTED SO THE APP CAN DECIDE WHAT ACTION TO TAKE. IT 
                 // SHOULD RE-ISSUE THE COMMAND WITH THE CMD FLAG SET TO IOCTL_RW_PENDING
                 if ( signal_pending(current) )
                 {

                    pDDIoctlInfo->returnCode = DDERR_PENDING_CMD_INTERRUPTED;

                    if(DBG_SIGNALS || DBG_IOCTL)
                    {
                        printk(KERN_CRIT "ibmasm: DBG_IOCTL -  Pending RW_IOCTL was interrupted by a signal.\n"); 
                    }

                    // DEQUEUE THE COMMAND FROM THE COMMAND QUEUE
                    spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );
                    sysSpDequeue(pDriverNode, pDriverNode->CmdQueue);
                    spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

                    // FREE MEMORY
                    if (pCmdBlk)
                    {
                       if (callerFlag == EXTERNAL_IOCTL)
                       {
                          if (pCmdBlk->pBuffer)
                          {
                             kfree(pCmdBlk->pBuffer);
                             pCmdBlk->pBuffer = NULL;
                          }
                       }

                       kfree(pCmdBlk);
                       pCmdBlk = NULL;
                    }

                    return 0;
                 }                                               
             }

          }

          break;  // exit IOCTL_RW case
      }                                                             
      case IOCTL_DISCONNECT:
      {
          if (DBG_IOCTL )
          {
              printk(KERN_CRIT "ibmasm: DBG_IOCTL - Received a IOCTL_DISCONNECT request. \n");
          }

          // FIRST REMOVE PID FROM REGISTRATION TABLE
          for (i = 0; i < MAXPIDS; i++)
          {
             if ((pDriverNode->PIDtable[i].valid)      && 
                 (pDriverNode->PIDtable[i].PID == PID) )
             {
                 pDriverNode->PIDtable[i].valid = 0;
             }
          }

          // PURGE ALL COMMANDS ASSOCIATED WITH THIS PID
          spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );

          if ( sysSpSeekQueue(pDriverNode->CmdQueue, PID) )
          {
             sysSpPurgeQueueByPID(pDriverNode, pDriverNode->CmdQueue, PID);

             spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );
                 
             if (DBG_IOCTL)
             {
                printk(KERN_CRIT "ibmasm: DBG_IOCTL - Purging commands for PID %d.\n", PID);
             }
          }
          else // NOTHING TO PURGE, JUST UNLOCK THE COMMAND QUEUE
          {
             spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );
          }

          // PURGE ALL EVENTS ASSOCIATED WITH THIS PID
          spin_lock_irqsave( &(pDriverNode->EventQueueLock), flags );

          if ( sysSpSeekQueue(pDriverNode->EventQueue, PID) )
          {
             sysSpPurgeQueueDeregisterEvents(pDriverNode, pDriverNode->EventQueue, PID);

             spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );
                 
             if (DBG_IOCTL)
             {
                printk(KERN_CRIT "ibmasm: DBG_IOCTL - Purging events for PID %d.\n", PID);
             }
          }
          else    // NO EVENTS TO PURGE, JUST UNLOCK THE COMMAND QUEUE 
          {
             spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );
          }

          break;
      }   // end of case IOCTL_DISCONNECT
      case IOCTL_WRITE_DESKTOP_STATUS:
      {
          if ( pDriverNode->AsmType != ASMTYPE_EAGLE)
          {
             pDDIoctlInfo->returnCode = DDERR_UNKNOWN_IOCTL;
             break;
          }

          if (DBG_MOUSE || DBG_IOCTL)
          {
              printk(KERN_CRIT "DBG_MOUSE: ibmasm: DBG_MOUSE -  Received IOCTL_WRITE_DESKTOP_STATUS.\n");
          }

          Resolution = 
              READ_REGISTER_ULONG( (PULONG)(pDriverNode->pRemoteDataBaseAddr + CONDOR_MOUSE_ISR_CONTROL_OFFSET) );

          if ( pDDIoctlInfo->bufferLength )
          {
             if ( *pDDIoctlInfo->pBuffer > '0' ) 
             {
                if(DBG_MOUSE && DBG_PHR )
                {
                   printk(KERN_CRIT "ibmasm: DBG_MOUSE - Writing 1 to CONDOR_MOUSE_ISR_CONTROL_OFFSET.\n");
                }

                WRITE_REGISTER_ULONG( (PULONG)(pDriverNode->pRemoteDataBaseAddr + CONDOR_MOUSE_ISR_CONTROL_OFFSET), 1);

                if ( Resolution )
                {
                   pDDIoctlInfo->returnCode = DDERR_SUCCESS;
                }
                else
                {
                   pDDIoctlInfo->returnCode = DDERR_SEND_RESOLUTION;
                }

                rc = 1;
                    // END OR PHR_188142
             }
             else
             {
                if (DBG_MOUSE)
                {
                    printk(KERN_CRIT "ibmasm: DBG_MOUSE - Writing 0 to CONDOR_MOUSE_ISR_CONTROL_OFFSET.\n");
                }

                WRITE_REGISTER_ULONG(
                   (PULONG)(pDriverNode->pRemoteDataBaseAddr + CONDOR_MOUSE_ISR_CONTROL_OFFSET) , 0);

                pDDIoctlInfo->returnCode = DDERR_SUCCESS;
             }
          }
          else
          {
             printk(KERN_CRIT "ibmasm: ERROR sysSpProcessIoctl() IOCTL_WRITE_DESKTOP_STATUS was called with invalid buffer length.\n");
             pDDIoctlInfo->returnCode = DDERR_ZALLOC_CMD_BUFFER; 
             break;
          }

          sysSpRunSpCmd(pDriverNode);

          return rc;
          
      }  // end case IOCTL_WRITE_DESKTOP_STATUS
      case IOCTL_GET_INPUT_DATA: 
      {
          // Temporary pointer used to traverse thru the callers buffer
          PREMOTE_INPUT_DATA  TempPtr;

          // The following local set variables are used to reset the device drivers global remote data 
          // queue elements after they have been handled (sent to the calling app)
          ULONG SetRemoteInputCount = 0;
          PREMOTE_INPUT_DATA pSetRemoteDataOut;

          // The following local get variables are used to take a snapshot of the device drivers
          // global remote data queue before they get handled (sent to the calling app)
          ULONG GetRemoteInputCount = 0;
          PREMOTE_INPUT_DATA pGetRemoteDataIn;
          PREMOTE_INPUT_DATA pGetRemoteDataOut;

          if (DBG_MOUSE)
          {
              printk(KERN_CRIT "DBG_MOUSE: sysSpProcessIoctl() Received IOCTL_GET_INPUT_DATA.\n");
          }

          if ( pDriverNode->AsmType != ASMTYPE_EAGLE )
          {
             pDDIoctlInfo->returnCode = DDERR_UNKNOWN_IOCTL;
             break;
          }

          pDDIoctlInfo->returnCode = DDERR_SUCCESS;
          
          spin_lock_irqsave( &(pDriverNode->RemoteDataLock), flags );
          GetRemoteInputCount = pDriverNode->RemoteInputCount;
          spin_unlock_irqrestore( &(pDriverNode->RemoteDataLock), flags );

          // IF THERE IS NO REMOTE DATA AVAILABLE AT THIS TIME WAIT FOR SOME TO BE SENT BY THE SP
          if ( GetRemoteInputCount == 0 )
          {
              if(DBG_MOUSE)
                 printk(KERN_CRIT "DBG_MOUSE: IOCTL_GET_INPUT_DATA will block until rmt data is available.\n");

              // INITIALIZE THE WAIT QUEUE AND WAIT FOR THE COMMAND RESPONSE FROM THE SP
              init_waitqueue_head(&(pDriverNode->pRmtDataWaitQueue));

              interruptible_sleep_on(&(pDriverNode->pRmtDataWaitQueue));

              // CHECK TO SEE IF WE WERE UNBLOCKED BY A SIGNAL. IF SO, RETURN TO THE CALLER WITH 
              // DDERR_INTERRUPTED SO THE APP CAN DECIDE WHAT ACTION TO TAKE. IT SHOULD RE_ISSUE
              // THE COMMAND 
              if ( signal_pending(current) )
              {
                 pDDIoctlInfo->returnCode = DDERR_INTERRUPTED;
                 break;
              }
          }

          // TAKE A SNAPSHOT OF THE REMOTE DATA QUEUE 
          spin_lock_irqsave( &(pDriverNode->RemoteDataLock), flags );

          pGetRemoteDataIn    = pDriverNode->pRemoteDataIn;
          pGetRemoteDataOut   = pDriverNode->pRemoteDataOut;
          GetRemoteInputCount = pDriverNode->RemoteInputCount;

          spin_unlock_irqrestore( &(pDriverNode->RemoteDataLock), flags );

          SetRemoteInputCount = 0;
          
          pSetRemoteDataOut = pGetRemoteDataOut;

          // Initialize the temporary pointer to point to the start of the callers buffer
          TempPtr = (PREMOTE_INPUT_DATA)(pDDIoctlInfo->pBuffer);
          
          for ( i = 0; i < GetRemoteInputCount; i++ ) 
          {
              // copy a remote mouse/keyboard block from the device drivers memory to the callers buffer
              memcpy( TempPtr, pSetRemoteDataOut, sizeof(REMOTE_INPUT_DATA));
              
              SetRemoteInputCount += 1;

              // bump the pointer to the next remote mouse/keyboard block available from the device driver
              pSetRemoteDataOut++;

              // Check for Wrap before next output/read
              if (pSetRemoteDataOut == pDriverNode->pRemoteDataEnd)
              {
                  pSetRemoteDataOut = pDriverNode->pRemoteDataStart;
              }

              TempPtr++;
          }

          // UPDATE THE REMOTE DATA QUEUE TO INDICATE THAT QUEUED ITEMS HAVE BEEN HANDLED 
          spin_lock_irqsave( &(pDriverNode->RemoteDataLock), flags );

          pDriverNode->pRemoteDataOut    = pSetRemoteDataOut;   
          pDriverNode->RemoteInputCount -= SetRemoteInputCount; 

          spin_unlock_irqrestore( &(pDriverNode->RemoteDataLock), flags );

          // Let the caller know how many remote mouse/keboard elements are available 
          pDDIoctlInfo->bufferLength = SetRemoteInputCount;

          pDDIoctlInfo->returnCode = DDERR_SUCCESS; 
          
          break;
      }    // end case IOCTL_GET_INPUT_DATA
      case IOCTL_WRITE_DISPLAY_STATUS:
      {
           if ( pDriverNode->AsmType != ASMTYPE_EAGLE )
           {
              pDDIoctlInfo->returnCode = DDERR_UNKNOWN_IOCTL;
              break;
           }

           if(DBG_MOUSE) printk(KERN_CRIT "DBG_MOUSE: sysSpProcessIoctl() - Received IOCTL_WRITE_DISPLAY_STATUS.\n");

           if( pDDIoctlInfo->bufferLength )
           {
                // Let SP know the display state.
                WRITE_REGISTER_ULONG( (PULONG)(pDriverNode->pRemoteDataBaseAddr + CONDOR_INPUT_DISPLAY_RESX_OFFSET), 
                                      ((PDISPLAY_DATA)pDDIoctlInfo->pBuffer)->max_x  );

                WRITE_REGISTER_ULONG( (PULONG)(pDriverNode->pRemoteDataBaseAddr + CONDOR_INPUT_DISPLAY_RESY_OFFSET), 
                                      ((PDISPLAY_DATA)pDDIoctlInfo->pBuffer)->max_y  );

                WRITE_REGISTER_ULONG( (PULONG)(pDriverNode->pRemoteDataBaseAddr + CONDOR_INPUT_DISPLAY_BITS_OFFSET), 
                                      ((PDISPLAY_DATA)pDDIoctlInfo->pBuffer)->bits_per_pixel  );
           }
           else
           {
               printk(KERN_CRIT "ibmasm: ERROR sysSpProcessIoctl() IOCTL_WRITE_DISPLAY_STATUS was called with invalid buffer length.\n");
               pDDIoctlInfo->returnCode = DDERR_ZALLOC_CMD_BUFFER; 
               break;
           }

           WRITE_REGISTER_ULONG( (PULONG)(pDriverNode->pRemoteDataBaseAddr + CONDOR_MOUSE_ISR_CONTROL_OFFSET) , 1);

           pDDIoctlInfo->returnCode = DDERR_SUCCESS;
           break;
      }                                    //  end case IOCTL_WRITE_DISPLAY_STATUS
      case IOCTL_GET_VNC_STATUS: 
      {
           if ( pDriverNode->AsmType != ASMTYPE_EAGLE )
           {
              pDDIoctlInfo->returnCode = DDERR_UNKNOWN_IOCTL;
              break;
           }

           if(DBG_MOUSE) printk(KERN_CRIT "DBG_MOUSE: sysSpProcessIoctl() - Received IOCTL_GET_VNC_STATUS.\n");

           // PHR_180930  - need to change isconnected to u32
           ((PREMOTE_CONNECTION_DATA)pDDIoctlInfo->pBuffer)->isconnected = 
                   READ_REGISTER_ULONG( (PULONG)(pDriverNode->pRemoteDataBaseAddr + CONDOR_OUTPUT_VNC_STATUS_OFFSET) );

            pDDIoctlInfo->returnCode = DDERR_SUCCESS;
            break;
      }                                    // end of case IOCTL_GET_VNC_STATUS
      case IOCTL_SEND_REVERSE_HB:
      {  
           if (DBG_REV_HB)
           {
               printk(KERN_CRIT "DBG_REV_HB: enterered IOCTL_SEND_REVERSE_HB case. \n");
           }

           // ALLOCATE AND INITIALIZE AN IOCTL WORK BLOCK 
           pCmdBlk = NULL;
           pCmdBlk = (PWORK_BLOCK)kmalloc((sizeof(WORK_BLOCK)), GFP_KERNEL);

           if ( !pCmdBlk )
           {
              printk(KERN_CRIT "ibmasm: ERROR - Could not allocate IOCTL work block, rc = -ENOMEM. \n");
              pDDIoctlInfo->returnCode =  DDERR_ZALLOC_CMD_BUFFER; 
              return 0;
           }

           memset(pCmdBlk, 0, sizeof(WORK_BLOCK));

           // ALLOCATE AND INITIALIZE A DATA BUFFER, THEN COPY THE USERS DATA INTO IT 
           pCmdBlk->pBuffer = NULL;
           pCmdBlk->pBuffer = (unsigned char *)kmalloc( pDDIoctlInfo->bufferLength, GFP_KERNEL);

           if ( !pCmdBlk->pBuffer )
           {
              printk(KERN_CRIT "ibmasm: ERROR - Could not allocate a data buffer, rc = -ENOMEM. \n");

              if (pCmdBlk)
              {
                  kfree(pCmdBlk);
                  pCmdBlk = NULL;
              }

              pDDIoctlInfo->returnCode =  DDERR_ZALLOC_CMD_BUFFER; 
              return 0;
           }

           memset(pCmdBlk->pBuffer, 0, pDDIoctlInfo->bufferLength);

           CopyRC = copy_from_user(pCmdBlk->pBuffer, pDDIoctlInfo->pBuffer, pCmdBlk->bufferLength);

           if ( CopyRC )   // PHR_174969
           {
               printk(KERN_CRIT "ibmasm: ERROR - Could not copy calling applications data to the send buffer, rc = -EFAULT. \n"); 
               pDDIoctlInfo->returnCode = DDERR_ZALLOC_CMD_BUFFER; 
           }
           else
           {
               while ( TRUE )                                                   
               {                                                      
                  if (DBG_REV_HB)
                  {
                      printk(KERN_CRIT "DBG_REV_HB: IOCTL_SEND_REVERSE_HB is sending Are U There to the SP. \n");
                  }

                  // PREPARE THE COMMAND BLOCK
                  pCmdBlk->PID            = PID;
                  pCmdBlk->bufferLength   = pDDIoctlInfo->bufferLength;
                  pCmdBlk->completionCode = DDERR_CMD_INITIATED;
                  pCmdBlk->pDDIoctlInfo   = pDDIoctlInfo;                       

                  // ADD THIS COMMAND TO THE COMMAND QUEUE - IF THE QUEUE WAS NON NULL (INDICATING THAT THERE
                  // IS A COMMAND IN FRONT OF THIS ONE), WAIT FOR THIS COMMAND'S TURN TO RUN BY BLOCKING THIS
                  // ROUTINE AND WAITING FOR sysSpRunSpCmd() TO KICK OFF THIS ROUTINE AGAIN. OTHERWISE, SEND
                  // THE COMMAND TO THE SP IMMEDIATELY
                  spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );

                  if ( (sysSpEnqueue(pDriverNode, pDriverNode->CmdQueue, pCmdBlk) != NULL))
                  { 
                      spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

                      // INITIALIZE THE WAIT QUEUE AND WAIT FOR THIS COMMANDS TURN TO RUN
                      init_waitqueue_head(&(pCmdBlk->pWaitQueueRun));

                      //interruptible_sleep_on(&(pCmdBlk->pWaitQueueRun));    
                      sleep_on(&(pCmdBlk->pWaitQueueRun));                    

                      // CHECK TO SEE IF WE WERE UNBLOCKED BY A SIGNAL. IF SO, RETURN TO THE CALLER WITH 
                      // DDERR_INTERRUPTED SO THE APP CAN DECIDE WHAT ACTION TO TAKE. 
                      if ( signal_pending(current) )
                      {
                         // REMOVE THE COMMAND FROM THE COMMAND QUEUE    PHR_TODO
                         pDDIoctlInfo->returnCode = DDERR_INTERRUPTED;

                         if(DBG_REV_HB)
                         {
                             printk(KERN_CRIT "DBG_REV_HB: initiated Reverse HB command was interrupted by a signal.\n"); 
                         }

                         break;   // exit while

                         // END OF PHR_187769 
                      }
                      else
                      {
                         spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );
                      }
                  }

                  // RETRIEVE THE COMMAND FROM THE COMMAND QUEUE
                  pCmdBlk = sysSpPeekQueue(pDriverNode->CmdQueue);

                  spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

                  if (pCmdBlk->completionCode == DDERR_CMD_INITIATED )
                  {
                     pCmdBuffer = (PCMD_BUFFER)pCmdBlk->pBuffer;

                     timeoutValue = CMD_TIMEOUT_45; 

                     // SEND THE IOCTL COMMAND REQUEST TO THE SP (ISA/PCI) 
                     if ( pDriverNode->AsmType == ASMTYPE_WISEMAN ||
                          pDriverNode->AsmType == ASMTYPE_EAGLE   )
                     {
                         sysSpProcessPciRequest(pDriverNode, pCmdBlk, timeoutValue);
                     }
                     #ifndef IA64     
                     else
                     {  
                         sysSpProcessIsaRequest(pCmdBlk, timeoutValue);
                     }
                     #endif           

                     if ( pCmdBlk->completionCode == DDERR_CMD_PENDING ) 
                     {
                         // INITIALIZE THE WAIT QUEUE AND WAIT FOR THE COMMAND RESPONSE FROM THE SP
                         init_waitqueue_head(&(pCmdBlk->pWaitQueueRun));

                         sleep_on(&(pCmdBlk->pWaitQueueRun));                    

                         // CHECK TO SEE IF WE WERE UNBLOCKED BY A SIGNAL. IF SO, RETURN TO THE CALLER WITH 
                         // DDERR_INTERRUPTED SO THE APP CAN DECIDE WHAT ACTION TO TAKE.
                         if ( (signal_pending(current)) || (pCmdBlk->completionCode != DDERR_SUCCESS) ) 
                         {

                            // DEQUEUE THE COMMAND FROM THE COMMAND QUEUE
                            spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );
                            sysSpDequeue(pDriverNode, pDriverNode->CmdQueue);
                            spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

                            pDDIoctlInfo->returnCode = DDERR_INTERRUPTED;

                            if(DBG_REV_HB)
                            {
                                printk(KERN_CRIT "DBG_REV_HB: Pending Reverse Heart Beat was interrupted by a signal.\n"); 
                            }

                            break;   // exit while
                         }  
                     }

                     // DEQUEUE THE COMMAND FROM THE COMMAND QUEUE
                     spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );
                     sysSpDequeue(pDriverNode, pDriverNode->CmdQueue);
                     spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );
                  }

                  if ( pCmdBlk->completionCode != DDERR_SUCCESS )
                  {
                      pDDIoctlInfo->returnCode = pCmdBlk->completionCode;

                      RetryCount++ ;

                      if ( RetryCount == 2 )
                      {
                          break;   // exit while
                      }
                  }
                  else 
                  {
                      pDDIoctlInfo->returnCode = DDERR_SUCCESS;
                      RetryCount = 0;
                  }

                  // INITIALIZE INFO STRUCTURE FOR THE REVERSE HEART BEAT COMMAND TIMEOUT ROUTINE
                  pDriverNode->pActiveRHBBlk = pCmdBlk;    // PHR_MULTI 

                  // SET UP THE REVERSE HEART BEAT TIMER - TIMEOUT VALUE IS 5 SECONDS
                  pDriverNode->RevHBtimer.function = (void *)sysSpRevHbTimeout;
                  pDriverNode->RevHBtimer.data     = (unsigned long)pDriverNode;   // PHR_MULTI
                  pDriverNode->RevHBtimer.expires  = jiffies + 5 * HZ;

                  add_timer(&(pDriverNode->RevHBtimer));

                  // INITIALIZE THE WAIT QUEUE AND WAIT FOR THE REVERSE HEARTBEAT TIMEOUT
                  init_waitqueue_head(&(pCmdBlk->pWaitQueueRun));
                  interruptible_sleep_on(&(pCmdBlk->pWaitQueueRun));    // PHR_187769

                  // STOP THE REVERSE HEART BEAT TIMER
                  del_timer(&(pDriverNode->RevHBtimer));

                  if ( signal_pending(current) )
                  {
                     pDDIoctlInfo->returnCode = DDERR_INTERRUPTED;

                     if(DBG_REV_HB)
                     {
                         printk(KERN_CRIT "DBG_REV_HB: Idle IOCTL_SEND_REVERSE_HB was interrupted by a signal.\n"); 
                     }

                     break;
                  }     
               }      // ehd while
           }

           if ( pDDIoctlInfo->returnCode != DDERR_SUCCESS )
           {
               rc = 0;
           }
           else
           {
               rc = 1;
           }

           // FREE MEMORY
           if (pCmdBlk)
           {
              if (pCmdBlk->pBuffer)
              {
                 kfree(pCmdBlk->pBuffer);
                 pCmdBlk->pBuffer = NULL;
              }

              kfree(pCmdBlk);
              pCmdBlk = NULL;
           }

           sysSpRunSpCmd(pDriverNode);   

           if(DBG_REV_HB)
           {
               printk(KERN_CRIT "DBG_REV_HB: exiting case IOCTL_SEND_REVERSE_HB.\n"); 
           }

           return rc;
      }   // End case IOCTL_SEND_REVERSE_HB           
      default:  // INVALID IOCTL RECEIVED
      {
            if (DBG_IOCTL) printk(KERN_CRIT "sysSpProcessIoctl: Invalid IOCTL function!\n");
            pDDIoctlInfo->returnCode = DDERR_UNKNOWN_IOCTL;
            rc = 0;
            break;
      }
   }     // end switch - cmd

   if ( (cmd == IOCTL_GET_INPUT_DATA)  &&  pCmdBlk )
   {
      kfree(pCmdBlk);
      pCmdBlk = NULL;
   }

   // FINISH PROCESSING THE COMMAND REQUESTS THAT WERE SENT TO THE ADAPETER
   if( ((cmd == IOCTL_RW)              || 
        (cmd == IOCTL_RW_PENDING)      ||
        (cmd == IOCTL_RW_INITIATED))   &&  pCmdBlk )  
   {
      switch (pCmdBlk->completionCode)
      {
         case DDERR_SUCCESS:

           if (callerFlag == EXTERNAL_IOCTL) 
           {
               CopyRC = 0; 

               CopyRC = !access_ok(VERIFY_WRITE, pDDIoctlInfo->pBuffer, pCmdBlk->bufferLength ); 

               if (CopyRC)
               {
                  if (DBG_IOCTL)
                  {
                     printk(KERN_CRIT "ibmasm: DBG_IOCTL - Device returned more data than the callers buffer can handle. Ioctl request failed.\n");
                  }

                  rc = 0;
                  break;
               }

               // COPY THE COMMAND RESPONSE BACK TO THE CALLER
               CopyRC = copy_to_user(pDDIoctlInfo->pBuffer, pDriverNode->pSpResponseBuffer, pCmdBlk->bufferLength);

               if ( CopyRC )
               {
                   printk(KERN_CRIT "ibmasm: ERROR - Could not copy data to the calling applications buffer, rc = -EFAULT. \n"); 
                   pCmdBlk->completionCode  = DDERR_ZALLOC_CMD_BUFFER; 
                   pDDIoctlInfo->returnCode = DDERR_ZALLOC_CMD_BUFFER; 
                   rc = 0;
                   break;
               }        
           }

           // SET THE RETURN CODE FOR THE CALLER
           pDDIoctlInfo->returnCode = DDERR_SUCCESS;

           rc = 1;

           break;

         case DDERR_INITIATED_CMD_INTERRUPTED:       // PHR_174969

           // SET THE RETURN CODE FOR THE CALLER
           pDDIoctlInfo->returnCode = DDERR_INITIATED_CMD_INTERRUPTED;

           rc = 0;

           break;

         case DDERR_PENDING_CMD_INTERRUPTED:       // PHR_174969
          
           // SET THE RETURN CODE FOR THE CALLER
           pDDIoctlInfo->returnCode = DDERR_PENDING_CMD_INTERRUPTED;

           rc = 0;

           break;

         case DDERR_RCV_TIMEOUT:        // CURRENT COMMAND HAS TIMED OUT. DEQUEUE THE COMMAND 
         
           // SET THE RETURN CODE FOR THE CALLER
           pDDIoctlInfo->returnCode = DDERR_RCV_TIMEOUT;

           rc = 0;

           break;

         case DDERR_DRIVER_CLOSED:  // THIS COMMAND'S ASSOCIATED PROCESS HAS CLOSED THE DRIVER.
          
           // SET THE RETURN CODE FOR THE CALLER
           pDDIoctlInfo->returnCode = DDERR_DRIVER_CLOSED;

           rc = 0;

           break;
          
         default:

           printk(KERN_CRIT "ibmasm: ERROR - Unexpected completion Code = %i\n", pCmdBlk->completionCode );
            
      } // end switch

      if (pDDIoctlInfo->returnCode != DDERR_PENDING_CMD_INTERRUPTED)      // PHR_174969
      {

         // DEQUEUE THE COMMAND FROM THE COMMAND QUEUE
         spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );
         sysSpDequeue(pDriverNode, pDriverNode->CmdQueue);
         spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

         // FREE MEMORY
         if (pCmdBlk)
         {
            if (callerFlag == EXTERNAL_IOCTL)
            {
               if (pCmdBlk->pBuffer)
               {
                  kfree(pCmdBlk->pBuffer);
                  pCmdBlk->pBuffer = NULL;
               }
            }

            kfree(pCmdBlk);
            pCmdBlk = NULL;
         }
      }
   }   // end if ( cmd == ... )  

   if ( pDDIoctlInfo->returnCode != DDERR_SUCCESS )
   {
       if ( pDDIoctlInfo->returnCode != DDERR_SEND_RESOLUTION )     // PHR_188142
       {
           rc = 0;
       }
   }

   sysSpRunSpCmd(pDriverNode);
   
   return rc;
}         // end sysSpProcessIoctl()


#undef UwsIoctl_c


