/*
 * $Id: vethcmd.c,v 1.9 2004/06/19 12:42:37 yumo Exp $
 * vethcfg -- veth device driver configurator program
 *
 * Author: Yumo (Katsuyuki Yumoto) 2000-2004
 *         yumo@st.rim.or.jp
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <netinet/in.h>
#include <linux/if_ether.h>

#ifdef ETHMAP
#include <ethmap.h>
#endif

#include "veth.h"
#include "vethcfg.h"


/* internal error codes */
#define ERR_FAILCNF -1
#define ERR_FEWARGS -2
#define ERR_INVALPIF -3
#define ERR_INVALPNAM -4
#define ERR_INVALPTR -5
#define ERR_INVALPVAL -6
#define ERR_INVALSD -7
#define ERR_INVALVIF -8
#define ERR_NODRV -9
#define ERR_NOMEM -10
#define ERR_UNKNCOMM -11

/* maximum name length */
#define MAX_NAME 8

char *get_field_n(char *d, char *s, int n)
{
	char *p, *q, c;
	int i;

	p = strrchr (s, '\n');
	if (p){
		*p = '\0';
	}
	for (p = q = s, i = 1; p && *p;){
		for (; (*p == ' ' || *p == '\t') && *p; p++)
			;
		q = strchr(p, ' ');
		if (!q){
			q = strchr(p, '\t');
		}
		if (!q){
			q = strchr(p, '\0');
		}
		if (i == n){
			c = *q;
			*q = '\0';
			strcpy(d, p);
			*q = c;
			return d;
		}
		i++;
		p = q;
	}
	return 0;
}



char *get_module_name(char *d, char *ifname)
{
	FILE *fp;
	char astr[MAXLINE][MAXCOL+1];
	char s1[MAXCOL+1];
	char s2[MAXCOL+1];
	char s3[MAXCOL+1];
	char s4[MAXCOL+1];
	char s5[MAXCOL+1];
	char cmd[128];
	char *p, *q;
	int i = 0, j = 0, n;

	if (!ifname || strlen(ifname) > MAX_NAME)
		return NULL;

	/* first stage */
	sprintf(cmd, "/sbin/ifconfig %s 0 up", ifname);

	if (system(cmd)){
		return NULL;
	}

	if ((fp = popen("/bin/cat /proc/modules", "r")) == NULL){
		return NULL;
	}
	for (i = 0; fgets(astr[i]+1, MAXCOL, fp) && i < MAXLINE; i++){
		if (get_field_n(s1, astr[i]+1, 3)){
			if (!atoi(s1)){
				astr[i][0] = 1; /* exclusive mark */
			} else {
				astr[i][0] = 0;
			}
		}
	}
	pclose (fp);

	n = i;

	/* second stage */
	sprintf(cmd, "/sbin/ifconfig %s 0 down", ifname);

	if (system(cmd)){
		return NULL;
	}
	
	if ((fp = popen("/bin/cat /proc/modules", "r")) == NULL){
		return NULL;
	}
	for (; fgets(s1, MAXCOL, fp);){
		if (get_field_n(s2, s1, 1)){
			for (i = 0; i < n; i++){
				if (!astr[i][0] && get_field_n(s3, astr[i]+1, 1)){
					if (strcmp(s2, s3) == 0){
						p = get_field_n(s4, astr[i]+1, 3);
						q = get_field_n(s5, s1, 3);
						if (p && q && atoi(p) != atoi(q)+1){
							astr[i][0] = 1; /* exclusive mark */
						}
					}
				}
			}
		} else {
			break;
		}
	}
	pclose (fp);


	/* third stage */
	sprintf(cmd, "/sbin/ifconfig %s 0 up", ifname);

	if (system(cmd)){
		return NULL;
	}
	
	if ((fp = popen("/bin/cat /proc/modules", "r")) == NULL){
		return NULL;
	}
	for (j = 0; fgets(s1, MAXCOL, fp);){
		if (get_field_n(s2, s1, 1)){
			for (i = 0; i < n; i++){
				if (!astr[i][0] && get_field_n(s3, astr[i]+1, 1)){
					if (strcmp(s2, s3) == 0){
						p = get_field_n(s4, astr[i]+1, 3);
						q = get_field_n(s5, s1, 3);
						if (p && q && atoi(p) != atoi(q)){
							astr[i][0] = 1; /* exclusive mark */
						} else {
						    j++;
							strcpy(d, s2);
						}
					}
				}
			}
		} else {
			break;
		}
	}
	pclose (fp);


	/* check if single line is remaining */
	if (j != 1){
		return NULL;
	}
	return d;
}







int veth_config(int sock, struct ifreq *req)
{
	int ret;

	ret = ioctl(sock, SIOCDEVPRIVATE, req);
	if (ret < 0)
		perror(PROG_NAME);

	return ret;
}

void print_version(struct u_drv_info di)
{
	printf ("%s %d.%d.%d\n", di.name, di.major, di.minor, di.patch);
	printf ("Copyright 2001-2004 Yumo (Katsuyuki Yumoto)\n");
	printf ("\
This is free software; see the source for copying conditions. There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n");
	printf ("\n");
	return;
}

int version_match(struct u_drv_info di)
{
	return((di.major == VETH_VER_MAJOR) && (di.minor == VETH_VER_MINOR))?1:0;
}


void print_general_info(char *s)
{
	char sbuf[1024];
	int len;

	if (!s)
		return;

	strcpy(sbuf, s);
	len = strlen(sbuf);
	sprintf(sbuf+len, "%s version            : %d.%d.%d\n",
		   PROG_NAME, VETH_VER_MAJOR, VETH_VER_MINOR, VETH_VER_PATCH);
	len = strlen(sbuf);
	sprintf(sbuf+len,"Please type \"%s -h\" to show usage.\n\n", PROG_NAME);
	printf("%s", sbuf);
	return;
}

int does_driver_exists(int s, struct u_drv_info *di)
{
	struct ifreq req;
	struct vethconf vc;
	int ret = 1;

	if (s < 0 || !di){
		ret = 0;
		goto exit;
	}
	(struct vethconf *)req.ifr_data = &vc;
	vc.buf = (unsigned char *)di;
	vc.size = sizeof(struct u_drv_info);

	/* check if veth device driver exists... */
	strcpy(req.ifr_name, DEFAULT_VETH_NAME);
	vc.action = VETHCTL_VERSION;
	vc.magic = VETH_MAGIC;
	if (veth_config(s, &req))
		ret = 0;

	if (ret && strcmp(di->name, "VETH") != 0)
		ret = 0;

exit:
	return ret;
}


int get_agg_info(struct u_agg_info *u, int s, char *vif)
{
	struct ifreq req;
	struct vethconf vc;
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!u || !vif){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	(struct vethconf *)req.ifr_data = &vc;
	strcpy(req.ifr_name, vif);
	(struct u_agg_info *)vc.buf = u;
	
	vc.action = VETHCTL_GET_AGG_INFO;
	vc.magic = VETH_MAGIC;
	strcpy(vc.vif_name, vif);
	vc.size = sizeof(struct u_agg_info);
	ret = veth_config(s, &req);

exit:
	return ret;
}

int set_agg_info(struct u_agg_info *u, int s, char *vif)
{
	struct ifreq req;
	struct vethconf vc;
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!u || !vif){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	(struct vethconf *)req.ifr_data = &vc;
	strcpy(req.ifr_name, vif);
	(struct u_agg_info *)vc.buf = u;
	
	vc.action = VETHCTL_SET_AGG_INFO;
	vc.magic = VETH_MAGIC;
	strcpy(vc.vif_name, vif);
	vc.size = sizeof(struct u_agg_info);
	ret = veth_config(s, &req);

exit:
	return ret;
}

int get_agg_key(int s, char *vif)
{
	unsigned short key;
	struct u_agg_info u;
	int ret = 0;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!vif){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	if (get_agg_info(&u, s, vif)){
		ret = 0;
		goto exit;
	}

	key = u.actor_admin_agg_key;
	ret = key;

exit:
	return ret;
}

int get_port_info(struct u_port_info *u, int s, char *pif)
{
	struct ifreq req;
	struct vethconf vc;
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!u || !pif){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(pif) > MAX_NAME){
		ret = ERR_INVALPIF;
		goto exit;
	}

	(struct vethconf *)req.ifr_data = &vc;
	strcpy(req.ifr_name, DEFAULT_VETH_NAME);
	strcpy(vc.pif_name, pif);
	(struct u_port_info *)vc.buf = u;
	
	vc.action = VETHCTL_GET_PORT_INFO;
	vc.magic = VETH_MAGIC;
	vc.size = sizeof(struct u_port_info);
	if (veth_config(s, &req))
		goto exit;

	ret = 0;
exit:
	return ret;
}

int set_port_info(struct u_port_info *u, int s, char *pif)
{
	struct ifreq req;
	struct vethconf vc;
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!u || !pif){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(pif) > MAX_NAME){
		ret = ERR_INVALPIF;
		goto exit;
	}

	(struct vethconf *)req.ifr_data = &vc;
	strcpy(req.ifr_name, DEFAULT_VETH_NAME);
	strcpy(vc.pif_name, pif);
	(struct u_port_info *)vc.buf = u;
	
	vc.action = VETHCTL_SET_PORT_INFO;
	vc.magic = VETH_MAGIC;
	vc.size = sizeof(struct u_port_info);
	if (veth_config(s, &req))
		goto exit;
	ret = 0;

exit:
	return ret;
}

int add_phyif(int s, char *pif, struct u_port_info *di)
{
	struct ifreq req;
	struct vethconf vc;
	char str[MAXCOL];
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!di || !pif){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(pif) > MAX_NAME){
		ret = ERR_INVALPIF;
		goto exit;
	}

	(struct vethconf *)req.ifr_data = &vc;
	strcpy(req.ifr_name, DEFAULT_VETH_NAME);
	strcpy(vc.pif_name, pif);
	if (get_module_name(str, pif)){
		strcpy(vc.pif_module, str);
	} else {
		vc.pif_module[0] = '\0';
	}
	(struct u_port_info *)vc.buf = di;
	
	vc.action = VETHCTL_ADD_PHYIF;
	vc.magic = VETH_MAGIC;
	vc.size = sizeof(struct u_port_info);
	if (veth_config(s, &req))
		goto exit;

	ret = 0;

exit:
	return ret;
}


int del_phyif(int s, char *pif)
{
	struct ifreq req;
	struct vethconf vc;
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!pif){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(pif) > MAX_NAME){
		ret = ERR_INVALPIF;
		goto exit;
	}

	(struct vethconf *)req.ifr_data = &vc;
	strcpy(req.ifr_name, DEFAULT_VETH_NAME);
	strcpy(vc.pif_name, pif);
	vc.buf = NULL;
	
	vc.action = VETHCTL_DEL_PHYIF;
	vc.magic = VETH_MAGIC;
	vc.size = 0;
	if (veth_config(s, &req))
		goto exit;

	ret = 0;

exit:
	return ret;
}

char *decode_selected_val(unsigned long s)
{
	switch (s){
	case UNSELECTED:
		return "Unselected";
	case SELECTED:
		return "Selected";
	case STANDBY:
		return "Standby";
	default:
		return "Unknown";
	}
}

int print_port(struct u_port_info u, char *pif)
{
	int i;
	int ret = 0;

	if (!pif){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(pif) > MAX_NAME){
		ret = ERR_INVALPIF;
		goto exit;
	}

	printf("\nPort related information\n\n");

    printf("  Actor part\n");
    printf("                     Administrative       Operational\n");
	printf("  port number           %4d                  \n",
		   u.actor_port_num);
	printf("  port priority         0x%04x                \n",
		   u.actor_port_pri);
	printf("  aggregator id         %4d                  \n",
		   u.actor_port_agg_id);
	printf("  port key              0x%04x            0x%04x\n",
		   u.actor_admin_port_key, u.actor_oper_port_key);
	printf("  port state\n");
	printf("     LACP activety         %c                 %c\n",
		   u.actor_admin_port_state & PS_LACP_ACTIVITY?'1':'.',
		   u.actor_oper_port_state & PS_LACP_ACTIVITY?'1':'.');
	printf("     LACP timeout          %c                 %c\n",
		   u.actor_admin_port_state & PS_LACP_TIMEOUT?'1':'.',
		   u.actor_oper_port_state & PS_LACP_TIMEOUT?'1':'.');
	printf("     Aggregation           %c                 %c\n",
		   u.actor_admin_port_state & PS_AGGREGATION?'1':'.',
		   u.actor_oper_port_state & PS_AGGREGATION?'1':'.');
	printf("     Synchronization       %c                 %c\n",
		   u.actor_admin_port_state & PS_SYNCHRONIZATION?'1':'.',
		   u.actor_oper_port_state & PS_SYNCHRONIZATION?'1':'.');
	printf("     Collecting            %c                 %c\n",
		   u.actor_admin_port_state & PS_COLLECTING?'1':'.',
		   u.actor_oper_port_state & PS_COLLECTING?'1':'.');
	printf("     Distributing          %c                 %c\n",
		   u.actor_admin_port_state & PS_DISTRIBUTING?'1':'.',
		   u.actor_oper_port_state & PS_DISTRIBUTING?'1':'.');
	printf("     Defaulted             %c                 %c\n",
		   u.actor_admin_port_state & PS_DEFAULTED?'1':'.',
		   u.actor_oper_port_state & PS_DEFAULTED?'1':'.');
	printf("     Expired               %c                 %c\n",
		   u.actor_admin_port_state & PS_EXPIRED?'1':'.',
		   u.actor_oper_port_state & PS_EXPIRED?'1':'.');

	
    printf("\n\n  Partner part\n");
    printf("                      Administrative      Operational\n");
	printf("  system               ");
	for (i = 0; i < ETH_ALEN/2; i++)
		   printf("%02x", u.partner_admin_sys[i]);
	printf("-");
	for (i = ETH_ALEN/2; i < ETH_ALEN; i++)
		   printf("%02x", u.partner_admin_sys[i]);
	printf("     ");
	for (i = 0; i < ETH_ALEN/2; i++)
		   printf("%02x", u.partner_oper_sys[i]);
	printf("-");
	for (i = ETH_ALEN/2; i < ETH_ALEN; i++)
		   printf("%02x", u.partner_oper_sys[i]);
	printf("\n");

	printf("  system priority      0x%04x             0x%04x\n",
		   u.partner_admin_sys_pri, u.partner_oper_sys_pri);
	printf("  key                  0x%04x             0x%04x\n",
		   u.partner_admin_key, u.partner_oper_key);
	printf("  port number          %4d               %4d\n",
		   u.partner_admin_port_num, u.partner_oper_port_num);
	printf("  port priority        0x%04x             0x%04x\n",
		   u.partner_admin_port_pri, u.partner_oper_port_pri);

	printf("  port state\n");
	printf("     LACP activety        %c                  %c\n",
		   (u.partner_admin_port_state & PS_LACP_ACTIVITY)?'1':'.',
		   (u.partner_oper_port_state & PS_LACP_ACTIVITY)?'1':'.');
	printf("     LACP timeout         %c                  %c\n",
		   (u.partner_admin_port_state & PS_LACP_TIMEOUT)?'1':'.',
		   (u.partner_oper_port_state & PS_LACP_TIMEOUT)?'1':'.');
	printf("     Aggregation          %c                  %c\n",
		   u.partner_admin_port_state & PS_AGGREGATION?'1':'.',
		   u.partner_oper_port_state & PS_AGGREGATION?'1':'.');
	printf("     Synchronization      %c                  %c\n",
		   u.partner_admin_port_state & PS_SYNCHRONIZATION?'1':'.',
		   u.partner_oper_port_state & PS_SYNCHRONIZATION?'1':'.');
	printf("     Collecting           %c                  %c\n",
		   u.partner_admin_port_state & PS_COLLECTING?'1':'.',
		   u.partner_oper_port_state & PS_COLLECTING?'1':'.');
	printf("     Distributing         %c                  %c\n",
		   u.partner_admin_port_state & PS_DISTRIBUTING?'1':'.',
		   u.partner_oper_port_state & PS_DISTRIBUTING?'1':'.');
	printf("     Defaulted            %c                  %c\n",
		   u.partner_admin_port_state & PS_DEFAULTED?'1':'.',
		   u.partner_oper_port_state & PS_DEFAULTED?'1':'.');
	printf("     Expired              %c                  %c\n",
		   u.partner_admin_port_state & PS_EXPIRED?'1':'.',
		   u.partner_oper_port_state & PS_EXPIRED?'1':'.');

	printf("\n");
	printf("  Selected                               %s\n",
		   decode_selected_val(u.selected));

exit:
	return ret;
}

char *decode_distmode(unsigned short m)
{
	if (m & DIST_ROUNDROBIN){
		return "Round Robin";
	} else if (m & DIST_TCPUDPHASH){
		return "TCP/UDP DA/SA";
	} else {
		return "DA/SA";
	}
}

int print_port_line(char *line)
{
#ifdef ETHMAP
	char *s, *p, *q;
	char *sticky_intf;
	char c;
#endif

	if (!line)
		return ERR_INVALPTR;
	if (strlen(line) > 255)
		return ERR_INVALPNAM;
#ifdef ETHMAP
	s = line;
	printf("%c", *s++);
	p = s;
	s += 5;
	c = *s;
	*s = '\0';
	for (; *p == ' '; p++)
		;
	q = strdup(p);
	if (!q)
		return ERR_NOMEM;
	*s = c;
	sticky_intf = ethmap_reverse_query(q);
	if (!sticky_intf)
		printf("%s", q);
	else
		printf("[%s]", sticky_intf);
	free(q);
	printf("%s", s);
#else
	printf("%s", line);
#endif
	return 0;
}

int print_agg(struct u_agg_info u, char *vif)
{
	int i;
	int ret = 0;

	if (!vif){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	printf("\nAggregator related information\n\n");
    printf("  Actor part\n");
    printf("                           Administrative      Operational\n");
    printf("  Aggregator MAC address                       ");
	for (i = 0; i < ETH_ALEN/2; i++)
		   printf("%02x", u.agg_mac[i]);
	printf("-");
	for (i = ETH_ALEN/2; i < ETH_ALEN; i++)
		   printf("%02x", u.agg_mac[i]);
	printf("\n");

	printf("  Aggregator ID                                %4d\n",
		   u.agg_id);
	printf("  Aggregator key             0x%04x            0x%04x\n",
		   u.actor_admin_agg_key, u.actor_oper_agg_key);

    printf("\n  Partner part\n");
    printf("                           Administrative      Operational\n");
	printf("  System                                       ");
	for (i = 0; i < ETH_ALEN/2; i++)
		   printf("%02x", u.partner_sys[i]);
	printf("-");
	for (i = ETH_ALEN/2; i < ETH_ALEN; i++)
		   printf("%02x", u.partner_sys[i]);
	printf("\n");

	printf("  System priority                               0x%04x\n",
		   u.partner_sys_pri);
	printf("  Aggregator key                                0x%04x\n",
		   u.partner_oper_agg_key);

	printf("\n");
	printf("  Distributing Algorithm   %s\n",
		   decode_distmode(u.distmode));

exit:
	return ret;
}

int port_param_set(struct u_port_info *u, char *pnam, char *pval)
{
	int ret = 0;

	if (!u || !pnam || !pval){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(pnam) > MAX_NAME){
		ret = ERR_INVALPNAM;
		goto exit;
	}
	if (strlen(pval) > MAX_NAME){
		ret = ERR_INVALPVAL;
		goto exit;
	}

	if (strcmp(pnam, "akey") == 0){
		u->actor_admin_port_key = atoi(pval)&0xff00;
	} else if (strcmp(pnam, "aps_ac") == 0){
		atoi(pval)?(u->actor_admin_port_state|=PS_LACP_ACTIVITY):
			(u->actor_admin_port_state&=~PS_LACP_ACTIVITY);
	} else if (strcmp(pnam, "aps_to") == 0){
		atoi(pval)?(u->actor_admin_port_state|=PS_LACP_TIMEOUT):
			(u->actor_admin_port_state&=~PS_LACP_TIMEOUT);
	} else if (strcmp(pnam, "aps_ag") == 0){
		atoi(pval)?(u->actor_admin_port_state|=PS_AGGREGATION):
			(u->actor_admin_port_state&=~PS_AGGREGATION);
	} else if (strcmp(pnam, "aps_sy") == 0){
		atoi(pval)?(u->actor_admin_port_state|=PS_SYNCHRONIZATION):
			(u->actor_admin_port_state&=~PS_SYNCHRONIZATION);
	} else if (strcmp(pnam, "aps_co") == 0){
		atoi(pval)?(u->actor_admin_port_state|=PS_COLLECTING):
			(u->actor_admin_port_state&=~PS_COLLECTING);
	} else if (strcmp(pnam, "aps_di") == 0){
		atoi(pval)?(u->actor_admin_port_state|=PS_DISTRIBUTING):
			(u->actor_admin_port_state&=~PS_DISTRIBUTING);
	} else if (strcmp(pnam, "aps_de") == 0){
		atoi(pval)?(u->actor_admin_port_state|=PS_DEFAULTED):
			(u->actor_admin_port_state&=~PS_DEFAULTED);
	} else if (strcmp(pnam, "aps_ex") == 0){
		atoi(pval)?(u->actor_admin_port_state|=PS_EXPIRED):
			(u->actor_admin_port_state&=~PS_EXPIRED);

	} else if (strcmp(pnam, "psyspri") == 0){
		u->partner_admin_sys_pri = atoi(pval)&0xffff;
	} else if (strcmp(pnam, "pkey") == 0){
		u->partner_admin_key = atoi(pval)&0xffff;
	} else if (strcmp(pnam, "ppnum") == 0){
		u->partner_admin_port_num = atoi(pval)&0xffff;
	} else if (strcmp(pnam, "pppri") == 0){
		u->partner_admin_port_pri = atoi(pval)&0xffff;
	} else if (strcmp(pnam, "pps_ac") == 0){
		atoi(pval)?(u->partner_admin_port_state|=PS_LACP_ACTIVITY):
			(u->partner_admin_port_state&=~PS_LACP_ACTIVITY);
	} else if (strcmp(pnam, "pps_to") == 0){
		atoi(pval)?(u->partner_admin_port_state|=PS_LACP_TIMEOUT):
			(u->partner_admin_port_state&=~PS_LACP_TIMEOUT);
	} else if (strcmp(pnam, "pps_ag") == 0){
		atoi(pval)?(u->partner_admin_port_state|=PS_AGGREGATION):
			(u->partner_admin_port_state&=~PS_AGGREGATION);
	} else if (strcmp(pnam, "pps_sy") == 0){
		atoi(pval)?(u->partner_admin_port_state|=PS_SYNCHRONIZATION):
			(u->partner_admin_port_state&=~PS_SYNCHRONIZATION);
	} else if (strcmp(pnam, "pps_co") == 0){
		atoi(pval)?(u->partner_admin_port_state|=PS_COLLECTING):
			(u->partner_admin_port_state&=~PS_COLLECTING);
	} else if (strcmp(pnam, "pps_di") == 0){
		atoi(pval)?(u->partner_admin_port_state|=PS_DISTRIBUTING):
			(u->partner_admin_port_state&=~PS_DISTRIBUTING);
	} else if (strcmp(pnam, "pps_de") == 0){
		atoi(pval)?(u->partner_admin_port_state|=PS_DEFAULTED):
			(u->partner_admin_port_state&=~PS_DEFAULTED);
	} else if (strcmp(pnam, "pps_ex") == 0){
		atoi(pval)?(u->partner_admin_port_state|=PS_EXPIRED):
			(u->partner_admin_port_state&=~PS_EXPIRED);
	}

exit:
	return ret;
}

int agg_param_set(struct u_agg_info *u, char *pnam, char *pval)
{
	int ret = 0;

	if (!u || !pnam || !pval){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (strlen(pnam) > MAX_NAME){
		ret = ERR_INVALPNAM;
		goto exit;
	}
	if (strlen(pval) > MAX_NAME){
		ret = ERR_INVALPVAL;
		goto exit;
	}

	if (strcmp(pnam, "key") == 0){
		u->actor_admin_agg_key = atoi(pval)&0xff00;
	} else if (strcmp(pnam, "distalgo") == 0){
		if (strcmp(pval, "tcpudp") == 0){
			u->distmode = DIST_TCPUDPHASH;
		} else if (strcmp(pval, "rrobin") == 0){
			u->distmode = DIST_ROUNDROBIN;
		} else {
			u->distmode = 0; /* DA/SA */
		}
	}

exit:
	return ret;
}

void print_reason(int code)
{
	switch(code){
	case ERR_FAILCNF:
		fprintf(stderr, "%s: Fail to ioctl.\n", PROG_NAME);
		break;
	case ERR_FEWARGS:
		fprintf(stderr, "%s: Too few arguments.\n", PROG_NAME);
		break;
	case ERR_INVALPIF:
		fprintf(stderr, "%s: Invalid physical interface name\n", PROG_NAME);
		break;
	case ERR_INVALPNAM:
		fprintf(stderr, "%s: Invalid parameter name\n", PROG_NAME);
		break;
	case ERR_INVALPTR:
		fprintf(stderr, "%s: Invalid pointer\n", PROG_NAME);
		break;
	case ERR_INVALPVAL:
		fprintf(stderr, "%s: Invalid parameter value\n", PROG_NAME);
		break;
	case ERR_INVALSD:
		fprintf(stderr, "%s: Invalid socket id\n", PROG_NAME);
		break;
	case ERR_INVALVIF:
		fprintf(stderr, "%s: Invalid virtual interface name\n", PROG_NAME);
		break;
	case ERR_NODRV:
		fprintf(stderr, "%s: veth driver does not exists.\n", PROG_NAME);
		break;
	case ERR_NOMEM:
		fprintf(stderr, "%s: Cannot allocate memory.\n", PROG_NAME);
		break;
	case ERR_UNKNCOMM:
		fprintf(stderr, "%s: Unknown command\n", PROG_NAME);
		break;
	}
}

int veth_command(int argc, char **argv)
{
	int sock;
	struct u_drv_info di;
	int key;
	int ret = 0;

	struct ifreq req;
	struct vethconf vc;
	char str[MAXCOL];
#ifdef ETHMAP
	char *sticky_intf = NULL;
#endif


	/* a silly raw socket just for ioctl()ling it */
	sock = socket(AF_INET, SOCK_DGRAM, 0);
	if (sock < 0) {
		perror(PROG_NAME);
		print_general_info("");
		return 1;
	}

	if (!does_driver_exists(sock, &di)){
		print_general_info("veth device driver does not exists.\n");
		return 1;
	}

	if (!version_match(di)){
		sprintf(str, "Version mismatch between %s and veth driver.\nUpdate %s please.\n\n", PROG_NAME, PROG_NAME);
		print_general_info(str);
		printf("veth device driver version : %d.%d.%d\n", di.major, di.minor, di.patch);
		printf("%s version            : %d.%d.%d\n",
		   PROG_NAME, VETH_VER_MAJOR, VETH_VER_MINOR, VETH_VER_PATCH);
		return 1;
	}


	if (strcmp(argv[1], "add") == 0){
		struct u_port_info di;

		if (argc < 4){
			ret = ERR_FEWARGS;
			goto close_and_exit;
		}

		/* get aggregator's key */
		if (strncmp(argv[3], "veth", 4)){
			ret = ERR_INVALVIF;
			goto close_and_exit;
		}
		key = get_agg_key(sock, argv[3]);
		if (key <= 0){
			fprintf (stderr, "%s: Cannot get aggregator's key\n", PROG_NAME);
			ret = 1;
			goto close_and_exit;
		}

		/* set physical i/f name */
		di.actor_admin_port_key = key;
#ifdef ETHMAP
		sticky_intf = ethmap_query(argv[2]);
		if (!sticky_intf)
			ret = ERR_INVALPIF;
		else
			ret = add_phyif(sock, sticky_intf, &di);
#else
		ret = add_phyif(sock, argv[2], &di);
#endif
		
	} else if (strcmp(argv[1], "del") == 0){
		if (argc < 3){
			ret = ERR_FEWARGS;
			goto close_and_exit;
		}
#ifdef ETHMAP
		sticky_intf = ethmap_query(argv[2]);
		if (!sticky_intf)
			ret = ERR_INVALPIF;
		else
			ret = del_phyif(sock, sticky_intf);
#else
		ret = del_phyif(sock, argv[2]);
#endif
		
	} else if (strcmp(argv[1], "move") == 0){
		struct u_port_info u;

		if (argc < 4){
			ret = ERR_FEWARGS;
			goto close_and_exit;
		}
		
		/* get aggregator's key */
		if (strncmp(argv[3], "veth", 4)){
			ret = ERR_INVALVIF;
			goto close_and_exit;
		}
		ret = get_agg_key(sock, argv[3]);
		if (ret < 0)
			goto close_and_exit;
		if (!ret){
			fprintf (stderr, "%s: Cannot get aggregator's key\n", PROG_NAME);
			ret = 1;
			goto close_and_exit;
		}
		key = ret;

#ifdef ETHMAP
		sticky_intf = ethmap_query(argv[2]);
		if (!sticky_intf)
			ret = ERR_INVALPIF;
		else
			ret = get_port_info(&u, sock, sticky_intf);
#else
		ret = get_port_info(&u, sock, argv[2]);
#endif
		if (ret)
			goto close_and_exit;

		u.actor_admin_port_key = key;
#ifdef ETHMAP
		sticky_intf = ethmap_query(argv[2]);
		if (!sticky_intf)
			ret = ERR_INVALPIF;
		else
			ret = set_port_info(&u, sock, sticky_intf);
#else
		ret = set_port_info(&u, sock, argv[2]);
#endif
		
	} else if (strcmp(argv[1], "showport") == 0){
		struct u_port_info u;

		if (argc < 3){
			ret = ERR_FEWARGS;
			goto close_and_exit;
		}
		
#ifdef ETHMAP
		sticky_intf = ethmap_query(argv[2]);
		if (!sticky_intf)
			ret = ERR_INVALPIF;
		else
			ret = get_port_info(&u, sock, sticky_intf);
#else
		ret = get_port_info(&u, sock, argv[2]);
#endif
		if (ret)
			goto close_and_exit;

		ret = print_port(u, argv[2]);

	} else if (strcmp(argv[1], "showagg") == 0){
		struct u_agg_info u;

		if (argc < 3){
			ret = ERR_FEWARGS;
			goto close_and_exit;
		}
		
		if (strncmp(argv[2], "veth", 4)){
			ret = ERR_INVALVIF;
			goto close_and_exit;
		}

		ret = get_agg_info(&u, sock, argv[2]);
		if (ret)
			goto close_and_exit;

		ret = print_agg(u, argv[2]);

	} else if (strcmp(argv[1], "setport") == 0){
		struct u_port_info u;

		if (argc < 5){
			ret = ERR_FEWARGS;
			goto close_and_exit;
		}

#ifdef ETHMAP
		sticky_intf = ethmap_query(argv[2]);
		if (!sticky_intf)
			ret = ERR_INVALPIF;
		else
			ret = get_port_info(&u, sock, sticky_intf);
#else
		ret = get_port_info(&u, sock, argv[2]);
#endif
		if (ret)
			goto close_and_exit;

		ret = port_param_set(&u, argv[3], argv[4]);
		if (ret)
			goto close_and_exit;

#ifdef ETHMAP
		sticky_intf = ethmap_query(argv[2]);
		if (!sticky_intf)
			ret = ERR_INVALPIF;
		else
			ret = set_port_info(&u, sock, sticky_intf);
#else
		ret = set_port_info(&u, sock, argv[2]);
#endif

	} else if (strcmp(argv[1], "setagg") == 0){
		struct u_agg_info u;

		if (argc < 5){
			ret = ERR_FEWARGS;
			goto close_and_exit;
		}
		
		if (strncmp(argv[2], "veth", 4)){
			ret = ERR_INVALVIF;
			goto close_and_exit;
		}

		ret = get_agg_info(&u, sock, argv[2]);
		if (ret)
			goto close_and_exit;

		ret = agg_param_set(&u, argv[3], argv[4]);
		if (ret)
			goto close_and_exit;

		ret = set_agg_info(&u, sock, argv[2]);

	} else if (strcmp(argv[1], "list") == 0){
		FILE *fp = fopen("/proc/net/veth", "r");
		int i = 0;
		if (!fp){
			fprintf(stderr, "veth driver does not exist\n");
			ret = 1;
			goto close_and_exit;
		}
		for (; fgets(str, MAXCOL, fp); i++){
			if (!i)
				printf("%s", str);
			else {
				ret = print_port_line(str);
				if (ret)
					goto close_and_exit;
			}
		}
		ret = 0;

	} else if (strcmp(argv[1], "ver") == 0){
		print_version(di);
		ret = 0;

	} else {
		ret = ERR_UNKNCOMM;

	}

close_and_exit:

	if (ret < 0)
		print_reason(ret);
	
	if (ret){
		ret = 1;
	}
	close(sock);
	return ret;
}

/*
 * Local variables:
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 *  indent-tabs-mode: t
 * End:
 */
