/*
 * Exceptions for specific devices. Usually work-arounds for fatal design flaws.
 * Derived from fixup.c of i386 tree.
 */

#include <linux/pci.h>
#include <linux/init.h>
#include <linux/vgaarb.h>
#include <linux/screen_info.h>

#include <asm/machvec.h>

/*
 * Fixup to mark boot BIOS video selected by BIOS before it changes
 *
 * From information provided by "Jon Smirl" <jonsmirl@gmail.com>
 *
 * The standard boot ROM sequence for an x86 machine uses the BIOS
 * to select an initial video card for boot display. This boot video
 * card will have it's BIOS copied to C0000 in system RAM.
 * IORESOURCE_ROM_SHADOW is used to associate the boot video
 * card with this copy. On laptops this copy has to be used since
 * the main ROM may be compressed or combined with another image.
 * See pci_map_rom() for use of this flag. Before marking the device
 * with IORESOURCE_ROM_SHADOW check if a vga_default_device is already set
 * by either arch cde or vga-arbitration, if so only apply the fixup to this
 * already determined primary video card.
 */

static void pci_fixup_video(struct pci_dev *pdev)
{
	struct pci_dev *bridge;
	struct pci_bus *bus;
	u16 config;

	if ((strcmp(ia64_platform_name, "dig") != 0)
	    && (strcmp(ia64_platform_name, "hpzx1")  != 0))
		return;
	/* Maybe, this machine supports legacy memory map. */

	if (!vga_default_device()) {
		resource_size_t start, end;
		int i;

		/* Does firmware framebuffer belong to us? */
		for (i = 0; i < DEVICE_COUNT_RESOURCE; i++) {
			if (!(pci_resource_flags(pdev, i) & IORESOURCE_MEM))
				continue;

			start = pci_resource_start(pdev, i);
			end  = pci_resource_end(pdev, i);

			if (!start || !end)
				continue;

			if (screen_info.lfb_base >= start &&
			    (screen_info.lfb_base + screen_info.lfb_size) < end)
				vga_set_default_device(pdev);
		}
	}

	/* Is VGA routed to us? */
	bus = pdev->bus;
	while (bus) {
		bridge = bus->self;

		/*
		 * From information provided by
		 * "David Miller" <davem@davemloft.net>
		 * The bridge control register is valid for PCI header
		 * type BRIDGE, or CARDBUS. Host to PCI controllers use
		 * PCI header type NORMAL.
		 */
		if (bridge && (pci_is_bridge(bridge))) {
			pci_read_config_word(bridge, PCI_BRIDGE_CONTROL,
						&config);
			if (!(config & PCI_BRIDGE_CTL_VGA))
				return;
		}
		bus = bus->parent;
	}
	if (!vga_default_device() || pdev == vga_default_device()) {
		pci_read_config_word(pdev, PCI_COMMAND, &config);
		if (config & (PCI_COMMAND_IO | PCI_COMMAND_MEMORY)) {
			pdev->resource[PCI_ROM_RESOURCE].flags |= IORESOURCE_ROM_SHADOW;
			dev_printk(KERN_DEBUG, &pdev->dev, "Boot video device\n");
			vga_set_default_device(pdev);
		}
	}
}
DECLARE_PCI_FIXUP_CLASS_FINAL(PCI_ANY_ID, PCI_ANY_ID,
				PCI_CLASS_DISPLAY_VGA, 8, pci_fixup_video);
