/*
 * STMicroelectronics magnetometers driver
 *
 * Copyright 2012-2013 STMicroelectronics Inc.
 *
 * Denis Ciocca <denis.ciocca@st.com>
 *
 * Licensed under the GPL-2.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/mutex.h>
#include <linux/interrupt.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/irq.h>
#include <linux/delay.h>
#include <linux/iio/iio.h>
#include <linux/iio/sysfs.h>
#include <linux/iio/buffer.h>

#include <linux/iio/common/st_sensors.h>
#include "st_magn.h"

#define ST_MAGN_NUMBER_DATA_CHANNELS		3

/* DEFAULT VALUE FOR SENSORS */
#define ST_MAGN_DEFAULT_OUT_X_H_ADDR		0X03
#define ST_MAGN_DEFAULT_OUT_Y_H_ADDR		0X07
#define ST_MAGN_DEFAULT_OUT_Z_H_ADDR		0X05

/* FULLSCALE */
#define ST_MAGN_FS_AVL_1300MG			1300
#define ST_MAGN_FS_AVL_1900MG			1900
#define ST_MAGN_FS_AVL_2500MG			2500
#define ST_MAGN_FS_AVL_4000MG			4000
#define ST_MAGN_FS_AVL_4700MG			4700
#define ST_MAGN_FS_AVL_5600MG			5600
#define ST_MAGN_FS_AVL_8000MG			8000
#define ST_MAGN_FS_AVL_8100MG			8100
#define ST_MAGN_FS_AVL_12000MG			12000
#define ST_MAGN_FS_AVL_15000MG			15000
#define ST_MAGN_FS_AVL_16000MG			16000

/* Special L addresses for Sensor 2 */
#define ST_MAGN_2_OUT_X_L_ADDR			0x28
#define ST_MAGN_2_OUT_Y_L_ADDR			0x2a
#define ST_MAGN_2_OUT_Z_L_ADDR			0x2c

/* Special L addresses for sensor 3 */
#define ST_MAGN_3_OUT_X_L_ADDR			0x68
#define ST_MAGN_3_OUT_Y_L_ADDR			0x6a
#define ST_MAGN_3_OUT_Z_L_ADDR			0x6c

static const struct iio_chan_spec st_magn_16bit_channels[] = {
	ST_SENSORS_LSM_CHANNELS(IIO_MAGN,
			BIT(IIO_CHAN_INFO_RAW) | BIT(IIO_CHAN_INFO_SCALE),
			ST_SENSORS_SCAN_X, 1, IIO_MOD_X, 's', IIO_BE, 16, 16,
			ST_MAGN_DEFAULT_OUT_X_H_ADDR),
	ST_SENSORS_LSM_CHANNELS(IIO_MAGN,
			BIT(IIO_CHAN_INFO_RAW) | BIT(IIO_CHAN_INFO_SCALE),
			ST_SENSORS_SCAN_Y, 1, IIO_MOD_Y, 's', IIO_BE, 16, 16,
			ST_MAGN_DEFAULT_OUT_Y_H_ADDR),
	ST_SENSORS_LSM_CHANNELS(IIO_MAGN,
			BIT(IIO_CHAN_INFO_RAW) | BIT(IIO_CHAN_INFO_SCALE),
			ST_SENSORS_SCAN_Z, 1, IIO_MOD_Z, 's', IIO_BE, 16, 16,
			ST_MAGN_DEFAULT_OUT_Z_H_ADDR),
	IIO_CHAN_SOFT_TIMESTAMP(3)
};

static const struct iio_chan_spec st_magn_2_16bit_channels[] = {
	ST_SENSORS_LSM_CHANNELS(IIO_MAGN,
			BIT(IIO_CHAN_INFO_RAW) | BIT(IIO_CHAN_INFO_SCALE),
			ST_SENSORS_SCAN_X, 1, IIO_MOD_X, 's', IIO_LE, 16, 16,
			ST_MAGN_2_OUT_X_L_ADDR),
	ST_SENSORS_LSM_CHANNELS(IIO_MAGN,
			BIT(IIO_CHAN_INFO_RAW) | BIT(IIO_CHAN_INFO_SCALE),
			ST_SENSORS_SCAN_Y, 1, IIO_MOD_Y, 's', IIO_LE, 16, 16,
			ST_MAGN_2_OUT_Y_L_ADDR),
	ST_SENSORS_LSM_CHANNELS(IIO_MAGN,
			BIT(IIO_CHAN_INFO_RAW) | BIT(IIO_CHAN_INFO_SCALE),
			ST_SENSORS_SCAN_Z, 1, IIO_MOD_Z, 's', IIO_LE, 16, 16,
			ST_MAGN_2_OUT_Z_L_ADDR),
	IIO_CHAN_SOFT_TIMESTAMP(3)
};

static const struct iio_chan_spec st_magn_3_16bit_channels[] = {
	ST_SENSORS_LSM_CHANNELS(IIO_MAGN,
			BIT(IIO_CHAN_INFO_RAW) | BIT(IIO_CHAN_INFO_SCALE),
			ST_SENSORS_SCAN_X, 1, IIO_MOD_X, 's', IIO_LE, 16, 16,
			ST_MAGN_3_OUT_X_L_ADDR),
	ST_SENSORS_LSM_CHANNELS(IIO_MAGN,
			BIT(IIO_CHAN_INFO_RAW) | BIT(IIO_CHAN_INFO_SCALE),
			ST_SENSORS_SCAN_Y, 1, IIO_MOD_Y, 's', IIO_LE, 16, 16,
			ST_MAGN_3_OUT_Y_L_ADDR),
	ST_SENSORS_LSM_CHANNELS(IIO_MAGN,
			BIT(IIO_CHAN_INFO_RAW) | BIT(IIO_CHAN_INFO_SCALE),
			ST_SENSORS_SCAN_Z, 1, IIO_MOD_Z, 's', IIO_LE, 16, 16,
			ST_MAGN_3_OUT_Z_L_ADDR),
	IIO_CHAN_SOFT_TIMESTAMP(3)
};

static const struct st_sensor_settings st_magn_sensors_settings[] = {
	{
		.wai = 0, /* This sensor has no valid WhoAmI report 0 */
		.wai_addr = ST_SENSORS_DEFAULT_WAI_ADDRESS,
		.sensors_supported = {
			[0] = LSM303DLH_MAGN_DEV_NAME,
		},
		.ch = (struct iio_chan_spec *)st_magn_16bit_channels,
		.odr = {
			.addr = 0x00,
			.mask = 0x1c,
			.odr_avl = {
				{ .hz = 1, .value = 0x00 },
				{ .hz = 2, .value = 0x01 },
				{ .hz = 3, .value = 0x02 },
				{ .hz = 8, .value = 0x03 },
				{ .hz = 15, .value = 0x04 },
				{ .hz = 30, .value = 0x05 },
				{ .hz = 75, .value = 0x06 },
				/* 220 Hz, 0x07 reportedly exist */
			},
		},
		.pw = {
			.addr = 0x02,
			.mask = 0x03,
			.value_on = 0x00,
			.value_off = 0x03,
		},
		.fs = {
			.addr = 0x01,
			.mask = 0xe0,
			.fs_avl = {
				[0] = {
					.num = ST_MAGN_FS_AVL_1300MG,
					.value = 0x01,
					.gain = 1100,
					.gain2 = 980,
				},
				[1] = {
					.num = ST_MAGN_FS_AVL_1900MG,
					.value = 0x02,
					.gain = 855,
					.gain2 = 760,
				},
				[2] = {
					.num = ST_MAGN_FS_AVL_2500MG,
					.value = 0x03,
					.gain = 670,
					.gain2 = 600,
				},
				[3] = {
					.num = ST_MAGN_FS_AVL_4000MG,
					.value = 0x04,
					.gain = 450,
					.gain2 = 400,
				},
				[4] = {
					.num = ST_MAGN_FS_AVL_4700MG,
					.value = 0x05,
					.gain = 400,
					.gain2 = 355,
				},
				[5] = {
					.num = ST_MAGN_FS_AVL_5600MG,
					.value = 0x06,
					.gain = 330,
					.gain2 = 295,
				},
				[6] = {
					.num = ST_MAGN_FS_AVL_8100MG,
					.value = 0x07,
					.gain = 230,
					.gain2 = 205,
				},
			},
		},
		.multi_read_bit = false,
		.bootime = 2,
	},
	{
		.wai = 0x3c,
		.wai_addr = ST_SENSORS_DEFAULT_WAI_ADDRESS,
		.sensors_supported = {
			[0] = LSM303DLHC_MAGN_DEV_NAME,
			[1] = LSM303DLM_MAGN_DEV_NAME,
		},
		.ch = (struct iio_chan_spec *)st_magn_16bit_channels,
		.odr = {
			.addr = 0x00,
			.mask = 0x1c,
			.odr_avl = {
				{ .hz = 1, .value = 0x00 },
				{ .hz = 2, .value = 0x01 },
				{ .hz = 3, .value = 0x02 },
				{ .hz = 8, .value = 0x03 },
				{ .hz = 15, .value = 0x04 },
				{ .hz = 30, .value = 0x05 },
				{ .hz = 75, .value = 0x06 },
				{ .hz = 220, .value = 0x07 },
			},
		},
		.pw = {
			.addr = 0x02,
			.mask = 0x03,
			.value_on = 0x00,
			.value_off = 0x03,
		},
		.fs = {
			.addr = 0x01,
			.mask = 0xe0,
			.fs_avl = {
				[0] = {
					.num = ST_MAGN_FS_AVL_1300MG,
					.value = 0x01,
					.gain = 909,
					.gain2 = 1020,
				},
				[1] = {
					.num = ST_MAGN_FS_AVL_1900MG,
					.value = 0x02,
					.gain = 1169,
					.gain2 = 1315,
				},
				[2] = {
					.num = ST_MAGN_FS_AVL_2500MG,
					.value = 0x03,
					.gain = 1492,
					.gain2 = 1666,
				},
				[3] = {
					.num = ST_MAGN_FS_AVL_4000MG,
					.value = 0x04,
					.gain = 2222,
					.gain2 = 2500,
				},
				[4] = {
					.num = ST_MAGN_FS_AVL_4700MG,
					.value = 0x05,
					.gain = 2500,
					.gain2 = 2816,
				},
				[5] = {
					.num = ST_MAGN_FS_AVL_5600MG,
					.value = 0x06,
					.gain = 3030,
					.gain2 = 3389,
				},
				[6] = {
					.num = ST_MAGN_FS_AVL_8100MG,
					.value = 0x07,
					.gain = 4347,
					.gain2 = 4878,
				},
			},
		},
		.multi_read_bit = false,
		.bootime = 2,
	},
	{
		.wai = 0x3d,
		.wai_addr = ST_SENSORS_DEFAULT_WAI_ADDRESS,
		.sensors_supported = {
			[0] = LIS3MDL_MAGN_DEV_NAME,
		},
		.ch = (struct iio_chan_spec *)st_magn_2_16bit_channels,
		.odr = {
			.addr = 0x20,
			.mask = 0x1c,
			.odr_avl = {
				{ .hz = 1, .value = 0x00 },
				{ .hz = 2, .value = 0x01 },
				{ .hz = 3, .value = 0x02 },
				{ .hz = 5, .value = 0x03 },
				{ .hz = 10, .value = 0x04 },
				{ .hz = 20, .value = 0x05 },
				{ .hz = 40, .value = 0x06 },
				{ .hz = 80, .value = 0x07 },
			},
		},
		.pw = {
			.addr = 0x22,
			.mask = 0x03,
			.value_on = 0x00,
			.value_off = 0x03,
		},
		.fs = {
			.addr = 0x21,
			.mask = 0x60,
			.fs_avl = {
				[0] = {
					.num = ST_MAGN_FS_AVL_4000MG,
					.value = 0x00,
					.gain = 146,
				},
				[1] = {
					.num = ST_MAGN_FS_AVL_8000MG,
					.value = 0x01,
					.gain = 292,
				},
				[2] = {
					.num = ST_MAGN_FS_AVL_12000MG,
					.value = 0x02,
					.gain = 438,
				},
				[3] = {
					.num = ST_MAGN_FS_AVL_16000MG,
					.value = 0x03,
					.gain = 584,
				},
			},
		},
		.drdy_irq = {
			/* drdy line is routed drdy pin */
			.stat_drdy = {
				.addr = ST_SENSORS_DEFAULT_STAT_ADDR,
				.mask = 0x07,
			},
		},
		.multi_read_bit = true,
		.bootime = 2,
	},
	{
		.wai = 0x40,
		.wai_addr = 0x4f,
		.sensors_supported = {
			[0] = LSM303AGR_MAGN_DEV_NAME,
			[1] = LIS2MDL_MAGN_DEV_NAME,
		},
		.ch = (struct iio_chan_spec *)st_magn_3_16bit_channels,
		.odr = {
			.addr = 0x60,
			.mask = 0x0c,
			.odr_avl = {
				{ .hz = 10, .value = 0x00 },
				{ .hz = 20, .value = 0x01 },
				{ .hz = 50, .value = 0x02 },
				{ .hz = 100, .value = 0x03 },
			},
		},
		.pw = {
			.addr = 0x60,
			.mask = 0x03,
			.value_on = 0x00,
			.value_off = 0x03,
		},
		.fs = {
			.fs_avl = {
				[0] = {
					.num = ST_MAGN_FS_AVL_15000MG,
					.gain = 1500,
				},
			},
		},
		.bdu = {
			.addr = 0x62,
			.mask = 0x10,
		},
		.drdy_irq = {
			.addr = 0x62,
			.mask_int1 = 0x01,
			.stat_drdy = {
				.addr = 0x67,
				.mask = 0x07,
			},
		},
		.multi_read_bit = false,
		.bootime = 2,
	},
};

static int st_magn_read_raw(struct iio_dev *indio_dev,
			struct iio_chan_spec const *ch, int *val,
							int *val2, long mask)
{
	int err;
	struct st_sensor_data *mdata = iio_priv(indio_dev);

	switch (mask) {
	case IIO_CHAN_INFO_RAW:
		err = st_sensors_read_info_raw(indio_dev, ch, val);
		if (err < 0)
			goto read_error;

		return IIO_VAL_INT;
	case IIO_CHAN_INFO_SCALE:
		*val = 0;
		if ((ch->scan_index == ST_SENSORS_SCAN_Z) &&
					(mdata->current_fullscale->gain2 != 0))
			*val2 = mdata->current_fullscale->gain2;
		else
			*val2 = mdata->current_fullscale->gain;
		return IIO_VAL_INT_PLUS_MICRO;
	case IIO_CHAN_INFO_SAMP_FREQ:
		*val = mdata->odr;
		return IIO_VAL_INT;
	default:
		return -EINVAL;
	}

read_error:
	return err;
}

static int st_magn_write_raw(struct iio_dev *indio_dev,
		struct iio_chan_spec const *chan, int val, int val2, long mask)
{
	int err;

	switch (mask) {
	case IIO_CHAN_INFO_SCALE:
		err = st_sensors_set_fullscale_by_gain(indio_dev, val2);
		break;
	case IIO_CHAN_INFO_SAMP_FREQ:
		if (val2)
			return -EINVAL;
		mutex_lock(&indio_dev->mlock);
		err = st_sensors_set_odr(indio_dev, val);
		mutex_unlock(&indio_dev->mlock);
		return err;
	default:
		err = -EINVAL;
	}

	return err;
}

static ST_SENSORS_DEV_ATTR_SAMP_FREQ_AVAIL();
static ST_SENSORS_DEV_ATTR_SCALE_AVAIL(in_magn_scale_available);

static struct attribute *st_magn_attributes[] = {
	&iio_dev_attr_sampling_frequency_available.dev_attr.attr,
	&iio_dev_attr_in_magn_scale_available.dev_attr.attr,
	NULL,
};

static const struct attribute_group st_magn_attribute_group = {
	.attrs = st_magn_attributes,
};

static const struct iio_info magn_info = {
	.driver_module = THIS_MODULE,
	.attrs = &st_magn_attribute_group,
	.read_raw = &st_magn_read_raw,
	.write_raw = &st_magn_write_raw,
	.debugfs_reg_access = &st_sensors_debugfs_reg_access,
};

#ifdef CONFIG_IIO_TRIGGER
static const struct iio_trigger_ops st_magn_trigger_ops = {
	.owner = THIS_MODULE,
	.set_trigger_state = ST_MAGN_TRIGGER_SET_STATE,
	.validate_device = st_sensors_validate_device,
};
#define ST_MAGN_TRIGGER_OPS (&st_magn_trigger_ops)
#else
#define ST_MAGN_TRIGGER_OPS NULL
#endif

int st_magn_common_probe(struct iio_dev *indio_dev)
{
	struct st_sensor_data *mdata = iio_priv(indio_dev);
	int irq = mdata->get_irq_data_ready(indio_dev);
	int err;

	indio_dev->modes = INDIO_DIRECT_MODE;
	indio_dev->info = &magn_info;
	mutex_init(&mdata->tb.buf_lock);

	err = st_sensors_power_enable(indio_dev);
	if (err)
		return err;

	err = st_sensors_check_device_support(indio_dev,
					ARRAY_SIZE(st_magn_sensors_settings),
					st_magn_sensors_settings);
	if (err < 0)
		goto st_magn_power_off;

	mdata->num_data_channels = ST_MAGN_NUMBER_DATA_CHANNELS;
	mdata->multiread_bit = mdata->sensor_settings->multi_read_bit;
	indio_dev->channels = mdata->sensor_settings->ch;
	indio_dev->num_channels = ST_SENSORS_NUMBER_ALL_CHANNELS;

	mdata->current_fullscale = (struct st_sensor_fullscale_avl *)
					&mdata->sensor_settings->fs.fs_avl[0];
	mdata->odr = mdata->sensor_settings->odr.odr_avl[0].hz;

	err = st_sensors_init_sensor(indio_dev, NULL);
	if (err < 0)
		goto st_magn_power_off;

	err = st_magn_allocate_ring(indio_dev);
	if (err < 0)
		goto st_magn_power_off;

	if (irq > 0) {
		err = st_sensors_allocate_trigger(indio_dev,
						ST_MAGN_TRIGGER_OPS);
		if (err < 0)
			goto st_magn_probe_trigger_error;
	}

	err = iio_device_register(indio_dev);
	if (err)
		goto st_magn_device_register_error;

	dev_info(&indio_dev->dev, "registered magnetometer %s\n",
		 indio_dev->name);

	return 0;

st_magn_device_register_error:
	if (irq > 0)
		st_sensors_deallocate_trigger(indio_dev);
st_magn_probe_trigger_error:
	st_magn_deallocate_ring(indio_dev);
st_magn_power_off:
	st_sensors_power_disable(indio_dev);

	return err;
}
EXPORT_SYMBOL(st_magn_common_probe);

void st_magn_common_remove(struct iio_dev *indio_dev)
{
	struct st_sensor_data *mdata = iio_priv(indio_dev);

	st_sensors_power_disable(indio_dev);

	iio_device_unregister(indio_dev);
	if (mdata->get_irq_data_ready(indio_dev) > 0)
		st_sensors_deallocate_trigger(indio_dev);

	st_magn_deallocate_ring(indio_dev);
}
EXPORT_SYMBOL(st_magn_common_remove);

MODULE_AUTHOR("Denis Ciocca <denis.ciocca@st.com>");
MODULE_DESCRIPTION("STMicroelectronics magnetometers driver");
MODULE_LICENSE("GPL v2");
