/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */

#include "dbviewmanager.h"
#include "../../utils/debugger.h"
#include "../../utils/stringutils.h"

	/**
 	 * Constructor
	 */
  DBViewManager::DBViewManager()
  	: DBBaseManager()
  {
  } // end constructor
	
	/**
 	 * Constructor
 	 * It is assumed that the database connection object will remain alive during
 	 * the life of this object. Be very careful to ensure that no methods are called
 	 * on this object if the connection object no longer exists. If the connection
 	 * object has already been destroyed then unpredictable results will be returned. 	
   */		
  DBViewManager::DBViewManager(DBConnection *poDBConn)
  	: DBBaseManager(poDBConn)
  {
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBViewManager::~DBViewManager()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to retrieve the list of views for this connection.
	 * @return	a DBViewSet object that contains the view details
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if not connected
	 */
	void DBViewManager::retrieveListOfViews(DBViewSet & roDBViewSet) throw (SQLException, DBConnectionException)
	{
		string strMethodName = "DBViewManager::retrieveListOfViews";
		
		Debugger::entered(strMethodName);
		
		// sql statement used to retrieve all the view details
		string strSQL = "SELECT v.viewname, v.viewowner, v.definition, d.description as comment "
										"FROM pg_views v, pg_class c, pg_description d "
										"WHERE v.viewname = c.relname "
										"AND c.oid = d.objoid "
										"AND v.viewname not like 'pg\\_%' "										
										"UNION ALL "
										"SELECT v.viewname, v.viewowner, v.definition, NULL as comment "
										"FROM pg_views v, pg_class c "
										"WHERE v.viewname = c.relname "
										"AND c.oid NOT IN (select objoid from pg_description) "
										"AND v.viewname not like 'pg\\_%' "
										"ORDER BY viewname";
										
		// execute query										
		m_poDBConn->executeQuery(strSQL, roDBViewSet.m_oViewList);										
		
		Debugger::exited(strMethodName);
		
	} // end retrieveListOfTables
	
	/**
	 * Used to modify the view comment.
	 * @param			rstrViewName		the name of the view
	 * @param			rstrComment			the comment to set for the view
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBViewManager::modifyViewComment(const string & rstrViewName, const string & rstrComment)
				throw (SQLException, DBConnectionException)
	{
		string strSQLComment;
 		if (rstrComment == "")
 		{
 			strSQLComment = "COMMENT ON VIEW " + rstrViewName + " IS NULL";
 		} // end if need to drop comment
 		else
 		{
 			strSQLComment = "COMMENT ON VIEW " + rstrViewName +
 												" IS '" + StringUtils::databasestr(rstrComment) + "'";			
 		} // end else need to set new comment
		// set comment
		m_poDBConn->execute(strSQLComment);												
	
	} // end modifyViewComment
