/*
*	rmgrLib.c - the replication manager communication library (wb)
*
*	Functions provideded in this file are used by backends
*	to communicate with the replication manager
*
*/

#include <string.h>
// DAR HACK this is needed for SUN#include <sys/filio.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include "postgres.h"
#include "miscadmin.h"
#include "replication/replication.h"


/*
*	RmgrConnect
*
*	connects to the ReplicaMgr and sends hostid/xid as 
*	identification.
*	
*	param:
*		rmgrSock - the socket to the rmgr (on return)
*		ReplicaMgrPort - well-known rmgr port number
*		hostid - host id identifying the replicated txn
*		xid - transaction id identifying the repl. txn
*	return:
*		int - STATUS_OK for success, STAUS_ERROR for failure
*/

int
RmgrConnect(bufsockptr rmgrSock, int ReplicaMgrPort, int hostid, int procid, int xid)
{	

	int flag = 1;
		rc_msg_t	msg_type;
	
	
	if(!rmgrSock)
	{
		printf("no socket\n");
		return STATUS_ERROR;
	}
	
	/* bk: now only the first time, a connection is established, 
	 * this connection is maintained throughout the lifetime of the process
	 * if the connection is already there we clean it up so that we do
	 * not read any old data
	*/
	/* DAR HACK This needs to be added to the proc struct
		in proc.h 
	if (MyProc->connection_state == CONNECTED)
	{
	*/
	  if(RmgrReceiveMsg(&msg_type, rmgrSock, FALSE) == STATUS_ERROR)
	  {
		RmgrDisconnect(MyRmgrSock);
		elog(ERROR, "Replication protocol error in clearing buffer transaction mgmt. module");
	  } 
	  /*elog(NOTICE,"MSG_NEW_TXN send xid %d", xid);*/
	  /* send the msg telling the host-id */
	  RmgrSendTxnId(rmgrSock, procid, xid); 
	  return STATUS_OK;
	/* DAR HACK This needs to be added to the proc struct
	        in proc.h

	}
	*/
	  
	  
	if(sockClientConnect(NULL, ReplicaMgrPort, rmgrSock) != STATUS_OK)
			 proc_exit(100);
	/*elog(NOTICE,"I have connected");	*/
	sockPutInt((int) MSG_OPENING, 4, rmgrSock);
	sockPutInt(12, 4, rmgrSock);
	sockPutInt(hostid, 4, rmgrSock);
	sockPutInt(procid, 4, rmgrSock);
	sockPutInt(xid, 4, rmgrSock);
	/*elog(NOTICE,"i have put on socket");*/
	
	sockFlush(rmgrSock);
	/*elog(NOTICE,"i have flushed socket");*/
	
	

	if(ioctl(rmgrSock->sock, FIONBIO, &flag)<0)
		return STATUS_ERROR;
	
	/* DAR HACK This needs to be added to the proc struct
		in proc.h 
	MyProc->connection_state = CONNECTED;
	*/
	
	return STATUS_OK;
}

/*
*	RmgrDisconnect
*
*	disconnects from the ReplicaMgr
*	
*	param:
*		rmgrSock - the socket to the rmgr 
*	return:
*		int - STATUS_OK for success, STAUS_ERROR for failure
*/
int
RmgrDisconnect(bufsockptr rmgrSock)
{
	/* DAR HACK This needs to be added to the proc struct
		in proc.h 
	MyProc->state = CLOSED;
	*/
	sockClose(rmgrSock);
	
	return STATUS_OK;
}

/*
*	RmgrSendWriteSet
*
*	Sends a write set to the rmgr. 
*	
*	param:
*		ws - pointer to the write set structure
*		rmgrSock - the socket to the rmgr 
*	return:
*		int - STATUS_OK for success, STAUS_ERROR for failure
*/
int
RmgrSendWriteSet(WriteSetPtr ws, bufsockptr rmgrSock)
{
	ebuf extbuf;
	
	sockInitE(&extbuf, WS_BUF_SIZE);
	
	WriteSetSerialize(ws, &extbuf);
	if((sockPutInt((int) MSG_WRITESET, 4, rmgrSock) != STATUS_OK) ||
	   (sockPutInt(sockGetCountE(&extbuf), 4, rmgrSock) != STATUS_OK))
		proc_exit(100);
	
	sockFlushE(&extbuf, rmgrSock, TRUE);
	
	/* DAR HACK This needs to be added to the proc struct
		in proc.h 
	MyProc->state = SEND_PHASE;
	*/
	
	sockDestroyE(&extbuf);
	//elog(NOTICE,"write set completely sent to rmgr\n");
	return STATUS_OK;
}

/*
*	RmgrReceiveWriteSet
*
*	Receives a write set from the rmgr. 
*	
*	param:
*		ws - pointer to the write set structure
*		rmgrSock - the socket to the rmgr 
*	return:
*		int - STATUS_OK for success, STAUS_ERROR for failure
*/
int
RmgrReceiveWriteSet(WriteSetPtr ws, bufsockptr rmgrSock)
{
	int 		nread = 0,
				datalen = 0;
	rc_msg_t	msg_type;
	
	do{
		sockWait(TRUE, FALSE, rmgrSock);
		nread += sockReadData(rmgrSock, RMGR_HDR_SIZE - nread, TRUE);
	}while(nread < RMGR_HDR_SIZE); 
	sockGetInt((int *) &msg_type, 4, rmgrSock);
	sockGetInt(&datalen, 4, rmgrSock);
	//elog(NOTICE,"read first part");	
	nread = 0;
	do{
		sockWait(TRUE, FALSE, rmgrSock);
		nread += sockReadData(rmgrSock, datalen - nread, TRUE);
	}while(nread < datalen);
	
	Assert(msg_type == MSG_WRITESET);
	WriteSetUnserialize(ws, rmgrSock);
	return STATUS_OK;
}

/*
*	RmgrSendMsg
*
*	Sends a msg (no data appended) to the rmgr.
*	
*	param:
*		msg - the message to send
*		rmgrSock - the socket to the rmgr
*		waitForAck - do we wait for MSG_ACK?
*			ATTN: recv buffer should be empty!
*	return:
*		int - STATUS_OK for success, STAUS_ERROR for failure
*/
int
RmgrSendMsg(rc_msg_t msg, bufsockptr rmgrSock, bool waitForAck)
{
	rc_msg_t ack;
	int		 datalen = -1;
	
	//elog(NOTICE, "[%d]Sending msg %d to rmgr", MyProcPid, (int)msg);
	if(!rmgrSock)
		return STATUS_ERROR;
	if((sockPutInt((int) msg, 4, rmgrSock) != STATUS_OK) ||
	   (sockPutInt(0, 4, rmgrSock) != STATUS_OK))
		proc_exit(100);
	sockFlush(rmgrSock);
	if(waitForAck){
		sockWait(TRUE, FALSE, rmgrSock);
		if((sockGetInt((int *)&ack, 4, rmgrSock) != STATUS_OK) ||
		   (sockGetInt(&datalen, 4, rmgrSock) != STATUS_OK))
			proc_exit(100);
		sockConsume(rmgrSock);
		if(ack != MSG_ACK){
			sockClose(rmgrSock);
			proc_exit(101);
		}
	}
	return STATUS_OK;
}

/*
*	RmgrReceiveMsg
*
*	Receives a msg (no data appended) from the rmgr.
*	
*	param:
*		msg - the received message
*		rmgrSock - the socket to the rmgr
*		blockingRead - do we block if no data available?
*			ATTN: recv buffer should be empty!
*	return:
*		int - STATUS_OK for success, STAUS_ERROR for failure
*/
int
RmgrReceiveMsg(rc_msg_t *msg, bufsockptr rmgrSock, bool blockingRead)
{
	int 		nread = 0,
				datalen = 0;
	
	do{
		if(blockingRead)
			sockWait(TRUE, FALSE, rmgrSock);
		nread += sockReadData(rmgrSock, RMGR_HDR_SIZE - nread , TRUE);
		/* bk: if we are non-blocking, we only want to test whether something
		 * is already there, in this case we read it, otherwise we
		 * do not read anything */
		if(!blockingRead && nread == 0)
		{
			return STATUS_OK;
		}	
	}while(nread < RMGR_HDR_SIZE);
	sockGetInt((int *) msg, 4, rmgrSock);
	sockGetInt(&datalen, 4, rmgrSock);
	sockConsume(rmgrSock);
	if(*msg == MSG_PROTO_ERROR)
	{
		sockClose(rmgrSock);
		return STATUS_ERROR;
	}

	return STATUS_OK;
}

int
RmgrSendTxnId(bufsockptr rmgrSock, int procid, int xid)
{


	if(!rmgrSock)
		return STATUS_ERROR;
		/* bk: put msg type */
	if((sockPutInt((int) MSG_NEW_TXN, 4, rmgrSock) != STATUS_OK) ||
	   /* bk: length of the following is 8 bytes */
	   (sockPutInt(8, 4, rmgrSock) != STATUS_OK) ||
		(sockPutInt(procid, 4, rmgrSock) != STATUS_OK) ||
		(sockPutInt(xid, 4, rmgrSock) != STATUS_OK))
		proc_exit(100);
	
	sockFlush(rmgrSock);

	return STATUS_OK;
}

/*
*	RmgrInitConn
* 
*	initializes the connection socket
*
*	param:
*		rmgrSock - ptr to ptr to the buffered socket to
*		initialize
*	return:
*		void
*/
void
RmgrInitConn(bufsockptr *rmgrSock)
{	
	printf("Rmgr Init Conn");
	*rmgrSock = sockInit(AF_UNIX_BUFSIZE);
}

/*
*	RmgrdestroyConn
* 
*	destroys connection socket
*
*	param:
*		rmgrSock - ptr to the buffered socket to
*		free
*	return:
*		void
*/
void
RmgrDestroyConn(int dummy, bufsockptr rmgrSock)
{
	/* bk: now first send msg to replicamanager */
	elog(NOTICE,"i send closing message");
	RmgrSendMsg(MSG_CLOSING, MyRmgrSock, TRUE);
	sockDestroy(rmgrSock);
}

