//
//  PostgreSQL.h
//  PostgresSQL
//
//  Created by Pascal on Sat Dec 14 2002.
//  Copyright (c) 2002 P3 Consulting. All rights reserved.
//

#ifndef	DEMO_MODE
#define	DEMO_MODE	0
#endif

#import <Foundation/Foundation.h>
#import "PgSQLBindingProtocol.h"
#import "PgSQLResultSetProtocol.h"
#import "PgSQLListener.h"
#import "libpq-fe.h"
#import "libpq/libpq-fs.h"
#import "pgtypes.h"

#if	DEMO_MODE
#define		MAX_RESULTS	50
#else
#define		MAX_RESULTS	-1
#endif


/*!
@header PostgreSQL.h

	@discussion	
 P3 Consulting, 2002-2003 All Rights Reserved<br><br>
 This is a Cocoa-wrapper around the libcpg functions for connecting to a PostgreSQL database.
 
 As some of you will notice most of the following API is designed to be 
 - as far as possible - source code compatible with OpenBase Basic API.<br>
 This is due to the simple fact that I also used OpenBase and want to migrate as easely 
 as possible some of my own code to PostgreSQL. 
 <br>
 In order to support <code>uniqueRowIdForTable</code> functionnality, you have to had a 
	<br><code>_rowid BIGSERIAL NOT NULL UNIQUE PRIMARY KEY</code><br>
 field to every table you plan to use with <code>uniqueRowIdForTable</code>.
 <br><br>
 <em>Differences with OpenBase API:</em><br>
<blockquote>  	
- <code>markRow</code> and <code>removeMarkOnRow</code> are not implemented - <br>
		LOCKING strategy is not impemented that way in PostgreSQL
		please refer to PostgreSQL doc and use executeCommand to
		achieve desired functionality.
	<br><br>
	- <code>databaseEncodingName</code> is not implemented
<br><br>	
	- notification mechanism is different <br>
		we use <code>LISTEN</code> (and <code>UNLISTEN</code> to disable a notification)
		to register a PgSQLListener object that will be called
		by an NSTimer at a user specified interval, if a PostgreSQL
		notification is detected at that time, the pgSQLNotify method
		(defined in PgSQLListener protocol) of a user specified object 
		will be called.
<br><br> 
	- large object support has different - but close - API<br>
	-	Oid are used instead of char[] to hold large object reference<br>
	-	more functionnality is available including reading and 
		writing part of large object.
</blockquote><br><br>		
<table width="100%" border="2" cellspacing="5" cellpadding="5"><tr><td bordercolor="#999999"><div align="LEFT"><strong>DEMO 
WARNING<br> </strong>demonstration version limited to:<br> no more 
than 50 records are returned from any query<br> methods affected are<br><code>nextRow</code>: 
returns false when 50th record is reached<br><code>afterLastRow</code>: position cursor on 
50th record<br><code>resultAsDictionary</code>: only the first 50 rows are returned<br> <br> 
Be careful: rowsAffected method returns the REAL result count of the originating query.<br>  
<br> Methods not implemented:<br> <code>beginTransaction, endTransaction, 
rollbackTransaction</code><br> </div></td></tr></table><br> <table width="100%" border="2" cellspacing="5" cellpadding="5"><tr><td><div align="LEFT"><strong>PostgreSQL 
limitation:<br> </strong>resultTableName is not implemented (returns <code>"NOT_SUPPORTED_BY_POSTGRESQL"</code>)<br> 
I don't have found yet a way to returns the name of the originating table of a query's resulting column...<br>
If anybody has an idea how to do it, please let me know.
</div></td></tr></table><br>
  		 
	<br><hr><br>	
	<strong>Methods implemented by the delegate</strong><br>
	-(void)postgreSQLAsyncResultAvailable:(PostgreSQL *)inConn;<br>
	-(void)postgreSQLAsyncResulTimedOut:(PostgreSQL *)inConn;<br>
	-(void)postgreSQL:(PostgreSQL *)inConn notification:(NSDictionary *)inDict;
	<br><br>
	<strong>Notifications sent</strong><br>
	<code>PostgreSQLNotification</code> when a NOTIFY is received from backend.
	<code>PostgreSQLResultAvailable</code> when result from an asynchronous query is available.
	<code>PostgreSQLTimeOut</code> when a timeout occured on an asynchronous query. 
	
*/

@class	PgASyncThread, PgSQLResult, NSWindow	;


/*!
@const	PgSQLColumnsKeyField
@discussion key for columns array
*/
extern	NSString	*PgSQLColumnsKeyField	;
/*!
@const	PgSQLNameKeyField 
@discussion key for column's name field (string)<br>
*/
extern	NSString	*PgSQLNameKeyField		;
/*!
@const	PgSQLTypeKeyField 
@discussion key for column's type field (integer)<br>
*/
extern	NSString	*PgSQLTypeKeyField		;
/*!
@const	PgSQLTypeNameKeyField 
@discussion key for column's type field (integer)<br>
*/
extern	NSString	*PgSQLTypeNameKeyField	;
/*!
@const	PgSQLRowsKeyField 
@discussion key for fields array
 */
extern	NSString	*PgSQLRowsKeyField	;
/*!
@const	PostgreSQLNotification
 */
extern	NSString *PostgreSQLNotification	;
/*!
@const	PostgreSQLResultAvailable
 */
extern	NSString *PostgreSQLResultAvailable	;
/*!
@const	PostgreSQLTimeOut
 */
extern	NSString *PostgreSQLTimeOut	;

/*!
@const	PgSQLuserInfoNotificationField
 */
extern	NSString *PgSQLuserInfoNotificationField	;
/*!
@const	PgSQLconditionNotificationField
 */
extern	NSString *PgSQLconditionNotificationField	;
/*!
@const	PgSQLconnectionNotificationField
 */
extern	NSString *PgSQLconnectionNotificationField	;


/*!
   @class PostgreSQL
   @abstract The wrapper class for PosgreSQL connections.
   Implements NSCoding,PgSQLBindingProtocol,PgSQLResultSetProtocol protocols.<br>
*/

@interface PostgreSQL : NSObject<NSCoding,PgSQLBindingProtocol,PgSQLResultSetProtocol> {
    PGconn			*_conn;
	BOOL			_debugMode;
	BOOL			_commandExecuted;
	BOOL			_asyncCommandInProgress;
	BOOL			_asyncCommandResultAvailable;
	FILE			*_traceFile	;
	PGresult 		*_lastResult	;
	int				_curRowIndex	;	
	int				_rowsInResult	;
	int				_fieldsInResult	;	
	int				_transactionLevel	;		
	NSMutableString	*_cmdBuffer	;
	NSMutableArray	*_bindingArray	;
	NSMutableDictionary	*_listeners		;
	NSTimer			*_listeningTimer	;
	PgASyncThread	*_asyncThread	;
	NSConnection	*_asyncConnection;
	long			_timeOut			;
	NSString		*_uniqueKeyID	;
	id				_delegate		;
	NSDictionary	*_localUSDict	;
	NSMutableArray	*_results	;
}

/*! 
   @method setDebug
   @abstract turns on|off global debugging. 
   @discussion Newly allocated PostgreSQL objects have their instance debug flag set the the global one.
   @param yn  as you expect.
*/
+ (void)setDebug:(BOOL)yn;

/*! 
   @method setDebugMode
   @abstract turns on|off instance debugging. 
   @param turnOn  as you expect.
*/
- (void)setDebugMode:(BOOL)turnOn;
/*! 
   @method debugMode
   @result current debug mode of the connection. 
*/
- (BOOL)debugMode;

/*! 
   @method startTracing
   @abstract start tracing to file traceFileName. 
   @param traceFileName full path name of file to dump tracing information.
   @result YES if successfull
*/
-(BOOL)startTracing:(const char *)traceFileName;

/*! 
   @method stopTracing
   @abstract turns off tracing. 
*/
-(void)stopTracing;

/*! 
   @method connectToDatabase
   @abstract specify parameters to start PostgreSQL database connection 
   @discussion The wrapper is disconnected from any opened database before establishing a new one.<br>
   Passing nil or empty string to any of the parameters will fail.<br>
   Use <code>(BOOL)connectToDatabase:(const char *)connectInfo return:(int *)returnCode</code> if you need to pass different arguments like options or tty or if you want to use less arguments and let PostgreSQL use default ones.
   @param dbName	databasename as defined in PostgreSQL.
   @param hostName	network name of host computer
   @param loginName user name for login
   @param password user password for login
   @param returnCode where to return int result code for connection
   @result YES if connection successful No otherwise
*/
- (BOOL)connectToDatabase:(const char *)dbName onHost:(const char *)hostName login:(const char *)loginName password:(const char *)password return:(int *)returnCode;


/*! 
	@method connectToDatabase
	@abstract specify parameters string to start PostgreSQL database connection. See PostgreSQL doc for more details on param string.
	@param connectInfo character buffer holding the conenction string to be used.
	@param return pointer an integer buffer o hold result code from backend.
	@result YES if connection successfull, NO otherwise. 
*/
- (BOOL)connectToDatabase:(const char *)connectInfo return:(int *)returnCode;
/*! 
   @method connectAskingUser
   @abstract present a dialog to let the user fill in fields to connect to a database.
   @discussion In case of error, you can get the reason of the trouble by calling <code>connectErrorMessage</code>.
   @param return: pointer an integer buffer to hold result code from backend.
   @param forWindow: if nil a login window will be presented, if not nil a panel attached to the specified window will be presented.
   @result YES if connection successfull, NO otherwise. 
*/
- (BOOL)connectAskingUser:(int *)returnCode forWindow:(NSWindow *)hostWindow;
/*! 
   @method connectAskingUserWithDefaults
   @abstract present a dialog to let the user fill in fields to connect to a database.
   @discussion In case of error, you can get the reason of the trouble by calling <code>connectErrorMessage</code>.
   @param defaults: is a dictionary with following non mandatory keys<br>
   <code>host, port, dbname, user, password, options </code>
   @param return: pointer an integer buffer to hold result code from backend.
   @param forWindow: if nil a login window will be presented, if not nil a panel attached to the specified window will be presented.
   @result YES if connection successfull, NO otherwise. 
*/
- (BOOL)connectAskingUserWithDefaults:(NSDictionary *)defaults return:(int *)returnCode forWindow:(NSWindow *)hostWindow;

/*!
   @method getVariables
   @discussion returned dictionary is cached: further calls wiil always return the same dictionary, even if Postgre's ones were changed. Use <code>loadVariables</code> to force reloading.
   @result returns a dictionary of Postgre variables ("SHOW ALL" 's result).
*/
-(NSDictionary *)getVariables;
/*!
   @method loadVariables
   @discussion reconstruced the "SHOW ALL" dictionary (always in sync with backend).
   @result returns a dictionary of Postgre variables ("SHOW ALL" 's result).
*/
-(NSDictionary *)loadVariables;
/*!
   @method showClientEncoding
   @discussion use internally dictionary returned by getVariables.
   @result returns the client encoding converted into NSStringEncoding.
*/
- (NSStringEncoding) showClientEncoding;
/*!
   @method showServerEncoding
   @discussion use internally dictionary returned by getVariables.
   @result returns the server encoding converted into NSStringEncoding.
*/
- (NSStringEncoding) showServerEncoding;

/*!
   @method uniqueID
   @abstract if you need to insert the PostgreSQL object in a NSDictionary, <br>
	this is a convenience method that is guaranted to return a unique string usable as key object.
	Uniqueness is guaranted for the lifetime of the program duration.
   @result	unique id string.
*/
-(NSString *)uniqueID;

/*! 
   @method postgreSQLVersion
   @abstract returns version of the running postmaster. 
   @discussion returns reulst of executing the <code>SELECT version()</code> query.
   @result	NSString containing the postgreSQL version.
*/
- (NSString *)postgreSQLVersion;

/*! 
   @method frameworkVersion
   @abstract returns version of the framework. 
   @result	NSString describing the framework version.
*/
- (NSString *)frameworkVersion;
/*! 
   @method databaseName
   @abstract returns database name of the connection. 
   @result	pointer on database name string buffer.
*/
-(const char *)databaseName;
/*! 
   @method hostName
   @abstract returns host name of the connection. 
   @result	pointer on host name string buffer.
*/
-(const char *)hostName;
/*! 
   @method loginName
   @abstract returns user name of the connection. 
   @result	pointer on user name string buffer.
*/
-(const char *)loginName;
/*! 
   @method password
   @abstract returns password of the connection. 
   @result	pointer on password string buffer.
*/
-(const char *)password;
/*! 
   @method port
   @abstract returns port of the connection. 
   @result	pointer on port string buffer.
*/
-(const char *)port;
/*! 
   @method tty
   @abstract returns tty of the connection. 
   @result	pointer on tty string buffer.
*/
-(const char *)tty;
/*! 
   @method options
   @abstract returns options of the connection. 
   @result	pointer on options string buffer.
*/
-(const char *)options;
/*! 
   @method socket
   @abstract returns socket of the database server. 
   @result	socket number.
*/
-(int)socket;
/*! 
   @method backendPID
   @abstract returns backend process id of the database server. 
   @result	backend process id.
*/
-(int)backendPID;
/*! 
   @method getPGconn
   @abstract returns pointer on PGconn record. 
   @discussion ALWAYS returns NIL in DEMO version. See PostgreSQL for advanced use.<br>
   You should never close yourselve a connection by calling PQfinish([pgconn getPGconn]).
   @result	pointer on PGconn record.
*/
-(PGconn *)getPGconn;

/*! 
   @method delegate
*/
-(id) delegate;

/*! 
   @method setDelegate
*/
-(void)setDelegate:(id)newDelegate;

/*! 
   @method disconnect
   @abstract disconnect cuurently opened connection. 
   @discussion Not strictly necessary in the API since connectTodatabase disconnects automatically from any previously opened connection.
*/
-(void)disconnect;

/*! 
   @method getResultSet
   @abstract Wrapping the last query result in a PgSQLResult object, makes possible to execute a new
   query without losing the result of the previous one. <br>
   @discussion However once done, it's no more possible to access result's information through methods of the PostgreSQL connection (rowsAffected, bindDouble, etc): you MUST use equivalent methods of the PgSQLResult wrapper object instead. Indeed, getResultSet has as side effect to make the PostgreSQL "forgets" about the last query result. PostgreSQL object maintains a list of PgSQLResult they originated, this list is freed when the PostgreSQL object is deallocated.
   @result	Wrapper object for last query result.
*/
-(PgSQLResult *)getResultSet;

/*! 
   @method clientEncoding
   @result	client encoding id.
*/
-(int)clientEncoding;
/*! 
   @method setClientEncoding
   @param encoding name
*/
-(int)setClientEncoding:(const char *)encoding;


/*!
	@method escapeString
	@result	nil if out of memory<br>
			autorelase NSString containing the escaped original
*/
-(NSString *)escapeString:(const char *)inSource;
/*!
	@method escapeBinary
	@result	nil if out of memory<br>
			autorelase NSData containing the escaped original
*/
-(NSData *)escapeBinary:(const unsigned char *)inSource length:(size_t)len;
#if	SUPPORT_73
/*!
	@method unescapeBinary
	@abstract	
	@discussion supported in 7.3
	@result	nil if out of memory<br>
			autorelase NSData containing the unescaped original
*/
-(NSData *)unescapeBinary:(const unsigned char *)inSource length:(size_t)len;
#endif

/*! 
   @method	serverMessage
   @result	PQerrorMessage()
*/
-(const char *)serverMessage;
/*! 
   @method connectErrorMessage
   @param errorCode to be converted into string.
   @result PQresStatus(errorCode)
*/
-(const char *)connectErrorMessage:(int)errorCode;

/*! 
   @method beginTransaction
   @abstract self explanatory
   @discussion DOES NOTHING (and returns YES) IN DEMO VERSION.
   @result YES if successful
*/
-(BOOL)beginTransaction;
/*! 
   @method commitTransaction
   @abstract self explanatory
   @discussion DOES NOTHING (and returns YES) IN DEMO VERSION.
   @result YES if successful
*/
-(BOOL)commitTransaction;
/*! 
   @method rollbackTransaction
   @abstract self explanatory
   @discussion DOES NOTHING (and returns YES) IN DEMO VERSION.
   @result YES if successful
*/
-(BOOL)rollbackTransaction;
/*! 
   @method transactionInProgress
   @abstract self explanatory
   @result YES if a transaction is in progress (beginTransaction has been called)
*/
- (BOOL)transactionInProgress;

/*! 
   @method clearCommands
   @abstract clears the command buffer
*/
-(void)clearCommands;
/*! 
   @method commandBuffer
   @result pointer on command buffer 
*/
-(const char *)commandBuffer;
/*! 
   @method makeCommand
   @abstract concat <code>cmd</code> to current command buffer
   @discussion makeCommand family methods don't escape automatically their arguments<br>
		Use escapeString if you need to.
*/
-(void)makeCommand:(const char *)cmd;
/*! 
   @method makeCommand
   @abstract concat <code>cmd</code> to current command buffer
   @discussion makeCommand family methods don't escape automatically their arguments<br>
		Use escapeString if you need to.
*/
-(void)makeCommandWithString:(NSString *)cmd;
/*! 
   @method makeCommandf
   @abstract same as makeCommand but with a format and variable arguments list (printf-like)
*/
-(void)makeCommandf:(const char *)format, ...;
/*! 
   @method makeCommandf
   @abstract same as makeCommand but with a format and variable arguments list (printf-like)
*/
-(void)vmakeCommandf:(const char *)format args:(va_list)ap;
/*! 
   @method executeCommand
   @abstract pass the current command buffer to the PostgreSQL server for execution
   @result	YES if successful
*/
-(BOOL)executeCommand;
/*! 
   @method executeCommandWithCursor
   @abstract pass the current command buffer to the PostgreSQL server for execution inside a cursor declaration
   @param cursorName the name of the cursor
   @param binary if cursor should be declared BINARY
   @result	YES if successful
*/
-(BOOL)executeCommandWithCursor:(const char *)cursorName binary:(BOOL)binary;
/*! 
   @method executeAsyncCommand
   @abstract pass the current command buffer to the PostgreSQL server for asynchronous execution
   @result	YES if successful
*/
-(BOOL)executeAsyncCommand;
/*! 
   @method executeAsyncCommandWithCursor
   @abstract pass the current command buffer to the PostgreSQL server for asynchronous execution inside a cursor declaration
   @param cursorName the name of the cursor
   @param binary if cursor should be declared BINARY
   @result	YES if successful
*/
-(BOOL)executeAsyncCommandWithCursor:(const char *)cursorName binary:(BOOL)binary;

/*!
	@method	cancelRequest
*/
-(BOOL)cancelRequest;

/*! 
   @method closeCursor
   @abstract close the cursor opened with executeCommandWithCursor
   @param cursorName
   @result YES if successful. 
*/
-(BOOL)closeCursor:(const char *)cursorName;

/*! 
   @method bufferHasCommands
   @result YES if command buffer contains not yet executed string. 
*/
-(BOOL)bufferHasCommands;

/*! 
   @method resultReturned
   @result YES if latest command executed returns rows.
*/
-(BOOL)resultReturned;

/*! 
   @method uniqueRowIdForTable
   @abstract uniqueRowIdForTable: requires you have a <br>
							<code>_rowid bigserial NOT NULL PRIMARY KEY</code> <br>
						field in the table
	@result the column value as a string.
*/
-(const char *)uniqueRowIdForTable:(const char *)tableName;
/*! 
   @method uniqueRowIdForTable
   @abstract let you specify the column name to be used for generating the unique "rowid".<br>
   Next rowid is generated by executing the following SQL statement:<br><code>
   SELECT max(colName)+1 FROM tableName</code><br>
	@result the column value as a string.
*/
-(const char *)uniqueRowIdForTable:(const char *)tableName column:(const char *)colName;

/*! 
   @method startNotificationFor
   @abstract  Execute a <code>LISTEN inTableName</code> query. 
   @discussion When the notification arrives, the <code>postgreSQL:notification:</code>
   selector of the listener delegate and of the application are invocated, (if thy exist). A PostgreSQLNotification is also posted.
	The dictionary of the notification contains:<br>
		PgSQLuserInfoNotificationField	userInfo<br>
		PgSQLconditionNotificationField tableName<br>
		PgSQLconnectionNotificationField self (PostgreSQL object)<br>
	Note that if a listener on the same table already exist, only the notificationDelegate is assigned to it and the existing one is returned.
   @result The listener object created to hold the notificationDelegate/inTableName couple/
*/
-(PgSQLListener *)startNotificationFor:(const char *)inTableName delegate:(id)notificatonDelegate userInfo:(id)userInfo;
/*! 
   @method removeNotificationFor   
   @abstract Removes the listener of the list of registered listener and execute a <code>UNLISTEN condition</code> query.
*/
-(void)removeNotificationFor:(PgSQLListener *)listener;

/*! 
   @method retrieveBinary
   @abstract	retrieve a BLOB into an newly created NSData object (autorelease)
   @param the Oid of the BLOB to be retrieved.
   @result	NSData object or nil.
*/
-(NSData *)retrieveBinary:(Oid)inOid;
/*! 
	@method retrieveBinaryFromStartLength
	@abstract	retrieve part of a BLOB into an newly created NSData object (autorelease)
    @param the Oid of the BLOB to be retrieved.
	@param start position from which to retrieve data
	@param length number of bytes to retrieve.
	@result	NSData object or nil.
*/
-(NSData *)retrieveBinaryFromStartLength:(Oid)inOid start:(int)inStart length:(int)length;
/*! 
   @method insertBinary
   @abstract insert a BLOB made from a memory buffer.
   @param inData pointer on the memory buffer
   @param size of the memory buffer
   @result	the Oid of the created BLOB, 0 if an error occurred.
*/
-(Oid)insertBinary:(unsigned char *)inData size:(int)size;
/*! 
	@method insertBinaryFromFile
    @abstract insert a BLOB made from NSFileHandle
	@param  the NSFileHandle on the file.
	@result	the Oid of the created BLOB, 0 if an error occurred.
*/
-(Oid)insertBinaryFromFile:(NSFileHandle *)inFile;
/*! 
   @method insertBinaryFromData
   @abstract insert a BLOB made from NSData
   @result	the Oid of the created BLOB, 0 if an error occurred.
*/
-(Oid)insertBinaryFromData:(NSData *)inData;
/*! 
   @method overWriteBinaryWithDataFromStart
   @abstract	overwrite part a of BLOB with bytes from a NSData object from a specified start point.
   @result	YES if successful (number of bytes written == size of NSData buffer).
*/
-(BOOL)overWriteBinaryWithDataFromStart:(Oid)inOid data:(NSData *)inData start:(int)inStart;
/*! 
   @method exportBinaryToFile
   @abstract	Export the BLOB to the file system.
   @result	the lo_export returned code.
*/
-(int)exportBinaryToFile:(Oid)inOid pathName:(const char *)inPathName;
/*! 
	@method importBinaryFromFile
	@abstract insert a BLOB made from a file specified with its pathname.
	@result	the Oid of the created BLOB, 0 if an error occurred.
*/
-(Oid)importBinaryFromFile:(const char *)inPathName;
/*! 
	@method unlinkBinary
	@abstract	delete a large object
	@result 	result code from lo_unlink
*/
-(int)unlinkBinary:(Oid)inOid;
/*! 
   @method asyncResultAvailable
   @abstract	use to check if an asynchronous query has returned results yet.
   @result	YES if data is available ([conn nextRow] will not block).
*/
-(BOOL)asyncResultAvailable;
/*! 
   @method getTimeOut
   @abstract	time out to be used when doing an asynchronous query. <br>
	Defaults to 10 sec.<br>
	If set to -1 no time out will be used in select().<br>
	DON'T DO THAT WHEN TESTING if you want to avoid to kill -9 your testing code... in case of trouble.
   @result	time out value in seconds.
*/
-(long)getTimeOut;
/*! 
   @method setTimeOut
   @abstract	set the time out to be used when doing an asynchronous query.
   @param	time out value in seconds.
*/
-(void)setTimeOut:(long)inTimeOut;

/*!
   @method setNoticeProcessor
*/
-(PQnoticeProcessor)setNoticeProcessor:(PQnoticeProcessor)proc userArg:(void *)arg;

@end
