/* $PostgresPy: pl/src/pl.c,v 1.15 2004/06/29 00:12:02 flaw Exp $
 * 
 * † Instrument:
 *     Copyright 2004, rhid development. All Rights Reserved.
 *     
 *     Usage of the works is permitted provided that this
 *     instrument is retained with the works, so that any entity
 *     that uses the works is notified of this instrument.
 *     
 *     DISCLAIMER: THE WORKS ARE WITHOUT WARRANTY.
 *     
 *     [2004, Fair License; rhid.com/fair]
 *     
 * Description:
 *    Abstract procedural language implementation.
 */
#include <dlfcn.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <setjmp.h>
#include <errno.h>

#include <postgres.h>
#include <fmgr.h>
#include <funcapi.h>
#include <tcop/tcopprot.h>

#include <access/htup.h>
#include <access/heapam.h>
#include <access/xact.h>

#include <catalog/catname.h>
#include <catalog/pg_class.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <catalog/pg_language.h>
#include <catalog/indexing.h>

#include <utils/array.h>
#include <utils/elog.h>
#include <utils/builtins.h>
#include <utils/hsearch.h>
#include <utils/syscache.h>
#include <utils/relcache.h>

#include <nodes/memnodes.h>

#include <pgpy/pputils.h>
#include <pgpy/PGExcept.h>
#include <pgpy/pg.h>

#include "pl.h"

TupleDesc
RelationOidGetDescr(Oid reloid)
{
	Relation rel = NULL;
	TupleDesc rtupd = NULL;

	rel = RelationIdGetRelation(reloid);
	rtupd = CreateTupleDescCopy(RelationGetDescr(rel));
	RelationClose(rel);

	return(rtupd);
}

bytea *
mkbytea(const char *buf, Size blen)
{
	bytea *b;
	size_t buflen = 0;
	buflen = blen + VARHDRSZ;

	b = (bytea*)palloc(buflen);

	VARATT_SIZEP(b) = buflen;
	memcpy(VARDATA(b), buf, blen);

	return(b);
}

HeapTuple
ProcTuple_FromOid(Oid fn_oid)
{
	HeapTuple procTuple = NULL;

	procTuple = SearchSysCache(PROCOID, ObjectIdGetDatum(fn_oid), 0, 0, 0);
	if (!HeapTupleIsValid(procTuple))
		ereport(ERROR,(
			errmsg("Failed to find procedure in system cache \"%u\".", fn_oid)
		));

	return(procTuple);
}

HeapTuple
ProcTuple_SetProbin(HeapTuple procTuple, Datum probin)
{
	Relation rel;
	HeapTuple tup;

	Datum	values[Natts_pg_proc] = {0};
	char nulls[Natts_pg_proc];
	char replace[Natts_pg_proc];

	memset(replace, ' ', Natts_pg_proc);
	memset(nulls, ' ', Natts_pg_proc);

	replace[Anum_pg_proc_probin-1] = 'r';
	values[Anum_pg_proc_probin-1] = probin;

	rel = heap_openr(ProcedureRelationName, RowExclusiveLock);
	if (!RelationIsValid(rel))
		ereport(ERROR,(
			errmsg("Failed to open pg_proc System relation.")
		));

	tup = heap_modifytuple(procTuple, rel, values, nulls, replace);
	if (!tup)
		ereport(ERROR,(
			errmsg("Failed to modify procedure's tuple.")
		));

	simple_heap_update(rel, &tup->t_self, tup);
	CatalogUpdateIndexes(rel,tup);

	heap_close(rel, RowExclusiveLock);
	return(tup);
}

unsigned short 	pl_inits = 0;
HeapTuple			Language = NULL;


static void pl_EOXact(bool, void*);
static void plcache_init(void);
void
pl_initialize(Oid prolang)
{
	Language = SearchSysCacheCopy(LANGOID, prolang, 0, 0, 0);
	if (Language == NULL)
		ereport(ERROR,(
			errmsg("Failed to find %d in pg_language.", prolang),
			errcontext("procedural language initialization")
		));

	plcache_init();
	lang_initialize();
	RegisterEOXactCallback(pl_EOXact, NULL);
}

static void plcache_destroy(void);
void
pl_finalize(void)
{
	UnregisterEOXactCallback(pl_EOXact, NULL);
	plcache_destroy();
	lang_finalize();

	heap_freetuple(Language);
	Language = NULL;
}

MemoryContext plFormerContext = NULL;
MemoryContext plcontext = NULL;

void *
plalloc(Size amt)
{
	return(MemoryContextAlloc(plcontext, amt));
}

void *
replalloc(void *ptr, Size amt)
{
	if (ptr == NULL)
		return(plalloc(amt));

	return(repalloc(ptr,amt));
}

/*
 * pl_compile - one stop for complete compilation/probin storage
 */
void plcache_invalidate(Oid);
void
pl_compile(Oid fn_oid)
{
	Datum src, srctext;
	bytea *probin;
	bool isnull = false;

	char *sproc;
	size_t sprocSize;

	HeapTuple procTuple;
	TupleDesc procDesc;

	plproc code;

	Assert(fn_oid != InvalidOid);

	procTuple = SearchSysCacheCopy(PROCOID, fn_oid, 0,0,0); /* leaks on err */
	if (Language == NULL)
		pl_initialize(PROCSTRUCT(procTuple)->prolang);

	procDesc = SysTupleDesc(proc);
	srctext = fastgetattr(procTuple, Anum_pg_proc_prosrc, procDesc, &isnull);
	FreeTupleDesc(procDesc);

	if (isnull == true || srctext == NULL)
		ereport(ERROR,(
			errmsg("failed to get valid source text"),
			errdetail("source@%p is said to be %s NULL",
				(char*)srctext, isnull?"":"not")
		));

	src = DirectFunctionCall1(textout, srctext);
	code = plproc_FromSource(procTuple, (char*)src);
	pfree(DatumGetPointer(src));
	if (code == NULL)
		ereport(ERROR,(
			errmsg("generic procedure compilation failure.")
		));


	sproc = plproc_ToBinary(code, &sprocSize);
	plproc_free(code);

	if (sproc == NULL || sprocSize == NULL)
		ereport(ERROR,(
			errmsg("failed to render stored procedure.")
		));

	probin = mkbytea(sproc, sprocSize);
	pfree(sproc);

	ProcTuple_SetProbin(procTuple, PointerGetDatum(probin));
	pfree(probin);

	plcache_invalidate(fn_oid);

	heap_freetuple(procTuple);
}

/*
 * plproc_FromHeapTuple
 *
 *	Get a "stored" procedure from a procedure tuple's probin.
 */
plproc
plproc_FromHeapTuple(HeapTuple procTuple)
{
	plproc code = NULL;

	TupleDesc procDesc = NULL;
	bool isnull = false;
	bytea *procb;
	char *bin;

	Size srcLen = 0;

	Assert(procTuple != NULL);

	procDesc = SysTupleDesc(proc);
	procb = (bytea*)fastgetattr(procTuple,
							Anum_pg_proc_probin, procDesc, &isnull);
	FreeTupleDesc(procDesc);

	srcLen = VARSIZE(procb) - VARHDRSZ;
	bin = VARDATA(procb);

	code = plproc_FromBinary(bin, srcLen);

	return(code);
}


/*
 * Function cache hash table
 */
HTAB *plcache = NULL;

/*
 * plcache_init - initialize the cache
 */
static int plcache_oidcmp(const void *k1, const void *k2, Size ks)
{ return( ( (*((Oid*)k1)) == (*((Oid*)k2)) )? 0 : 1 ); }
static void
plcache_init()
{
	static HASHCTL hc = {0};

	hc.keysize = sizeof(Oid);
	hc.entrysize = sizeof(struct plcache);
	hc.match = plcache_oidcmp;
	hc.hash = tag_hash;
	hc.hcxt = plcontext;

	plcache = hash_create("Function Cache", 32, &hc,
					HASH_CONTEXT | HASH_COMPARE | HASH_ELEM | HASH_FUNCTION);
}

/*
 * plcache_free - free up plcache contents NOT the plcache pointer
 */
static void
plcache_free(struct plcache *plc)
{
	plproc_free(plc->code);

#ifdef Anum_pg_proc_proargnames
{
	int nargs = PROCSTRUCT(plc->proc)->pronargs;
	int carg;

	for (carg = 0; carg < nargs; ++carg)
	if (plc->argnames[carg])
	{
		pfree(plc->argnames[carg]);
		plc->argnames[carg] = 0;
	}
}
#endif
	
	heap_freetuple(plc->proc);
}

/*
 * plcache_invalidate - used by pl_compile to invalidate existing cache
 */
void
plcache_invalidate(Oid fn_oid)
{
	struct plcache *plc = NULL;
	plc = (struct plcache *) hash_search(plcache, &fn_oid, HASH_FIND, NULL);

	if (plc)
	{
		plcache_free(plc);
		hash_search(plcache, &fn_oid, HASH_REMOVE, NULL);
	}
}

/*
 * plcache_destroy - completely wipe cache
 */
static void
plcache_destroy(void)
{
	HASH_SEQ_STATUS hss;
	struct plcache *plc = NULL;

	if (!plcache) return;

	hash_seq_init(&hss, plcache);

	while ((plc = hash_seq_search(&hss)))
		plcache_free(plc);

	hash_destroy(plcache);
	plcache = NULL;
}

/*
 * plcache_fetch - fetch a function's cache, existing or not
 */
static struct plcache *
plcache_fetch(Oid fn_oid)
{
	bool exists = false;
	struct plcache *cache;
	HeapTuple procTuple = ProcTuple_FromOid(fn_oid);

	if (Language == NULL)
		pl_initialize(PROCSTRUCT(procTuple)->prolang);

	/*
	 * Fetch or Form an entry
	 */
	cache = (struct plcache *) hash_search(plcache,
											&fn_oid, HASH_ENTER, &exists);
	if (exists)
	{
		/*
		 * Only check Xmin; the validator would have invalidated the cache
		 * if there was a change within the same transaction/postmaster
		 */
		if (HeapTupleGetXmin(cache->proc) == HeapTupleGetXmin(procTuple))
			goto finish;
		
		plcache_free(cache);
	}
	
	cache->fn_oid = fn_oid; /* keep HASH_COMPARE happy */
	cache->proc = heap_copytuple(procTuple);

/*
 * proargnames
 *    Caching the argnames is done to ease later application
 */
#ifdef Anum_pg_proc_proargnames
	{
		TupleDesc procDesc = SysTupleDesc(proc);
		Datum		 *argnames = NULL;
		ArrayType *Argnames = NULL;
		bool isnull;
		int elms = 0, celm;

		Argnames = (ArrayType*)fastgetattr(procTuple,
				Anum_pg_proc_proargnames, procDesc, &isnull);
		FreeTupleDesc(procDesc);
			
		if (!isnull)
			deconstruct_array(Argnames, ARR_ELEMTYPE(Argnames), -1, false,
						'i', &argnames, &elms);

		Assert(elms < FUNC_MAX_ARGS);

		for (celm = 0; celm < elms; ++celm)
		{
			cache->argnames[celm] = DatumGetPointer(
					DirectFunctionCall1(textout, argnames[celm])
			);
			pfree(argnames[celm]);
		}
		pfree(argnames);
	}
#endif
	cache->code = plproc_FromHeapTuple(procTuple);

finish:
	ReleaseSysCache(procTuple);
	return(cache);
}

void
pl_error()
{
	if (!lang_error()) return;

	errstart(NOTICE, __FILE__, __LINE__, PG_FUNCNAME_MACRO);
	errmsg("additional information");
	errdetail("%s", lang_errorstr());
	errfinish(0);
}

/*
 * pl_EOXact - If InError, finalize it
 * 	pl_finalize will remove pl_EOXact from EOXactCallbacks
 * 	pl_initialize will add it again, when necessary
 */
static void
pl_EOXact(bool isCommit, void *arg)
{
	if (InError)
		pl_finalize();
}


/*
 * pl_finalizer
 *		Used to completely reinitialize the procedural language
 *		Normally useful to reset persistent data involved in the language.
 */
PG_FUNCTION_INFO_V1(pl_finalizer);
Datum
pl_finalizer(PG_FUNCTION_ARGS)
{
	EXC_DECLARE();
	EXCEPT( pl_error(); );

	plcontext = TopMemoryContext;
	plFormerContext = MemoryContextSwitchTo(plcontext);

	if (Language != NULL)
		pl_finalize();

	MemoryContextSwitchTo(plFormerContext);

	RETURN((Datum)0);
}


/*
 * pl_validator
 *		fmgr interface to pl_compile()
 */
PG_FUNCTION_INFO_V1(pl_validator);
Datum
pl_validator(PG_FUNCTION_ARGS)
{
	EXC_DECLARE();
	EXCEPT( pl_error(); );

	plcontext = TopMemoryContext;
	plFormerContext = MemoryContextSwitchTo(plcontext);

	pl_compile(PG_GETARG_OID(0));
	
	MemoryContextSwitchTo(plFormerContext);

	RETURN((Datum)0);
}

Datum pl_main(PG_FUNCTION_ARGS);

PG_FUNCTION_INFO_V1(pl_handler);
Datum
pl_handler(PG_FUNCTION_ARGS)
{
	PGFunction fn_addr;
	Datum	rv;
	EXC_DECLARE();
	EXCEPT( pl_error(); );

	plcontext = TopMemoryContext;
	plFormerContext = MemoryContextSwitchTo(plcontext);
	
	Assert(
		fcinfo->context &&
		nodeTag(fcinfo->context) == T_TriggerData &&
		fcinfo->resultinfo == NULL /* Triggers don't return sets */
	);

	fn_addr = fcinfo->flinfo->fn_addr;

	fcinfo->flinfo->fn_addr = (PGFunction) plcache_fetch(FC_FNOID);
	rv = pl_main(fcinfo);
	fcinfo->flinfo->fn_addr = fn_addr;

	MemoryContextSwitchTo(plFormerContext);

	RETURN(rv);
}
