/*
 *
 *	pg_migrator.c
 *
 */

#include "pg_migrator.h"

#ifdef HAVE_LANGINFO_H
#include <langinfo.h>
#endif


#define ALL_DUMP_FILE		"pg_migrator_dump_all.sql"
/* contains both global db information and CREATE DATABASE commands */
#define GLOBALS_DUMP_FILE	"pg_migrator_dump_globals.sql"
#define DB_DUMP_FILE		"pg_migrator_dump_db.sql"


static void install_support_functions(migratorContext *ctx);
static void generate_old_dump(migratorContext *ctx);
static void split_old_dump(migratorContext *ctx);
static void copy_clog_xlog_xid(migratorContext *ctx);
static void	set_frozenxids(migratorContext *ctx);
static void check_cluster_versions(migratorContext *ctx);
static void check_cluster_compatibility(migratorContext *ctx);
static void setup(migratorContext *ctx, char *argv0);
static void cleanup(migratorContext *ctx);
static void rename_old_pg_control(migratorContext *ctx);



/*
 *	main
 */
int
main(int argc, char **argv)
{
	migratorContext ctx;
	char *sequence_script_file_name = NULL;
	
	memset(&ctx, 0, sizeof(ctx));

	parseCommandLine(&ctx, argc, argv);

	pg_log(&ctx, PG_REPORT, "Performing consistency checks\n");
	pg_log(&ctx, PG_REPORT, "-----------------------------\n");
	setup(&ctx, argv[0]);

	check_cluster_versions(&ctx);
	check_cluster_compatibility(&ctx);


	/* -- OLD -- */
	start_postmaster(&ctx, CLUSTER_OLD, false);

	set_locale_and_encoding(&ctx, CLUSTER_OLD);
	
	/* Get the pg_database and pg_largeobject relation OID's */
	get_misc_relfilenodes(&ctx, CLUSTER_OLD);

	/* Extract a list of databases and tables from the old cluster */
	get_db_and_rel_infos(&ctx, &ctx.old.dbarr, CLUSTER_OLD);

	/* And some other stuff too */
	get_tablespace_paths(&ctx);
	get_loadable_libraries(&ctx);

	/* Check for various failure cases */
	v8_3_check_for_isn_and_int8_passing_mismatch(&ctx, CLUSTER_OLD);
	if (GET_MAJOR_VERSION(ctx.old.pg_version) <= 803 &&
		GET_MAJOR_VERSION(ctx.new.pg_version) >= 804)
	{
		v8_3_check_for_name_data_type_usage(&ctx, CLUSTER_OLD);
		v8_3_check_for_tsquery_usage(&ctx, CLUSTER_OLD);
		if (ctx.check)
		{
			v8_3_rebuild_tsvector_tables(&ctx, true, CLUSTER_OLD);
			v8_3_invalidate_hash_gin_indexes(&ctx, true, CLUSTER_OLD);
			v8_3_invalidate_bpchar_pattern_ops_indexes(&ctx, true, CLUSTER_OLD);
		}
		else
			/*
			 *	While we have the old server running, create the script
			 *	to properly restore its sequence values but we report this
			 *	at the end.
			 */
			sequence_script_file_name =
				v8_3_create_sequence_script(&ctx, CLUSTER_OLD);
	}

	/* Looks okay so far.  Prepare the pg_dump output */
	generate_old_dump(&ctx);
	split_old_dump(&ctx);

	stop_postmaster(&ctx, false, false);


	/* -- NEW -- */
	start_postmaster(&ctx, CLUSTER_NEW, false);

	set_locale_and_encoding(&ctx, CLUSTER_NEW);
	
	check_new_db_is_empty(&ctx);

	check_loadable_libraries(&ctx);

	stop_postmaster(&ctx, false, false);

	check_locale_and_encoding(&ctx, &ctx.old.controldata, &ctx.new.controldata);

	if (ctx.transfer_mode == TRANSFER_MODE_LINK)
		check_hard_link(&ctx);
	
	if (ctx.check)
	{
		pg_log(&ctx, PG_REPORT, "\n*Clusters are compatible*\n");
		exit_nicely(&ctx, false);
	}

	pg_log(&ctx, PG_REPORT, "*Checks complete*\n\n"
		"| If pg_migrator fails after this point, you must\n"
		"| re-initdb the new cluster before continuing.\n");

	if (ctx.num_tablespaces)
		pg_log(&ctx, PG_REPORT,
			"| You will also need to remove the \".old\" suffix\n"
			"| from old $PGDATA/global/pg_control and old\n"
			"| tablespace directory names before continuing.\n");
		
	pg_log(&ctx, PG_REPORT, "\nPerforming migration\n");
	pg_log(&ctx, PG_REPORT, "--------------------\n");
	

	/* rename pg_control so old server cannot be accidentally started */
	rename_old_pg_control(&ctx);
	
	/* -- NEW -- */
	start_postmaster(&ctx, CLUSTER_NEW, false);

	/*
	 * It would make more sense to freeze after loading the schema, but that
	 * would cause us to lose the frozenids restored by the load.
	 * We use --analyze so autovacuum doesn't update statistics later
	 */
	prep_status(&ctx, "Analyzing all rows on the new cluster");
	exec_prog(&ctx, true,
			  SYSTEMQUOTE "\"%s/vacuumdb\" --port %d --all --analyze >> %s 2>&1" SYSTEMQUOTE,
			  ctx.new.bindir, ctx.new.port, ctx.logfile);
	check_ok(&ctx);

	/*
	 * We do freeze after analyze so pg_statistic is also frozen
	 */
	prep_status(&ctx, "Freezing all rows on the new cluster");
	exec_prog(&ctx, true,
			  SYSTEMQUOTE "\"%s/vacuumdb\" --port %d --all --freeze >> %s 2>&1" SYSTEMQUOTE,
			  ctx.new.bindir, ctx.new.port, ctx.logfile);
	check_ok(&ctx);

	stop_postmaster(&ctx, false, false);


	/*
	 *	Rename all tablespace paths to .old so we can copy them later.
	 *	DESTRUCTIVE of old cluster
	 */
	prep_status(&ctx, "Renaming tablespaces to *.old");
	rename_tablespaces(&ctx);
	check_ok(&ctx);

	copy_clog_xlog_xid(&ctx);
	/* New now using xid of old system */


	/* -- NEW -- */
	start_postmaster(&ctx, CLUSTER_NEW, false);

	set_frozenxids(&ctx);
	get_misc_relfilenodes(&ctx, CLUSTER_NEW);

	/*
	 * We have to create the databases first so we can create the
	 * toast table placeholder relfiles.
	 */
	prep_status(&ctx, "Creating databases in new cluster");
	exec_prog(&ctx, true,
			  SYSTEMQUOTE "\"%s/%s\" --set ON_ERROR_STOP=on --port %d "
			  "-f \"%s/%s\" --dbname template1 >> \"%s\"" SYSTEMQUOTE,
			  ctx.new.bindir, ctx.new_psql_exe, ctx.new.port,
			  ctx.home_dir, GLOBALS_DUMP_FILE, ctx.logfile);
	check_ok(&ctx);

	get_db_and_rel_infos(&ctx, &ctx.new.dbarr, CLUSTER_NEW);

	install_support_functions(&ctx);

	stop_postmaster(&ctx, false, false);

	prep_status(&ctx, "Creating placeholder relfiles for toast relations\n");
	reserve_relfilenode_names(&ctx);
	prep_status(&ctx, "");
	check_ok(&ctx);


	/* -- NEW -- */
	start_postmaster(&ctx, CLUSTER_NEW, false);

	prep_status(&ctx, "Restoring database schema");
	exec_prog(&ctx, true,
			  SYSTEMQUOTE "\"%s/%s\" --set ON_ERROR_STOP=on --port %d "
			  "-f \"%s/%s\" --dbname template1 >> \"%s\"" SYSTEMQUOTE,
			  ctx.new.bindir, ctx.new_psql_exe, ctx.new.port,
			  ctx.home_dir, DB_DUMP_FILE, ctx.logfile);
	check_ok(&ctx);

	/* regenerate now that we have db schemas */
	dbarr_free(&ctx.new.dbarr);
	get_db_and_rel_infos(&ctx, &ctx.new.dbarr, CLUSTER_NEW);

	prep_status(&ctx, "Restoring relations to use fixed toast file names\n");
	adjust_rels_to_use_fixed_toast_relfilenodes(&ctx, &ctx.old.dbarr, &ctx.new.dbarr);
	prep_status(&ctx, "");	/* in case nothing printed */
	check_ok(&ctx);

	/*
	 * adjust_rels_to_use_fixed_toast_relfilenodes would have created new
	 * entries in the new database; hence regenerate new_dbinfos to get in sync.
	 */
	dbarr_free(&ctx.new.dbarr);
	get_db_and_rel_infos(&ctx, &ctx.new.dbarr, CLUSTER_NEW);

	stop_postmaster(&ctx, false, false);


	prep_status(&ctx, "Restoring user relations\n");
	transfer_all_new_dbs(&ctx, &ctx.old.dbarr, &ctx.new.dbarr,
						 ctx.old.pgdata, ctx.new.pgdata);
	prep_status(&ctx, "");	/* in case nothing printed */
	check_ok(&ctx);

	/*
	 * Assuming OIDs are only used in system tables, there is no need to
	 * restore the OID counter because we have not transferred any OIDs from
	 * the old system, but we do it anyway just in case.  We do it late here
	 * because there is no need to have the schema load use new oids.
	 */
	prep_status(&ctx, "Setting next oid for new cluster");
	exec_prog(&ctx, true, SYSTEMQUOTE "\"%s/pg_resetxlog\" -o %u \"%s\" > " DEVNULL SYSTEMQUOTE,
		  ctx.new.bindir, ctx.old.controldata.chkpnt_nxtoid, ctx.new.pgdata);
	check_ok(&ctx);

	if (GET_MAJOR_VERSION(ctx.old.pg_version) <= 803 &&
		GET_MAJOR_VERSION(ctx.new.pg_version) >= 804)
	{
		start_postmaster(&ctx, CLUSTER_NEW, true);
		/* restore proper sequence values using file created from old server */
		if (strlen(sequence_script_file_name) > 0)
		{
			prep_status(&ctx, "Adjusting sequences");
			exec_prog(&ctx, true,
					  SYSTEMQUOTE "\"%s/%s\" --set ON_ERROR_STOP=on --port %d "
					  "-f \"%s\" --dbname template1 >> \"%s\"" SYSTEMQUOTE,
					  ctx.new.bindir, ctx.new_psql_exe, ctx.new.port,
					  sequence_script_file_name, ctx.logfile);
			unlink(sequence_script_file_name);
			check_ok(&ctx);
		}
		pg_free(sequence_script_file_name);

		v8_3_rebuild_tsvector_tables(&ctx, false, CLUSTER_NEW);
		v8_3_invalidate_hash_gin_indexes(&ctx, false, CLUSTER_NEW);
		v8_3_invalidate_bpchar_pattern_ops_indexes(&ctx, false, CLUSTER_NEW);
		stop_postmaster(&ctx, false, true);
	}
	
	pg_log(&ctx, PG_REPORT, "\n*Upgrade complete*\n");

	pg_log(&ctx, PG_REPORT, "\n"
	  		"| Optimizer statistics and free space information\n"
			"| are not transferred by pg_migrator, so consider\n"
			"| running:\n"
			"| \tvacuumdb --all --analyze\n" 
			"| on the newly-upgraded cluster.\n");

	cleanup(&ctx);

	return 0;
}


/*
 *	generate_old_dump
 */
static void
generate_old_dump(migratorContext *ctx)
{
	/* run new pg_dumpall binary */
	prep_status(ctx, "Creating catalog dump");

	/*
	 * --binary-upgrade records the width of dropped columns in pg_class, and
	 * restores the frozenid's for databases and relations.
	 */
	exec_prog(ctx, true,
			SYSTEMQUOTE "\"%s/pg_dumpall\" --port %d --schema-only "
			"--binary-upgrade > \"%s/" ALL_DUMP_FILE "\"" SYSTEMQUOTE,
		    ctx->new.bindir, ctx->old.port, ctx->home_dir);
	check_ok(ctx);
}



/*
 *	split_old_dump
 *
 *	This function splits pg_dumpall output into global values and
 *	database creation, and per-db scemas.  This allows us to create
 *	the toast place holders between retoring these two parts of the
 *	dump.  We split on the first "\connect " after a CREATE ROLE
 *	username match;  this is where the per-db restore starts.
 *	
 *	We suppress recreation of our own username so we don't generate
 *	an error during restore
 */
static void
split_old_dump(migratorContext *ctx)
{
	FILE	*all_dump, *globals_dump, *db_dump;
	FILE	*current_output;
	char	line[LINE_ALLOC];
	bool	start_of_line = true;
	char	create_role_str[MAX_STRING];
	char	create_role_str_quote[MAX_STRING];
	char	filename[MAX_STRING];
	bool	suppressed_username = false;
	
	prep_status(ctx, "Splitting old dump file");

	snprintf(filename, sizeof(filename), "%s/%s", ctx->home_dir, ALL_DUMP_FILE);
	if ((all_dump =	fopen(filename, "r")) == NULL)
		pg_log(ctx, PG_FATAL, "Cannot open dump file %s\n", filename);
	snprintf(filename, sizeof(filename), "%s/%s", ctx->home_dir, GLOBALS_DUMP_FILE);
	if ((globals_dump =	fopen(filename, "w")) == NULL)
		pg_log(ctx, PG_FATAL, "Cannot write to dump file %s\n", filename);
	snprintf(filename, sizeof(filename), "%s/%s", ctx->home_dir, DB_DUMP_FILE);
	if ((db_dump =	fopen(filename, "w")) == NULL)
		pg_log(ctx, PG_FATAL, "Cannot write to dump file %s\n", filename);
	current_output = globals_dump;
	
	/* patterns used to prevent our own username from being recreated */
	snprintf(create_role_str, sizeof(create_role_str),
			 "CREATE ROLE %s;", ctx->user);
	snprintf(create_role_str_quote, sizeof(create_role_str_quote),
			 "CREATE ROLE %s;", quote_identifier(ctx, ctx->user));
		
	while(fgets(line, sizeof(line), all_dump) != NULL)
	{
		/* switch to db_dump file output? */
		if (current_output == globals_dump && start_of_line &&
			suppressed_username &&
			strncmp(line, "\\connect ", strlen("\\connect ")) == 0)
			current_output = db_dump;

		/* output unless we are recreating our own username */
		if (current_output != globals_dump || !start_of_line ||
			(strncmp(line, create_role_str, strlen(create_role_str)) != 0 &&
			 strncmp(line, create_role_str_quote, strlen(create_role_str_quote)) != 0))
				fputs(line, current_output);
		else
			suppressed_username = true;
					
		if (strlen(line) > 0 && line[strlen(line) - 1] == '\n')
			start_of_line = true;
		else
			start_of_line = false;
	}	

	fclose(all_dump);
	fclose(globals_dump);
	fclose(db_dump);

	check_ok(ctx);
}



/*
 *	copy_clog_xlog_xid()
 */
static void
copy_clog_xlog_xid(migratorContext *ctx)
{
	char		old_clog_path[MAXPGPATH];
	char		new_clog_path[MAXPGPATH];

	/* copy old commit logs to new data dir */
	prep_status(ctx, "Deleting old commit clogs");

	snprintf(old_clog_path, sizeof(old_clog_path), "%s/pg_clog", ctx->old.pgdata);
	snprintf(new_clog_path, sizeof(new_clog_path), "%s/pg_clog", ctx->new.pgdata);
	if (rmtree(new_clog_path, true) != true)
		pg_log(ctx, PG_FATAL, "Unable to delete directory %s\n", new_clog_path);
	check_ok(ctx);

	prep_status(ctx, "Copying commit clogs");
	/* libpgport's copydir() doesn't work in FRONTEND code */
#ifndef WIN32
	exec_prog(ctx, true, SYSTEMQUOTE "%s \"%s\" \"%s\"" SYSTEMQUOTE,
				"cp -Rf",
#else
				/* flags: everything, no confirm, quiet, overwrite read-only */
	exec_prog(ctx, true, SYSTEMQUOTE "%s \"%s\" \"%s\\\"" SYSTEMQUOTE,
				"xcopy /e /y /q /r",
#endif
				 old_clog_path, new_clog_path);
	check_ok(ctx);

	/* set the next transaction id of the new cluster */
	prep_status(ctx, "Setting next transaction id for new cluster");
	exec_prog(ctx, true, SYSTEMQUOTE "\"%s/pg_resetxlog\" -f -x %u \"%s\" > " DEVNULL SYSTEMQUOTE,
	   ctx->new.bindir, ctx->old.controldata.chkpnt_nxtxid, ctx->new.pgdata);
	check_ok(ctx);

	/* now reset the wal archives in the new cluster */
	prep_status(ctx, "Resetting WAL archives");
	exec_prog(ctx, true, SYSTEMQUOTE "\"%s/pg_resetxlog\" -l %u,%u,%u \"%s\" >> \"%s\" 2>&1" SYSTEMQUOTE,
			  ctx->new.bindir, ctx->old.controldata.chkpnt_tli,
			  ctx->old.controldata.logid, ctx->old.controldata.nxtlogseg,
			  ctx->new.pgdata, ctx->logfile);
	check_ok(ctx);
}


/*
 *	set_frozenxids()
 *
 *	We have frozen all xids, so set relfrozenxid and datfrozenxid
 *	to be the old cluster's xid counter, which we just set in the new
 *	cluster.  User-table frozenxid values will be set by pg_dumpall
 *	--binary-upgrade, but objects not set by the pg_dump must have
 *	proper frozen counters.
 */
static void	set_frozenxids(migratorContext *ctx)
{
	int			dbnum;
	PGconn	   *conn;
	PGresult   *dbres;
	int			ntups;
	
	prep_status(ctx, "Setting frozenxid counters in new cluster");

	conn = connectToServer(ctx, "template1", CLUSTER_NEW);

	/* set pg_database.datfrozenxid */
	PQclear(executeQueryOrDie(ctx, conn,
							"UPDATE pg_catalog.pg_database "
							"SET	datfrozenxid = '%u' "
							/* cannot connect to 'template0', so ignore */
							"WHERE	datname != 'template0'",
							ctx->old.controldata.chkpnt_nxtxid));

	/* get database names */
	dbres = executeQueryOrDie(ctx, conn,
							"SELECT	datname "
							"FROM	pg_catalog.pg_database "
							"WHERE	datname != 'template0'");
	/* free dbres below */

	PQfinish(conn);
	
	ntups = PQntuples(dbres);
	for (dbnum = 0; dbnum < ntups; dbnum++)
	{
		conn = connectToServer(ctx,  PQgetvalue(dbres, dbnum, 0), CLUSTER_NEW);

		/* set pg_class.relfrozenxid */
		PQclear(executeQueryOrDie(ctx, conn,
								"UPDATE	pg_catalog.pg_class "
								"SET	relfrozenxid = '%u' "
								/* only heap and TOAST are vacuumed */
								"WHERE	relkind = 'r' OR "
								"		relkind = 't'",
								ctx->old.controldata.chkpnt_nxtxid));
		PQfinish(conn);
	}

	PQclear(dbres);

	check_ok(ctx);
}


/*
 *	setup()
 */
static void
setup(migratorContext *ctx, char *argv0)
{
	char		exec_path[MAXPGPATH];	/* full path to my executable */

	/* disallow execution by the root user	*/
	pg_free(get_user_id(ctx));

	/*
	 * make sure the user has a clean environment, otherwise, we may confuse
	 * libpq when we connect to one (or both) of the servers.
	 */
	check_for_libpq_envvars(ctx);

	verify_directories(ctx);

	/* no postmasters should be running */
	if (is_server_running(ctx, ctx->old.pgdata))
	{
		pg_log(ctx, PG_FATAL, "There seems to be a postmaster servicing the old cluster.\n"
					  "Please shutdown that postmaster and try again.\n");
	}

	/* same goes for the new postmaster */
	if (is_server_running(ctx, ctx->new.pgdata))
	{
		pg_log(ctx, PG_FATAL, "There seems to be a postmaster servicing the new cluster.\n"
			   "Please shutdown that postmaster and try again.\n");
	}

	/* get path to pg_migrator executable */
	if (find_my_exec(argv0, exec_path) < 0)
		pg_log(ctx, PG_FATAL, "Could not get pathname to pg_migrator: %s\n", strerror(errno));

	/* Trim off program name and keep just path */
	*last_dir_separator(exec_path) = '\0';
	canonicalize_path(exec_path);
	ctx->exec_path = pg_strdup(ctx, exec_path);
}


/*
 *	check_cluster_versions()
 */
static void
check_cluster_versions(migratorContext *ctx)
{
	/* get old and new cluster versions */
	ctx->old.pg_version = get_postgres_version(ctx, &ctx->old.pg_version_str, CLUSTER_OLD);
	ctx->new.pg_version = get_postgres_version(ctx, &ctx->new.pg_version_str, CLUSTER_NEW);

	/* We allow migration from/to the same major version for beta upgrades */

	if (GET_MAJOR_VERSION(ctx->old.pg_version) < 803)
		pg_log(ctx, PG_FATAL, "This utility can only upgrade from PostgreSQL version 8.3 and later.\n");

	/* Only PG 8.4+ has enhancements to pg_dump, pg_dumpall, and vacuumdb */
	if (GET_MAJOR_VERSION(ctx->new.pg_version) < 804)
		pg_log(ctx, PG_FATAL, "This utility can only upgrade to PostgreSQL version 8.4 and later.\n");

	/*
	 * We can't allow downgrading because we use the target pg_dumpall, and
	 * pg_dumpall cannot operate on new datbase versions, only older versions.
	 */
	if (ctx->old.pg_version > ctx->new.pg_version)
		pg_log(ctx, PG_FATAL, "This utility cannot be used to downgrade to older major PostgreSQL versions.\n");

	/* Must be built using _new_ server source */
	if (GET_MAJOR_VERSION(PG_VERSION_NUM) !=
		GET_MAJOR_VERSION(ctx->new.pg_version))
		pg_log(ctx, PG_FATAL,
				"This binary was built using %s PostgreSQL sources; it must be\n"
				"built using %s PostgreSQL sources to match the new cluster.\n",
				PG_MAJORVERSION, ctx->new.pg_version_str);
}


/*
 *	check_cluster_compatibility()
 */
static void
check_cluster_compatibility(migratorContext *ctx)
{
	char		libfile[MAXPGPATH];
	FILE 		*lib_test;

	/*
	 *	Test pg_migrator.so is in the proper place.   We cannot copy it
	 *	ourselves because install directories are typically root-owned.
	 */
	snprintf(libfile, sizeof(libfile), "%s/pg_migrator%s", ctx->new.libpath,
				DLSUFFIX);

	if ((lib_test = fopen(libfile, "r")) == NULL)
		pg_log(ctx, PG_FATAL,
			"\npg_migrator%s must be created and installed in %s\n", DLSUFFIX, libfile);
	else
		fclose(lib_test);

	/* get/check pg_control data of servers */
	get_control_data(ctx, ctx->old.bindir, ctx->old.pgdata,
					 &ctx->old.controldata, ctx->old.pg_version);
	get_control_data(ctx, ctx->new.bindir, ctx->new.pgdata,
					 &ctx->new.controldata, ctx->new.pg_version);
	check_control_data(ctx, &ctx->old.controldata, &ctx->new.controldata);
}


/*
 *	cleanup()
 */
static void
cleanup(migratorContext *ctx)
{
	int			tblnum;
	char	filename[MAX_STRING];

	for (tblnum = 0; tblnum < ctx->num_tablespaces; tblnum++)
		pg_free(ctx->tablespaces[tblnum]);
	pg_free(ctx->tablespaces);

	dbarr_free(&ctx->old.dbarr);
	dbarr_free(&ctx->new.dbarr);
	pg_free(ctx->logfile);
	pg_free(ctx->user);
	pg_free(ctx->old.pg_version_str);
	pg_free(ctx->new.pg_version_str);
	pg_free(ctx->old.controldata.lc_collate);
	pg_free(ctx->new.controldata.lc_collate);
	pg_free(ctx->old.controldata.lc_ctype);
	pg_free(ctx->new.controldata.lc_ctype);
	pg_free(ctx->old.controldata.encoding);
	pg_free(ctx->new.controldata.encoding);
	
	if (ctx->log_fd != NULL)
	{
		fclose(ctx->log_fd);
		ctx->log_fd = NULL;
	}

	if (ctx->debug_fd)
		fclose(ctx->debug_fd);

	snprintf(filename, sizeof(filename), "%s/%s", ctx->home_dir, ALL_DUMP_FILE);
	unlink(filename);
	snprintf(filename, sizeof(filename), "%s/%s", ctx->home_dir, GLOBALS_DUMP_FILE);
	unlink(filename);
	snprintf(filename, sizeof(filename), "%s/%s", ctx->home_dir, DB_DUMP_FILE);
	unlink(filename);
}


/*
 * install_support_functions()
 *
 * pg_migrator requires some support functions, which should be dynamically
 * linked with the server. This routine installs those functions.
 */
static void
install_support_functions(migratorContext *ctx)
{
	int			dbnum;

	prep_status(ctx, "Adding support functions to new cluster");

	for (dbnum = 0; dbnum < ctx->new.dbarr.ndbs; dbnum++)
	{
		DbInfo	   *newdb = &ctx->new.dbarr.dbs[dbnum];
		PGconn	   *conn = connectToServer(ctx, newdb->db_name, CLUSTER_NEW);

		PQclear(executeQueryOrDie(ctx, conn,
					"CREATE FUNCTION public.pg_toasttbl_recreate(OID,OID) "
					"RETURNS VOID "
					"AS '$libdir/pg_migrator' "
					"LANGUAGE C STRICT;"));
		PQclear(executeQueryOrDie(ctx, conn,
					"CREATE FUNCTION public.pg_toasttbl_drop(OID) "
					"RETURNS VOID "
					"AS '$libdir/pg_migrator' "
					"LANGUAGE C STRICT;"));
		PQfinish(conn);
	}
	check_ok(ctx);
}



static void
rename_old_pg_control(migratorContext *ctx)
{
	char	old_path[MAX_STRING], new_path[MAX_STRING];

	prep_status(ctx, "Adding \".old\" suffix to old global/pg_control");

	snprintf(old_path, sizeof(old_path), "%s/global/pg_control", ctx->old.pgdata);
	snprintf(new_path, sizeof(new_path), "%s/global/pg_control.old", ctx->old.pgdata);
	if (pg_mv_file(old_path, new_path) != 0)
	{
#ifdef WIN32
		_dosmaperr(GetLastError());
#endif
		pg_log(ctx, PG_FATAL, "Unable to rename %s to %s.\n", old_path, new_path);
	}
	check_ok(ctx);
}
