/*
 *	tablespace.c
 *
 *	tablespace functions
 */

#include "pg_migrator.h"

static void get_tablespace_paths(migratorContext *ctx);
static void set_tablespace_directory_suffix(migratorContext *ctx,
							Cluster whichCluster);


/*
 * rename_tablespaces()
 *
 * rename tablespace paths in "tablespaces" array to *.old
 *
 * This will fail if the directory is a mount point, i.e. cannot be renamed
 */
void
rename_tablespaces(migratorContext *ctx)
{
	int			tblnum;
	char		old_path[MAXPGPATH];
	char		new_path[MAXPGPATH];

	prep_status(ctx, "Adding \".old\" suffix to tablespace directory names");

	for (tblnum = 0; tblnum < ctx->num_tablespaces; tblnum++)
	{
		/* rename the old tablespace directory to *.old */
		snprintf(old_path, sizeof(old_path), "%s", ctx->tablespaces[tblnum]);
		snprintf(new_path, sizeof(new_path), "%s.old", ctx->tablespaces[tblnum]);

		pg_log(ctx, PG_INFO, "moving %s to %s\n", old_path, new_path);
		if (pg_mv_file(old_path, new_path) != 0)
			pg_log(ctx, PG_FATAL,
			  "error while renaming old tablespace directory %s to %s: %s\n",
				   old_path, new_path, getErrorText(errno));

		/* create an empty directory for the current tablespace */
		if (mkdir(old_path, 0x755) == -1)
			pg_log(ctx, PG_FATAL,
				   "error while creating tablespace directory %s: %s\n",
				   old_path, getErrorText(errno));
	}

	check_ok(ctx);
}


void
init_tablespaces(migratorContext *ctx)
{
	get_tablespace_paths(ctx);

	set_tablespace_directory_suffix(ctx, CLUSTER_OLD);
	set_tablespace_directory_suffix(ctx, CLUSTER_NEW);

	if (ctx->num_tablespaces > 0 &&
		strcmp(ctx->old.tablespace_suffix, ctx->new.tablespace_suffix) == 0)
		pg_log(ctx, PG_FATAL,
			   "Cannot migrate to/from the same system catalog version when\n"
			   "using tablespaces.\n");
}


/*
 * get_tablespace_paths()
 *
 * Scans pg_tablespace and returns a malloc'ed array of all tablespace
 * paths. Its the caller's responsibility to free the array.
 */
static void
get_tablespace_paths(migratorContext *ctx)
{
	PGconn	   *conn = connectToServer(ctx, "template1", CLUSTER_OLD);
	PGresult   *res;
	int			ntups;
	int			tblnum;
	int			i_spclocation;

	res = executeQueryOrDie(ctx, conn,
							"SELECT	spclocation "
							"FROM	pg_catalog.pg_tablespace "
							"WHERE	spcname != 'pg_default' AND "
							"		spcname != 'pg_global'");

	ctx->num_tablespaces = ntups = PQntuples(res);
	ctx->tablespaces = (char **) pg_malloc(ctx, ntups * sizeof(char *));

	i_spclocation = PQfnumber(res, "spclocation");

	for (tblnum = 0; tblnum < ntups; tblnum++)
		ctx->tablespaces[tblnum] = pg_strdup(ctx,
									 PQgetvalue(res, tblnum, i_spclocation));

	PQclear(res);

	PQfinish(conn);

	return;
}


static void
set_tablespace_directory_suffix(migratorContext *ctx, Cluster whichCluster)
{
	ClusterInfo *cluster = (whichCluster == CLUSTER_OLD) ? &ctx->old : &ctx->new;

	/* Did we rename the old tablespace to ".old"? */
	if (GET_MAJOR_VERSION(ctx->new.major_version) <= 804)
	{
		if (whichCluster == CLUSTER_OLD)
			cluster->tablespace_suffix = pg_strdup(ctx, ".old");
		else
			cluster->tablespace_suffix = pg_strdup(ctx, "");
	}
	/* Does this cluster lack a version-specific subdirectory? */
	else if (GET_MAJOR_VERSION(cluster->major_version) <= 804)
		cluster->tablespace_suffix = pg_strdup(ctx, "");
	else
	{
		/* This cluster has a version-specific subdirectory */
		cluster->tablespace_suffix = pg_malloc(ctx, 4 + strlen(cluster->major_version_str) +
											   10 /* OIDCHARS */ + 1);

		/* The leading slash is needed to start a new directory. */
		sprintf(cluster->tablespace_suffix, "/PG_%s_%d", cluster->major_version_str,
				cluster->controldata.cat_ver);
	}
}
