/*
 *	pg_migrator.h
 */

#define	PG_MIGRATOR_VERSION	"8.4.7"

#include "postgres.h"

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include <dirent.h>
#include <sys/time.h>
#include <sys/types.h>
#ifdef WIN32
#include <shlobj.h>
#endif

#include "libpq-fe.h"

/* Allocate for null byte */
#define NAMEDATASIZE	(NAMEDATALEN + 1)

#define USER_NAME_SIZE	128

#define MAX_STRING		1024
#define LINE_ALLOC		4096
#define QUERY_ALLOC		8192
  
#define MIGRATOR_API_VERSION	1

#define MESSAGE_WIDTH		"60"

#define OVERWRITE_MESSAGE	"  %-" MESSAGE_WIDTH "." MESSAGE_WIDTH "s\r"
#define GET_MAJOR_VERSION(v)	((v) / 100)

#ifndef WIN32
#define pg_copy_file		copy_file
#define pg_mv_file			rename
#define pg_link_file		link
#define DEVNULL "/dev/null"
#define DEVTTY "/dev/tty"
#else
#define pg_copy_file		CopyFile
#define pg_mv_file		pgrename
#define pg_link_file		win32_pghardlink
#define EXE_EXT				".exe"
#define sleep(x)			Sleep(x * 1000)
#define DEVNULL "nul"
/* "con" does not work from the Msys 1.0.10 console (part of MinGW). */
#define DEVTTY  "con"
/* from pgport */
extern int  pgrename(const char *from, const char *to);
extern int  pgunlink(const char *path);
#define rename(from, to)	pgrename(from, to)
#define unlink(path)		pgunlink(path)
#endif

/* from pgport */
extern void copydir(char *fromdir, char *todir, bool recurse);
#if !defined(EDB_NATIVE_LANG) || GET_MAJOR_VERSION(PG_VERSION_NUM) >= 804
extern bool rmtree(const char *path, bool rmtopdir);
#else
/* Used for EDB AS 8.3 compiles */
extern bool rmtree(char *path, bool rmtopdir);
#endif

extern char pathSeparator;

/*
 * Each relation is represented by a relinfo structure.
 */
typedef struct
{
	char		nspname[NAMEDATASIZE];		/* namespace name */
	char		relname[NAMEDATASIZE];		/* relation name */
	Oid			reloid;			/* relation oid				 */
	Oid			relfilenode;	/* relation relfile node	 */
	Oid			toastrelid;		/* oid of the toast relation */
	char		tablespace[MAXPGPATH]; /* relations tablespace path */
} RelInfo;

typedef struct
{
	RelInfo    *rels;
	int			nrels;
} RelInfoArr;

/*
 * The following structure represents a relation mapping.
 */
typedef struct
{
	Oid			old;			/* Relfilenode of the old relation */
	Oid			new;			/* Relfilenode of the new relation */
	char		old_file[MAXPGPATH];
	char		new_file[MAXPGPATH];
	char		old_nspname[NAMEDATASIZE];	/* old name of the namespace */
	char		old_relname[NAMEDATASIZE];	/* old name of the relation */
	char		new_nspname[NAMEDATASIZE];	/* new name of the namespace */
	char		new_relname[NAMEDATASIZE];	/* new name of the relation */
} FileNameMap;

/*
 * Structure to store database information
 */
typedef struct
{
	Oid			db_oid;			/* oid of the database */
	char		db_name[NAMEDATASIZE];	/* database name */
	RelInfoArr	rel_arr;		/* array of all user relinfos */
} DbInfo;

typedef struct
{
	DbInfo	   *dbs;			/* array of db infos */
	int			ndbs;			/* number of db infos */
} DbInfoArr;

/*
 * The following structure is used to hold pg_control information.
 * Rather than using the backend's control structure we use our own
 * structure to avoid pg_control version issues between releases.
 */
typedef struct
{
	uint32		ctrl_ver;
	uint32		cat_ver;
	uint32		logid;
	uint32		nxtlogseg;
	uint32		chkpnt_tli;
	uint32		chkpnt_nxtxid;
	uint32		chkpnt_nxtoid;
	uint32		align;
	uint32		blocksz;
	uint32		largesz;
	uint32		walsz;
	uint32		walseg;
	uint32		ident;
	uint32		index;
	uint32		toast;
	bool		date_is_int;
	bool		float8_pass_by_value;
	char	   *lc_collate;
	char	   *lc_ctype;
	char	   *encoding;
} ControlData;

/*
 * Enumeration to denote link modes
 */
typedef enum
{
	TRANSFER_MODE_COPY,
	TRANSFER_MODE_LINK
} transferMode;

/*
 * Enumeration to denote pg_log modes
 */
typedef enum
{
	PG_INFO,
	PG_REPORT,
	PG_WARNING,
	PG_FATAL,
	PG_DEBUG
} eLogType;

/*
 * Enumeration to distinguish between old cluster and new cluster
 */
typedef enum
{
	NONE = 0,	/* used for no running servers */
	CLUSTER_OLD,
	CLUSTER_NEW
} Cluster;

typedef long pgpid_t;


/*
 * cluster
 *
 *	information about each cluster
 */
typedef struct
{
	ControlData controldata;	/* pg_control information */
	DbInfoArr	dbarr;		/* dbinfos array */
	char	   *pgdata;		/* pathname for cluster's $PGDATA directory */
	char	   *bindir;		/* pathname for cluster's executable directory */
	const char *psql_exe;	/* name of the psql command to execute
							 * in the cluster */
	unsigned short port;	/* port number where postmaster is waiting */
	uint32		pg_version;		/* PG_VERSION of cluster */
	char	   *pg_version_str;	/* string PG_VERSION of cluster */
	Oid			pg_database_oid;		/* OID of pg_database relation */
	char	   *libpath;	/* pathname for cluster's pkglibdir */
	/* EDB AS is PG 8.2 with 8.3 enhancements backpatched. */
	bool		is_edb_as;	/* EnterpriseDB's Postgres Plus Advanced Server? */
} ClusterInfo;


/*
 * migratorContext
 *
 *	We create a migratorContext object to store all of the information
 *	that we need to migrate a single cluster.
 */
typedef struct
{
	ClusterInfo		old, new;	/* old and new cluster information */
	const char *progname;		/* complete pathname for this program */
	char	   *exec_path;		/* full path to my executable */
	char	   *user;			/* username for clusters */
	char		home_dir[MAXPGPATH];	/* name of user's home directory - we
										 * dump the old cluster schema here */
	char	  **tablespaces;	/* tablespaces */
	int			num_tablespaces;
	char	  **libraries;		/* loadable libraries */
	int			num_libraries;
	pgpid_t		postmasterPID;	/* PID of currently running postmaster */
	Cluster	running_cluster;

	char	   *logfile;		/* name of log file (may be /dev/null) */
	FILE	   *log_fd;			/* log FILE */
	FILE	   *debug_fd;		/* debug-level log FILE */
	bool		check;			/* TRUE -> ask user for permission to make
								 * changes */
	bool		verbose;		/* TRUE -> be verbose in messages */
	bool		debug;			/* TRUE -> log more information */
	transferMode	transfer_mode;		/* copy files or link them? */
} migratorContext;


/*
 * Global variables
 */
char		scandir_file_pattern[MAX_STRING + 1];

/* controldata.c */

void get_control_data(migratorContext *ctx, ClusterInfo *cluster, bool live_check);
void check_control_data(migratorContext *ctx, ControlData *oldctrl,
				   ControlData *newctrl);
void set_locale_and_encoding(migratorContext *ctx, Cluster whichCluster);
void check_locale_and_encoding(migratorContext *ctx, ControlData *oldctrl,
				   ControlData *newctrl);


/* exec.c */

int			exec_prog(migratorContext *ctx, bool throw_error,
		  		const char *cmd,...);
void		verify_directories(migratorContext *ctx);
bool		is_server_running(migratorContext *ctx, const char *datadir);


/* file.c */

#ifdef PAGE_CONVERSION
typedef const char *(*pluginStartup) (uint16 migratorVersion,
								uint16 *pluginVersion, uint16 newPageVersion,
								   uint16 oldPageVersion, void **pluginData);
typedef const char *(*pluginConvertFile) (void *pluginData,
								   const char *dstName, const char *srcName);
typedef const char *(*pluginConvertPage) (void *pluginData,
								   const char *dstPage, const char *srcPage);
typedef const char *(*pluginShutdown) (void *pluginData);

typedef struct
{
	uint16		oldPageVersion;		/* Page layout version of the old
									 * cluster		 */
	uint16		newPageVersion;		/* Page layout version of the new
									 * cluster		 */
	uint16		pluginVersion;	/* API version of converter plugin */
	void	   *pluginData;	/* Plugin data (set by plugin) */
	pluginStartup startup;	/* Pointer to plugin's startup function */
	pluginConvertFile convertFile;	/* Pointer to plugin's file converter
										 * function */
	pluginConvertPage convertPage;	/* Pointer to plugin's page converter
										 * function */
	pluginShutdown shutdown;	/* Pointer to plugin's shutdown function */
} pageCnvCtx;

const char *setupPageConverter(migratorContext *ctx, pageCnvCtx **result);

#else
/* dummy */
typedef void *pageCnvCtx;
#endif

void		transfer_relfile(migratorContext *ctx, pageCnvCtx *pageConverter,
				 const char *fromfile, const char *tofile,
				 const char *oldnspname, const char *oldrelname,
				 const char *newnspname, const char *newrelname);
int			copy_file(const char *fromfile, const char *tofile, bool force);
void transfer_single_new_db(migratorContext *ctx, pageCnvCtx *pageConverter,
					   FileNameMap *maps, int size);
void check_hard_link(migratorContext *ctx);


/* info.c */

void		check_new_db_is_empty(migratorContext *ctx);
FileNameMap *gen_db_file_maps(migratorContext *ctx, DbInfo *old_db,
				DbInfo *new_db, int *nmaps, const char *old_pgdata,
				const char *new_pgdata);
void		get_db_and_rel_infos(migratorContext *ctx, DbInfoArr *db_arr,
			Cluster whichCluster);
void		get_loadable_libraries(migratorContext *ctx);
void		check_loadable_libraries(migratorContext *ctx);
DbInfo	   *dbarr_lookup_db(DbInfoArr *db_arr, const char *db_name);
RelInfo    *relarr_lookup_rel(migratorContext *ctx, RelInfoArr *rel_arr,
				const char *nspname, const char *relname,
				Cluster whichCluster);
void		dbarr_free(DbInfoArr *db_arr);
void print_maps(migratorContext *ctx, FileNameMap *maps, int n,
		   const char *dbName);

/* option.c */

void		parseCommandLine(migratorContext *ctx, int argc, char *argv[]);

/* relfilenode.c */

void		reserve_relfilenode_names(migratorContext *ctx);
void		get_pg_database_relfilenode(migratorContext *ctx, Cluster whichCluster);
void		get_tablespace_paths(migratorContext *ctx);
void		rename_tablespaces(migratorContext *ctx);
PGresult   *getDatabaseOIDs(PGconn *conn);
void		adjust_rels_to_use_fixed_toast_relfilenodes(migratorContext *ctx,
			DbInfoArr *olddb_arr, DbInfoArr *newdb_arr);
const char *transfer_all_new_dbs(migratorContext *ctx, DbInfoArr *olddb_arr,
				   DbInfoArr *newdb_arr, char *old_pgdata, char *new_pgdata);




/* server.c */

PGconn *connectToServer(migratorContext *ctx, const char *db_name,
				Cluster whichCluster);
PGresult *executeQueryOrDie(migratorContext *ctx, PGconn *conn,
				  const char *fmt,...);

void 		start_postmaster(migratorContext *ctx, Cluster whichCluster, bool quiet);
void		stop_postmaster(migratorContext *ctx, bool fast, bool quiet);
uint32		get_postgres_version(migratorContext *ctx, char **verstr,
								 Cluster whichCluster);
void		check_for_libpq_envvars(migratorContext *ctx);


/* util.c */

void		exit_nicely(migratorContext *ctx, bool need_cleanup);
void	   *pg_malloc(migratorContext *ctx, int n);
void		pg_free(void *p);
char	   *pg_strdup(migratorContext *ctx, const char *s);
char	   *quote_identifier(migratorContext *ctx, const char *s);
char	   *get_user_id(migratorContext *ctx);
void		check_ok(migratorContext *ctx);
void		report_status(migratorContext *ctx, eLogType type, const char *fmt,...);
void		pg_log(migratorContext *ctx, eLogType type, char *fmt,...);
void		prep_status(migratorContext *ctx, const char *fmt,...);
void		check_ok(migratorContext *ctx);
char	   *pg_strdup(migratorContext *ctx, const char *s);
void	   *pg_malloc(migratorContext *ctx, int size);
void		pg_free(void *ptr);

/* version.c */
void		v8_3_check_for_name_data_type_usage(migratorContext *ctx,
							Cluster whichCluster);
void		v8_3_check_for_tsquery_usage(migratorContext *ctx,
							Cluster whichCluster);
void		 v8_3_check_for_isn_and_int8_passing_mismatch(migratorContext *ctx,
							Cluster whichCluster);
void		v8_3_rebuild_tsvector_tables(migratorContext *ctx,
							bool check_mode, Cluster whichCluster);
void		v8_3_invalidate_hash_gin_indexes(migratorContext *ctx,
							bool check_mode, Cluster whichCluster);
void		v8_3_invalidate_bpchar_pattern_ops_indexes(migratorContext *ctx,
							bool check_mode, Cluster whichCluster);
void		v8_4_populate_pg_largeobject_metadata(migratorContext *ctx,
							bool check_mode, Cluster whichCluster);
char 		*v8_3_create_sequence_script(migratorContext *ctx,
							Cluster whichCluster);
void		v_8_4_check_for_composite_types(migratorContext *ctx,
							Cluster whichCluster);
void		v_8_4_check_for_array_types(migratorContext *ctx,
							Cluster whichCluster);
void		v_8_4_check_for_enum_types(migratorContext *ctx,
							Cluster whichCluster);
