/*--------------------------------------------------------------------
 * FILE:
 *     replicate.c
 *
 * NOTE:
 *     This file is composed of the functions to call with the source
 *     at backend for the replication.
 *     Low level I/O functions that called by in these functions are 
 *     contained in 'replicate_com.c'.
 *
 *--------------------------------------------------------------------
 */

/*--------------------------------------
 * INTERFACE ROUTINES
 *
 * setup/teardown:
 *      PGR_Init_Replicate_Server_Data
 *      PGR_Set_Replicate_Server_Socket
 *      PGR_delete_shm
 * I/O call:
 *      PGR_Send_Replicate_Command
 * table handling:
 *      PGR_get_replicate_server_info
 * status distinction:
 *      PGR_Is_Replicated_Command
 *      Xlog_Check_Replicatec
 * replicateion main:
 *      PGR_replication 
 *-------------------------------------
 */
#ifdef USE_REPLICATION

#include "postgres.h"

#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <netdb.h>
#include <netinet/in.h>
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#include <arpa/inet.h>
#include <sys/file.h>
#include <netdb.h>

#include "pg_config.h"
#include "libpq/libpq.h"
#include "libpq/pqformat.h"
#include "miscadmin.h"
#include "nodes/nodes.h"
#include "nodes/print.h"
#include "utils/guc.h"
#include "parser/parser.h"
#include "access/xact.h"
#include "replicate.h"

/* the source of this value is 'access/transam/varsup.c' */
#define VAR_OID_PREFETCH                (8192)

PGR_ClusterDB_Info ClusterDB_Info;
PGR_Partial_Info Partial_Info;
PGR_QueryString_Info QueryString_Info;
PGR_Lifecheck_Pid_Info Lifecheck_Pid_Info;
PGR_ReplicationServer_Info ReplicationServer_Info;
PGR_ReplicationLog_Info ReplicationLog_Info;
PGR_ClusterDB_Mode_Info ClusterDB_Mode_Info;

bool autocommit = true;

/*--------------------------------------
 * PROTOTYPE DECLARATION
 *--------------------------------------
 */
int PGR_Init_Replicate_Server_Data(char * path);
int PGR_Set_Replicate_Server_Socket(void);
char * PGR_Send_Replicate_Command(const char * query_string, int query_len, char cmdSts ,char cmdType);
bool PGR_Is_Replicated_Command(const char * query);
int Xlog_Check_Replicate(int operation);
int PGR_Replicate_Function_Call(void);
void PGR_delete_shm(void);
ReplicateServerInfo * PGR_get_replicate_server_info(void);
bool PGR_Is_System_Command(const char * query);
int PGR_Call_System_Command(char * command);
int PGR_GetTimeOfDay(struct timeval *tp, struct timezone *tpz);
long PGR_Random(void);
int PGR_Set_Current_Time(char * sec, char * usec);
int PGR_Send_Copy(CopyData * copy,int end);
CopyData * PGR_Set_Copy_Data(CopyData * copy, char * str, int len, int end);
char * PGR_scan_terminate(const char * str);
bool PGR_Is_Stand_Alone(void);
void PGR_Send_Message_To_Frontend(char * msg);
void PGR_Notice_Transaction_Query_Done(void);
void PGR_Notice_Transaction_Query_Aborted(void);
int PGR_Notice_Conflict(void);
int PGR_Recv_Trigger (int user_timeout);
void PGR_Set_Replication_Server_Status( ReplicateServerInfo * sp, int status);
int PGR_Is_Skip_Replication(const char * query);
bool PGR_Did_Commit_Transaction(void);
bool PGR_Set_Transaction_Mode(bool mode,const char * commandTag);
char * PGR_Remove_Comment(const char * str);
void PGR_Force_Replicate_Query(void);
void PGR_Notice_DeadLock(void);
void PGR_Set_Cluster_Status(int status);
int PGR_Get_Cluster_Status(void);
int PGR_lo_create(int flags);
int PGR_lo_open(Oid lobjId,int32 mode);
int PGR_lo_close(int32 fd);
int PGR_lo_write(int fd, char *buf, int len);
int PGR_lo_lseek(int32 fd, int32 offset, int32 whence);
int PGR_lo_unlink(Oid lobjId);
void PGR_init_replication(const char * query, int len);
void PGR_init_each_query(void);
int PGR_exec_replication(const char * commandTag,CommandDest dest );
int PGR_startup_lifecheck(void);
void PGR_finish_lifecheck(int sig);
int PGR_Wait_Reliable_Trigger(const char * commandTag);
void PGR_Set_Replication_Flag(Node * parsetree);
int PGR_Init_PGCluster_Info(void);
int PGR_Get_Cluster_Conf_Data(char * path);
void PGR_Clear_Children(int sig);
void PGR_Recovery_Failed(void);
void PGR_Elog_Abort_Copy(void);
void PGR_Elog_Abort_Proc(int timeout);
void PGR_Elog_Warn_Proc(char * msg, char * fix);
void PGR_ExceptionalCondition(void);
bool PGR_Is_This_Transaction_Committed(void);
void PGR_Set_Copy_Char(int ch);

static int PGR_replication(const char * query_string, CommandDest dest, const char * commandTag);
static int set_command_args(char argv[PGR_CMD_ARG_NUM][256],char *str);
static bool is_same_replication_server(ReplicateServerInfo * sp1, ReplicateServerInfo * sp2 );
static ReplicateServerInfo * search_new_replication_server ( ReplicateServerInfo * sp , int socket_type );

static int close_replicate_server_socket ( ReplicateServerInfo * sp , int socket_type );
static int recv_replicate_result(int sock,char * result,int user_timeout);
static int recv_message(int sock,char * buf,int flag);
static void clear_recv_queue(int sock);
static int send_replicate_packet(int sock,ReplicateHeader * header, const char * query_string);
static int get_replicate_server_socket ( ReplicateServerInfo * sp , int socket_type );
static bool is_copy_from(const char * query);
static int is_session_authorization(const char * query);
static int get_words( char words[MAX_WORDS][MAX_WORD_LETTERS] ,const char * string,int length,int upper);
static int get_table_name(char * table_name, const char * query, int position );
static bool is_not_replication_query(const char * query_string, int query_len, char cmdType);
static int Comp_Not_Replicate(PGR_Not_Replicate_Type * nrp1,PGR_Not_Replicate_Type* nrp2);
static bool is_serial_control_query(char cmdType, const char * query);
static bool is_select_into_query(char cmdType, const char * query);
static int send_response_to_replication_server(const char * notice);
static bool is_transaction_command(const char * commandTag);
static bool do_not_replication_command(const char * commandTag);
static bool is_create_temp_table(const char * query);
static int add_replication_server(char * hostname,char * port, char * recovery_port, char * lifecheck_port);
static int change_replication_server(char * hostname,char * port, char * recovery_port);
static int get_new_replication_socket( ReplicateServerInfo * base, ReplicateServerInfo * sp, int socket_type);
/*
static bool confirm_query_done(int sock, ReplicateHeader * header, char *result);
*/
static char * get_hostName(char * str);
static unsigned int get_next_request_id(void);
static bool is_this_query_replicated(char * id);
static int set_replication_id(char * id);
static void set_response_mode(char * mode);
#ifdef CONTROL_LOCK_CONFLICT
static int wait_lock_answer(void);
static int read_trigger(char * result, int buf_size);
#endif
static int return_current_oid(void);
static int sync_oid(char * oid);

extern ssize_t secure_read(Port *, void *, size_t);
/*--------------------------------------------------------------------
 * SYMBOL
 *    PGR_Init_Replicate_Server_Data()
 * NOTES
 *    Read Configuration file and create ReplicationServer_Info.ReplicateServerData table
 * ARGS
 *    void
 * RETURN
 *    OK: STATUS_OK
 *    NG: STATUS_ERROR
 *--------------------------------------------------------------------
 */
int
PGR_Init_Replicate_Server_Data(char * path)
{
	int table_size,str_size;
	ReplicateServerInfo  *sp;
	PGR_Not_Replicate_Type * nrp;
	ConfDataType * conf;
	int rec_no,cnt;
	char fname[256];
	int status = STATUS_OK;

	if (PGCluster_Info == NULL)
	{
		return STATUS_ERROR;
	}
	if (PGCluster_Info->Com_Info.ConfData_Info.ConfData_Top == (ConfDataType *)NULL)
	{
		return STATUS_ERROR;
	}

	/* allocate replication server information table */
	table_size = sizeof(ReplicateServerInfo) * MAX_SERVER_NUM;
	ClusterDB_Info.ReplicateServerShmid = shmget(IPC_PRIVATE,table_size,IPC_CREAT | IPC_EXCL | 0600);
	if (ClusterDB_Info.ReplicateServerShmid < 0)
	{
		return STATUS_ERROR;
	}
	ReplicationServer_Info.ReplicateServerData = (ReplicateServerInfo *)shmat(ClusterDB_Info.ReplicateServerShmid,0,0);
	if (ReplicationServer_Info.ReplicateServerData == (ReplicateServerInfo *)-1)
	{
		return STATUS_ERROR;
	}
	memset(ReplicationServer_Info.ReplicateServerData,0,table_size);
	sp = ReplicationServer_Info.ReplicateServerData;

	/* allocate cluster db information table */
	ClusterDB_Info.ClusterDBShmid = shmget(IPC_PRIVATE,sizeof(ClusterDBInfo),IPC_CREAT | IPC_EXCL | 0600);
	if (ClusterDB_Info.ClusterDBShmid < 0)
	{
		return STATUS_ERROR;
	}
	ClusterDB_Info.ClusterDBData = (ClusterDBInfo *)shmat(ClusterDB_Info.ClusterDBShmid,0,0);
	if (ClusterDB_Info.ClusterDBData == (ClusterDBInfo *)-1)
	{
		return STATUS_ERROR;
	}
	memset(ClusterDB_Info.ClusterDBData,0,sizeof(ClusterDBInfo));
	PGR_Set_Cluster_Status(STATUS_REPLICATE);

	/* allocate partial replicate table */
	table_size = sizeof(PGR_Not_Replicate_Type) * MAX_SERVER_NUM;
	Partial_Info.PGR_Not_Replicate = malloc(table_size);
	if (Partial_Info.PGR_Not_Replicate == (PGR_Not_Replicate_Type*)NULL)
	{
		return STATUS_ERROR;
	}
	memset(Partial_Info.PGR_Not_Replicate, 0, table_size);
	nrp = Partial_Info.PGR_Not_Replicate;
	cnt = 0;
	conf = PGCluster_Info->Com_Info.ConfData_Info.ConfData_Top;
	while ((conf != (ConfDataType *)NULL) && (cnt < MAX_SERVER_NUM))
	{
		/* set replication server table */
		if (!strcmp(conf->table,REPLICATION_SERVER_INFO_TAG))
		{
			rec_no = conf->rec_no;
			cnt = rec_no;
			if (!strcmp(conf->key,HOST_NAME_TAG))
			{
				strncpy((sp + rec_no)->hostName,conf->value,sizeof(sp->hostName));
				conf = (ConfDataType *)conf->next;
				continue;
			}
			if (!strcmp(conf->key,PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					(sp + rec_no)->portNumber = atoi(conf->value);
				}
				else
				{
					(sp + rec_no)->portNumber = DEFAULT_PGRP_PORT;
				}
				(sp + rec_no)->sock = -1;
				if ((sp + rec_no)->useFlag != DATA_USE)
				{
					PGR_Set_Replication_Server_Status((sp+rec_no), DATA_USE);
				}
				memset((sp + rec_no + 1)->hostName,0,sizeof(sp->hostName));
				(sp + rec_no + 1)->useFlag = DATA_END;
				conf = (ConfDataType *)conf->next;
				continue;
			}
			if (!strcmp(conf->key,LIFECHECK_PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					(sp + rec_no)->lifecheckPortNumber = atoi(conf->value);
				}
				else
				{
					(sp + rec_no)->lifecheckPortNumber = DEFAULT_PGRP_LIFECHECK_PORT;
				}
				conf = (ConfDataType *)conf->next;
				continue;
			}
			if (!strcmp(conf->key,RECOVERY_PORT_TAG))
			{
				if (atoi(conf->value) > 0)
				{
					(sp + rec_no)->recoveryPortNumber = atoi(conf->value);
				}
				else
				{
					(sp + rec_no)->recoveryPortNumber = DEFAULT_PGRP_RECOVERY_PORT;
				}
				if ((sp + rec_no)->useFlag != DATA_USE)
				{
					PGR_Set_Replication_Server_Status((sp+rec_no), DATA_USE);
				}
				memset((sp + rec_no + 1)->hostName,0,sizeof(sp->hostName));
				(sp + rec_no + 1)->useFlag = DATA_END;
				conf = (ConfDataType *)conf->next;
				continue;
			}
		}
		/* set part replication table */
		if (!strcmp(conf->table,NOT_REPLICATE_INFO_TAG))
		{
			rec_no = conf->rec_no;
			cnt = rec_no;
			if (Partial_Info.PGR_Not_Replicate_Rec_Num < rec_no +1)
			{
				Partial_Info.PGR_Not_Replicate_Rec_Num = rec_no +1;
			}
			if (!strcmp(conf->key,DB_NAME_TAG))
			{
				strncpy((nrp + rec_no)->db_name,conf->value,sizeof(nrp->db_name));
				conf = (ConfDataType *)conf->next;
				continue;
			}
			if (!strcmp(conf->key,TABLE_NAME_TAG))
			{
				strncpy((nrp + rec_no)->table_name,conf->value,sizeof(nrp->table_name));
				conf = (ConfDataType *)conf->next;
				continue;
			}
		}

		/* set cluster db data */
		if (!strcmp(conf->key,RECOVERY_PORT_TAG))
		{
			if (atoi(conf->value))
			{
				PGCluster_Info->Com_Info.ConfData_Info.Recovery_Port_Number = atoi(conf->value);
			}
			else
			{
				PGCluster_Info->Com_Info.ConfData_Info.Recovery_Port_Number = DEFAULT_CLUSTER_RECOVERY_PORT;
			}
		}
		if (!strcmp(conf->key,LIFECHECK_PORT_TAG))
		{
			if (atoi(conf->value) > 0)
			{
				PGCluster_Info->Com_Info.ConfData_Info.LifeCheck_Port_Number = atoi(conf->value);
			}
			else
			{
				PGCluster_Info->Com_Info.ConfData_Info.LifeCheck_Port_Number = DEFAULT_CLUSTER_LIFECHECK_PORT;
			}
		}
		if (!strcmp(conf->key,STATUS_LOG_FILE_TAG))
		{
			PGCluster_Info->Com_Info.Log_Info.PGRStatusFileName = strdup(conf->value);
			conf = (ConfDataType*)conf->next;
			continue;
		}
		if (!strcmp(conf->key,ERROR_LOG_FILE_TAG))
		{
			PGCluster_Info->Com_Info.Log_Info.PGRLogFileName = strdup(conf->value);
			conf = (ConfDataType*)conf->next;
			continue;
		}
		if (!strcmp(conf->key,RSYNC_PATH_TAG))
		{
			str_size = strlen(conf->value) ;
			PGCluster_Info->RsyncPath = malloc(str_size + 1);
			if (PGCluster_Info->RsyncPath == NULL)
			{
				return STATUS_ERROR;
			}
			memset(PGCluster_Info->RsyncPath,0,str_size + 1);
			strncpy(PGCluster_Info->RsyncPath,conf->value,str_size);
		}
		if (!strcmp(conf->key,RSYNC_OPTION_TAG))
		{
			str_size = strlen(conf->value) ;
			PGCluster_Info->RsyncOption = malloc(str_size + 1);
			if (PGCluster_Info->RsyncOption == NULL)
			{
				return STATUS_ERROR;
			}
			memset(PGCluster_Info->RsyncOption,0,str_size + 1);
			strncpy(PGCluster_Info->RsyncOption,conf->value,str_size);
		}
		if (!strcmp(conf->key,STAND_ALONE_TAG))
		{
			if (PGCluster_Info->PGR_Stand_Alone == (PGR_Stand_Alone_Type*)NULL)
			{
				PGCluster_Info->PGR_Stand_Alone = (PGR_Stand_Alone_Type*)malloc(sizeof(PGR_Stand_Alone_Type));
			}
			if (PGCluster_Info->PGR_Stand_Alone == (PGR_Stand_Alone_Type *)NULL)
			{
				return STATUS_ERROR;
			}
			memset(PGCluster_Info->PGR_Stand_Alone, 0, sizeof(PGR_Stand_Alone_Type));
			PGCluster_Info->PGR_Stand_Alone->is_stand_alone = false;
			if (!strcmp(conf->value,READ_WRITE_IF_STAND_ALONE))
			{
				PGCluster_Info->PGR_Stand_Alone->permit = PERMIT_READ_WRITE;
			}
			else
			{
				PGCluster_Info->PGR_Stand_Alone->permit = PERMIT_READ_ONLY;
			}
		}
		conf = (ConfDataType *)conf->next;
	}
	if (PGCluster_Info->Com_Info.Log_Info.Log_Print)
	{
		if (PGCluster_Info->Com_Info.Log_Info.PGRLogFileName == NULL)
		{
			snprintf(fname,sizeof(fname),"%s/%s",path,CLUSTER_ERROR_LOG_FILE);
		}
		else
		{
			strncpy(fname,PGCluster_Info->Com_Info.Log_Info.PGRLogFileName,sizeof(fname));
		}
		PGCluster_Info->Com_Info.Log_Info.LogFp = fopen(fname,"a");
		if (PGCluster_Info->Com_Info.Log_Info.LogFp == NULL)
		{
			if (Debug_pretty_print)
				elog(DEBUG1, "fopen failed: (%s)",strerror(errno));
			return STATUS_ERROR;
		}
	}

	if (PGCluster_Info->Com_Info.Log_Info.PGRStatusFileName == NULL)
	{
		snprintf(fname,sizeof(fname),"%s/%s",path,CLUSTER_STATUS_LOG_FILE);
	}
	else
	{
		strncpy(fname,PGCluster_Info->Com_Info.Log_Info.PGRStatusFileName,sizeof(fname));
	}
	PGCluster_Info->Com_Info.Log_Info.StatusFp = fopen(fname,"a");
	if (PGCluster_Info->Com_Info.Log_Info.StatusFp == NULL)
	{
		if (Debug_pretty_print)
			elog(DEBUG1, "fopen failed: (%s)",strerror(errno));
		return STATUS_ERROR;
	}
	ReplicationServer_Info.TransactionSock = -1;
	if (PGCluster_Info->ReplicateCurrentTime == (ReplicateNow *)NULL)
	{
		PGCluster_Info->ReplicateCurrentTime = (ReplicateNow *)malloc(sizeof(ReplicateNow));
	}
	if (PGCluster_Info->ReplicateCurrentTime == (ReplicateNow *)NULL)
	{
		return STATUS_ERROR;
	}
	memset(PGCluster_Info->ReplicateCurrentTime,0,sizeof(ReplicateNow));
	if (PGCluster_Info->PGRCopyData == (CopyData *)NULL)
	{
		PGCluster_Info->PGRCopyData = (CopyData *)malloc(sizeof(CopyData));
	}
	if (PGCluster_Info->PGRCopyData == (CopyData *)NULL)
	{
		return STATUS_ERROR;
	}
	memset(PGCluster_Info->PGRCopyData,0,sizeof(CopyData));

	if (Partial_Info.PGR_Not_Replicate_Rec_Num  == 0)
	{
		free(Partial_Info.PGR_Not_Replicate);
		Partial_Info.PGR_Not_Replicate = NULL;
	}
	else
	{
		qsort((char *)Partial_Info.PGR_Not_Replicate,Partial_Info.PGR_Not_Replicate_Rec_Num,sizeof(PGR_Not_Replicate_Type), (int (*)(const void*,const void*))Comp_Not_Replicate);
	}

	ClusterDB_Info.PGRSelfHostName = malloc(HOSTNAME_MAX_LENGTH);
	if (ClusterDB_Info.PGRSelfHostName == NULL)
	{
		return STATUS_ERROR;
	}
	memset(ClusterDB_Info.PGRSelfHostName,0,HOSTNAME_MAX_LENGTH);
	if (gethostname(ClusterDB_Info.PGRSelfHostName,HOSTNAME_MAX_LENGTH) < 0)
	{
		return STATUS_ERROR;
	}

	/* initialize global tables */
	status = PGRsyn_init();
	if (status != STATUS_OK)
	{
		return STATUS_ERROR;
	}

	return	STATUS_OK;
}

/*--------------------------------------------------------------------
 * SYMBOL
 *    PGR_Set_Replicate_Server_Socket()
 * NOTES
 *    Create new socket and set ReplicationServer_Info.ReplicateServerData table
 * ARGS
 *    void
 * RETURN
 *    OK: STATUS_OK
 *    NG: STATUS_ERROR
 *--------------------------------------------------------------------
 */
int
PGR_Set_Replicate_Server_Socket(void)
{
	ReplicateServerInfo * sp;
	if (PGCluster_Info == NULL)
	{
		return STATUS_ERROR;
	}
	if (ReplicationServer_Info.ReplicateServerData == NULL)
	{
		return STATUS_ERROR;
	}
	sp = ReplicationServer_Info.ReplicateServerData;
	while (sp->useFlag != DATA_END){
		sp->sock = -1;
		PGR_Create_Socket_Connect(&(sp->sock),sp->hostName,sp->portNumber);
		sp ++;
	}
	return	STATUS_OK;
}

/*--------------------------------------------------------------------
 * SYMBOL
 *    get_replicate_server_socket()
 * NOTES
 *    search or create a socket to connect with the replication server
 * ARGS
 *    ReplicateServerInfo * sp: replication server data (I)
 *    int socket_type: socket type (I)
 *                       -PGR_TRANSACTION_SOCKET:
 *                       -PGR_QUERY_SOCKET:
 * RETURN
 *    OK: >0(socket)
 *    NG: -1
 *--------------------------------------------------------------------
 */
static int
get_replicate_server_socket ( ReplicateServerInfo * sp , int socket_type )
{
	ReplicateServerInfo * tmp;

	tmp = sp;
	if (tmp == (ReplicateServerInfo *) NULL)
	{
		return -1;
	}
	if (tmp->hostName[0] == '\0')
	{
		return -1;
	}

	if (ReplicationServer_Info.TransactionSock > 0)
	{
		return ReplicationServer_Info.TransactionSock;
	}

	while(PGR_Create_Socket_Connect(&ReplicationServer_Info.TransactionSock,tmp->hostName,tmp->portNumber) != STATUS_OK)
	{
		if (ReplicationServer_Info.TransactionSock > 0)
			close(ReplicationServer_Info.TransactionSock);
		ReplicationServer_Info.TransactionSock = -1;
		PGR_Set_Replication_Server_Status(tmp, DATA_ERR);
		tmp = PGR_get_replicate_server_info();
		if (tmp == (ReplicateServerInfo *)NULL)
		{
			return -1;
		}
		PGR_Set_Replication_Server_Status(tmp, DATA_USE);
	}
	return ReplicationServer_Info.TransactionSock;
}

/*--------------------------------------------------------------------
 * SYMBOL
 *    close_replicate_server_socket()
 * NOTES
 *    close the socket connected with the replication server
 * ARGS
 *    ReplicateServerInfo * sp: replication server data (I)
 *    int socket_type: socket type (I)
 *                       -PGR_TRANSACTION_SOCKET:
 *                       -PGR_QUERY_SOCKET:
 * RETURN
 *    OK: STATUS_OK
 *    NG: STATUS_ERROR
 *--------------------------------------------------------------------
 */
static int
close_replicate_server_socket ( ReplicateServerInfo * sp , int socket_type )
{
	if (sp == (ReplicateServerInfo *)NULL )
	{
		return STATUS_ERROR;
	}
	if (sp->hostName[0] == '\0')
	{
		return STATUS_ERROR;
	}
	if (ReplicationServer_Info.TransactionSock != -1)
	{
		PGR_Close_Sock(&(ReplicationServer_Info.TransactionSock));
		ReplicationServer_Info.TransactionSock = -1;
	}
	switch (socket_type)
	{
		case PGR_TRANSACTION_SOCKET:
			if (ReplicationServer_Info.TransactionSock != -1)
			{
				PGR_Close_Sock(&(ReplicationServer_Info.TransactionSock));
			}
			ReplicationServer_Info.TransactionSock = -1;
			break;
		case PGR_QUERY_SOCKET:
			if (sp->sock != -1)
			{
				PGR_Close_Sock(&(sp->sock));
			}
			sp->sock = -1;
			break;
	}
	PGR_Set_Replication_Server_Status(sp, DATA_INIT);
	return STATUS_OK;
}

static bool
is_same_replication_server(ReplicateServerInfo * sp1, ReplicateServerInfo * sp2 )
{
	if ((sp1 == NULL) || (sp2 == NULL))
	{
		return false;
	}
	if ((!strcmp(sp1->hostName,sp2->hostName)) &&
		(sp1->portNumber == sp2->portNumber) &&
		(sp1->recoveryPortNumber == sp2->recoveryPortNumber))
	{
		return true;
	}
	return false;
}

static ReplicateServerInfo *
search_new_replication_server ( ReplicateServerInfo * sp , int socket_type )
{
	ReplicateHeader dummy_header;
	ReplicateServerInfo * rs_tbl;
	char command[256];
	int sock = -1;

	if ((ReplicationServer_Info.ReplicateServerData == NULL) || ( sp == NULL))
	{
		return NULL;
	}
	rs_tbl = sp;
	close_replicate_server_socket ( sp , socket_type);
	sp ++;
	while (is_same_replication_server(sp,rs_tbl) != true)
	{
		if (sp->useFlag == DATA_END)
		{
			sp = ReplicationServer_Info.ReplicateServerData;
		}
		sock = get_replicate_server_socket( sp , socket_type);
		if (sock <= 0 )
		{
			if (is_same_replication_server(sp,rs_tbl) == true)
			{
				return NULL;
			}
			else
			{
				sp++;
			}
			continue;
		}
		memset(&dummy_header, 0, sizeof(ReplicateHeader));
		memset(command,0,sizeof(command));
		snprintf(command,sizeof(command)-1,"SELECT %s(%d,%s,%d,%d)",
				PGR_SYSTEM_COMMAND_FUNC,
				PGR_CHANGE_REPLICATION_SERVER_FUNC_NO,
				sp->hostName,
				sp->portNumber,
				sp->recoveryPortNumber);
		dummy_header.cmdSys = CMD_SYS_CALL;
		dummy_header.cmdSts = CMD_STS_NOTICE;
		dummy_header.query_size = htonl(strlen(command));
		if (send_replicate_packet(sock,&dummy_header,command) != STATUS_OK)
		{
			close_replicate_server_socket ( sp , socket_type);
			PGR_Set_Replication_Server_Status(sp, DATA_ERR);
		}
		else
		{
			PGR_Set_Replication_Server_Status(sp, DATA_USE);
			return sp;
		}
		sp++;
	}
	return NULL;
}

static int
get_table_name(char * table_name, const char * query, int position )
{
	
	int i,wc;
	char * p;
	char * sp;
	int length;

	if ((table_name == NULL) || (query == NULL) || (position < 1))
	{
		return STATUS_ERROR;
	}
	length = strlen(query);
	p = (char *)query;
	wc = 1;
	sp = table_name;
	for (i = 0 ; i < length ; i ++)
	{
		while(isspace(*p))
		{
			p++;
			i++;
		}
		while((*p != '\0') && (! isspace(*p)))
		{
			if ((*p == ';') || (*p == '('))
				break;
			if (wc == position)
			{
				*sp = *p;
				sp++;
			}
			p++;
			i++;
		}
		if (wc == position)
		{
			*sp = '\0';
			break;
		}
		wc++;
	}
	return STATUS_OK;
}

static bool 
is_not_replication_query(const char * query_string, int query_len, char cmdType)
{
	PGR_Not_Replicate_Type key;
	PGR_Not_Replicate_Type * ptr = NULL;

	if (Partial_Info.PGR_Not_Replicate_Rec_Num <= 0)
		return false;
	if (query_string == NULL)
		return true;
	memset(&key,0,sizeof(PGR_Not_Replicate_Type));
	strncpy(key.db_name ,(char *)(MyProcPort->database_name),sizeof(key.db_name));
	switch (cmdType)
	{
		case CMD_TYPE_INSERT:
			get_table_name(key.table_name,query_string,3);
			break;
		case CMD_TYPE_UPDATE:
			get_table_name(key.table_name,query_string,2);
			break;
		case CMD_TYPE_DELETE:
			get_table_name(key.table_name,query_string,3);
			break;
		case CMD_TYPE_COPY:
			get_table_name(key.table_name,query_string,2);
			break;
		default:
			return false;
	}
	ptr = (PGR_Not_Replicate_Type*)bsearch((void*)&key,(void*)Partial_Info.PGR_Not_Replicate,Partial_Info.PGR_Not_Replicate_Rec_Num,sizeof(PGR_Not_Replicate_Type), (int (*)(const void*,const void*))Comp_Not_Replicate);
	if (ptr == NULL)
	{
		return false;
	}
	return true;

}

/*--------------------------------------------------------------------
 * SYMBOL
 *    PGR_Send_Replicate_Command()
 * NOTES
 *    create new socket
 * ARGS
 *    char * query_string: query strings (I)
 *    char cmdSts: 
 *    char cmdType:
 * RETURN
 *    OK: result
 *    NG: NULL
 *--------------------------------------------------------------------
 */
char *
PGR_Send_Replicate_Command(const char * query_string, int query_len, char cmdSts ,char cmdType)
{
	int sock = 0;
	int cnt = 0;
	ReplicateHeader header;
	char * serverName = NULL;
	int portNumber=0;
	char * result = NULL;
	ReplicateServerInfo * sp = NULL;
	ReplicateServerInfo * base = NULL;
	int socket_type = 0;
	char argv[ PGR_CMD_ARG_NUM ][256];
	int argc = 0;
	int func_no = 0;
	int status = 0;
	int check_flag =0;

	/*
	 * check query string
	 */
	if (PGCluster_Info == NULL)
	{
		return NULL;
	}
	if ((query_string == NULL)  ||
		(query_len < 0))
	{
		return NULL;
	}
	/* check not replication query */
	if (is_not_replication_query(query_string, query_len, cmdType) == true)
	{
		ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate = false;
		return NULL;
	}

	if ((cmdSts == CMD_STS_TRANSACTION ) ||
		(cmdSts == CMD_STS_SET_SESSION_AUTHORIZATION ) ||
		(cmdSts == CMD_STS_TEMP_TABLE ))
	{
		socket_type = PGR_TRANSACTION_SOCKET ;
	}
	else
	{
		socket_type = PGR_QUERY_SOCKET ;
	}
	sp = PGR_get_replicate_server_info();
	if (sp == NULL)
	{
		if (Debug_pretty_print)
			elog(DEBUG1,"PGR_get_replicate_server_info get error");
		return NULL;
	}
	sock = get_replicate_server_socket( sp , socket_type);
	if (sock <= 0)
	{
		if (Debug_pretty_print)
			elog(DEBUG1,"get_replicate_server_socket fail");
		return NULL;
	}
	result = malloc(PGR_MESSAGE_BUFSIZE + 4);
	if (result == NULL)
	{
		return NULL;
	}
	memset(result,0,PGR_MESSAGE_BUFSIZE + 4);

	serverName = sp->hostName;
	portNumber = (int)sp->portNumber;
	memset(&header,0,sizeof(ReplicateHeader));
	header.cmdSys = CMD_SYS_REPLICATE;
	header.cmdSts = cmdSts;
	header.cmdType = cmdType;
	header.port = htons(PostPortNumber);
	header.pid = htons(getpid());
	header.query_size = htonl(query_len); 
	strncpy(header.dbName ,(char *)(MyProcPort->database_name),sizeof(header.dbName));
	strncpy(header.userName , (char *)(MyProcPort->user_name),sizeof(header.userName));
	gethostname(header.from_host,sizeof(header.from_host));
	header.request_id = htonl(get_next_request_id());
	header.rlog = 0;

	base = sp;
	ReplicationServer_Info.PGR_Sock_To_Replication_Server = sock;

retry_send_replicate_packet:

	clear_recv_queue(sock); 
	memset(result,0,PGR_MESSAGE_BUFSIZE + 4);
	cnt = 0;
	while (send_replicate_packet(sock,&header,query_string) != STATUS_OK)
	{
		if (cnt > MAX_RETRY_TIMES )
		{
			sock = get_new_replication_socket( base, sp, socket_type);
			if (sock < 0)
			{
				if (Debug_pretty_print)
					elog(DEBUG1,"all replication servers may be down");
				if (PGCluster_Info != NULL)
				{
					PGCluster_Info->PGR_Stand_Alone->is_stand_alone = true;
				}
				if (cmdSts == CMD_STS_TRANSACTION )
				{
					strcpy(result,PGR_REPLICATION_ABORT_MSG);
					return result;
				}
				free(result);
				result = NULL;
				return NULL;
			}
			ReplicationServer_Info.PGR_Sock_To_Replication_Server = sock;
			header.rlog = CONNECTION_SUSPENDED_TYPE;
			cnt = 0;
		}
		cnt ++;
	}

	memset(result,0,PGR_MESSAGE_BUFSIZE);
	if ((status = recv_replicate_result(sock,result,0)) <= 0)
	{
		/* replication server should be down */
		sock = get_new_replication_socket( base, sp, socket_type);
		if (sock < 0)
		{
			if (Debug_pretty_print)
				elog(DEBUG1,"all replication servers may be down");
			if (PGCluster_Info != NULL)
			{
				PGCluster_Info->PGR_Stand_Alone->is_stand_alone = true;
			}
			if (cmdSts == CMD_STS_TRANSACTION )
			{
				strcpy(result,PGR_REPLICATION_ABORT_MSG);
				return result;
			}
			if (result != NULL)
			{
				free(result);
				result = NULL;
			}
			return NULL;
		}
		ReplicationServer_Info.PGR_Sock_To_Replication_Server = sock;
		header.rlog = CONNECTION_SUSPENDED_TYPE;
		goto retry_send_replicate_packet;

		/* connection closed */
	}
	argc = set_command_args(argv,result);
	if (argc >= 1)
	{
		func_no = atoi(argv[0]);
		/* set current system time */
		if (func_no == PGR_SET_CURRENT_TIME_FUNC_NO)
		{
			if (atol(argv[1]) == 0)
			{
				CreateCheckPoint(false,true);
			}
			else
			{
				PGR_Set_Current_Time(argv[1],argv[2]);
				set_replication_id(argv[3]);
				set_response_mode(argv[4]);
			}
		}
		else if (func_no == PGR_NOTICE_DEADLOCK_DETECTION_FUNC_NO)
		{
			memset(result,0,PGR_MESSAGE_BUFSIZE);
			strcpy(result,PGR_DEADLOCK_DETECTION_MSG);
		}
		else if (func_no == PGR_QUERY_CONFIRM_ANSWER_FUNC_NO)
		{
			check_flag = atoi(argv[1]);
			if (check_flag == PGR_ALREADY_COMMITTED )
			{
				PGR_Set_Current_Time(argv[2],argv[3]);
				set_replication_id(argv[4]);
			}
			else
			{
				PGR_Set_Current_Time(argv[1],argv[2]);
				set_replication_id(argv[3]);
				/* this query is not replicated */
				/*
				free(result);
				return NULL;
				*/
			}
		}
	}
	return result;
}

static int
get_new_replication_socket( ReplicateServerInfo * base, ReplicateServerInfo * sp, int socket_type)
{
	int sock = 0;

	if (( base == NULL) ||
		( sp == NULL))
	{
		return -1;
	}
	close_replicate_server_socket ( sp , socket_type);
	PGR_Set_Replication_Server_Status(sp, DATA_ERR);
	sp = search_new_replication_server(base, socket_type);
	if (sp == NULL)
	{
		if (Debug_pretty_print)
			elog(DEBUG1,"all replication servers may be down");
		if (PGCluster_Info != NULL)
		{
			PGCluster_Info->PGR_Stand_Alone->is_stand_alone = true;
		}
		return -1;
	}
	sock = get_replicate_server_socket( sp , socket_type);
	return sock;
}


static int
recv_replicate_result(int sock,char * result,int user_timeout)
{
	fd_set      rmask;
	struct timeval timeout;
	int rtn;

	if (result == NULL)
	{
		return -1;
	}
	if (user_timeout == 0)
	{
		timeout.tv_sec = PGR_RECV_TIMEOUT * 3;
	}
	else
	{
		timeout.tv_sec = user_timeout;
	}
	timeout.tv_usec = 0;

	/*
	 * Wait for something to happen.
	 */
	rtn = 1;
	while (rtn)
	{
		FD_ZERO(&rmask);
		FD_SET(sock,&rmask);
		rtn = select(sock+1, &rmask, (fd_set *)NULL, (fd_set *)NULL, &timeout);
		if (rtn && FD_ISSET(sock, &rmask))
		{
			return (recv_message(sock, result,0));
		}
		else
		{
			if (errno != EINTR)
			{
				return -1;
			}
		}
	}
	return -1;
}

static int
recv_message(int sock,char * buf,int flag)
{
	int cnt = 0;
	int r = 0;
	char * read_ptr;
	int read_size = 0;
	cnt = 0;
	read_ptr = buf;

	for (;;)
	{
		r = recv(sock,read_ptr + read_size ,PGR_MESSAGE_BUFSIZE - read_size, flag); 
		if (r < 0)
		{
			if (errno == EINTR)
			{
				usleep(PGR_RECV_WAIT_MSEC);
				continue;
			}
#ifdef EAGAIN
			else if (errno == EAGAIN)
			{
				usleep(PGR_RECV_WAIT_MSEC);
				continue;
			}
#endif /* EAGAIN */
#ifdef ECONNREFUSED
			else if (errno == ECONNREFUSED)
			{
				usleep(PGR_RECV_WAIT_MSEC);
				continue;
			}
#endif /* ECONNREFUSED */
#ifdef ENOTCONN
			else if (errno == ENOTCONN)
			{
				usleep(PGR_RECV_WAIT_MSEC);
				continue;
			}
#endif /* ENOTCONN */
			if (cnt < PGR_RECV_RETRY_CNT )
			{
				cnt ++;
				usleep(PGR_RECV_WAIT_MSEC);
				continue;
			}
			else
			{
				if (read_size == 0)
				{
					return -1;
				}
			}
		}
		if (r > 0)
		{
			read_size += r;
			if (read_size == PGR_MESSAGE_BUFSIZE)
			{
				return read_size;
			}
		}
		if ((r == 0) && (read_size == 0))
		{
			return -1;
		}
		if (cnt < PGR_RECV_RETRY_CNT )
		{
			cnt ++;
			usleep(PGR_RECV_WAIT_MSEC);
			continue;
		}
		else
		{
			return -1;
		}
	}
}

static void
clear_recv_queue(int sock)
{
	fd_set      rmask;
	struct timeval timeout;
	char buf[PGR_MESSAGE_BUFSIZE];
	int rtn = 0;

	timeout.tv_sec = 0;
	timeout.tv_usec = 0;

	FD_ZERO(&rmask);
	FD_SET(sock,&rmask);

	memset(buf,0,sizeof(buf));
	rtn = 1;
	while (rtn)
	{
		FD_ZERO(&rmask);
		FD_SET(sock,&rmask);
		rtn = select(sock+1, &rmask, (fd_set *)NULL, (fd_set *)NULL, &timeout);
		if ((rtn > 0) && (FD_ISSET(sock, &rmask)))
		{
			if (recv_message(sock, buf,0) <= 0)
				break;
		}
		else if (errno == EINTR)
		{
			continue;
		}
		else
		{
			break;
		}
	}
}

static int
send_replicate_packet(int sock,ReplicateHeader * header, const char * query_string)
{
	int s = 0;
	int cnt = 0;
	char * send_ptr = NULL;
	char * buf = NULL;
	int send_size = 0;
	int buf_size = 0;
	int header_size = 0;
	int rtn = 0;
	fd_set      wmask;
	struct timeval timeout;
	int query_size = 0;

	/* check parameter */
	if ((sock <= 0) || (header == NULL))
	{
		return STATUS_ERROR;
	}
	query_size = ntohl(header->query_size);
	header_size = sizeof(ReplicateHeader);
	buf_size = header_size + query_size + 4;
	buf = (char *)malloc(buf_size);
	if (buf == (char *)NULL)
	{
		return STATUS_ERROR;
	}
	memset(buf,0,buf_size);
	buf_size -= 4;
	memcpy(buf,header,header_size);
	if (query_string != NULL)
	{
		memcpy((char *)(buf+header_size),query_string,query_size+1);
	}
	send_ptr = buf;

	timeout.tv_sec = PGR_SEND_TIMEOUT;
	timeout.tv_usec = 0;

	/*
	 * Wait for something to happen.
	 */
	rtn = 1;
	while (rtn)
	{
		FD_ZERO(&wmask);
		FD_SET(sock,&wmask);
		rtn = select(sock+1, (fd_set *)NULL, &wmask, (fd_set *)NULL, &timeout);
		if (rtn && FD_ISSET(sock, &wmask))
		{
			cnt = 0;
			send_size = 0;
			for (;;)
			{
				/* s = send(sock,send_ptr + send_size,buf_size - send_size ,MSG_DONTWAIT); */
				s = send(sock,send_ptr + send_size,buf_size - send_size ,0);
				if (s < 0){
					if (errno == EINTR)
					{
						usleep(PGR_SEND_WAIT_MSEC);
						continue;
					}
	#ifdef EAGAIN
					if (errno == EAGAIN)
					{
						usleep(PGR_SEND_WAIT_MSEC);
						continue;
					}
	#endif /* EAGAIN */
	#ifdef EWOULDBLOCK
					if (errno == EWOULDBLOCK )
					{
						usleep(PGR_SEND_WAIT_MSEC);
						continue;
					}
	#endif /* EWOULDBLOCK */
	#ifdef ENOBUF
					if (errno == ENOBUF )
					{
						usleep(PGR_SEND_WAIT_MSEC);
						continue;
					}
	#endif /* ENOBUF */
	#ifdef ENOMEM
					if (errno == ENOMEM )
					{
						usleep(PGR_SEND_WAIT_MSEC);
						continue;
					}
	#endif /* ENOMEM */
					if (cnt < PGR_SEND_RETRY_CNT )
					{
						cnt ++;
						usleep(PGR_SEND_WAIT_MSEC);
						continue;
					}
					else
					{
						if (send_size == 0)
						{
							free(buf);
							buf = NULL;
							return STATUS_ERROR;
						}
					}
				}
				if (s > 0)
				{
					send_size += s;
					if (send_size == buf_size)
					{
						free(buf);
						buf = NULL;
						return STATUS_OK;
					}
				}
				usleep(PGR_SEND_WAIT_MSEC);
			}
		}
	}
	if (buf != NULL)
	{
		free(buf);
		buf = NULL;
	}
	return STATUS_ERROR;
}

bool
PGR_Is_Replicated_Command(const char * query)
{

	return (PGR_Is_System_Command(query));
}

int
Xlog_Check_Replicate(int operation)
{
	if ((PGR_Get_Cluster_Status() == STATUS_RECOVERY)	&&
		(ClusterDB_Mode_Info.Transaction_Mode == false ) )
	{
		elog(WARNING, "This query is not permitted while recovery db ");
	}
	else if ((operation == CMD_UTILITY ) ||
		(operation == CMD_INSERT )  ||
		(operation == CMD_UPDATE )  ||
		(operation == CMD_DELETE ))
	{
		return (PGR_Replicate_Function_Call());
	}
	return STATUS_OK;
}

int 
PGR_Replicate_Function_Call(void)
{
	char *result = NULL;
	int status = STATUS_OK;

	if ((PGCluster_Info == NULL) ||
		(PGCluster_Info->PGR_Stand_Alone == NULL))
	{
		 return STATUS_OK;
	}
	if (QueryString_Info.Query_String != NULL)
	{
		if (PGR_Is_Stand_Alone() == true)
		{
			if (PGCluster_Info->PGR_Stand_Alone->permit == PERMIT_READ_ONLY)
			{
				QueryString_Info.Query_String = NULL;
				return STATUS_ERROR;
			}
		}
		PGCluster_Info->PGR_Need_Notice = true;
		PGCluster_Info->PGR_Check_Lock.check_lock_conflict = true;
        result = PGR_Send_Replicate_Command(QueryString_Info.Query_String,strlen(QueryString_Info.Query_String), CMD_STS_QUERY,CMD_TYPE_SELECT);
		if (result != NULL)
		{
			PGR_Reload_Start_Time();
			if (!strncmp(result,PGR_DEADLOCK_DETECTION_MSG,strlen(PGR_DEADLOCK_DETECTION_MSG)))
			{
				status = STATUS_DEADLOCK_DETECT;
			}
			free(result);
			result = NULL;
		}
		else
		{
			status = STATUS_ERROR;
		}
		QueryString_Info.Query_String = NULL;
    }
	return status;
}

void
PGR_delete_shm(void)
{
	/*
	if (PGCluster_Info == NULL)
	{
		return;
	}
	*/

	if (ReplicationServer_Info.ReplicateServerData != NULL)
	{
		shmdt(ReplicationServer_Info.ReplicateServerData);
		ReplicationServer_Info.ReplicateServerData = NULL;
		shmctl(ClusterDB_Info.ReplicateServerShmid,IPC_RMID,(struct shmid_ds *)NULL);
	}
	if (ClusterDB_Info.ClusterDBData != NULL)
	{
		shmdt(ClusterDB_Info.ClusterDBData);
		ClusterDB_Info.ClusterDBData = NULL;
		shmctl(ClusterDB_Info.ClusterDBShmid,IPC_RMID,(struct shmid_ds *)NULL);
	}
	if (ReplicationServer_Info.TransactionSock != -1)
	{
		close(ReplicationServer_Info.TransactionSock);
	}
	if (ClusterDB_Info.PGRSelfHostName != NULL)
	{
		free(ClusterDB_Info.PGRSelfHostName);
		ClusterDB_Info.PGRSelfHostName = NULL;
	}
	if (Partial_Info.PGR_Not_Replicate != NULL)
	{
		free(Partial_Info.PGR_Not_Replicate);
		Partial_Info.PGR_Not_Replicate = NULL;
	}
	PGRsyn_quit();
	if (PGCluster_Info == NULL) 
	{
		return;
	}

	if (PGCluster_Info->RsyncPath != NULL)
	{
		free(PGCluster_Info->RsyncPath);
		PGCluster_Info->RsyncPath = NULL;
	}
	if (PGCluster_Info->RsyncOption != NULL)
	{
		free(PGCluster_Info->RsyncOption);
		PGCluster_Info->RsyncOption = NULL;
	}

	if (PGCluster_Info->ReplicateCurrentTime != NULL)
	{
		free(PGCluster_Info->ReplicateCurrentTime);
		PGCluster_Info->ReplicateCurrentTime = NULL;
	}

	if (PGCluster_Info->PGRCopyData != NULL)
	{
		free (PGCluster_Info->PGRCopyData);
		PGCluster_Info->PGRCopyData = NULL;
	}

	if (PGCluster_Info->PGR_Stand_Alone != NULL)
	{
		free(PGCluster_Info->PGR_Stand_Alone);
		PGCluster_Info->PGR_Stand_Alone = NULL;
	}

	if (PGCluster_Info->Com_Info.Log_Info.StatusFp != NULL)
	{
		fflush(PGCluster_Info->Com_Info.Log_Info.StatusFp);
		fclose(PGCluster_Info->Com_Info.Log_Info.StatusFp);
		PGCluster_Info->Com_Info.Log_Info.StatusFp = NULL;
	}
	if (PGCluster_Info->Com_Info.Log_Info.LogFp != NULL)
	{
		fflush(PGCluster_Info->Com_Info.Log_Info.LogFp);
		fclose(PGCluster_Info->Com_Info.Log_Info.LogFp);
		PGCluster_Info->Com_Info.Log_Info.LogFp = NULL;
	}
}

ReplicateServerInfo * 
PGR_get_replicate_server_info(void)
{
	ReplicateServerInfo * sp;

	if (ReplicationServer_Info.ReplicateServerData == NULL)
	{
		return (ReplicateServerInfo *)NULL;
	}
	sp = ReplicationServer_Info.ReplicateServerData;
	while (sp->useFlag != DATA_END)
	{
		if (sp->useFlag == DATA_USE )
		{
			if (ReplicationServer_Info.CurrentReplicateServer != NULL)
			{
				ReplicationServer_Info.LastReplicateServer = ReplicationServer_Info.CurrentReplicateServer;
			}
			ReplicationServer_Info.CurrentReplicateServer = sp;
			return (ReplicationServer_Info.CurrentReplicateServer);
		}
		sp++;
	}
	sp = ReplicationServer_Info.ReplicateServerData;
	while (sp->useFlag != DATA_END)
	{
		if (sp->useFlag != DATA_ERR )
		{
			if (ReplicationServer_Info.CurrentReplicateServer != NULL)
			{
				ReplicationServer_Info.LastReplicateServer = ReplicationServer_Info.CurrentReplicateServer;
			}
			ReplicationServer_Info.CurrentReplicateServer = sp;
			return (ReplicationServer_Info.CurrentReplicateServer);
		}
		sp++;
	}
	if (PGCluster_Info != NULL)
	{
		PGCluster_Info->PGR_Stand_Alone->is_stand_alone = true;
	}
	if (ReplicationServer_Info.CurrentReplicateServer != NULL)
	{
		ReplicationServer_Info.LastReplicateServer = ReplicationServer_Info.CurrentReplicateServer;
	}
	ReplicationServer_Info.CurrentReplicateServer = NULL;
	return (ReplicateServerInfo *)NULL;
}

int
PGR_Send_Copy(CopyData * copy,int end )
{
	char cmdSts,cmdType;
	char * p = NULL;
	char *result = NULL;
	char term[8];
	int i;
	/*int status = 0; */
	fflush(stderr);
	if (copy == NULL)
	{
	fflush(stderr);
		return STATUS_ERROR;
	}

	cmdSts = CMD_STS_COPY;

	if (ClusterDB_Mode_Info.Transaction_Mode)
	{
		cmdSts = CMD_STS_TRANSACTION ;
	}
	if (ClusterDB_Mode_Info.Session_Authorization_Mode)
	{
		cmdSts = CMD_STS_SET_SESSION_AUTHORIZATION ;
	}
	cmdType = CMD_TYPE_COPY_DATA;
	fflush(stderr);
	if (end)
	{
		memset(term,0,sizeof(term));
		cmdType = CMD_TYPE_COPY_DATA_END;
		i = copy->cnt-1;
		if ((copy->copy_data[i]) == EOF)
		{
			p = &(copy->copy_data[i]);
			strcpy(term,"\\.\n");
			strncpy(p,term,sizeof(term));
			copy->cnt += 2;
		}
	}
	fflush(stderr);
	result = PGR_Send_Replicate_Command(copy->copy_data, copy->cnt, cmdSts, cmdType);
	memset(copy,0,sizeof(CopyData));

	if (result != NULL)
	{
		PGR_Reload_Start_Time();
		free(result);
		result = NULL;
		return STATUS_OK;
	}
	else
	{
		return STATUS_ERROR;
	}
}

CopyData * 
PGR_Set_Copy_Data(CopyData * copy, char *str, int len,int end)
{
	int copy_size, send_size;
	char * ep;
	int rest_len;
	CopyData rest_data;
	int status = STATUS_OK;
	char buf[4096];
	memset (buf,0,sizeof(buf));

	if (PGCluster_Info == NULL)
	{
		return (CopyData *)NULL;
	}
	if ((ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate == false) ||
		(copy == NULL))
	{
		return (CopyData *)NULL;
	}
	if (len > 0)
	{
memcpy(buf,str,len);
		copy_size = COPYBUFSIZ - copy->cnt -4;
		send_size = 0;
		if (copy_size < len)
		{
			memcpy(&(copy->copy_data[copy->cnt]) ,str + send_size,copy_size);
			copy->cnt  += copy_size;
			send_size += copy_size;
			len -= copy_size;
			ep = strrchr(copy->copy_data,'\n');
			if (ep != NULL)
			{
				*ep = '\0';
				rest_len = copy->cnt - strlen(copy->copy_data) -1;
				memset(&rest_data,0,sizeof(CopyData));
				memcpy(rest_data.copy_data,(ep +1),rest_len);
				copy->cnt -= (rest_len );
				*ep = '\n';
				*(ep+1) = '\0';
				status = PGR_Send_Copy(copy,0);
				memcpy(copy->copy_data,rest_data.copy_data,rest_len);
				copy->cnt = rest_len;
			}
			status = PGR_Send_Copy(copy,0);
			/*copy_size = COPYBUFSIZ - copy->cnt -1;*/
		}
		memcpy(&(copy->copy_data[copy->cnt]) ,str + send_size,len);
		copy->cnt  += len;
	}
	if (end)
	{
		status = PGR_Send_Copy(copy,end);
	}

	if (status != STATUS_OK)
	{
		return (CopyData *)NULL;
	}

	return copy;
}

static int
PGR_replication(const char * query_string, CommandDest dest, const char * commandTag)
{
	char *result = NULL;
	char cmdSts = CMD_STS_OTHER;
	char cmdType = CMD_TYPE_OTHER;
	int query_len = 0;
	int session_type;

	if ((PGCluster_Info == NULL)   ||
		(query_string == NULL) ||
		(commandTag == NULL))
	{
		return STATUS_ERROR;
	}

	QueryString_Info.Query_String = NULL;
	query_len = strlen(query_string);

	/* save query data for retry */
	PGCluster_Info->PGR_Retry_Query.query_string = query_string;
	PGCluster_Info->PGR_Retry_Query.query_len = query_len;
	PGCluster_Info->PGR_Retry_Query.useFlag = DATA_USE;
	PGCluster_Info->PGR_Retry_Query.cmdSts = CMD_STS_QUERY ;

	if (PGCluster_Info->PGR_Retry_Query.cmdType == CMD_TYPE_COPY) 
	{
		if (is_copy_from(query_string))
		{
			PGCluster_Info->PGR_Retry_Query.cmdSts = CMD_STS_COPY;
			ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate = true;
		}
		else
		{
			ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate = false;
			return STATUS_NOT_REPLICATE;
		}
	}
	else if (PGCluster_Info->PGR_Retry_Query.cmdType == CMD_TYPE_SET) 
	{
		session_type = is_session_authorization(query_string);
		switch (session_type)
		{
			case SESSION_AUTHORIZATION_BEGIN:
				PGCluster_Info->PGR_Retry_Query.cmdType = CMD_TYPE_SESSION_AUTHORIZATION_BEGIN ;
				ClusterDB_Mode_Info.Session_Authorization_Mode = true;
				break;
			case SESSION_AUTHORIZATION_END:
				PGCluster_Info->PGR_Retry_Query.cmdType = CMD_TYPE_SESSION_AUTHORIZATION_END ;
				ClusterDB_Mode_Info.Session_Authorization_Mode = false;
				break;
		}
	}
	else if (!strcmp(commandTag,"CREATE TABLE")) 
	{
		if (is_create_temp_table(query_string))
		{
			ClusterDB_Mode_Info.Create_Temp_Table_Mode = true;
		}
	}
	if (ClusterDB_Mode_Info.Create_Temp_Table_Mode)
	{
		PGCluster_Info->PGR_Retry_Query.cmdSts = CMD_STS_TEMP_TABLE ;
	}

	/* check partitional replication table */
	if (is_not_replication_query(query_string, query_len, cmdType)== true )
	{
		ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate = false;
		return STATUS_NOT_REPLICATE;
	}

	if ((is_transaction_command(commandTag) == true) ||
		(autocommit == false ))
	{
		ClusterDB_Mode_Info.Transaction_Mode = true;
	}
	if (ClusterDB_Mode_Info.Transaction_Mode)
	{
		PGCluster_Info->PGR_Retry_Query.cmdSts = CMD_STS_TRANSACTION ;
		if ((PGCluster_Info->PGR_Retry_Query.cmdType == CMD_TYPE_COMMIT ) ||
			(PGCluster_Info->PGR_Retry_Query.cmdType == CMD_TYPE_ROLLBACK ))
		{
			if (autocommit == true)
				ClusterDB_Mode_Info.Transaction_Mode = false;
			if (PGCluster_Info->ReplicateCurrentTime != NULL)
			{
				PGCluster_Info->ReplicateCurrentTime->useFlag = DATA_INIT;
				PGCluster_Info->ReplicateCurrentTime->use_seed = 0;
				PGCluster_Info->ReplicateCurrentTime->use_time = 0;
			}
		}
	}
	if (ClusterDB_Mode_Info.Session_Authorization_Mode)
	{
		PGCluster_Info->PGR_Retry_Query.cmdSts = CMD_STS_SET_SESSION_AUTHORIZATION ;
		if (PGCluster_Info->PGR_Retry_Query.cmdType == CMD_TYPE_SESSION_AUTHORIZATION_END)
		{
			ClusterDB_Mode_Info.Session_Authorization_Mode = false;
		}
	}
	if ((PGCluster_Info->PGR_Retry_Query.cmdSts == CMD_STS_TRANSACTION ) ||
		(PGCluster_Info->PGR_Retry_Query.cmdSts == CMD_STS_SET_SESSION_AUTHORIZATION ) ||
		(PGCluster_Info->PGR_Retry_Query.cmdSts == CMD_STS_TEMP_TABLE ))
	{
		QueryString_Info.Query_String = NULL;
		if (( do_not_replication_command(commandTag) == true) &&
			(PGCluster_Info->PGR_Retry_Query.cmdType != CMD_TYPE_SELECT)) 
		{
			return STATUS_NOT_REPLICATE;
		}

		if (Debug_pretty_print)
			elog(DEBUG1,"transaction query send :%s",(char *)query_string);
		cmdSts = PGCluster_Info->PGR_Retry_Query.cmdSts ;
		cmdType = PGCluster_Info->PGR_Retry_Query.cmdType;
		result = PGR_Send_Replicate_Command(query_string,query_len, cmdSts,cmdType);
		if (result != NULL)
		{
			if (!strncmp(result,PGR_DEADLOCK_DETECTION_MSG,strlen(PGR_DEADLOCK_DETECTION_MSG)))
			{
				/*
				PGR_Send_Message_To_Frontend(result);
				*/
				free(result);
				result = NULL;
				return STATUS_DEADLOCK_DETECT;
			}
			else if (!strncmp(result,PGR_REPLICATION_ABORT_MSG,strlen(PGR_REPLICATION_ABORT_MSG)))
			{
				free(result);
				result = NULL;
				return STATUS_REPLICATION_ABORT;
			}
			free(result);
			result = NULL;
			return STATUS_CONTINUE;
		}
		else
		{
			return STATUS_ERROR;
		}
	}
	else
	{
		cmdSts = PGCluster_Info->PGR_Retry_Query.cmdSts ;
		cmdType = PGCluster_Info->PGR_Retry_Query.cmdType;
		if ( do_not_replication_command(commandTag) == false)
		{
			QueryString_Info.Query_String = NULL;
			result = PGR_Send_Replicate_Command(query_string,query_len,cmdSts,cmdType);
			if (result != NULL)
			{
				if (!strncmp(result,PGR_DEADLOCK_DETECTION_MSG,strlen(PGR_DEADLOCK_DETECTION_MSG)))
				{
					free(result);
					result = NULL;
					return STATUS_DEADLOCK_DETECT;
				}
				else if (!strncmp(result,PGR_REPLICATION_ABORT_MSG,strlen(PGR_REPLICATION_ABORT_MSG)))
				{
					free(result);
					result = NULL;
					return STATUS_REPLICATION_ABORT;
				}
				free(result);
				result = NULL;
				return STATUS_CONTINUE;
			}
			else
			{
				return STATUS_ERROR;
			}
		}
		else
		{
			if (( is_serial_control_query(cmdType,query_string) == true) ||
				( is_select_into_query(cmdType,query_string) == true))
			{
				QueryString_Info.Query_String = NULL;
				PGCluster_Info->PGR_Need_Notice = true;
				PGCluster_Info->PGR_Check_Lock.check_lock_conflict = true;
				result = PGR_Send_Replicate_Command(query_string,query_len,cmdSts,cmdType);
				if (result != NULL)
				{
					/*
					PGR_Send_Message_To_Frontend(result);
					*/
					if (!strncmp(result,PGR_DEADLOCK_DETECTION_MSG,strlen(PGR_DEADLOCK_DETECTION_MSG)))
					{
						free(result);
						return STATUS_DEADLOCK_DETECT;
					}
					free(result);
					result = NULL;
					return STATUS_CONTINUE;
				}
				else
				{
					return STATUS_ERROR;
				}
			}
			else
			{
				QueryString_Info.Query_String = query_string;
				/*ReplicationServer_Info.PGR_Sock_To_Replication_Server = -1;*/
			}
			return STATUS_CONTINUE_SELECT;
		}
	}
	return STATUS_CONTINUE;
}


bool
PGR_Is_System_Command(const char * query)
{
	char * ptr;

	if (query == NULL)
	{
		return false;
	}
	ptr = strstr(query,PGR_SYSTEM_COMMAND_FUNC);
	if (ptr != NULL)
	{
		ptr = strchr(ptr,'(');
		if (ptr == NULL)
			return false;
		return true;
	}
	return false;
}

static int
set_command_args(char argv[ PGR_CMD_ARG_NUM ][256],char *str)
{
	int i,j,cnt,len;
	char * ptr = str;

	if (str == NULL)
	{
		return 0;
	}
	len = strlen(str);
	cnt = j = 0;
	for ( i = 0 ; i < len ; i++,ptr++)
	{
		if (cnt >= PGR_CMD_ARG_NUM)
			break;
		if (( *ptr == ',') || (*ptr == ')'))
		{
			argv[cnt][j] = '\0';
			cnt ++;
			j = 0;
			continue;
		}
		argv[cnt][j] = *ptr;
		j++;
	}
	if (cnt < PGR_CMD_ARG_NUM)
		argv[cnt][j] = '\0';
	cnt ++;

	return cnt;
}

static int
add_replication_server(char * hostname,char * port, char * recovery_port, char * lifecheck_port)
{
	int cnt = 0;
	uint16_t portNumber = 0;
	uint16_t recoveryPortNumber = 0;
	uint16_t lifecheckPortNumber = 0;
	ReplicateServerInfo * sp = NULL;
	unsigned int h1,h2;

	if ((PGCluster_Info == NULL) ||
		(hostname == NULL)   ||
		(port == NULL )      ||
		(recovery_port == NULL ))
	{
		return STATUS_ERROR;
	}
	if (ReplicationServer_Info.ReplicateServerData == NULL)
	{
		return STATUS_ERROR;
	}
	if (atoi(port) > 0)
	{
		portNumber = atoi(port);
	}
	else
	{
		portNumber = DEFAULT_PGRP_PORT;
	}
	if (atoi(recovery_port) > 0)
	{
		recoveryPortNumber = atoi(recovery_port);
	}
	else
	{
		recoveryPortNumber = DEFAULT_PGRP_RECOVERY_PORT;
	}
	if (atoi(lifecheck_port) > 0)
	{
		lifecheckPortNumber = atoi(lifecheck_port);
	}
	else
	{
		lifecheckPortNumber = DEFAULT_PGRP_RECOVERY_PORT;
	}
	h1 = PGRget_ip_by_name(hostname);
	cnt = 0;
	sp = ReplicationServer_Info.ReplicateServerData;
	while (sp->useFlag != DATA_END){
		h2 = PGRget_ip_by_name(sp->hostName);
		if((h1 == h2) &&
			(sp->portNumber == portNumber) &&
			(sp->recoveryPortNumber == PGCluster_Info->Com_Info.ConfData_Info.Recovery_Port_Number))
		{
			if (sp->useFlag != DATA_USE)
			{
				PGR_Set_Replication_Server_Status(sp, DATA_INIT);
			}
			return STATUS_OK;
		}
		sp ++;
		cnt ++;
	}
	if (cnt < MAX_SERVER_NUM)
	{
		strncpy(sp->hostName,hostname,sizeof(sp->hostName));
		sp->portNumber = portNumber;
		sp->recoveryPortNumber = PGCluster_Info->Com_Info.ConfData_Info.Recovery_Port_Number;
		sp->lifecheckPortNumber = lifecheckPortNumber;
		PGR_Set_Replication_Server_Status(sp, DATA_INIT);
		memset((sp+1),0,sizeof(ReplicateServerInfo));
		(sp + 1)->useFlag = DATA_END;
	}
	else
	{
		return STATUS_ERROR;
	}
	return	STATUS_OK;
}

static int
change_replication_server(char * hostname,char * port, char * recovery_port)
{
	int cnt;
	uint16_t portNumber;
	uint16_t recoveryPortNumber;
	ReplicateServerInfo * sp;
	uint32_t h1,h2;

	if ((PGCluster_Info == NULL)  ||
		(hostname == NULL)    ||
		(port == NULL )       ||
		(recovery_port == NULL ))
	{
		return STATUS_ERROR;
	}
	if (ReplicationServer_Info.ReplicateServerData == NULL)
	{
		return STATUS_ERROR;
	}

	h1 = PGRget_ip_by_name(hostname);
	portNumber = atoi(port);
	recoveryPortNumber = atol(recovery_port);
	cnt = 0;
	sp = ReplicationServer_Info.ReplicateServerData;
	while (sp->useFlag != DATA_END){
		h2 = PGRget_ip_by_name(sp->hostName);
		if((h1 == h2) &&
			(sp->portNumber == portNumber) &&
			(sp->recoveryPortNumber == PGCluster_Info->Com_Info.ConfData_Info.Recovery_Port_Number))
		{
			PGR_Set_Replication_Server_Status(sp, DATA_USE);
		}
		else
		{
			if (sp->useFlag == DATA_USE)
			{
				PGR_Set_Replication_Server_Status(sp, DATA_INIT);
			}
		}
		sp ++;
		cnt ++;
	}
	return	STATUS_OK;
}

int
PGR_Set_Current_Time(char * sec, char * usec)
{
	struct timeval orig_tv;
	struct timezone tpz;
	struct timeval tv;

	if ((PGCluster_Info == NULL) ||
		(sec == NULL)  ||
		(usec == NULL) ||
		(PGCluster_Info->ReplicateCurrentTime == (ReplicateNow *)NULL))
	{
		return STATUS_ERROR;
	}


	gettimeofday(&orig_tv, &tpz);
	tv.tv_sec = atol(sec);
	tv.tv_usec = atol(usec);
	PGCluster_Info->ReplicateCurrentTime->tp.tv_sec = tv.tv_sec;
	PGCluster_Info->ReplicateCurrentTime->tp.tv_usec = tv.tv_usec;
	PGCluster_Info->ReplicateCurrentTime->useFlag = DATA_USE;
	PGCluster_Info->ReplicateCurrentTime->use_seed = 0;
	/* set offset */
	PGCluster_Info->ReplicateCurrentTime->use_time = 0;
	PGCluster_Info->ReplicateCurrentTime->offset_sec = tv.tv_sec - orig_tv.tv_sec;
	PGCluster_Info->ReplicateCurrentTime->offset_usec = tv.tv_usec - orig_tv.tv_usec;
	return	STATUS_OK;
}

static int
set_replication_id(char * id)
{
	if (id == NULL)
	{
		return STATUS_ERROR;
	}
	/*set replicate id in this process */
	ReplicationLog_Info.PGR_Replicate_ID = atol(id);

	if (ReplicationServer_Info.CurrentReplicateServer == NULL)
	{
		PGR_get_replicate_server_info();
	}
	if (ReplicationServer_Info.CurrentReplicateServer != NULL)
	{
		/* set replicate id in this system */
		ReplicationServer_Info.CurrentReplicateServer->replicate_id = atol(id);
	}
	
	return STATUS_OK;
}

static void
set_response_mode(char * mode)
{
	int response_mode = 0;

	if (mode == NULL)
		return;
	response_mode = atoi(mode);
	if (response_mode < 0)
		return;
	if (ReplicationServer_Info.CurrentReplicateServer == NULL)
	{
		PGR_get_replicate_server_info();
		if (ReplicationServer_Info.CurrentReplicateServer == NULL)
		{
			return;
		}
	}
	if (ReplicationServer_Info.CurrentReplicateServer->response_mode != response_mode)
	{
		ReplicationServer_Info.CurrentReplicateServer->response_mode = response_mode;
	}
}

int
PGR_Call_System_Command(char * command)
{
	char * ptr;
	char * args;
	char argv[ PGR_CMD_ARG_NUM ][256];
	int argc = 0;
	int func_no;
	char * hostName = NULL;

	if ((PGCluster_Info == NULL) ||
		(command == NULL)    || 
		(PGCluster_Info->ReplicateCurrentTime == NULL))
	{
		return STATUS_ERROR;
	}

	ptr = strstr(command,PGR_SYSTEM_COMMAND_FUNC);
	if (ptr == NULL)
		return STATUS_ERROR;
	ptr = strchr(ptr,'(');
	if (ptr == NULL)
		return STATUS_ERROR;
	args = ptr+1;
	ptr = strchr(ptr,')');
	if (ptr == NULL)
		return STATUS_ERROR;
	*ptr = '\0';
	argc = set_command_args(argv,args);
	if (argc < 1)
		return STATUS_ERROR;
	func_no = atoi(argv[0]);
	switch (func_no)
	{
		/* set current system time */
		case PGR_SET_CURRENT_TIME_FUNC_NO:
			if ((atoi(argv[3]) > 0) &&
				(is_this_query_replicated(argv[3]) == true))
			{
				/* skip this query */
				return STATUS_SKIP_QUERY;
			}
			PGR_Set_Current_Time(argv[1],argv[2]);
			set_replication_id(argv[3]);
			set_response_mode(argv[4]);
			break;
		/* add new replication server data */
		case PGR_STARTUP_REPLICATION_SERVER_FUNC_NO:
			hostName = get_hostName(argv[1]);
			add_replication_server(hostName,argv[2],argv[3],argv[4]);
			break;
		/* change new replication server */
		case PGR_CHANGE_REPLICATION_SERVER_FUNC_NO:
			hostName = get_hostName(argv[1]);
			change_replication_server(hostName,argv[2],argv[3]);
			break;
		/* check replication id */
		case PGR_QUERY_CONFIRM_ANSWER_FUNC_NO:
			if ((atoi(argv[3]) > 0) &&
				(is_this_query_replicated(argv[3]) == true))
			{
				/* skip this query */
				return STATUS_SKIP_QUERY;
			}
			else
			{
				PGR_Set_Current_Time(argv[1],argv[2]);
				set_replication_id(argv[3]);
			}
			break;
		/* get current oid */
		case PGR_GET_OID_FUNC_NO:
			return_current_oid();
			break;
		/* set current oid */
		case PGR_SET_OID_FUNC_NO:
			sync_oid(argv[1]);
			break;
	}
	return STATUS_OK;
}

int
PGR_GetTimeOfDay(struct timeval *tp, struct timezone *tpz)
{

	int rtn;

	rtn = gettimeofday(tp, tpz);
	if ((PGCluster_Info == NULL) ||
		(PGCluster_Info->ReplicateCurrentTime == NULL))
	{
		return rtn;
	}
	if (PGCluster_Info->ReplicateCurrentTime->useFlag == DATA_USE)
	{
		if ( PGCluster_Info->ReplicateCurrentTime->use_time <= 1)
		{
			tp->tv_sec = PGCluster_Info->ReplicateCurrentTime->tp.tv_sec;
			tp->tv_usec = PGCluster_Info->ReplicateCurrentTime->tp.tv_usec;
			if ((ClusterDB_Mode_Info.Transaction_Mode == false) && 
				(ClusterDB_Mode_Info.Session_Authorization_Mode == false))
			{
				PGCluster_Info->ReplicateCurrentTime->use_time ++;
			}
		}
		else
		{
			tp->tv_sec += PGCluster_Info->ReplicateCurrentTime->offset_sec;
		}
		rtn = 0;
	}
	return rtn;
}

long
PGR_Random(void)
{
	double rtn;
	if ((PGCluster_Info != NULL) &&
		(PGCluster_Info->ReplicateCurrentTime != NULL))
	{
		if ( PGCluster_Info->ReplicateCurrentTime->use_seed == 0)
		{
			srand( PGCluster_Info->ReplicateCurrentTime->tp.tv_sec );
			PGCluster_Info->ReplicateCurrentTime->use_seed = 1;
		}
	}
	rtn = random();
	return rtn;
}

char *
PGR_scan_terminate(const char * str)
{
	char * p;
	int sflag = 0;
	int dflag = 0;

	if (str == NULL)
		return NULL;
	p = (char *)str;
	while ( *p != '\0' )
	{
		if ((!strncmp(p,"--",2)) ||
			(!strncmp(p,"//",2)))
		{
			while (( *p != '\n') && (*p != '\0'))
			{
				p++;
			}
			continue;
		}

		switch (*p)
		{
			case '\'':
				sflag ^= 1;
				break;
			case '\"':
				dflag ^= 1;
				break;
			case '\\':
				p +=2;
				continue;
				break;
			case ';':
				if ((!sflag) && (!dflag))
					return p;
				break;
		}
		p++;
	}
	return NULL;
}

static bool
is_copy_from(const char * query)
{
	char * p;
	int i;
	char buf[12];
	int c_flag = 0;
	if (query == NULL)
		return false;
	p = (char *)query;
	for ( i = 0 ; i <= 1 ; i ++)
	{
		/* get 'copy table_name' string */
		while(isspace(*p))
			p++;
		while ((*p != '\0') && (*p  != '(') && (!isspace(*p)))
			p++;
	}
	while(isspace(*p))
		p++;
	/* skip table column */
	if (*p == '(')
	{
		c_flag = 1;
		p++;
		while (*p != '\0') 
		{
			if (*p == '(')
				c_flag ++;
			if (*p == ')')
				c_flag --;
			if (c_flag == 0)
			{
				p++;
				break;
			}
			p++;
		}
		while(isspace(*p))
			p++;
	}
	/* get 'from' or 'to' */
	i = 0;
	memset(buf,0,sizeof(buf));
	while ((*p != '\0') && (!isspace(*p)) && ( i < sizeof(buf)-1))
	{
		buf[i] = (char)toupper(*p);
		p++;
		i++;
	}
	if (!strcmp(buf,"FROM"))
	{
		return true;
	}
	else
	{
		return false;
	}
}

static bool
is_create_temp_table(const char * query)
{
	int len,wc;
	char buf[MAX_WORDS][MAX_WORD_LETTERS];

	if (query == NULL)
		return false;
	len = strlen(query);
	wc = get_words(buf,query,len,1);
	if (wc < 4)
		return false;
	if ((!strncmp(buf[0],"CREATE", strlen("CREATE"))) &&
		(!strncmp(buf[1],"TEMP",strlen("TEMP"))) &&
		(!strncmp(buf[2],"TABLE",strlen("TABLE"))))
	{
		return true;
	}
	return false;
}
	
static int
is_session_authorization(const char * query)
{

	int len,wc;
	int cnt,rtn;
	char buf[MAX_WORDS][MAX_WORD_LETTERS];

	if (query == NULL)
		return NOT_SESSION_AUTHORIZATION;

	rtn = NOT_SESSION_AUTHORIZATION;
	len = strlen(query);
	wc = get_words(buf,query,len,1);
	if (wc < 4)
	{
		return NOT_SESSION_AUTHORIZATION;
	}
	if ((!strncmp(buf[0],"RESET", strlen("RESET"))) &&
		(!strncmp(buf[1],"SESSION",strlen("SESSION"))) &&
		(!strncmp(buf[2],"AUTHORIZATION",strlen("AUTHORIZATION"))))
	{
		return SESSION_AUTHORIZATION_END;
	}
	cnt = 1;
	if (!strcmp(buf[cnt],"SESSION"))
	{
		cnt ++;
		if (!strcmp(buf[cnt],"AUTHORIZATION"))
		{
			rtn = SESSION_AUTHORIZATION_BEGIN;
		}
		else
		{
			cnt ++;
			if (!strcmp(buf[cnt],"AUTHORIZATION"))
			{
				rtn = SESSION_AUTHORIZATION_BEGIN;
			}
		}
		cnt ++;
		if (!strcmp(buf[cnt],"DEFAULT"))
		{
			rtn = SESSION_AUTHORIZATION_END;
		}
	}
	return rtn;
}

static int
get_words( char words[MAX_WORDS][MAX_WORD_LETTERS] ,const char * string,int length,int upper)
{
	int i,wc,lc;
	char * p = NULL;
	char * buf = NULL;

	if (string == NULL)
		return STATUS_ERROR;
	buf = (char *)malloc(length);
	if (buf == (char *)NULL)
		return STATUS_ERROR;

	memset(buf,0,length);
	p = (char *)string;
	wc = 0;
	for (i = 0 ; i < length ; i ++)
	{
		if ((*p == '\0') || (wc >= MAX_WORDS))
			break;
		while (isspace(*p))
		{
			p++;
			i++;
		}
		lc = 0;
		while ((*p != '\0') && (! isspace(*p)))
		{
			if (upper)
				*(buf+lc) = (char)toupper(*p);
			else
				*(buf+lc) = *p;

			p++;
			i++;
			lc++;
		}
		memset(words[wc],0,MAX_WORD_LETTERS);
		memcpy(words[wc],buf,lc);
		memset(buf,0,length);
		wc++;
	}
	free(buf);
	buf = NULL;
	return wc;
}

static int
Comp_Not_Replicate(PGR_Not_Replicate_Type * nrp1,PGR_Not_Replicate_Type* nrp2)
{
	int rtn;

	if ((nrp1 == NULL) ||
		(nrp2 == NULL))
	{
		return 0;
	}
	rtn = strcmp(nrp1->table_name,nrp2->table_name);
	if (rtn == 0)
	{
		rtn = strcmp(nrp1->db_name,nrp2->db_name);
	}
	return rtn;
}

bool
PGR_Is_Stand_Alone(void)
{
	if ((PGCluster_Info == NULL) || 
		(PGCluster_Info->PGR_Stand_Alone == NULL))
	{
		return true;
	}
	if (PGCluster_Info->PGR_Stand_Alone->is_stand_alone == true)
	{
		return true;
	}
	return false;
}

void
PGR_Send_Message_To_Frontend(char * msg)
{
	StringInfoData buf;

	if (msg == NULL)
		return;
	switch (*msg)
	{
		/* normal message */
		case 'C':
			pq_puttextmessage(*msg,msg+1);
			break;
		/* error message */
		case 'E':
		case 'N':
			initStringInfo(&buf);
			pq_sendbyte(&buf,*msg);
			pq_sendstring(&buf, msg+1);
			pq_endmessage(&buf);
			break;
	}
	pq_flush();
}

static bool
is_serial_control_query(char cmdType, const char * query)
{
	char * buf = NULL;
	int len = 0;
	int i = 0;
	char * p = NULL;

	if ((cmdType != CMD_TYPE_SELECT ) ||
		( query == NULL))
	{
		return false;
	}

	p = (char *)query;
	len = strlen(query) +1;
	buf = (char *)malloc(len);
	if (buf == (char *)NULL)
		return false;

	memset(buf,0,len);
	for ( i = 0 ; i < len ; i ++)
	{
		*(buf+i) = toupper(*(query+i));
	}
	if ((strstr(buf,"NEXTVAL") != NULL) ||
		(strstr(buf,"SETVAL") != NULL))
	{
		free(buf);
		buf = NULL;
		return true;
	}
	free(buf);
	buf = NULL;
	return false;
}

static bool
is_select_into_query(char cmdType, const char * query)
{
	char * buf = NULL;
	int len = 0;
	int i = 0;
	char * p = NULL;

	if ((cmdType != CMD_TYPE_SELECT ) ||
		( query == NULL))
	{
		return false;
	}

	p = (char *)query;
	len = strlen(query) +1;
	buf = malloc(len);
	if (buf == (char *)NULL)
		return false;

	memset(buf,0,len);
	for ( i = 0 ; i < len ; i ++)
	{
		*(buf+i) = toupper(*(query+i));
	}
	if (strstr(buf,"INTO") != NULL)
	{
		free(buf);
		buf = NULL;
		return true;
	}
	free(buf);
	buf = NULL;
	return false;
}

static int
send_response_to_replication_server(const char * notice)
{
	ReplicateHeader header;
	int status;

	if ((PGCluster_Info == NULL) ||
		(PGCluster_Info->PGR_Lock_Noticed))
	{
		return STATUS_OK;
	}
	if ((notice == NULL) ||
		(ReplicationServer_Info.PGR_Sock_To_Replication_Server <= 0))
	{
		return STATUS_ERROR;
	}

	memset(&header,0,sizeof(ReplicateHeader));
	header.cmdSys = CMD_SYS_CALL;
	header.cmdSts = CMD_STS_RESPONSE;
	if (!strcmp(notice,PGR_QUERY_ABORTED_NOTICE_CMD))
	{
		header.cmdType = CMD_TYPE_FRONTEND_CLOSED;
	}
	header.query_size = htonl(strlen(notice));
	status = send_replicate_packet(ReplicationServer_Info.PGR_Sock_To_Replication_Server,&header,(char *)notice);
	return status;
}

void
PGR_Notice_Transaction_Query_Done(void)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}
	if (PGCluster_Info->PGR_Need_Notice == true)
	{
		send_response_to_replication_server(PGR_QUERY_DONE_NOTICE_CMD);
	}
}

void
PGR_Notice_Transaction_Query_Aborted(void)
{
	ClusterDB_Mode_Info.Transaction_Mode = false;
	ClusterDB_Mode_Info.Session_Authorization_Mode = false;
	send_response_to_replication_server(PGR_QUERY_ABORTED_NOTICE_CMD);
}

int
PGR_Notice_Conflict(void)
{
	const char * msg = NULL ;
	int rtn = STATUS_OK;

	if (PGCluster_Info == NULL)
		return STATUS_ERROR;
	msg = PGR_LOCK_CONFLICT_NOTICE_CMD ;
	if (PGCluster_Info->PGR_Check_Lock.deadlock == true)
	{
		msg = PGR_DEADLOCK_DETECT_NOTICE_CMD ;
	}
	if (PGCluster_Info->PGR_Check_Lock.dest == TO_FRONTEND)
	{
		pq_puttextmessage('C',msg);
		pq_flush();
#ifdef CONTROL_LOCK_CONFLICT
		rtn = wait_lock_answer();
#endif
	}
	else
	{
		send_response_to_replication_server(msg);
#ifdef CONTROL_LOCK_CONFLICT
		rtn = PGR_Recv_Trigger (PGR_RECV_TIMEOUT);
#endif
	}
	return rtn;
}

#ifdef CONTROL_LOCK_CONFLICT
static int
wait_lock_answer(void)
{
	char result[PGR_MESSAGE_BUFSIZE+4];
	int rtn = 0;

	memset(result,0,sizeof(result));
	rtn = read_trigger(result, PGR_MESSAGE_BUFSIZE);
	if (rtn < 0)
		return STATUS_ERROR;
	return STATUS_OK;
}

static int
read_trigger(char * result, int buf_size)
{
	int i = 0;
	char c;
	int r = 0;

	if ((result == NULL) || (buf_size <= 0 ))
	{
		return EOF;
	}
	/*
	pq_getbytes(result,buf_size);
	*/
	while ((r = pq_getbytes(&c,1)) == 0)
	{
		if (i < buf_size -1)
		{
			*(result + i) = c;
		}
		else
		{
			break;
		}
		if (c == '\0')
			break;
		i++;
	}

	return r;
}
#endif

int
PGR_Recv_Trigger (int user_timeout)
{
	char result[PGR_MESSAGE_BUFSIZE];
	int rtn = 0;
	int func_no = 0;

	if (PGCluster_Info == NULL)
		return STATUS_ERROR;
	if (PGCluster_Info->PGR_Lock_Noticed)
	{
		return STATUS_OK;
	}
	if (ReplicationServer_Info.PGR_Sock_To_Replication_Server < 0)
		return STATUS_ERROR;
	memset(result,0,sizeof(result));
	rtn = recv_replicate_result(ReplicationServer_Info.PGR_Sock_To_Replication_Server,result,user_timeout);
	if (rtn > 0)
	{
		func_no = atoi(result);
		if (func_no  <= 0)
		{
			func_no = STATUS_OK;
		}
		return func_no;
	}
	else 
	{
		if (user_timeout == 0)
		{
			PGR_Set_Replication_Server_Status(ReplicationServer_Info.CurrentReplicateServer, DATA_ERR);
		}
		return STATUS_ERROR;
	}
	return STATUS_OK;
}

bool
PGR_Set_Transaction_Mode(bool mode,const char * commandTag)
{
	if (commandTag == NULL)
	{
		return mode;
	}
	if (autocommit == false)
	{
		return true;
	}
	if (mode == false)
	{
		if ((!strcmp(commandTag,"BEGIN")) ||
			(!strcmp(commandTag,"START TRANSACTION")) )
		{
			return true;
		}
	}
	if (mode == true)
	{
		if ((!strcmp(commandTag,"COMMIT")) ||
			(!strcmp(commandTag,"ROLLBACK")))
		{
			return false;
		}
	}
	return mode;
}


static bool
is_transaction_command(const char * commandTag)
{
	if (commandTag == NULL)
	{
		return false;
	}
	if ((!strcmp(commandTag,"BEGIN")) ||
		(!strcmp(commandTag,"START TRANSACTION")))
	{
		return true;
	}
	else
	{
		return false;
	}
}

static bool
do_not_replication_command(const char * commandTag)
{
	if (commandTag == NULL)
	{
		return true;
	}
	if ((!strcmp(commandTag,"SELECT")) ||
		(!strcmp(commandTag,"CLOSE CURSOR")) ||
		(!strcmp(commandTag,"MOVE")) ||
		(!strcmp(commandTag,"FETCH")) ||
		(!strcmp(commandTag,"SHOW")) ||
		(!strcmp(commandTag,"REINDEX")) ||
		(!strcmp(commandTag,"ANALYZE")) ||
		(!strcmp(commandTag,"VACUUM")) ||
		(!strcmp(commandTag,"EXPLAIN")))
	{
		return true;
	}
	else
	{
		return false;
	}
}

void
PGR_Set_Replication_Server_Status( ReplicateServerInfo * sp, int status)
{
	if (sp == NULL)
	{
		return;
	}
	if (sp->useFlag != status)
	{
		sp->useFlag = status;
	}
}

int
PGR_Is_Skip_Replication(const char * query)
{
	char skip_2[256];

	if ((query == NULL) ||
		(MyProcPort == NULL))
	{
		return -1;
	}
	sprintf(skip_2,SKIP_QUERY_2,MyProcPort->user_name);
	if ((strncmp(query,SKIP_QUERY_1,strlen(SKIP_QUERY_1)) == 0) ||
		(strncmp(query,skip_2,strlen(skip_2)) == 0))
	{
		return 3;
	}
	if ((strncmp(query,SKIP_QUERY_3,strlen(SKIP_QUERY_3)) == 0) ||
		(strncmp(query,SKIP_QUERY_4,strlen(SKIP_QUERY_4)) == 0))
	{
		return 1;
	}
	return 0;
}

bool
PGR_Did_Commit_Transaction(void)
{
	int sock = 0;
	int cnt = 0;
	ReplicateHeader header;
	char * serverName = NULL;
	int portNumber=0;
	char * result = NULL;
	ReplicateServerInfo * sp = NULL;
	ReplicateServerInfo * base = NULL;
	int socket_type = 0;
	char argv[ PGR_CMD_ARG_NUM ][256];
	int argc = 0;
	int func_no = 0;

	if ((PGCluster_Info == NULL ) ||
		(PGCluster_Info->ReplicateCurrentTime == NULL) ||
		(PGCluster_Info->ReplicateCurrentTime->useFlag != DATA_USE))
	{
		return false;
	}
	sp = PGR_get_replicate_server_info();
	if (sp == NULL)
	{
		if (Debug_pretty_print)
			elog(DEBUG1,"PGR_get_replicate_server_info get error");
		return false;
	}
	sock = get_replicate_server_socket( sp , PGR_QUERY_SOCKET);
	if (sock < 0)
	{
		if (Debug_pretty_print)
			elog(DEBUG1,"get_replicate_server_socket fail");
		return false;
	}
	serverName = sp->hostName;
	portNumber = (int)sp->portNumber;
	header.cmdSys = CMD_SYS_CALL;
	header.cmdSts = CMD_STS_TRANSACTION_ABORT;
	header.cmdType = CMD_TYPE_COMMIT_CONFIRM;
	header.port = htons(PostPortNumber);
	header.pid = htons(getpid());
	header.tv.tv_sec = htonl(PGCluster_Info->ReplicateCurrentTime->tp.tv_sec);
	header.tv.tv_usec = htonl(PGCluster_Info->ReplicateCurrentTime->tp.tv_usec);
	header.query_size = htonl(0); 
	strncpy(header.dbName ,(char *)(MyProcPort->database_name),sizeof(header.dbName));
	strncpy(header.userName , (char *)(MyProcPort->user_name),sizeof(header.userName));
	gethostname(header.from_host,sizeof(header.from_host));
	header.replicate_id = htonl(ReplicationLog_Info.PGR_Replicate_ID);
	header.request_id = 0;
	base = sp;
	ReplicationServer_Info.PGR_Sock_To_Replication_Server = sock;
	result = (char *)malloc(PGR_MESSAGE_BUFSIZE);
	if (result == (char *)NULL)
	{
		return false;
	}
	memset(result,0,PGR_MESSAGE_BUFSIZE);
retry_send_confirm_packet:

	cnt = 0;
	while (send_replicate_packet(sock,&header,"") != STATUS_OK)
	{
		if (cnt > MAX_RETRY_TIMES )
		{
			sock = get_new_replication_socket( base, sp, socket_type);
			if (sock < 0)
			{
				if (Debug_pretty_print)
					elog(DEBUG1,"all replication servers may be down");
				PGCluster_Info->PGR_Stand_Alone->is_stand_alone = true;
				free(result);
				result = NULL;
				return false;
			}
			ReplicationServer_Info.PGR_Sock_To_Replication_Server = sock;
			cnt = 0;
		}
		cnt ++;
	}

	if (recv_replicate_result(sock,result,0) < 0)
	{
		/* replication server should be down */
		sock = get_new_replication_socket( base, sp, socket_type);
		if (sock < 0)
		{
			if (Debug_pretty_print)
				elog(DEBUG1,"all replication servers may be down");
			PGCluster_Info->PGR_Stand_Alone->is_stand_alone = true;
			free(result);
			result = NULL;
			return false;
		}
		ReplicationServer_Info.PGR_Sock_To_Replication_Server = sock;
		goto retry_send_confirm_packet;
	}
	/* read answer */
	argc = set_command_args(argv,result);
	if (argc >= 1)
	{
		func_no = atoi(argv[0]);
		if (func_no == PGR_TRANSACTION_CONFIRM_ANSWER_FUNC_NO)
		{
			/* the transaction was commited in other server */
			if (atoi(argv[1]) == PGR_ALREADY_COMMITTED)
			{
				free(result);
				result = NULL;
				return true;
			}
		}
	}
	free(result);
	result = NULL;
	return false;
}

#if 0
static bool
confirm_query_done(int sock, ReplicateHeader * header, char *result)
{
	char argv[ PGR_CMD_ARG_NUM ][256];
	int argc = 0;
	int func_no = 0;
	int status = 0;

	header->cmdSys = CMD_SYS_CALL;
	header->cmdSts = CMD_STS_QUERY_SUSPEND;
	header->cmdType = CMD_TYPE_QUERY_CONFIRM;
	if (send_replicate_packet(sock,header,"") != STATUS_OK)
	{
		return false;
	}
	if (recv_replicate_result(sock,result,0) <= 0)
	{
		return false;
	}
	/* read answer */
	argc = set_command_args(argv,result);
	if (argc >= 2)
	{
		func_no = atoi(argv[0]);
		status = atoi(argv[1]);
		if ((func_no == PGR_QUERY_CONFIRM_ANSWER_FUNC_NO) &&
			(status == PGR_ALREADY_COMMITTED))
		{
			/* the query was done in other server */
			return true;
		}
	}
	return false;
}
#endif

static char *
get_hostName(char * str)
{
	char * top = NULL;
	char * p = NULL;

	p = str;
	while ( *p != '\0')
	{
		if (*p == '\'')
		{
			*p = '\0';
			p++;
			if (top == NULL)
			{
				top = p;
			}
		}
		p++;
	}
	return top;
}

static unsigned int
get_next_request_id(void)
{
	if (ReplicationLog_Info.PGR_Request_ID +1 < PGR_MAX_COUNTER)
	{
		ReplicationLog_Info.PGR_Request_ID ++;
	}
	else
	{
		ReplicationLog_Info.PGR_Request_ID = 0;
	}
	return ReplicationLog_Info.PGR_Request_ID ;
		
}

static bool
is_this_query_replicated(char * id)
{
	uint32_t replicate_id = 0;
	uint32_t saved_id = 0;
	uint32_t comp_id = 0;
	uint32_t saved_over_flow_flag = 0;
	uint32_t comp_over_flow_flag = 0;
	ReplicateServerInfo * replicate_server_info = NULL;

	if (id == NULL)
	{
		return false;
	}
	replicate_id = atol(id);
	comp_id = (replicate_id & PGR_GET_DATA_FILTER);
	comp_over_flow_flag = (replicate_id & PGR_GET_OVER_FLOW_FILTER);

	if (ReplicationServer_Info.CurrentReplicateServer == NULL)
	{
		PGR_get_replicate_server_info();
	}
	if (ReplicationServer_Info.CurrentReplicateServer != NULL)
	{
		replicate_server_info = ReplicationServer_Info.CurrentReplicateServer;
	}
	else if (ReplicationServer_Info.LastReplicateServer != NULL)
	{
		replicate_server_info = ReplicationServer_Info.LastReplicateServer;
	}
	if (replicate_server_info != NULL)
	{
		saved_id = (ReplicationServer_Info.CurrentReplicateServer->replicate_id & PGR_GET_DATA_FILTER);
		saved_over_flow_flag = (ReplicationServer_Info.CurrentReplicateServer->replicate_id & PGR_GET_OVER_FLOW_FILTER);
		if (saved_over_flow_flag == comp_over_flow_flag)
		{
			if ( saved_id < comp_id)
			{
				return false;
			}
		}
		else
		{
			if ( saved_id > comp_id)
			{
				return false;
			}
		}
		return true;
	}
	else
	{
		return false;
	}
}

char *
PGR_Remove_Comment(const char * str)
{
	char * p = NULL;
	p = (char *)str;
	while( *p != '\0')
	{
		while(isspace(*p))
		{
			p++;
		}
		if ((!memcmp(p,"--",2)) ||
			(!memcmp(p,"//",2)))
		{
			while((*p != '\n') && (*p != '\0'))
			{
				p++;
			}
			continue;
		}
		break;
	}
	return p;
}

void
PGR_Force_Replicate_Query(void)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}
	if (PGCluster_Info->PGR_Retry_Query.useFlag == DATA_USE)
	{
		PGR_Send_Replicate_Command(PGCluster_Info->PGR_Retry_Query.query_string,
			PGCluster_Info->PGR_Retry_Query.query_len,
			PGCluster_Info->PGR_Retry_Query.cmdSts,
			PGCluster_Info->PGR_Retry_Query.cmdType);
	}
}

void
PGR_Notice_DeadLock(void)
{
	ReplicateHeader header;

	if (PGCluster_Info == NULL)
	{
		return;
	}

	memset(&header,0,sizeof(ReplicateHeader));
	header.cmdSys = CMD_SYS_CALL;
	header.cmdSts = CMD_STS_NOTICE;
	header.cmdType = CMD_TYPE_DEADLOCK_DETECT;
	header.query_size = 0;
	send_replicate_packet(ReplicationServer_Info.PGR_Sock_To_Replication_Server,&header,(char *)NULL);
}

void
PGR_Set_Cluster_Status(int status)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}
	if (ClusterDB_Info.ClusterDBData != NULL)
	{
		if (ClusterDB_Info.ClusterDBData->status != status)
		{
			ClusterDB_Info.ClusterDBData->status = status;
		}
	}
}

int
PGR_Get_Cluster_Status(void)
{
	if (PGCluster_Info == NULL)
	{
		return 0;
	}
	if (ClusterDB_Info.ClusterDBData != NULL)
	{
		return (ClusterDB_Info.ClusterDBData->status);
	}
	return 0;
}

static int
return_current_oid(void)
{
	char msg[PGR_MESSAGE_BUFSIZE];

	LWLockAcquire(OidGenLock, LW_EXCLUSIVE);

	if (ShmemVariableCache->nextOid < ((Oid) BootstrapObjectIdData))
	{
		ShmemVariableCache->nextOid = BootstrapObjectIdData;
		ShmemVariableCache->oidCount = 0;
	}

	if (ShmemVariableCache->oidCount == 0)
	{
		XLogPutNextOid(ShmemVariableCache->nextOid + VAR_OID_PREFETCH);
		ShmemVariableCache->oidCount = VAR_OID_PREFETCH;
	}

	memset(msg,0,sizeof(msg));
	snprintf(msg, sizeof(msg), "%u", ShmemVariableCache->nextOid);
	if ((PGCluster_Info != NULL) &&
		(PGCluster_Info->PGR_Check_Lock.dest == TO_FRONTEND))
	{
		pq_puttextmessage('C',msg);
		pq_flush();
	}
	else
	{
		send_response_to_replication_server(msg);
	}
	return STATUS_OK;
}

static int
sync_oid(char * oid)
{
	uint32_t next_oid = 0;
	int offset = 0;
	char msg[PGR_MESSAGE_BUFSIZE];

	next_oid =  strtoul(oid, NULL, 10);
	if (next_oid <= 0)
		return STATUS_ERROR;
	next_oid ++;
	offset = next_oid - ShmemVariableCache->nextOid ;
	if (offset <= 0)
		return STATUS_ERROR;

	if (next_oid < BootstrapObjectIdData)
	{
		ShmemVariableCache->nextOid = BootstrapObjectIdData;
		ShmemVariableCache->oidCount = 0;
	}

	/* If we run out of logged for use oids then we must log more */
	while (ShmemVariableCache->oidCount - offset <= 0)
	{
		offset -= (ShmemVariableCache->oidCount) ;
		(ShmemVariableCache->nextOid) += (ShmemVariableCache->oidCount);
		XLogPutNextOid(ShmemVariableCache->nextOid + VAR_OID_PREFETCH);
		ShmemVariableCache->oidCount = VAR_OID_PREFETCH;
	}

	(ShmemVariableCache->nextOid) += offset;
	(ShmemVariableCache->oidCount) -= offset;
	
	LWLockRelease(OidGenLock);

	memset(msg,0,sizeof(msg));
	snprintf(msg, sizeof(msg), "%u", ShmemVariableCache->nextOid);
	if (PGCluster_Info->PGR_Check_Lock.dest == TO_FRONTEND)
	{
		pq_puttextmessage('C',msg);
		pq_flush();
	}
	else
	{
		send_response_to_replication_server(msg);
	}
	return STATUS_OK;
}

int
PGR_lo_create(int flags)
{
	char * result = NULL;
	LOArgs lo_args;
	
	if ((PGCluster_Info == NULL)	||
		(PGCluster_Info->PGR_Is_Replicated_Query == true) ||
		(PGCluster_Info->PGR_Retry_Query.cmdSts != CMD_STS_TRANSACTION))
	{
		return STATUS_OK;
	}
	memset(&lo_args, 0, sizeof(LOArgs));
	lo_args.arg1 = htonl(flags);

	result = PGR_Send_Replicate_Command((char *)&lo_args,
		sizeof(LOArgs),
		CMD_STS_LARGE_OBJECT,
		CMD_TYPE_LO_CREATE);

	if (result != NULL)
	{
		free(result);
		return STATUS_OK;
	}
	
	return STATUS_ERROR;
}

int
PGR_lo_open(Oid lobjId,int32 mode)
{
	char * result = NULL;
	LOArgs lo_args;
	
	if ((PGCluster_Info == NULL)	||
		(PGCluster_Info->PGR_Is_Replicated_Query == true) ||
		(PGCluster_Info->PGR_Retry_Query.cmdSts != CMD_STS_TRANSACTION))
	{
		return STATUS_OK;
	}
	memset(&lo_args, 0, sizeof(LOArgs));
	lo_args.arg1 = htonl((uint32_t)lobjId);
	lo_args.arg2 = htonl((uint32_t)mode);

	result = PGR_Send_Replicate_Command((char *)&lo_args,
		sizeof(LOArgs),
		CMD_STS_LARGE_OBJECT,
		CMD_TYPE_LO_OPEN);
	
	if (result != NULL)
	{
		free(result);
		return STATUS_OK;
	}
	
	return STATUS_ERROR;
}

int
PGR_lo_close(int32 fd)
{
	char * result = NULL;
	LOArgs lo_args;
	
	if ((PGCluster_Info == NULL)	||
		(PGCluster_Info->PGR_Is_Replicated_Query == true) ||
		(PGCluster_Info->PGR_Retry_Query.cmdSts != CMD_STS_TRANSACTION))
	{
		return STATUS_OK;
	}
	memset(&lo_args, 0, sizeof(LOArgs));
	lo_args.arg1 = htonl((uint32_t)fd);

	result = PGR_Send_Replicate_Command((char *)&lo_args,
		sizeof(LOArgs),
		CMD_STS_LARGE_OBJECT,
		CMD_TYPE_LO_CLOSE);

	if (result != NULL)
	{
		free(result);
		return STATUS_OK;
	}
	
	return STATUS_ERROR;
}

int
PGR_lo_write(int fd, char *buf, int len)
{
	char * result = NULL;
	LOArgs *lo_args = NULL;
	int buf_size = 0;
	
	if ((PGCluster_Info == NULL)	||
		(PGCluster_Info->PGR_Is_Replicated_Query == true) ||
		(PGCluster_Info->PGR_Retry_Query.cmdSts != CMD_STS_TRANSACTION))
	{
		return STATUS_OK;
	}
	buf_size = sizeof(LOArgs) + len;
	lo_args = malloc(buf_size + 4);
	if (lo_args == (LOArgs *)NULL)
	{
		return STATUS_ERROR;
	}
	memset(lo_args, 0, buf_size + 4);
	lo_args->arg1 = htonl((uint32_t)fd);
	lo_args->arg2 = htonl((uint32_t)len);
	memcpy(lo_args->buf, buf, len);
	result = PGR_Send_Replicate_Command((char *)lo_args,
		buf_size,
		CMD_STS_LARGE_OBJECT,
		CMD_TYPE_LO_WRITE);

	free(lo_args);
	if (result != NULL)
	{
		free(result);
		return STATUS_OK;
	}
	
	return STATUS_ERROR;
}

int
PGR_lo_lseek(int32 fd, int32 offset, int32 whence)
{
	char * result = NULL;
	LOArgs lo_args;
	
	if ((PGCluster_Info == NULL)	||
		(PGCluster_Info->PGR_Is_Replicated_Query == true) ||
		(PGCluster_Info->PGR_Retry_Query.cmdSts != CMD_STS_TRANSACTION))
	{
		return STATUS_OK;
	}
	memset(&lo_args, 0, sizeof(LOArgs));
	lo_args.arg1 = htonl((uint32_t)fd);
	lo_args.arg2 = htonl((uint32_t)offset);
	lo_args.arg3 = htonl((uint32_t)whence);

	result = PGR_Send_Replicate_Command((char *)&lo_args,
		sizeof(LOArgs),
		CMD_STS_LARGE_OBJECT,
		CMD_TYPE_LO_LSEEK);

	if (result != NULL)
	{
		free(result);
		return STATUS_OK;
	}
	
	return STATUS_ERROR;
}

int
PGR_lo_unlink(Oid lobjId)
{
	char * result = NULL;
	LOArgs lo_args;
	
	if ((PGCluster_Info == NULL)	||
		(PGCluster_Info->PGR_Is_Replicated_Query == true) ||
		(PGCluster_Info->PGR_Retry_Query.cmdSts != CMD_STS_TRANSACTION))
	{
		return STATUS_OK;
	}
	memset(&lo_args, 0, sizeof(LOArgs));
	lo_args.arg1 = htonl((uint32_t)lobjId);

	result = PGR_Send_Replicate_Command((char *)&lo_args,
		sizeof(LOArgs),
		CMD_STS_LARGE_OBJECT,
		CMD_TYPE_LO_UNLINK);

	if (result != NULL)
	{
		free(result);
		return STATUS_OK;
	}
	
	return STATUS_ERROR;
}

void
PGR_init_replication(const char * query, int len)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}
	QueryString_Info.Skip_Cnt = 0;
	QueryString_Info.PGR_Status = STATUS_OK;
	QueryString_Info.Transaction_Aborted = false;
	QueryString_Info.Query_Ptr = query;

	ClusterDB_Mode_Info.PGR_Reliable_Mode_Wait = false;
	if (PGCluster_Info == NULL)
		return;
	if (PGCluster_Info->PGR_Is_Replicated_Query == false)
	{
		PGCluster_Info->PGR_Is_Replicated_Query = PGR_Is_Replicated_Command(QueryString_Info.Query_Ptr);
	}
	PGCluster_Info->PGR_Retry_Query.query_string = query;
	PGCluster_Info->PGR_Retry_Query.query_len = len;
	PGCluster_Info->PGR_Retry_Query.cmdType = CMD_TYPE_OTHER;
}

void
PGR_init_each_query(void)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}

	QueryString_Info.PGR_Not_Replication_Query = false;
	ClusterDB_Mode_Info.PGR_Reliable_Mode_Wait = false;

	PGCluster_Info->PGR_Retry_Query.query_string = NULL;
	PGCluster_Info->PGR_Retry_Query.query_len = 0;
	PGCluster_Info->PGR_Retry_Query.cmdSts = CMD_STS_OTHER;
	PGCluster_Info->PGR_Retry_Query.cmdType = CMD_TYPE_OTHER;
	PGCluster_Info->PGR_Retry_Query.useFlag = DATA_INIT;

	PGCluster_Info->PGR_Lock_Noticed = false;
}

int
PGR_exec_replication(const char * commandTag,CommandDest dest)
{
	char *null_ptr = NULL;
	int status = STATUS_OK;

	if (PGCluster_Info == NULL)
	{
		return STATUS_OK;
	}
	QueryString_Info.Query_Ptr = PGR_Remove_Comment(QueryString_Info.Query_Ptr);
	PGCluster_Info->PGR_Check_Lock.dest = TO_FRONTEND;
	PGCluster_Info->PGR_Need_Notice = false;
	PGCluster_Info->PGR_Check_Lock.check_lock_conflict = false;
	if (QueryString_Info.Skip_Cnt == 0)
	{
		QueryString_Info.Skip_Cnt = PGR_Is_Skip_Replication(QueryString_Info.Query_Ptr);
	}
	null_ptr = PGR_scan_terminate (QueryString_Info.Query_Ptr);
	if(null_ptr != NULL)
	{
		*null_ptr = '\0';
	}
	if ((PGCluster_Info->PGR_Is_Replicated_Query ) ||
		(QueryString_Info.Skip_Cnt != 0))
	{
		if (QueryString_Info.Skip_Cnt > 0)
		{
			QueryString_Info.Skip_Cnt --;
		}
		else
		{
			QueryString_Info.Skip_Cnt = 0;
		}
		ClusterDB_Mode_Info.Transaction_Mode = PGR_Set_Transaction_Mode(ClusterDB_Mode_Info.Transaction_Mode,commandTag);
		ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate = false;
		if (!strcmp(commandTag,"SELECT"))
		{
			if (PGR_Is_System_Command(QueryString_Info.Query_Ptr))
			{
				status = PGR_Call_System_Command((char *)QueryString_Info.Query_Ptr);
				if (status == STATUS_SKIP_QUERY)
				{
					pq_puttextmessage('C',PGR_ALREADY_REPLICATED_NOTICE_CMD);
					pq_flush();
					return status;
				}
				else
				{
					return STATUS_CONTINUE;
				}
			}
		}
		PGCluster_Info->PGR_Check_Lock.status_lock_conflict = STATUS_OK;
		PGCluster_Info->PGR_Check_Lock.dest = TO_FRONTEND;
	}
	else
	{
retry_replication:
		ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate = false;

		/* check cluster db status */
		if ((PGR_Get_Cluster_Status() == STATUS_RECOVERY)	&&
			(QueryString_Info.PGR_Not_Replication_Query == false)			&&
			(ClusterDB_Mode_Info.Transaction_Mode == false ) )
		{
			elog(WARNING, "This query is not permitted while recovery db ");
			if(null_ptr != NULL)
			{
				*null_ptr = ';';
				QueryString_Info.Query_Ptr = null_ptr +1;
			}
			return STATUS_CONTINUE;
		}
		if (PGR_Is_Stand_Alone() == true)
		{
			if (PGCluster_Info->PGR_Stand_Alone->permit == PERMIT_READ_ONLY)
			{
				if (QueryString_Info.PGR_Not_Replication_Query == false)
				{
					elog(ERROR, "This query is not permitted when all replication servers fell down ");
					if(null_ptr != NULL)
					{
						*null_ptr = ';';
						QueryString_Info.Query_Ptr = null_ptr +1;
					}
					return STATUS_CONTINUE;
				}
			}
		}
		else if ((QueryString_Info.PGR_Not_Replication_Query == false ) ||
			(!strcmp(commandTag,"SELECT")))
		{
			QueryString_Info.PGR_Status = PGR_replication(QueryString_Info.Query_Ptr,dest,commandTag);
			if (QueryString_Info.PGR_Status == STATUS_REPLICATED)
			{
				CommandCounterIncrement();
				return STATUS_CONTINUE;
			}
			else if (QueryString_Info.PGR_Status == STATUS_ERROR)
			{
				goto retry_replication;
			}
			else if (QueryString_Info.PGR_Status == STATUS_DEADLOCK_DETECT)
			{
				PGCluster_Info->PGR_Need_Notice = false;
				elog(ERROR, "postmaster deadlock detected");
				return STATUS_CONTINUE;
			}
			else if (QueryString_Info.PGR_Status == STATUS_REPLICATION_ABORT)
			{
				PGCluster_Info->PGR_Need_Notice = false;
				elog(ERROR, "replication server should be down, transaction aborted.");
				return STATUS_CONTINUE;
			}
			else if (QueryString_Info.PGR_Status != STATUS_CONTINUE)
			{
				PGCluster_Info->PGR_Check_Lock.dest = TO_FRONTEND;
			}
			else
			{
				PGCluster_Info->PGR_Check_Lock.dest = TO_REPLICATION_SERVER;
				ClusterDB_Mode_Info.PGR_Reliable_Mode_Wait = true;
			}
		}
	}
	if(null_ptr != NULL)
	{
		*null_ptr = ';';
		QueryString_Info.Query_Ptr = null_ptr +1;
	}
	if (!PGCluster_Info->PGR_Is_Replicated_Query )
	{
		if ((!strcmp(commandTag,"BEGIN")) ||
			(!strcmp(commandTag, "START TRANSACTION")) ||
			(ClusterDB_Mode_Info.Transaction_Mode == false ) )
		{
			PGR_Reload_Start_Time();
		}
	}
	if ((ClusterDB_Mode_Info.Transaction_Mode == true)            ||
		(ClusterDB_Mode_Info.Create_Temp_Table_Mode == true)      ||
		(ClusterDB_Mode_Info.Session_Authorization_Mode == true)  ||
		(!strcmp(commandTag,"COPY")))
	{
		PGCluster_Info->PGR_Need_Notice = true;
		PGCluster_Info->PGR_Check_Lock.check_lock_conflict = true;
	}
	else
	{
		if (QueryString_Info.PGR_Not_Replication_Query == false)
		{
			PGCluster_Info->PGR_Need_Notice = true;
			PGCluster_Info->PGR_Check_Lock.check_lock_conflict = true;
		}
		else
		{
			if ((PGCluster_Info->PGR_Is_Replicated_Query ) &&
			(!strcmp(commandTag,"SELECT")))
			{
				PGCluster_Info->PGR_Need_Notice = true;
				PGCluster_Info->PGR_Check_Lock.check_lock_conflict = true;
			}
		}
	}
	return STATUS_OK;
}

int PGR_startup_lifecheck(void)
{
	if (PGCluster_Info == NULL) 
	{
		return STATUS_OK;
	}

	Lifecheck_Pid_Info.Master_Pid = PGR_Master_Main();
	if (Lifecheck_Pid_Info.Master_Pid < 0)
	{
		elog(DEBUG1,"PGR_Master_Main failed");
		return STATUS_ERROR;
	}
	if (PGCluster_Info == NULL)
	{
		return STATUS_ERROR;
	}
	if (PGCluster_Info->PGR_Recovery_Option)
	{
		PGR_recovery_finish_send();
		PGCluster_Info->PGR_Recovery_Option = false;
	}
	/* life check process start */
	Lifecheck_Pid_Info.SYN_Pid = PGR_SYN_Main(SYN_TO_REPLICATION_SERVER, (void *)ReplicationServer_Info.ReplicateServerData);
	if (Lifecheck_Pid_Info.SYN_Pid < 0)
	{
		elog(DEBUG1,"PGR_SYN_Main failed");
		return STATUS_ERROR;
	}
	Lifecheck_Pid_Info.ACK_Pid = PGR_ACK_Main(PGCluster_Info->Com_Info.ConfData_Info.LifeCheck_Port_Number);
	if (Lifecheck_Pid_Info.ACK_Pid < 0)
	{
		elog(DEBUG1,"PGR_ACK_Main failed");
		return STATUS_ERROR;
	}
	return STATUS_OK;
}

void PGR_finish_lifecheck(int sig)
{
	int status = 0;

	if (PGCluster_Info == NULL)
	{
		return ;
	}

	if (PGCluster_Info->PGR_Recovery_Option)
	{
		PGR_recovery_error_send();
		PGCluster_Info->PGR_Recovery_Option = false;
	}
	if (Lifecheck_Pid_Info.Master_Pid > 0)
	{
		kill (Lifecheck_Pid_Info.Master_Pid,sig);
		waitpid(Lifecheck_Pid_Info.Master_Pid,&status,WNOHANG);
	}
	if (Lifecheck_Pid_Info.SYN_Pid > 0)
	{
		kill (Lifecheck_Pid_Info.SYN_Pid,sig);
		waitpid(Lifecheck_Pid_Info.SYN_Pid,&status,WNOHANG);
	}
	if (Lifecheck_Pid_Info.ACK_Pid > 0)
	{
		kill (Lifecheck_Pid_Info.ACK_Pid,sig);
		waitpid(Lifecheck_Pid_Info.ACK_Pid,&status,WNOHANG);
	}
	PGR_delete_shm();
}

int PGR_Wait_Reliable_Trigger(const char * commandTag)
{
	int status = STATUS_OK;

	if (PGCluster_Info == NULL)
	{
		return status;
	}
	if (ReplicationServer_Info.CurrentReplicateServer != NULL)
	{
		if ((PGCluster_Info != NULL) &&
			(!PGCluster_Info->PGR_Is_Replicated_Query )				&&
			(PGCluster_Info->PGR_Check_Lock.dest != TO_FRONTEND)	&&
			(ClusterDB_Mode_Info.PGR_Reliable_Mode_Wait == true)		&&
			(strcmp(commandTag,"COPY"))				&&
			(ReplicationServer_Info.CurrentReplicateServer->response_mode == PGR_RELIABLE_MODE))
		{
			status = PGR_Recv_Trigger(PGR_RECV_TIMEOUT);
		}
	}
	return status;
}

void 
PGR_Set_Replication_Flag(Node * parsetree)
{
	char cmdType = CMD_TYPE_OTHER;

	if (PGCluster_Info == NULL)
	{
		return;
	}
	QueryString_Info.PGR_Not_Replication_Query = false;
	/* set need to replication */
	switch (nodeTag(parsetree))
	{
		case T_SelectStmt:
		case T_ClosePortalStmt:
		case T_FetchStmt:
		case T_VacuumStmt:
		case T_ExplainStmt:
		case T_VariableShowStmt:
		case T_CheckPointStmt:
			QueryString_Info.PGR_Not_Replication_Query = true;
			break;
		default:
			QueryString_Info.PGR_Not_Replication_Query = false;
			break;
	}
	/* set query type */
	switch (nodeTag(parsetree))
	{
		case T_InsertStmt:
			cmdType = CMD_TYPE_INSERT ;
			break;
		case T_DeleteStmt:
			cmdType = CMD_TYPE_DELETE ;
			break;
		case T_UpdateStmt:
			cmdType = CMD_TYPE_UPDATE ;
			break;
		case T_SelectStmt:
			cmdType = CMD_TYPE_SELECT ;
			break;
		case T_TransactionStmt:
		{
			TransactionStmt *stmt = (TransactionStmt *) parsetree;
			switch (stmt->kind)
			{
				case TRANS_STMT_BEGIN:
				case TRANS_STMT_START:
					cmdType = CMD_TYPE_BEGIN ;
					break;
				case TRANS_STMT_COMMIT:
					cmdType = CMD_TYPE_COMMIT ;
					break;
				case TRANS_STMT_ROLLBACK:
					cmdType = CMD_TYPE_ROLLBACK ;
					break;
				case TRANS_STMT_ROLLBACK_TO:
					cmdType = CMD_TYPE_ROLLBACK_TO_SAVEPOINT ;
					break;
				case TRANS_STMT_SAVEPOINT:
					cmdType = CMD_TYPE_SAVEPOINT ;
					break;
				case TRANS_STMT_RELEASE:
					cmdType = CMD_TYPE_RELEASE_SAVEPOINT ;
					break;
				default:
					cmdType = CMD_TYPE_OTHER;
					break;
			}
		}
		case T_CreateStmt:
			break;
		case T_CreateTableSpaceStmt:
			break;
		case T_CopyStmt:
			cmdType = CMD_TYPE_COPY ;
			break;
		case T_VariableResetStmt:
			cmdType = CMD_TYPE_RESET ;
			break;
		case T_VariableSetStmt:
			cmdType = CMD_TYPE_SET;
			break;
		case T_VacuumStmt:
			cmdType = CMD_TYPE_VACUUM ;
			break;
		case T_ReindexStmt:
			cmdType = CMD_TYPE_REINDEX ;
			break;
		default:
			break;
	}
	PGCluster_Info->PGR_Retry_Query.cmdType = cmdType;
}

static void
init_ClusterDB_Mode_Info(PGR_ClusterDB_Mode_Info * p)
{
	if (p != (PGR_ClusterDB_Mode_Info *)NULL)
	{
		p->Transaction_Mode = false;
		p->Session_Authorization_Mode = false;
		p->Create_Temp_Table_Mode = false;
		p->PGR_Copy_Data_Need_Replicate = false;
		p->PGR_Reliable_Mode_Wait = false;
	}
}


int
PGR_Init_PGCluster_Info(void)
{
	/*
	if (PGCluster_Info == NULL)
	{
		return STATUS_OK;
	}
	*/
	PGCluster_Info = (PGR_PGCluster_Info *)malloc(sizeof(PGR_PGCluster_Info));
	if (PGCluster_Info == (PGR_PGCluster_Info *)NULL)
	{
		return STATUS_ERROR;
	}
	memset(PGCluster_Info, 0, sizeof(PGR_PGCluster_Info));
	PGR_Init_Com_Info(&(PGCluster_Info->Com_Info));
	init_ClusterDB_Mode_Info(&ClusterDB_Mode_Info);

	PGCluster_Info->RsyncPath = NULL;
	PGCluster_Info->RsyncOption = NULL;
	PGCluster_Info->PGR_Recovery_Option= false;

	PGCluster_Info->PGRCopyData = malloc(sizeof(CopyData));
	if (PGCluster_Info->PGRCopyData == (CopyData *)NULL)
	{
		return STATUS_ERROR;
	}
	PGCluster_Info->PGRCopyData->cnt = 0;	
	memset(PGCluster_Info->PGRCopyData->copy_data,0,COPYBUFSIZ);

	PGCluster_Info->PGR_Retry_Query.query_string = NULL;
	PGCluster_Info->PGR_Retry_Query.query_len = 0;
	PGCluster_Info->PGR_Retry_Query.cmdSts = 0;
	PGCluster_Info->PGR_Retry_Query.cmdType = 0;
	PGCluster_Info->PGR_Retry_Query.useFlag = DATA_END;

	PGCluster_Info->PGR_Stand_Alone = (PGR_Stand_Alone_Type *)malloc(sizeof(PGR_Stand_Alone_Type));
	if (PGCluster_Info->PGR_Stand_Alone == (PGR_Stand_Alone_Type *)NULL)
	{
		return STATUS_ERROR;
	}
	PGCluster_Info->PGR_Stand_Alone->is_stand_alone = false;
	PGCluster_Info->PGR_Stand_Alone->permit = PERMIT_READ_ONLY;
	PGCluster_Info->PGR_Is_Replicated_Query = false;

	PGCluster_Info->PGR_Need_Notice = false;
	PGCluster_Info->PGR_Lock_Noticed = false;
	PGCluster_Info->PGR_Check_Lock.check_lock_conflict = false;
	PGCluster_Info->PGR_Check_Lock.deadlock = false;
	PGCluster_Info->PGR_Check_Lock.status_lock_conflict = STATUS_OK;
	PGCluster_Info->PGR_Check_Lock.dest = 0;

	return STATUS_OK;
}

int
PGR_Get_Cluster_Conf_Data(char * path)
{
	if (PGCluster_Info == NULL) 
	{
		return STATUS_OK;
	}
	if (PGR_Get_Conf_Data( path, CLUSTER_CONF_FILE ) == STATUS_OK)
	{
		if (PGCluster_Info == (PGR_PGCluster_Info *)NULL)
		{
			return STATUS_ERROR;
		}
		PGR_Init_Replicate_Server_Data(path);
		PGR_Set_Replicate_Server_Socket();
		PGR_Free_Conf_Data();
		if (PGCluster_Info->PGR_Recovery_Option)
		{
			fprintf(stderr,"Start in recovery mode! \n");
			fprintf(stderr,"Please wait until a data synchronization finishes from Master DB... \n");
			if (PGR_Recovery_Main() != STATUS_OK)
			{
				return STATUS_ERROR;
			}
		}
		return STATUS_OK;
	}
	else
	{
		return STATUS_ERROR;
	}
}

void
PGR_Clear_Children(int sig)
{
	int status = 0;

	if (PGCluster_Info == NULL)
	{
		return;
	}
	if (PGCluster_Info->PGR_Recovery_Option)
	{
		PGR_recovery_error_send();
		PGCluster_Info->PGR_Recovery_Option = false;
	}
	if (Lifecheck_Pid_Info.Master_Pid > 0)
	{
		kill (Lifecheck_Pid_Info.Master_Pid,sig);
		waitpid(Lifecheck_Pid_Info.Master_Pid,&status,WNOHANG);
	}
	if (Lifecheck_Pid_Info.SYN_Pid > 0)
	{
		kill (Lifecheck_Pid_Info.SYN_Pid,sig);
		waitpid(Lifecheck_Pid_Info.SYN_Pid,&status,WNOHANG);
	}
	if (Lifecheck_Pid_Info.ACK_Pid > 0)
	{
		kill (Lifecheck_Pid_Info.ACK_Pid,sig);
		waitpid(Lifecheck_Pid_Info.ACK_Pid,&status,WNOHANG);
	}
	PGR_delete_shm();
}

void
PGR_Recovery_Failed(void)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}
	if (PGCluster_Info->PGR_Recovery_Option)
	{
		PGR_recovery_error_send();
		PGCluster_Info->PGR_Recovery_Option = false;
	}
	PGR_delete_shm();
}

void
PGR_Elog_Abort_Copy(void)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}
	if (ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate)
	{
		PGR_Set_Copy_Data(PGCluster_Info->PGRCopyData,NULL,0,1);
		ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate = false;
	}
}

void
PGR_Elog_Abort_Proc(int timeout)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}
	if (ReplicationServer_Info.CurrentReplicateServer != NULL)
	{
		if (PGCluster_Info->PGR_Need_Notice == true)
		{
			PGR_Notice_Transaction_Query_Aborted();
		}
		if (ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate)
		{
			PGR_Set_Copy_Data(PGCluster_Info->PGRCopyData,NULL,0,1);
		}
		else
		{
			if ((! PGCluster_Info->PGR_Is_Replicated_Query ) &&
				(PGCluster_Info->PGR_Check_Lock.dest != TO_FRONTEND) &&
				(ClusterDB_Mode_Info.PGR_Reliable_Mode_Wait == true) &&
				(ReplicationServer_Info.CurrentReplicateServer->response_mode == PGR_RELIABLE_MODE))
			{
				PGR_Recv_Trigger(0);
			}
		}
	}
	if (ReplicationServer_Info.TransactionSock != -1)
	{
		close (ReplicationServer_Info.TransactionSock);
		ReplicationServer_Info.TransactionSock = -1;
	}
}

void
PGR_Elog_Warn_Proc(char * msg, char * fix)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}
	if ((strstr(msg,"parse error") != NULL) ||
		(strstr(msg,"syntax error") != NULL))
	{
		if ((PGCluster_Info->PGR_Check_Lock.dest != TO_FRONTEND) &&
			(ClusterDB_Mode_Info.Transaction_Mode == true))
		{
			PGR_Force_Replicate_Query();
		}
	}
	PGR_Notice_Transaction_Query_Done();
	if (ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate)
	{
		PGR_Set_Copy_Data(PGCluster_Info->PGRCopyData,"\\.\n",3,1);
		/*PGR_Set_Copy_Data(PGCluster_Info->PGRCopyData,NULL,0,1);*/
	}
}

void
PGR_ExceptionalCondition(void)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}
	if ((PGCluster_Info->PGR_Check_Lock.dest == TO_REPLICATION_SERVER ) &&
		(PGCluster_Info->PGR_Need_Notice == true))
	{
		PGR_Notice_Transaction_Query_Aborted();
	}
	if (ClusterDB_Mode_Info.PGR_Copy_Data_Need_Replicate)
	{
		PGR_Set_Copy_Data(PGCluster_Info->PGRCopyData,NULL,0,1);
	}
}

bool
PGR_Is_This_Transaction_Committed(void)
{
	if (PGCluster_Info == NULL)
	{
		return false;
	}
	if ((ClusterDB_Mode_Info.Transaction_Mode == true)  &&
		(PGR_Did_Commit_Transaction() == true))
	{
		return true;
	}
	return false;
}

void
PGR_Set_Copy_Char(int ch)
{
	if (PGCluster_Info == NULL)
	{
		return;
	}
	if (ch != EOF)
	{
		PGR_Set_Copy_Data(PGCluster_Info->PGRCopyData,(char *)&ch,1,0);
	}
	else
	{
		PGR_Set_Copy_Data(PGCluster_Info->PGRCopyData,NULL,0,1);
	}
}
#endif /* USE_REPLICATION */
